/**
 * @brief       TIMR Manager
 * @author      AXELL CORPORATION
 * @description TIMR Manager Layer
 * @note        none
 * @history     2017_02_22  
 * @history     2017_10_26  Ver2.0
 * @history     2019_12_27  [SDK3.0] Timr}l[WɃ`lwł̃nh擾ǉ (#2160)
*/
/* DOM-IGNORE-BEGIN */
/*
* This program was created by AXELL CORPORATION.
* Copyright (C) 2017-2019 AXELL CORPORATION, all rights reserved.
*/
/* DOM-IGNORE-END */

#include "AG903_errno.h"
#include "AG903_intno.h"
#include "timr/timrmgr.h"
#include "timr/timrprm.h"
#include "int/intmgr.h"

typedef void (*AG903_TIMRMgrIntHdr)(void);	/** R[obN֐^ */

typedef struct _AG903_TIMRMgrInthdrStat {
	int32_t	hdrid;		/** nhID */
	void*	func;		/** ݃nh̋NԒn */
}AG903_TIMRMgrInthdrStat;

typedef struct _AG903_TIMRMgrChStat{
	AG903_TIMRMgrInthdrStat	inthdr[AG903_TIMR_INTHDR_NUM];
	uint8_t		mode;		/** 샂[h */
	uint8_t		outmode;	/** o̓[h */
	uint8_t		reserve[2];	/** \ */
} AG903_TIMRMgrChStat;

typedef struct _AG903_TIMRMgrHandleStat{
	uint8_t		lock;		/** bN */
	uint8_t		reserve[3];	/** \ */
} AG903_TIMRMgrHandleStat;

static AG903_TIMRMgrChStat		TimrChStat[AG903_TIMR_CH_NUM];		/** ch */
static AG903_TIMRMgrHandleStat	TimrHandleStat[AG903_TIMR_CH_NUM];	/** handle */

static const uint8_t TimrResolutionTypeTbl[AG903_TIMR_CNT_TYPENUM] =
{AG903_TIMR_RES_SYSCLK, AG903_TIMR_RES_TICK0, AG903_TIMR_RES_TICK1};	/** TvONbN */
static const uint8_t TimrOutModeTbl[AG903_TIMR_OUT_MODENUM] =
{AG903_TIMR_OUTMOD_PLUSE, AG903_TIMR_OUTMOD_LEVEL, AG903_TIMR_OUTMOD_PWM};	/** o̓[h */
static const uint8_t TimrInputReTbl[AG903_TIMR_INPUT_RETYPENUM][2] =
{{0,0},{0,2},{1,0},{1,2}};	/** [^GR[_͎ [lo,ch] */
static const uint8_t TimrInputTbl[AG903_TIMR_INPUT_TYPENUM][2] =
{{0,0},{0,1},{0,2},{0,3},{1,0},{1,1},{1,2},{1,3}};	/** ^C}͎ [lo,ch] */
static const uint8_t TimrResetTriggerTbl[AG903_TIMR_RSTTRG_TYPENUM] =
{AG903_TIMR_RSTEN_EXTPORT, AG903_TIMR_RSTEN_EVENT};	/** ZbggK */

static void TIMRMgr_Init(uint8_t ch);
static int32_t TIMRMgr_CheckHandle(AG903_TIMRMgrHandle* handle, uint8_t* ch);
static _Bool TIMRMgr_CheckIdle(uint8_t ch);

/**
 * @brief           TIMRManager
 * @param           none
 * @return          
 * @retval          AG903_ENONE I
 * @description     ϐ܂BVXeNCALLĉB<p>
 *                  ^C}́u~vԂɂȂ܂B
 * @note            ̊֐INTMgrĂĂяoĉB
*/
int32_t	AG903_TIMRMgrInit(void)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t loop;
	
	for(loop=0; loop<AG903_TIMR_CH_NUM; loop++) {
		TIMRMgr_Init(loop);	/* ԏ */
	}

	return retval;
}

/**
 * @brief           TIMRnh擾
 * @param           handle [out] TIMRnh
 * @return          nh擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  gpłnhs.
 * @description     CHănhmۂ܂B
*/
int32_t	AG903_TIMRMgrGetHandle(AG903_TIMRMgrHandle** handle)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t loop;
	
	if(NULL == handle) {
		return -AG903_EINVAL;
	}
	
	for(loop=0; loop<AG903_TIMR_CH_NUM; loop++) {
		if(false == TimrHandleStat[loop].lock) {
			TimrHandleStat[loop].lock = true;
			break;
		}
	}
	if(AG903_TIMR_CH_NUM <= loop) {
		return -AG903_EBUSY;
	}
	
	(*handle) = (AG903_TIMRMgrHandle*)&TimrHandleStat[loop];
	
	return retval;
}

/**
 * @brief           `lwłTIMRnh擾
 * @param           handle [out] TIMRnh
 * @return          nh擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  w̃`lgp
 * @description     mɎw̃`l擾邽߁A
 *                  ʏAG903_TIMRMgrGetHandleɎsĉB
*/
int32_t	AG903_TIMRMgrGetHandleCh(AG903_TIMRMgrHandle** handle, int ch)
{
	int32_t	 retval = AG903_ENONE;
	
	if(NULL == handle) {
		return -AG903_EINVAL;
	}
	if (ch < 0 || AG903_TIMR_CH_NUM <= ch) {
		return -AG903_EINVAL;
	}
	
	if(false == TimrHandleStat[ch].lock) {
		TimrHandleStat[ch].lock = true;
	} else {
		return -AG903_EBUSY;
	}
	
	(*handle) = (AG903_TIMRMgrHandle*)&TimrHandleStat[ch];
	
	return retval;
}

/**
 * @brief           TIMRnh
 * @param           handle [in] TIMRnhval
 * @return          nh
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgp(^C}쒆).
 * @description     nh܂B<p>쒆Ƀnhꂽꍇ̓G[Ԃ܂B
 * @note            KvɉAvAG903_INTMgrDisableIntɂTIMR̊荞݂𖳌ɂĉB
*/
int32_t	AG903_TIMRMgrReleaseHandle(AG903_TIMRMgrHandle* handle)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	uint8_t	 idle;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	idle = TIMRMgr_CheckIdle(ch);
	if(true != idle) {
		return -AG903_EBUSY;
	}
	
	TimrHandleStat[ch].lock = false;
	
	return retval;
}

/**
 * @brief           ݃nh̓o^
 * @param           handle [in] TIMRnh
 * @param           func [in] ݃nh̋NԒn
 * @param           param [in] p[^
 * @return          荞݃nho^
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EFAULT  s (nho^oȂ).
 * @description     荞݃nho^܂B<p>
 *                  param̓R[obN֐ɈƂēn܂Bvoid (*func)(void* param)<p>
 *                  ЂƂ̃^C}ɓo^\Ȋ݃nh AG903_TIMR_INTHDR_NUM Œ`܂B
 * @note            `lɐݒł銄荞݃nh̍ő吔AG903_OSWprCreIsrAutoł
 *                  ̍ő吔AG903_TIMR_INTHDR_NUMł̒`l̏قɂȂ܂B
 *                  (T-Kernelł͒ʏ1܂łł)
 * @note            ̃nh𕡐o^ꍇ͂̓o^R[obN܂B<p>
 *                  ܂̏ꍇo^nh폜ꍇ AG903_TIMRMgrDeleteIntHandler 
 *                  o^ƓsĉBꕔ̃nh폜ꍇ͐ɓ삵ȂȂ邱Ƃ܂B
 * @note            AG903_TIMR_INTHDR_NUM͕KvɉĕύX\ł(݂2œmFĂ܂)
 * @note            TIMRMgrIɊ荞݂𐧌䂵Ă邽ߓ쒆̓Av
                    AG903_INTMgrDisableInt Ń^C}荞݂ɊւݒύXȂŉB
 * @note            ̊֐͊荞݃nh̒ĂяoƂ͂ł܂B
*/
int32_t	AG903_TIMRMgrSetIntHandler(AG903_TIMRMgrHandle* handle, void* func, void* param)
{
	AG903_INTMgrHdrPrm inthdr;
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	int32_t	 hdrid;
	uint32_t loop;
	uint8_t	 ch;

	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == func) {
		return -AG903_EINVAL;
	}
	for(loop=0; loop<AG903_TIMR_INTHDR_NUM;	loop++) {
		if(NULL == TimrChStat[ch].inthdr[loop].func) {
			break;
		}
	}
	if(AG903_TIMR_INTHDR_NUM <= loop) {
		return -AG903_EFAULT;
	}
	
	inthdr.atr   = AG903_INT_HLNG;
	inthdr.intno = AG903_IRQ0_TIM0+ch;
	inthdr.func  = func;
	inthdr.exinf = param;
	hdrid = AG903_INTMgrSetHandler(&inthdr);
	if(0 >= hdrid) {
		return -AG903_EFAULT;
	}
	AG903_INTMgrEnableInt(AG903_IRQ0_TIM0+ch);
	TimrChStat[ch].inthdr[loop].hdrid = hdrid;
	TimrChStat[ch].inthdr[loop].func  = func;
	
	return retval;
}

/**
 * @brief           ݃nh̍폜
 * @param           handle [in] TIMRnh
 * @param           func [in] ݃nh̋NԒn
 * @return          荞݃nh̍폜
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     荞݃nh폜܂B
*/
int32_t	AG903_TIMRMgrDeleteIntHandler(AG903_TIMRMgrHandle* handle, void* func)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t loop;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == func) {
		return -AG903_EINVAL;
	}
	for(loop=0; loop<AG903_TIMR_INTHDR_NUM;	loop++) {
		if(func == TimrChStat[ch].inthdr[loop].func) {
			break;
		}
	}
	if(AG903_TIMR_INTHDR_NUM <= loop) {
		return -AG903_EINVAL;
	}
	
	retval = AG903_INTMgrDeleteHandler(TimrChStat[ch].inthdr[loop].hdrid);
	if(AG903_ENONE == retval) {
		TimrChStat[ch].inthdr[loop].hdrid = 0;
		TimrChStat[ch].inthdr[loop].func  = NULL;
	}

	return retval;
}

/**
 * @brief           [^[GR[_̓[hݒ
 * @param           handle [in] TIMRnh
 * @param           param [in] p[^
 * @return          [^[GR[_̓[hݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgp (^C}쒆)
 * @description     [^[GR[_̓[hݒ肵܂B<p>
 *                  {[h̃^C}ڍׂ́wAX51903dlxQƂĉB
*/
int32_t	AG903_TIMRMgrSetRotaryEncoderMode(AG903_TIMRMgrHandle* handle, AG903_TIMRMgrRotaryEncParam* param)
{
	AG903_TIMRPrmInModeParam		inprm  = {0};
	AG903_TIMRPrmOutModeParam		outprm = {0};
	AG903_TIMRPrmResetTriggerParam 	rstprm = {0};
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	uint8_t	 idle;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == param) {
		return -AG903_EINVAL;
	}
	if( (AG903_TIMR_INPUT_RETYPENUM <= param->intype) ||
		(AG903_TIMR_CNT_TYPENUM <= param->sampling) ) {
		return -AG903_EINVAL;
	}
	if(NULL != param->output) {
		if( (AG903_TIMR_EXTOUT_NUM  <= param->output->portch)   ||
		    (AG903_TIMR_POL_TYPENUM <= param->output->polarity) ||
		    (AG903_TIMR_OUT_PWM <= param->output->mode)		    || /* PWMo ݒs */
		    ((AG903_TIMR_OUT_PLUSE == param->output->mode)&&(AG903_TIMR_PLUSEWIDTH_MAX < param->output->plusewidth)) ) {
			return -AG903_EINVAL;
		}
	}
	idle = TIMRMgr_CheckIdle(ch);
	if(true != idle) {
		return -AG903_EBUSY;
	}
	
	inprm.res = TimrResolutionTypeTbl[param->sampling];
	inprm.lo  = TimrInputReTbl[param->intype][0];
	inprm.ch  = TimrInputReTbl[param->intype][1];
	AG903_TIMRPrmSetInMode(ch, &inprm);
	AG903_TIMRPrmSetPeriod(ch, param->range);
	AG903_TIMRPrmSetMatch(ch, param->match);

	if(NULL == param->output) {
		outprm.mod = AG903_TIMR_OUTMOD_DISABLE;
	}
	else {
		outprm.ovf = 1;	/* RotaryEncoder̓[hOVFoȂ */
		outprm.ch  = param->output->portch;
		outprm.pol = param->output->polarity;
		outprm.mod = TimrOutModeTbl[param->output->mode];
	}
	AG903_TIMRPrmSetOutMode(ch, &outprm);
	if(AG903_TIMR_OUTMOD_PLUSE == outprm.mod) {
		AG903_TIMRPrmSetPluseWidth(ch, param->output->plusewidth);
	}
	
 	rstprm.en = AG903_TIMR_RSTEN_DISABLE;
	AG903_TIMRPrmSetResetTrigger(ch, &rstprm);
	
	TimrChStat[ch].outmode = outprm.mod;
	TimrChStat[ch].mode    = AG903_TIMR_CTRLMOD_REIN;

	return retval;
}

/**
 * @brief           PWM̓[hݒ
 * @param           handle [in] TIMRnh
 * @param           param [in] p[^
 * @return          PWM̓[hݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgp (^C}쒆).
 * @description     PWM̓[hݒ肵܂B<p>
 *                  {[h̃^C}ڍׂ́wAX51903dlxQƂĉB
*/
int32_t	AG903_TIMRMgrSetPwmInMode(AG903_TIMRMgrHandle* handle, AG903_TIMRMgrPwmInParam* param)
{
	AG903_TIMRPrmInModeParam		inprm  = {0};
	AG903_TIMRPrmOutModeParam		outprm = {0};
	AG903_TIMRPrmResetTriggerParam 	rstprm = {0};
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	uint8_t	 idle;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == param) {
		return -AG903_EINVAL;
	}
	if( (AG903_TIMR_INPUT_TYPENUM <= param->intype)   ||
		(AG903_TIMR_CNT_TYPENUM <= param->resolution) ||
		(AG903_TIMR_POL_TYPENUM <= param->polarity) ) {
		return -AG903_EINVAL;
	}
	if(NULL != param->output) {
		if( (AG903_TIMR_EXTOUT_NUM  <= param->output->portch)   ||
		    (AG903_TIMR_POL_TYPENUM <= param->output->polarity) ||
		    (AG903_TIMR_OUT_PWM <= param->output->mode)		    || /* PWMo ݒs */
		    ((AG903_TIMR_OUT_PLUSE == param->output->mode)&&(AG903_TIMR_PLUSEWIDTH_MAX < param->output->plusewidth)) ) {
			return -AG903_EINVAL;
		}
	}
	idle = TIMRMgr_CheckIdle(ch);
	if(true != idle) {
		return -AG903_EBUSY;
	}
	
	inprm.res = TimrResolutionTypeTbl[param->resolution];
	inprm.lo  = TimrInputTbl[param->intype][0];
	inprm.ch  = TimrInputTbl[param->intype][1];
	inprm.pol = param->polarity;
	AG903_TIMRPrmSetInMode(ch, &inprm);
	if(true == param->oneshot) {
		AG903_TIMRPrmEnableOneshot(ch);
	}
	else {
		AG903_TIMRPrmDisableOneshot(ch);
	}

	if(NULL == param->output) {
		outprm.mod = AG903_TIMR_OUTMOD_DISABLE;
	}
	else {
		outprm.cm  = 1;	/* PWM̓[hCMoȂ */
		outprm.ch  = param->output->portch;
		outprm.pol = param->output->polarity;
		outprm.mod = TimrOutModeTbl[param->output->mode];
	}
	AG903_TIMRPrmSetOutMode(ch, &outprm);
	if(AG903_TIMR_OUTMOD_PLUSE == outprm.mod) {
		AG903_TIMRPrmSetPluseWidth(ch, param->output->plusewidth);
	}

 	rstprm.en = AG903_TIMR_RSTEN_DISABLE;
	AG903_TIMRPrmSetResetTrigger(ch, &rstprm);

	TimrChStat[ch].outmode = outprm.mod;
	TimrChStat[ch].mode    = AG903_TIMR_CTRLMOD_PWMIN;

	return retval;
}

/**
 * @brief           NbNETickv[hݒ
 * @param           handle [in] TIMRnh
 * @param           param [in] p[^
 * @return          AG903_ENONE   I
 * @return          -AG903_EINVAL ُ
 * @return          -AG903_EBUSY  nhgp (^C}쒆).
 * @description     NbNETickv[hݒ肵܂B<p>
 *                  {[h̃^C}ڍׂ́wAX51903dlxQƂĉB
*/
int32_t	AG903_TIMRMgrSetTickCountMode(AG903_TIMRMgrHandle* handle, AG903_TIMRMgrTickCntParam* param)
{
	AG903_TIMRPrmInModeParam		inprm  = {0};
	AG903_TIMRPrmOutModeParam		outprm = {0};
	AG903_TIMRPrmResetTriggerParam 	rstprm = {0};
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	uint8_t	 idle;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == param) {
		return -AG903_EINVAL;
	}
	if(AG903_TIMR_CNT_TYPENUM <= param->resolution) {
		return -AG903_EINVAL;
	}
	if(NULL != param->output) {
		if( (AG903_TIMR_EXTOUT_NUM  <= param->output->portch)   ||
		    (AG903_TIMR_POL_TYPENUM <= param->output->polarity) ||
		    (AG903_TIMR_OUT_MODENUM <= param->output->mode)		||
		    ((AG903_TIMR_OUT_PLUSE == param->output->mode)&&(AG903_TIMR_PLUSEWIDTH_MAX < param->output->plusewidth)) ) {
			return -AG903_EINVAL;
		}
	}
	idle = TIMRMgr_CheckIdle(ch);
	if(true != idle) {
		return -AG903_EBUSY;
	}
	
	inprm.res = TimrResolutionTypeTbl[param->resolution];
	AG903_TIMRPrmSetInMode(ch, &inprm);
	AG903_TIMRPrmSetPeriod(ch, param->period);
	AG903_TIMRPrmSetMatch(ch, param->match);
	if(true == param->oneshot) {
		AG903_TIMRPrmEnableOneshot(ch);
	}
	else {
		AG903_TIMRPrmDisableOneshot(ch);
	}
	
	if(NULL == param->output) {
		outprm.mod = AG903_TIMR_OUTMOD_DISABLE;
	}
	else {
		if(AG903_TIMR_COMPARE_BIT & param->output->mask) {
			outprm.cm  = 1;	/* 1bF}XNL */
		}
		if(AG903_TIMR_OVERFLOW_BIT & param->output->mask) {
			outprm.ovf = 1;	/* 1bF}XNL */
		}
		outprm.ch  = param->output->portch;
		outprm.pol = param->output->polarity;
		outprm.mod = TimrOutModeTbl[param->output->mode];
	}
	AG903_TIMRPrmSetOutMode(ch, &outprm);
	if(AG903_TIMR_OUTMOD_PLUSE == outprm.mod) {
		AG903_TIMRPrmSetPluseWidth(ch, param->output->plusewidth);
	}
	
 	rstprm.en   = AG903_TIMR_RSTEN_DISABLE;
 	rstprm.boot = AG903_TIMR_RSTBOOT_TICKCNT;
	AG903_TIMRPrmSetResetTrigger(ch, &rstprm);
	
	TimrChStat[ch].outmode = outprm.mod;
	TimrChStat[ch].mode    = AG903_TIMR_CTRLMOD_TICKCNT;
	
	return retval;
}

/**
 * @brief           Cxgv[hݒ
 * @param           handle [in] TIMRnh
 * @param           param [in] p[^
 * @return          Cxgv[hݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgp (^C}쒆).
 * @description     Cxgv[hݒ肵܂B<p>
 *                  {[h̃^C}ڍׂ́wAX51903dlxQƂĉB
*/
int32_t	AG903_TIMRMgrSetEventCountMode(AG903_TIMRMgrHandle* handle, AG903_TIMRMgrEventCntParam* param)
{
	AG903_TIMRPrmInModeParam		inprm  = {0};
	AG903_TIMRPrmOutModeParam		outprm = {0};
	AG903_TIMRPrmResetTriggerParam 	rstprm = {0};
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	uint8_t	 idle;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == param) {
		return -AG903_EINVAL;
	}
	if(AG903_TIMR_EVENT_NUM <= param->event) {
		return -AG903_EINVAL;
	}
	if(NULL != param->output) {
		if( (AG903_TIMR_EXTOUT_NUM  <= param->output->portch)   ||
		    (AG903_TIMR_POL_TYPENUM <= param->output->polarity) ||
		    (AG903_TIMR_OUT_PWM <= param->output->mode)		    || /* PWMo ݒs */
		    ((AG903_TIMR_OUT_PLUSE == param->output->mode)&&(AG903_TIMR_PLUSEWIDTH_MAX < param->output->plusewidth)) ) {
			return -AG903_EINVAL;
		}
	}
	idle = TIMRMgr_CheckIdle(ch);
	if(true != idle) {
		return -AG903_EBUSY;
	}
	
	inprm.eve = param->event;
	AG903_TIMRPrmSetInMode(ch, &inprm);
	AG903_TIMRPrmSetPeriod(ch, param->period);
	AG903_TIMRPrmSetMatch(ch, param->match);
	if(true == param->oneshot) {
		AG903_TIMRPrmEnableOneshot(ch);
	}
	else {
		AG903_TIMRPrmDisableOneshot(ch);
	}

	if(NULL == param->output) {
		outprm.mod = AG903_TIMR_OUTMOD_DISABLE;
	}
	else {
		if(AG903_TIMR_COMPARE_BIT & param->output->mask) {
			outprm.cm  = 1;	/* 1bF}XNL */
		}
		if(AG903_TIMR_OVERFLOW_BIT & param->output->mask) {
			outprm.ovf = 1;	/* 1bF}XNL */
		}
		outprm.ch  = param->output->portch;
		outprm.pol = param->output->polarity;
		outprm.mod = TimrOutModeTbl[param->output->mode];
	}
	AG903_TIMRPrmSetOutMode(ch, &outprm);
	if(AG903_TIMR_OUTMOD_PLUSE == outprm.mod) {
		AG903_TIMRPrmSetPluseWidth(ch, param->output->plusewidth);
	}
	
 	rstprm.en   = AG903_TIMR_RSTEN_DISABLE;
 	rstprm.boot = AG903_TIMR_RSTBOOT_EVENTCNT;
	AG903_TIMRPrmSetResetTrigger(ch, &rstprm);
	
	TimrChStat[ch].outmode = outprm.mod;
	TimrChStat[ch].mode    = AG903_TIMR_CTRLMOD_EVENTCNT;

	return retval;
}

/**
 * @brief           ͒[qpXv[hݒ
 * @param           handle [in] TIMRnh
 * @param           param [in] p[^
 * @return          ͒[qpXv[hݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgp (^C}쒆).
 * @description     ͒[qpXv[hݒ肵܂B<p>
 *                  {[h̃^C}ڍׂ́wAX51903dlxQƂĉB
*/
int32_t	AG903_TIMRMgrSetPulseCountMode(AG903_TIMRMgrHandle* handle, AG903_TIMRMgrPulseCntParam* param)
{
	AG903_TIMRPrmInModeParam		inprm  = {0};
	AG903_TIMRPrmOutModeParam		outprm = {0};
	AG903_TIMRPrmResetTriggerParam 	rstprm = {0};
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	uint8_t	 idle;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == param) {
		return -AG903_EINVAL;
	}
	if( (AG903_TIMR_INPUT_TYPENUM <= param->intype) ||
		(AG903_TIMR_CNT_TYPENUM <= param->sampling) ||
		(AG903_TIMR_POL_TYPENUM <= param->polarity) ) {
		return -AG903_EINVAL;
	}
	if(NULL != param->output) {
		if( (AG903_TIMR_EXTOUT_NUM  <= param->output->portch)   ||
		    (AG903_TIMR_POL_TYPENUM <= param->output->polarity) ||
		    (AG903_TIMR_OUT_PWM <= param->output->mode)		    || /* PWMo ݒs */
		    ((AG903_TIMR_OUT_PLUSE == param->output->mode)&&(AG903_TIMR_PLUSEWIDTH_MAX < param->output->plusewidth)) ) {
			return -AG903_EINVAL;
		}
	}
	idle = TIMRMgr_CheckIdle(ch);
	if(true != idle) {
		return -AG903_EBUSY;
	}
	
	inprm.res = TimrResolutionTypeTbl[param->sampling];
	inprm.lo  = TimrInputTbl[param->intype][0];
	inprm.ch  = TimrInputTbl[param->intype][1];
	inprm.pol = param->polarity;
	AG903_TIMRPrmSetInMode(ch, &inprm);
	AG903_TIMRPrmSetPeriod(ch, param->period);
	AG903_TIMRPrmSetMatch(ch, param->match);
	if(true == param->oneshot) {
		AG903_TIMRPrmEnableOneshot(ch);
	}
	else {
		AG903_TIMRPrmDisableOneshot(ch);
	}
	
	if(NULL == param->output) {
		outprm.mod = AG903_TIMR_OUTMOD_DISABLE;
	}
	else {
		if(AG903_TIMR_COMPARE_BIT & param->output->mask) {
			outprm.cm  = 1;	/* 1bF}XNL */
		}
		if(AG903_TIMR_OVERFLOW_BIT & param->output->mask) {
			outprm.ovf = 1;	/* 1bF}XNL */
		}
		outprm.ch  = param->output->portch;
		outprm.pol = param->output->polarity;
		outprm.mod = TimrOutModeTbl[param->output->mode];
	}
	AG903_TIMRPrmSetOutMode(ch, &outprm);
	if(AG903_TIMR_OUTMOD_PLUSE == outprm.mod) {
		AG903_TIMRPrmSetPluseWidth(ch, param->output->plusewidth);
	}

 	rstprm.en   = AG903_TIMR_RSTEN_DISABLE;
 	rstprm.boot = AG903_TIMR_RSTBOOT_PLUSECNT;
	AG903_TIMRPrmSetResetTrigger(ch, &rstprm);

	TimrChStat[ch].outmode = outprm.mod;
	TimrChStat[ch].mode    = AG903_TIMR_CTRLMOD_PLUSECNT;

	return retval;
}

/**
 * @brief           ͒[qxv[hݒ
 * @param           handle [in] TIMRnh
 * @param           param [in] p[^
 * @return          ͒[qxv[hݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgp (^C}쒆).
 * @description     ͒[qxv[hݒ肵܂B<p>
 *                  {[h̃^C}ڍׂ́wAX51903dlxQƂĉB
*/
int32_t	AG903_TIMRMgrSetLevelCountMode(AG903_TIMRMgrHandle* handle, AG903_TIMRMgrLevelCntParam* param)
{
	AG903_TIMRPrmInModeParam		inprm  = {0};
	AG903_TIMRPrmOutModeParam		outprm = {0};
	AG903_TIMRPrmResetTriggerParam 	rstprm = {0};
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	uint8_t	 idle;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == param) {
		return -AG903_EINVAL;
	}
	if( (AG903_TIMR_INPUT_TYPENUM <= param->intype) ||
		(AG903_TIMR_CNT_TYPENUM <= param->sampling) ||
		(AG903_TIMR_POL_TYPENUM <= param->polarity) ) {
		return -AG903_EINVAL;
	}
	if(NULL != param->output) {
		if( (AG903_TIMR_EXTOUT_NUM  <= param->output->portch)   ||
		    (AG903_TIMR_POL_TYPENUM <= param->output->polarity) ||
		    (AG903_TIMR_OUT_PWM <= param->output->mode)		    || /* PWMo ݒs */
		    ((AG903_TIMR_OUT_PLUSE == param->output->mode)&&(AG903_TIMR_PLUSEWIDTH_MAX < param->output->plusewidth)) ) {
			return -AG903_EINVAL;
		}
	}
	idle = TIMRMgr_CheckIdle(ch);
	if(true != idle) {
		return -AG903_EBUSY;
	}
	
	inprm.res = TimrResolutionTypeTbl[param->sampling];
	inprm.lo  = TimrInputTbl[param->intype][0];
	inprm.ch  = TimrInputTbl[param->intype][1];
	inprm.pol = param->polarity;
	AG903_TIMRPrmSetInMode(ch, &inprm);
	AG903_TIMRPrmSetPeriod(ch, param->period);
	AG903_TIMRPrmSetMatch(ch, param->match);
	if(true == param->oneshot) {
		AG903_TIMRPrmEnableOneshot(ch);
	}
	else {
		AG903_TIMRPrmDisableOneshot(ch);
	}
	
	if(NULL == param->output) {
		outprm.mod = AG903_TIMR_OUTMOD_DISABLE;
	}
	else {
		if(AG903_TIMR_COMPARE_BIT & param->output->mask) {
			outprm.cm  = 1;	/* 1bF}XNL */
		}
		if(AG903_TIMR_OVERFLOW_BIT & param->output->mask) {
			outprm.ovf = 1;	/* 1bF}XNL */
		}
		outprm.ch  = param->output->portch;
		outprm.pol = param->output->polarity;
		outprm.mod = TimrOutModeTbl[param->output->mode];
	}
	AG903_TIMRPrmSetOutMode(ch, &outprm);
	if(AG903_TIMR_OUTMOD_PLUSE == outprm.mod) {
		AG903_TIMRPrmSetPluseWidth(ch, param->output->plusewidth);
	}
	
 	rstprm.en   = AG903_TIMR_RSTEN_DISABLE;
 	rstprm.boot = AG903_TIMR_RSTBOOT_LEVELCNT;
	AG903_TIMRPrmSetResetTrigger(ch, &rstprm);
	
	TimrChStat[ch].outmode = outprm.mod;
	TimrChStat[ch].mode    = AG903_TIMR_CTRLMOD_LEVELCNT;

	return retval;
}

/**
 * @brief           ^C}N
 * @param           handle [in] TIMRnh
 * @return          ^C}N
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgp (^C}쒆).
 * @retval          -AG903_EFAULT s (샂[hs).
 * @description     ^C}N̐ݒ܂B<p>
 *                  ̊֐Őݒ肳ꂽ[hŃ^C}N܂B<p>
 *                  AG903_TIMRMgrSetRotaryEncoderMode , AG903_TIMRMgrSetPwmInMode , AG903_TIMRMgrSetTickCountMode,<p>
 *                  AG903_TIMRMgrSetEventCountMode , AG903_TIMRMgrSetPulseCountMode , AG903_TIMRMgrSetLevelCountMode
 * @note            ɓ샂[h̐ݒsKv܂B
*/
int32_t	AG903_TIMRMgrStart(AG903_TIMRMgrHandle* handle)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	uint8_t	 idle;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(AG903_TIMR_CTRLMOD_MODENUM <= TimrChStat[ch].mode) {
		return -AG903_EFAULT;
	}
	idle = TIMRMgr_CheckIdle(ch);
	if(true != idle) {
		return -AG903_EBUSY;
	}
	
	AG903_TIMRPrmSetCntMode(ch, TimrChStat[ch].mode);
	
	return retval;
}

/**
 * @brief           ^C}~
 * @param           handle [in] TIMRnh
 * @return          ^C}~
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     ^C}𓮍~ɂ܂B
*/
int32_t	AG903_TIMRMgrStop(AG903_TIMRMgrHandle* handle)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}

	AG903_TIMRPrmSetCntMode(ch, AG903_TIMR_CTRLMOD_STOP);
	
	return retval;
}

/**
 * @brief           ^C}Niwj
 * @param           chbit [in] TIMR`lirbgwArbgj
 * @return          ^C}Niwj
 * @retval          AG903_ENONE   I
 * @description     ^C}Niwj̐ݒ܂B<p>
 *                  TIMBOOTTRG  chbit Write܂B
 * @note            ȉ̓샂[h̃^C}͎w肵ȂŉB<p>
 *                  E[^[GR[_̓[h<p>
 *                  EPWM̓[h
*/
int32_t	AG903_TIMRMgrStartMulti(uint8_t chbit)
{
	int32_t	 retval = AG903_ENONE;
	
	AG903_TIMRPrmSetBootTrigger(chbit);
	
	return retval;
}

/**
 * @brief           ZbggKݒiEnablej
 * @param           handle [in] TIMRnh
 * @param           param [in] p[^
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EFAULT s (샂[hs).
 * @description     ZbggKݒiEnablejɂ܂B
 * @note            ɓ샂[h̐ݒsKv܂B<p>
 *                  샂[hݒyу[^[GR[_̓[hAPWM̓[hݒ肳Ă<p>
 *                  ^C}ɂ̓ZbggK͐ݒo܂
*/
int32_t	AG903_TIMRMgrEnableResetTrigger(AG903_TIMRMgrHandle* handle, AG903_TIMRMgrResetParam* param)
{
	AG903_TIMRPrmResetTriggerParam 	rstprm = {0};
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == param) {
		return -AG903_EINVAL;
	}
	if(AG903_TIMR_RSTTRG_TYPENUM <= param->trigger) {
		return -AG903_EINVAL;
	}
	if(AG903_TIMR_RSTTRG_EXTPORT == param->trigger) {
		if( (AG903_TIMR_INPUT_TYPENUM <= param->intype)   ||
			(AG903_TIMR_CNT_TYPENUM   <= param->sampling) ||
			(AG903_TIMR_POL_TYPENUM   <= param->polarity) ) {
			return -AG903_EINVAL;
		}
	}
	else  {
		if(AG903_TIMR_EVENT_NUM <= param->event) {
			return -AG903_EINVAL;
		}
	}
	
	switch(TimrChStat[ch].mode) {
		case AG903_TIMR_CTRLMOD_TICKCNT:
			rstprm.boot = AG903_TIMR_RSTBOOT_TICKCNT;
			break;
		case AG903_TIMR_CTRLMOD_EVENTCNT:
			rstprm.boot = AG903_TIMR_RSTBOOT_EVENTCNT;
			break;
		case AG903_TIMR_CTRLMOD_PLUSECNT:
			rstprm.boot = AG903_TIMR_RSTBOOT_PLUSECNT;
			break;
		case AG903_TIMR_CTRLMOD_LEVELCNT:
			rstprm.boot = AG903_TIMR_RSTBOOT_LEVELCNT;
			break;
		default:
			retval = -AG903_EFAULT;	/* ZbgNΏۂ͏L̃[ĥ */
			break;
	}

	if(AG903_ENONE == retval) {
		rstprm.en = TimrResetTriggerTbl[param->trigger];
		if(AG903_TIMR_RSTEN_EXTPORT == rstprm.en) {
			rstprm.res = TimrResolutionTypeTbl[param->sampling];
			rstprm.lo  = TimrInputTbl[param->intype][0];
			rstprm.ch  = TimrInputTbl[param->intype][1];
			rstprm.pol = param->polarity;
		}
		else {
			rstprm.eve = param->event;
		}
		AG903_TIMRPrmSetResetTrigger(ch, &rstprm);
	}

	return retval;
}

/**
 * @brief           ZbggKݒiDisablej
 * @param           handle [in] TIMRnh
 * @return          ZbggKݒiDisablej
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     ZbggK𖳌ݒiDisablejɂ܂B<p>
 *                  AG903_TIMRMgrEnableResetTrigger Őݒ肳ꂽZbggK𖳌܂B
*/
int32_t	AG903_TIMRMgrDisableResetTrigger(AG903_TIMRMgrHandle* handle)
{
	AG903_TIMRPrmResetTriggerParam 	rstprm = {0};
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}

 	rstprm.en = AG903_TIMR_RSTEN_DISABLE;
	AG903_TIMRPrmSetResetTrigger(ch, &rstprm);

	return retval;
}

/**
 * @brief           VbgݒiEnablej
 * @param           handle [in] TIMRnh
 * @return          VbgݒiEnableǰ
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EFAULT s (샂[hs).
 * @description     VbgݒiEnablej܂B<p>
 *                  ^C}삪sIfBbNꍇAVbgɕύX܂B
 * @note            ɓ샂[h̐ݒsKv܂B<p>
 *                  [^[GR[_̓[h͖łB
*/
int32_t	AG903_TIMRMgrEnableOneShot(AG903_TIMRMgrHandle* handle)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(AG903_TIMR_CTRLMOD_REIN == TimrChStat[ch].mode) {
		return -AG903_EFAULT;
	}
	
	AG903_TIMRPrmEnableOneshot(ch);
	
	return retval;
}

/**
 * @brief           Vbg얳ݒiDisablej
 * @param           handle [in] TIMRnh
 * @return          Vbg얳ݒiDisablej
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Vbg얳ݒiDisablej܂B<p>
 *                  ^C}삪VbgꍇAsIfBbNɕύX܂B
*/
int32_t	AG903_TIMRMgrDisableOneShot(AG903_TIMRMgrHandle* handle)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	AG903_TIMRPrmDisableOneshot(ch);
	
	return retval;
}

/**
 * @brief           ݃}XNݒ
 * @param           handle [in] TIMRnh
 * @param           target [in] Ώہirbgwj
 * @return          ݃}XNݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     ݃}XNݒ肵܂B<p>
 *                  targetŎw肵݂}XN܂B<p>
 *                  <table>
 *                  <tr><th>bit</th><th></th></tr>
 *                  <tr><td>1</td><td>RyAE}b`</td></tr>
 *                  <tr><td>0</td><td>I[o[t[</td></tr>
 *                  </table>
 *                  1ݒ肷ƃ}XNΏۂɂȂ܂B
*/
int32_t	AG903_TIMRMgrEnableIntMask(AG903_TIMRMgrHandle* handle, uint8_t target)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t setbit = 0;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	if(AG903_TIMR_COMPARE_BIT & target) {
		setbit |= AG903_TIMR_INTMSK_CM_BIT;
	}
	if(AG903_TIMR_OVERFLOW_BIT & target) {
		setbit |= AG903_TIMR_INTMSK_OVF_BIT;
	}
	
	if(0 != setbit) {
		AG903_TIMRPrmEnableMask(ch, setbit);
	}
	
	return retval;
}

/**
 * @brief           ݃}XNݒi}XNj
 * @param           handle [in] TIMRnh
 * @param           target [in] Ώہirbgwj
 * @return          ݃}XNݒi}XNj
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     ݃}XNݒi}XNj܂B<p>
 *                  targetŎw肵݂}XN܂B<p>
 *                  <table>
 *                  <tr><th>bit</th><th></th></tr>
 *                  <tr><td>1</td><td>RyAE}b`</td></tr>
 *                  <tr><td>0</td><td>I[o[t[</td></tr>
 *                  </table>
 *                  1ݒ肷ƃ}XNɂȂ܂B
*/
int32_t	AG903_TIMRMgrDisableIntMask(AG903_TIMRMgrHandle* handle, uint8_t target)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t  clrbit = 0;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	if(AG903_TIMR_COMPARE_BIT & target) {
		clrbit |= AG903_TIMR_INTMSK_CM_BIT;
	}
	if(AG903_TIMR_OVERFLOW_BIT & target) {
		clrbit |= AG903_TIMR_INTMSK_OVF_BIT;
	}
	
	if(0 != clrbit) {
		AG903_TIMRPrmDisableMask(ch, clrbit);
	}
	
	return retval;
}

/**
 * @brief           CxggKݒiEnablej
 * @param           handle [in] TIMRnh
 * @param           target [in] Ώہirbgwj
 * @return          CxggKݒiEnablej
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     CxggKݒiEnablej܂B<p>
 *                  targetŎw肵ŃCxggK܂B<p>
 *                  <table>
 *                  <tr><th>bit</th><th></th></tr>
 *                  <tr><td>1</td><td>RyAE}b`</td></tr>
 *                  <tr><td>0</td><td>I[o[t[</td></tr>
 *                  </table>
 *                  1ݒ肷ƃCxggK܂B
*/
int32_t	AG903_TIMRMgrEnableEventTrigger(AG903_TIMRMgrHandle* handle, uint8_t target)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t setbit = 0;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	if(AG903_TIMR_COMPARE_BIT & target) {
		setbit |= AG903_TIMR_TRIGGER_CM_BIT;
	}
	if(AG903_TIMR_OVERFLOW_BIT & target) {
		setbit |= AG903_TIMR_TRIGGER_OVF_BIT;
	}
	
	if(0 != setbit) {
		AG903_TIMRPrmEnableEventTrigger(ch, setbit);
	}
	
	return retval;
}

/**
 * @brief           CxggKݒiDisablej
 * @param           handle [in] TIMRnh
 * @param           target [in] Ώہirbgwj
 * @return          CxggKݒiDisablej
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     CxggKݒiDisablej܂B<p>
 *                  targetŎw肵ŃCxggK܂B<p>
 *                  <table>
 *                  <tr><th>bit</th><th></th></tr>
 *                  <tr><td>1</td><td>RyAE}b`</td></tr>
 *                  <tr><td>0</td><td>I[o[t[</td></tr>
 *                  </table>
 *                  1ݒ肷ƃCxggK܂B
*/
int32_t	AG903_TIMRMgrDisableEventTrigger(AG903_TIMRMgrHandle* handle, uint8_t target)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t clrbit = 0;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	if(AG903_TIMR_COMPARE_BIT & target) {
		clrbit |= AG903_TIMR_TRIGGER_CM_BIT;
	}
	if(AG903_TIMR_OVERFLOW_BIT & target) {
		clrbit |= AG903_TIMR_TRIGGER_OVF_BIT;
	}
	
	if(0 != clrbit) {
		AG903_TIMRPrmDisableEventTrigger(ch, clrbit);
	}
	
	return retval;
}

/**
 * @brief           DMANGXgݒiEnablej
 * @param           handle [in] TIMRnh
 * @param           target [in] Ώہirbgwj
 * @return          DMANGXgݒiEnablej
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     DMANGXgݒiEnablej܂<p>
 *                  targetŎw肵DMANGXg܂B<p>
 *                  <table>
 *                  <tr><th>bit</th><th></th></tr>
 *                  <tr><td>1</td><td>RyAE}b`</td></tr>
 *                  <tr><td>0</td><td>I[o[t[</td></tr>
 *                  </table>
 *                  1ݒ肷DMANGXg܂B
*/
int32_t	AG903_TIMRMgrEnableDmaRequest(AG903_TIMRMgrHandle* handle, uint8_t target)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t setbit = 0;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	if(AG903_TIMR_COMPARE_BIT & target) {
		setbit |= AG903_TIMR_DMAREQ_CM_BIT;
	}
	if(AG903_TIMR_OVERFLOW_BIT & target) {
		setbit |= AG903_TIMR_DMAREQ_OVF_BIT;
	}
	
	if(0 != setbit) {
		AG903_TIMRPrmEnableDmaRequest(ch, setbit);
	}
	
	return retval;
}

/**
 * @brief           DMANGXgݒiDisablej
 * @param           handle [in] TIMRnh
 * @param           target [in] Ώہirbgwj
 * @return          DMANGXgݒiDisablej
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     DMANGXgݒiDisablej܂B<p>
 *                  targetŎw肵DMANGXg܂B<p>
 *                  <table>
 *                  <tr><th>bit</th><th></th></tr>
 *                  <tr><td>1</td><td>RyAE}b`</td></tr>
 *                  <tr><td>0</td><td>I[o[t[</td></tr>
 *                  </table>
 *                  1ݒ肷DMANGXg܂B
*/
int32_t	AG903_TIMRMgrDisableDmaRequest(AG903_TIMRMgrHandle* handle, uint8_t target)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t clrbit = 0;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	if(AG903_TIMR_COMPARE_BIT & target) {
		clrbit |= AG903_TIMR_DMAREQ_CM_BIT;
	}
	if(AG903_TIMR_OVERFLOW_BIT & target) {
		clrbit |= AG903_TIMR_DMAREQ_OVF_BIT;
	}
	
	if(0 != clrbit) {
		AG903_TIMRPrmDisableDmaRequest(ch, clrbit);
	}
	
	return retval;
}

/**
 * @brief           Xe[^X擾
 * @param           handle [in] TIMRnh
 * @param           status [out] 
 * @return          Xe[^X擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Xe[^X擾܂B<p>
 *                  TIMnSTAT ̓e status Read܂B
*/
int32_t	AG903_TIMRMgrGetStatus(AG903_TIMRMgrHandle* handle, uint32_t* status)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == status) {
		return -AG903_EINVAL;
	}
	
	AG903_TIMRPrmGetStatus(ch, status);
	
	return retval;
}

/**
 * @brief           Xe[^XNA
 * @param           handle [in] TIMRnh
 * @param           target [in] Ώہirbgwj
 * @return          Xe[^XNA
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Xe[^XNA܂B<p>
 *                  targetŎw肵ŃXe[^XNA܂B<p>
 *                  <table>
 *                  <tr><th>bit</th><th>Xe[^X</th></tr>
 *                  <tr><td>1</td><td>RyAE}b`</td></tr>
 *                  <tr><td>0</td><td>I[o[t[</td></tr>
 *                  </table>
 *                  1ݒ肷ƃXe[^XNA܂B
*/
int32_t	AG903_TIMRMgrClearStatus(AG903_TIMRMgrHandle* handle, uint32_t target)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t clrbit = 0;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	if(AG903_TIMR_COMPARE_BIT & target) {
		clrbit |= AG903_TIMR_STAT_CM_BIT;
	}
	if(AG903_TIMR_OVERFLOW_BIT & target) {
		clrbit |= AG903_TIMR_STAT_OVF_BIT;
	}
	
	if(0 != clrbit) {
		AG903_TIMRPrmClearStatus(ch, clrbit);
	}
	
	return retval;
}

/**
 * @brief           JEgl擾
 * @param           handle [in] TIMRnh
 * @param           count [out] JEgl
 * @return          JEgl擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     JEgl擾܂B
*/
int32_t	AG903_TIMRMgrGetCount(AG903_TIMRMgrHandle* handle, uint32_t* count)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == count) {
		return -AG903_EINVAL;
	}
	
	AG903_TIMRPrmGetCount(ch, count);
	
	return retval;
}

/**
 * @brief           JEglݒ
 * @param           handle [in] TIMRnh
 * @param           count [in] JEgl
 * @return          JEglݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgp (^C}쒆).
 * @description     JEglݒ肵܂B<p>
 *                  ~̂ݐݒ\łB
*/
int32_t	AG903_TIMRMgrSetCount(AG903_TIMRMgrHandle* handle, uint32_t count)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	uint8_t	 idle;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	idle = TIMRMgr_CheckIdle(ch);
	if(true != idle) {
		return -AG903_EBUSY;
	}
	
	AG903_TIMRPrmSetCount(ch, count);
	
	return retval;
}

/**
 * @brief           l擾
 * @param           handle [in] TIMRnh
 * @param           period [out] l
 * @return          l擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     l擾܂B
*/
int32_t	AG903_TIMRMgrGetPeriodValue(AG903_TIMRMgrHandle* handle, uint32_t* period)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == period) {
		return -AG903_EINVAL;
	}
	
	AG903_TIMRPrmGetPeriod(ch, period);
	
	return retval;
}

/**
 * @brief           lݒ
 * @param           handle [in] TIMRnh
 * @param           period [in] l
 * @return          lݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     lݒ肵܂B
*/
int32_t	AG903_TIMRMgrSetPeriodValue(AG903_TIMRMgrHandle* handle, uint32_t period)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	AG903_TIMRPrmSetPeriod(ch, period);
	
	return retval;
}

/**
 * @brief           rl擾
 * @param           handle [in] TIMRnh
 * @param           match [out] rl
 * @return          rl擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     rl擾܂B
*/
int32_t	AG903_TIMRMgrGetMatchValue(AG903_TIMRMgrHandle* handle, uint32_t* match)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == match) {
		return -AG903_EINVAL;
	}
	
	AG903_TIMRPrmGetMatch(ch, match);
	
	return retval;
}

/**
 * @brief           rlݒ
 * @param           handle [in] TIMRnh
 * @param           match [in] rl
 * @return          rlݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     rlݒ肵܂B
*/
int32_t	AG903_TIMRMgrSetMatchValue(AG903_TIMRMgrHandle* handle, uint32_t match)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = TIMRMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	AG903_TIMRPrmSetMatch(ch, match);
	
	return retval;
}

/**
 * @brief           `lԍ擾
 * @param           handle [in] TIMRnh
 * @param           ch [out] `lԍ
 * @return          `lԍ擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     `lԍ擾܂B
*/
int32_t	AG903_TIMRMgrGetChannel(AG903_TIMRMgrHandle* handle, uint8_t* ch)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	
	if(NULL == ch) {
		return -AG903_EINVAL;
	}

	result = TIMRMgr_CheckHandle(handle, ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}

	return retval;
}

/*
	ԏ.
	chFTIMR`l.
*/
static void TIMRMgr_Init(uint8_t ch)
{
	uint32_t loop;
	
	if(AG903_TIMR_CH_NUM <= ch) {
		return;
	}

	for(loop=0; loop<AG903_TIMR_INTHDR_NUM; loop++) {
		TimrChStat[ch].inthdr[loop].hdrid = 0;
		TimrChStat[ch].inthdr[loop].func  = NULL;
	}
	TimrChStat[ch].mode    = AG903_TIMR_CTRLMOD_STOP;
	TimrChStat[ch].outmode = AG903_TIMR_OUTMOD_DISABLE;
	AG903_TIMRPrmSetCntMode(ch, AG903_TIMR_CTRLMOD_STOP);

	TimrHandleStat[ch].lock = false;

	return;
}

/*
	nh`FbN.
	handleFTIMRnh.
	chFTIMR`l.
*/
static int32_t TIMRMgr_CheckHandle(AG903_TIMRMgrHandle* handle, uint8_t* ch)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t get_ch;
	
	get_ch = ((uint32_t)handle - (uint32_t)TimrHandleStat) / sizeof(AG903_TIMRMgrHandleStat);
	
	if( (AG903_TIMR_CH_NUM <= get_ch) ||
		(&TimrHandleStat[get_ch] != (AG903_TIMRMgrHandleStat*)handle) ) {
		return -AG903_EINVAL;
	}
	(*ch) = (uint8_t)get_ch;
	
	return retval;
}

/*
	IDLE`FbN.
	chFTIMR`l.
	returnFtrueiIDLEj
*/
static _Bool TIMRMgr_CheckIdle(uint8_t ch)
{
	uint8_t	mode=0;
	_Bool	retval=false;
	
	AG903_TIMRPrmGetCntMode(ch, &mode);
	if(AG903_TIMR_CTRLMOD_STOP == mode) {
		retval=true;
	}
	
	return retval;
}
