/**
 * @brief       UART Manager
 * @author      AXELL CORPORATION
 * @description UART Manager Layer
 * @note        none
 * @history     2017_02_22  
 * @history     2017_10_26  Ver2.0
*/
/* DOM-IGNORE-BEGIN */
/*
 * This program was created by AXELL CORPORATION.
 * Copyright (C) 2017-2019 AXELL CORPORATION, all rights reserved.
 */
/* DOM-IGNORE-END */

#include "AG903_errno.h"
#include "AG903_intno.h"
#include "uart/uartmgr.h"
#include "uart/uartprm.h"
#include "int/intmgr.h"

typedef void (*AG903_UARTMgrIntHdr)(void);	/** R[obN֐^ */

typedef struct _AG903_UARTMgrSndStat{
	uint32_t	stat;		/**  */
	uint8_t*	buf;		/** Mobt@ */
	uint32_t	size;		/** MTCY */
	uint32_t	cnt;		/** MJE^ */
	uint8_t		dma;		/** DMAL */
	uint8_t		reserve[3];	/** \ */
} AG903_UARTMgrSndStat;

typedef struct _AG903_UARTMgrRcvStat{
	uint32_t	stat;		/**  */
	uint8_t		dma;		/** DMAL */
	uint8_t		reserve[3];	/** \ */
} AG903_UARTMgrRcvStat;

typedef struct _AG903_UARTMgrChStat{
	AG903_UARTMgrSndStat snd;	/** M */
	AG903_UARTMgrRcvStat rcv;	/** M */
	uint32_t	hdlnum;			/** nhԍ */
	int32_t		hdrid;			/** nhID */
	uint8_t		rs485;			/** RS485 ON/OFF */
	uint8_t		reserve[3];		/** \ */
} AG903_UARTMgrChStat;

typedef struct _AG903_UARTMgrHandleStat{
	AG903_UARTMgrClbk	clbk;	/** R[obN */
	uint32_t	event;			/** ʒmCxg */
	uint8_t		lock;			/** bN */
	uint8_t		reserve[3];		/** \ */
} AG903_UARTMgrHandleStat;

enum _AG903_UartStatusNum{
	AG903_UART_STAT_IDLE = 0,
	AG903_UART_STAT_SEND,
	AG903_UART_STAT_RECEIVE,
};

static AG903_UARTMgrChStat		UartChStat[AG903_UART_CH_NUM];		/** ch */
static AG903_UARTMgrHandleStat	UartHandleStat[AG903_UART_CH_NUM];	/** handle */
static uint8_t	UartSendBuf[AG903_UART_CH_NUM][AG903_UART_BUFSIZE];		/** Mobt@ */

static void UARTMgr_InitState(uint8_t ch);
static int32_t UARTMgr_CheckHandle(AG903_UARTMgrHandle* handle, uint8_t* ch);
static int32_t UARTMgr_SetParam(uint8_t ch, AG903_UARTMgrParam* param);
static int32_t UARTMgr_SetBaudrate(uint8_t ch, uint32_t baud);
static int32_t UARTMgr_SetForm(uint8_t ch, uint8_t parity, uint8_t stopbit, uint8_t databit);
static int32_t UARTMgr_SetFifo(uint8_t ch, uint8_t rx_trgl, uint8_t tx_trgl, _Bool enable);
static int32_t UARTMgr_SetFlow(uint8_t ch, uint8_t flow);
static void UARTMgr_Inthdr0(void);
static void UARTMgr_Inthdr1(void);
static void UARTMgr_Inthdr2(void);
static void UARTMgr_Inthdr3(void);
static void UARTMgr_IntProcess(uint8_t ch);
static int32_t UARTMgr_IntSend(uint8_t ch, uint8_t inttype, uint32_t* event);
static int32_t UARTMgr_IntReceive(uint8_t ch, uint8_t inttype, uint32_t* event);
static int32_t UARTMgr_IntLineStatus(uint8_t ch, uint32_t* event);
static int32_t UARTMgr_IntRs485Status(uint8_t ch, uint32_t* event);

static const AG903_UARTMgrIntHdr UartIntHdr[AG903_UART_CH_NUM] =
{ UARTMgr_Inthdr0, UARTMgr_Inthdr1, UARTMgr_Inthdr2, UARTMgr_Inthdr3 };


/**
 * @brief           UARTManager
 * @param           ch [in] UART`lԍi0`3j
 * @return          
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EFAULT  s (݃nho^s).
 * @description     Ԃ̏ƒʐMp[^̐ݒ܂BVXeNCALLĉB<p>
 *                  ʐMp[^͈ȉɏ]Đݒ肳܂B<p>
 *                  AG903_UART_DFLT_BAUD , AG903_UART_DFLT_PARITY , AG903_UART_DFLT_STOPBIT , AG903_UART_DFLT_DATBIT , AG903_UART_DFLT_FLOW
*/
int32_t	AG903_UARTMgrInit(uint8_t ch)
{
	AG903_INTMgrHdrPrm inthdr;
	AG903_UARTMgrParam param;
	int32_t	 retval = AG903_ENONE;
	int32_t	 hdrid;
	
	if(AG903_UART_CH_NUM <= ch) {
		return -AG903_EINVAL;
	}
	
	UARTMgr_InitState(ch);  /* ԏ */
	
	if(0 >= UartChStat[ch].hdrid) {
		inthdr.atr   = AG903_INT_HLNG;
		inthdr.intno = AG903_IRQ8_UART0+ch;
		inthdr.func  = (void*)UartIntHdr[ch];
		hdrid = AG903_INTMgrSetHandler(&inthdr);
		if(0 >= hdrid) {
			return -AG903_EFAULT;
		}
		UartChStat[ch].hdrid = hdrid;
	}
	
	AG903_INTMgrEnableInt(AG903_IRQ8_UART0+ch);
	
	param.baud    = AG903_UART_DFLT_BAUD;
	param.parity  = AG903_UART_DFLT_PARITY;
	param.stopbit = AG903_UART_DFLT_STOPBIT;
	param.databit = AG903_UART_DFLT_DATBIT;
	param.flow	  = AG903_UART_DFLT_FLOW;
	retval = UARTMgr_SetParam(ch, &param);	/* ݒ */
	
	return retval;
}

/**
 * @brief           UARTnh擾
 * @param           ch [in] UART`lԍi0`3j
 * @param           handle [out] YUARTnh
 * @return          nh擾
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EBUSY   gpłnhs.
 * @description     w肵CH̃nh擾܂B
*/
int32_t	AG903_UARTMgrGetHandle(uint8_t ch, AG903_UARTMgrHandle** handle)
{
	int32_t	 retval = AG903_ENONE;

	if( (AG903_UART_CH_NUM <= ch) ||
		(NULL == handle) ) {
		return -AG903_EINVAL;
	}
	if(true == UartHandleStat[ch].lock) {
		return -AG903_EBUSY;
	}
	
	UARTMgr_InitState(ch);	/* ԏ */
	
	UartHandleStat[ch].lock = true;
	
	(*handle) = (AG903_UARTMgrHandle*)&UartHandleStat[ch];
	
	return retval;
}

/**
 * @brief           UARTnh
 * @param           handle [in] UARTnh
 * @return          nh
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EBUSY   nhgp.
 * @description     nh܂B<p>쒆Ƀnhꂽꍇ̓G[Ԃ܂B
 * @note            KvɉAvAG903_INTMgrDisableIntɂUART̊荞݂𖳌ɂĉB
*/
int32_t	AG903_UARTMgrReleaseHandle(AG903_UARTMgrHandle* handle)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = UARTMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if( (AG903_UART_STAT_IDLE != UartChStat[ch].snd.stat) ||
		(AG903_UART_STAT_IDLE != UartChStat[ch].rcv.stat) ) {
		return -AG903_EBUSY;
	}
	
	UartHandleStat[ch].lock = false;
	
	return retval;
}

/**
 * @brief           R[obNo^
 * @param           handle [in] UARTnh
 * @param           clbk [in] R[obN֐|C^
 * @return          R[obNo^
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @description     R[obN֐o^܂B
*/
int32_t	AG903_UARTMgrSetCallback(AG903_UARTMgrHandle* handle, AG903_UARTMgrClbk clbk)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = UARTMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	UartHandleStat[ch].clbk = clbk;

	return retval;
}

/**
 * @brief           ʐMp[^ݒ
 * @param           handle [in] UARTnh
 * @param           param [in] p[^
 * @return          ʐMp[^ݒ茋
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EBUSY   nhgp.
 * @description     ʐMp[^ݒ肵܂B
*/
int32_t AG903_UARTMgrSetParam(AG903_UARTMgrHandle*handle, AG903_UARTMgrParam* param)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = UARTMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == param) {
		return -AG903_EINVAL;
	}
	if( (AG903_UART_STAT_IDLE != UartChStat[ch].snd.stat) ||
		(AG903_UART_STAT_IDLE != UartChStat[ch].rcv.stat) ) {
		return -AG903_EBUSY;
	}

	retval = UARTMgr_SetParam(ch, param);

	return retval;
}

/**
 * @brief           RS485p[^ݒ
 * @param           handle [in] UARTnh
 * @param           param [in] p[^
 * @return          RS485p[^ݒ茋
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EBUSY   nhgp.
 * @description     RS485p[^̐ݒ<p>
 *                  {֐̎gpRS485ʐMLɂȂ܂B
*/
int32_t AG903_UARTMgrSetRs485Param(AG903_UARTMgrHandle*handle, AG903_UARTMgrRs485Param *param)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint8_t	 ch;
	
	result = UARTMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == param) {
		return -AG903_EINVAL;
	}
	if( (AG903_UART_STAT_IDLE != UartChStat[ch].snd.stat) ||
		(AG903_UART_STAT_IDLE != UartChStat[ch].rcv.stat) ) {
		return -AG903_EBUSY;
	}

	AG903_UARTPrmSetSetuptime(ch, param->setuptime);
	AG903_UARTPrmSetHoldtime(ch, param->holdtime);

	if(param->event & (AG903_UART_RCVTIMEOUT_BIT|AG903_UART_CHARATIMEOUT_BIT)) { /* ^CAEgmL */
		AG903_UARTPrmDisableTimeoutDetect(ch);	/* U~ */
		AG903_UARTPrmEnableTimeoutIntMask(ch, (AG903_UART_DETECT_RTO_BIT|AG903_UART_DETECT_CTO_BIT));
		
		AG903_UARTPrmEnableTimeoutIntMask(ch, AG903_UART_DETECT_ALL_BIT);
		AG903_UARTPrmClerTimeoutStatus(ch, AG903_UART_DETECT_ALL_BIT);
		
		AG903_UARTPrmSetReceivetime(ch, param->rcvtime);
		AG903_UARTPrmSetCharainterval(ch, param->interval);
		AG903_UARTPrmSetTimeout(ch, param->timeout);
	}

	AG903_UARTPrmSetAutoMode(ch, true, true);
	AG903_UARTPrmEnableRS485(ch);	/* RS485 L */

	UartHandleStat[ch].event = param->event;
	UartChStat[ch].rs485 = true;

	return retval;
}

/**
 * @brief           f[^MJn
 * @param           handle [in] UARTnh
 * @param           buf [in] Mf[^擪AhX
 * @param           size [in] Mf[^TCY
 * @param           dma [in] DMA̎gpL
 * @return          f[^M
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EBUSY   nhgp.
 * @retval          -AG903_ENOBUFS obt@s.
 * @description     f[^MJn̐ݒ܂B<p>
 *                  DMAgp͎wTCỸf[^MŃR[obN܂B<p>
 *                  DMAgp͑MFIFOŃR[obN܂B<p>
 *                  DMAgpbuf,size̓e͖܂B
 * @note            DMAgpꍇ͕ʓrDMAC̐ݒsĉB
*/
int32_t AG903_UARTMgrSend(AG903_UARTMgrHandle* handle, uint8_t* buf, uint32_t size, _Bool dma)
{
	int32_t	retval = AG903_ENONE;
	int32_t	result;
	uint32_t sndsz;
	uint32_t loop;
	uint8_t	 ch;

	result = UARTMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(false == dma) {
		if((NULL == buf) || (0 >= size)) {
			return -AG903_EINVAL;
		}
		if(AG903_UART_BUFSIZE < size) {
			return -AG903_ENOBUFS;
		}
	}
	if(AG903_UART_STAT_IDLE != UartChStat[ch].snd.stat){
		/* M */
		return -AG903_EBUSY;
	}

	UartChStat[ch].snd.dma  = dma;
	UartChStat[ch].snd.stat = AG903_UART_STAT_SEND;
	if(true == dma) {
		AG903_UARTPrmEnableInt(ch, AG903_UART_IER_THREMP_BIT);	/* ݋ (THR Empty) */
	}
	else  {
		UartChStat[ch].snd.buf  = &UartSendBuf[ch][0];
		UartChStat[ch].snd.size = size;
		for(loop=0; loop<UartChStat[ch].snd.size; loop++) {
			*(UartChStat[ch].snd.buf+loop) = *(buf+loop);
		}
		if(AG903_UART_FIFO_SIZE <= size) {
			sndsz = AG903_UART_FIFO_SIZE;
		}
		else {
			sndsz = size;
		}
		UartChStat[ch].snd.cnt  = sndsz;
		AG903_UARTPrmSendData(ch, buf, sndsz);
		AG903_UARTPrmEnableInt(ch, AG903_UART_IER_THREMP_BIT);	/* ݋ */
	}
	
	
	return retval;
}

/**
 * @brief           f[^MJn
 * @param           handle [in] UARTnh
 * @param           dma [in] DMA̎gpL
 * @return          f[^M
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EBUSY   nhgp.
 * @description     f[^MJn̐ݒ܂B<p>
 *                  DMAgp̓f[^MŃR[obN܂<p>
 *                  DMAgp̓G[ŃR[obN܂B
 * @note            DMAgpꍇ͕ʓrDMAC̐ݒsĉB
*/
int32_t AG903_UARTMgrReceive(AG903_UARTMgrHandle* handle, _Bool dma)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t dismsk = 0;
	uint8_t	 ch;

	result = UARTMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}

	if( (AG903_UART_STAT_IDLE != UartChStat[ch].rcv.stat) ||
		((true == UartChStat[ch].rs485) && (AG903_UART_STAT_IDLE != UartChStat[ch].snd.stat)) ){
		/* M or RS485M */
		return -AG903_EBUSY;
	}

	UartChStat[ch].rcv.dma = dma;
	UartChStat[ch].rcv.stat = AG903_UART_STAT_RECEIVE;
	
	if(UartHandleStat[ch].event & AG903_UART_RCVTIMEOUT_BIT) {
		dismsk |= AG903_UART_DETECT_RTO_BIT;
	}
	if(UartHandleStat[ch].event & AG903_UART_CHARATIMEOUT_BIT) {
		dismsk |= AG903_UART_DETECT_CTO_BIT;
	}
	if(0 != dismsk) {
		AG903_UARTPrmDisableTimeoutIntMask(ch, dismsk);	/* }XN */
		AG903_UARTPrmEnableTimeoutDetect(ch);			/* T.O.m L */
	}
	
	AG903_UARTPrmEnableInt(ch, (AG903_UART_IER_LINEST_BIT|AG903_UART_IER_DATARDY_BIT));	/* ݋ */


	return retval;
}

/**
 * @brief           Mf[^擾
 * @param           handle [in] UARTnh
 * @param           buf [out] Mf[^i[AhX
 * @param           size [in] Mf[^TCY
 * @return          Mf[^擾
 * @retval          TCY
 * @retval          -AG903_EINVAL  ُ
 * @description     Mf[^擾܂B<p>
 *                  Mf[^Ȃꍇ0Ԃ܂B
*/
int32_t AG903_UARTMgrGetReceiveData(AG903_UARTMgrHandle* handle, uint8_t* buf, uint32_t size)
{
	int32_t	result;
	int32_t cnt;
	uint8_t	ch;
	uint8_t loop;
	uint8_t stat;

	result = UARTMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if( (NULL == buf) || (0 >= size) ){
		return -AG903_EINVAL;
	}

	cnt = 0;
	for(loop=0; loop<AG903_UART_FIFO_SIZE; loop++) {
		AG903_UARTPrmGetLineStatus(ch, &stat);
		if(0 == (AG903_UART_LSR_DATARDY_BIT&stat)) {
			break;
		}
		AG903_UARTPrmGetReceiveData(ch, buf);
		buf++;
		cnt++;
		if(size <= (uint32_t)cnt) {
			break;
		}
	}

	return cnt;
}

/**
 * @brief           f[^M~
 * @param           handle [in] UARTnh
 * @return          f[^M~
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @description     M~܂BiR[obNʒm~܂܂Bj
*/
int32_t AG903_UARTMgrStopSend(AG903_UARTMgrHandle* handle)
{
	int32_t	retval = AG903_ENONE;
	int32_t	result;
	uint8_t	 ch;

	result = UARTMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	AG903_UARTPrmDisableInt(ch, AG903_UART_IER_THREMP_BIT);
	UartChStat[ch].snd.stat = AG903_UART_STAT_IDLE;

	return retval;
}

/**
 * @brief           f[^M~
 * @param           handle [in] UARTnh
 * @return          f[^M~
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @description     M~܂BiR[obNʒm~܂܂Bj
*/
int32_t AG903_UARTMgrStopReceive(AG903_UARTMgrHandle* handle)
{
	int32_t	retval = AG903_ENONE;
	int32_t	result;
	uint8_t	 ch;

	result = UARTMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	AG903_UARTPrmDisableInt(ch, (AG903_UART_IER_LINEST_BIT|AG903_UART_IER_DATARDY_BIT));
	AG903_UARTPrmEnableTimeoutIntMask(ch, AG903_UART_DETECT_ALL_BIT);
	UartChStat[ch].rcv.stat = AG903_UART_STAT_IDLE;

	return retval;
}

/**
 * @brief           FIFOZbg
 * @param           handle [in] UARTnh
 * @param           rstbit [in] ZbgFIFOw
 * @return          FIFOZbg
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @description     FIFOZbg܂
*/
int32_t AG903_UARTMgrResetFifo(AG903_UARTMgrHandle* handle, uint8_t rstbit)
{
	int32_t	retval = AG903_ENONE;
	int32_t	result;
	uint8_t	fifo = 0;
	uint8_t	ch;

	result = UARTMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	if(AG903_UART_RESET_RXFIFO&rstbit) {
		fifo |= AG903_UART_RXFIFO_BIT;
	}
	if(AG903_UART_RESET_TXFIFO&rstbit) {
		fifo |= AG903_UART_TXFIFO_BIT;
	}
	
	if(0 != fifo) {
		AG903_UARTPrmResetFifo(ch, fifo);
	}
	
	return retval;
}

/**
 * @brief           Break̐ݒ
 * @param           handle [in] UARTnh
 * @param           enable [in] o/o
 * @return          Break̐ݒ茋
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @description     Break̐ݒ܂B
 * @note            BreakMWbN͓삵܂B
*/
int32_t AG903_UARTMgrSetBreak(AG903_UARTMgrHandle* handle, _Bool enable)
{
	int32_t	retval = AG903_ENONE;
	int32_t	result;
	uint8_t	 ch;
	
	result = UARTMgr_CheckHandle(handle, &ch);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	AG903_UARTPrmSetBreak(ch, enable);
	
	return retval;
}

/*
	ԏ
	chFUART`lԍ
*/
static void UARTMgr_InitState(uint8_t ch)
{
	if(AG903_UART_CH_NUM <= ch) {
		return;
	}

	UartHandleStat[ch].clbk = NULL;
	UartHandleStat[ch].event = 0;
	UartHandleStat[ch].lock  = false;
	
	UartChStat[ch].snd.stat = AG903_UART_STAT_IDLE;
	UartChStat[ch].snd.buf  = NULL;
	UartChStat[ch].snd.size = 0;
	UartChStat[ch].snd.cnt	= 0;
	UartChStat[ch].snd.dma  = false;
	UartChStat[ch].rcv.stat = AG903_UART_STAT_IDLE;
	UartChStat[ch].rcv.dma  = false;
	UartChStat[ch].rs485  = false;
	UartChStat[ch].hdlnum = 0;

	AG903_UARTPrmDisableRS485(ch);	/* RS485 Disable */

	return;
}

/*
	nh`FbN.
	handleFUARTnh.
	chFUART`lԍ.
*/
static int32_t UARTMgr_CheckHandle(AG903_UARTMgrHandle* handle, uint8_t* ch)
{
	uint32_t get_ch;
	
	get_ch = ((uint32_t)handle - (uint32_t)UartHandleStat) / sizeof(AG903_UARTMgrHandleStat);
	
	if( (AG903_UART_CH_NUM <= get_ch) ||
		(&UartHandleStat[get_ch] != (AG903_UARTMgrHandleStat*)handle) ) {
		return -AG903_EINVAL;
	}
	(*ch) = (uint8_t)get_ch;
	
	return AG903_ENONE;
}

/*
	ʐMp[^ݒichwj.
	chFUART`lԍ.
	paramFp[^
*/
static int32_t	UARTMgr_SetParam(uint8_t ch, AG903_UARTMgrParam* param)
{
	int32_t	 retval = AG903_ENONE;

	if( (AG903_UART_CH_NUM <= ch) ||
		(NULL == param) ){
		return -AG903_EINVAL;
	}
	
	do {
		retval = UARTMgr_SetBaudrate(ch, param->baud);
		if(AG903_ENONE != retval) {
			break;
		}
		
		retval = UARTMgr_SetForm(ch, param->parity, param->stopbit, param->databit);
		if(AG903_ENONE != retval) {
			break;
		}
		
		retval = UARTMgr_SetFifo(ch, AG903_UART_RXTRGL_1, AG903_UART_TXTRGL_3, true);
		if(AG903_ENONE != retval) {
			break;
		}
		
		retval = UARTMgr_SetFlow(ch, param->flow);
		if(AG903_ENONE != retval) {
			break;
		}
		
		AG903_UARTPrmSetDmaMode(ch, AG903_UART_DMA_MODE0); /* DMA Mode0 */
		
	} while(0);
	
	return retval;
}

/*
	{[E[gݒ.
	chFUART`l.
	baudF{[E[g.
*/
static int32_t UARTMgr_SetBaudrate(uint8_t ch, uint32_t baud)
{
	uint32_t limit;
	uint32_t div;
	uint8_t	 pscale;

	if(AG903_UART_CH_NUM <= ch) {
		return -AG903_EINVAL;
	}
	limit = (AG903_UART_CLK >> 4);
	if(limit < baud) {
		return -AG903_EINVAL;
	}
	
	if(115200 <= baud) {	/* 115200bpsȏPSRݒlύX */
		pscale = 1;
	}
	else {
		pscale = 4;
	}
	div = (AG903_UART_CLK/(pscale*baud*16));
	if(div & 0xFFFF0000) {
		return -AG903_EINVAL;
	}
	AG903_UARTPrmSetDivisor(ch, pscale, (uint16_t)div);

	return AG903_ENONE;
}

/*
	tH[}bgݒ.
	chFUART`l.
	parityFpeB.
	stopbitFXgbvrbg
	databitFf[^rbg
*/
static int32_t UARTMgr_SetForm(uint8_t ch, uint8_t parity, uint8_t stopbit, uint8_t databit)
{
	uint8_t	parity_type;
	uint8_t	length_type;

	if(AG903_UART_CH_NUM <= ch) {
		return -AG903_EINVAL;
	}

	if(AG903_UART_PARITY_NON == parity) {
		parity_type = AG903_UART_PARITY_TYPE_NON;
	}
	else if(AG903_UART_PARITY_EVEN == parity) {
		parity_type = AG903_UART_PARITY_TYPE_EVEN;
	}
	else if(AG903_UART_PARITY_ODD == parity) {
		parity_type = AG903_UART_PARITY_TYPE_ODD;
	}
	else {
		return -AG903_EINVAL;
	}
	
	if((AG903_UART_STOPBIT_1 == stopbit) && (AG903_UART_DATBIT_5 == databit)) {
		length_type = AG903_UART_WORDLEN5_STOP1;
	}
	else if((AG903_UART_STOPBIT_1 == stopbit) && (AG903_UART_DATBIT_6 == databit)) {
		length_type = AG903_UART_WORDLEN6_STOP1;
	}
	else if((AG903_UART_STOPBIT_1 == stopbit) && (AG903_UART_DATBIT_7 == databit)) {
		length_type = AG903_UART_WORDLEN7_STOP1;
	}
	else if((AG903_UART_STOPBIT_1 == stopbit) && (AG903_UART_DATBIT_8 == databit)) {
		length_type = AG903_UART_WORDLEN8_STOP1;
	}
	else if((AG903_UART_STOPBIT_1_5 == stopbit) && (AG903_UART_DATBIT_5 == databit)) {
		length_type = AG903_UART_WORDLEN5_STOP15;
	}
	else if((AG903_UART_STOPBIT_2 == stopbit) && (AG903_UART_DATBIT_6 == databit)) {
		length_type = AG903_UART_WORDLEN6_STOP2;
	}
	else if((AG903_UART_STOPBIT_2 == stopbit) && (AG903_UART_DATBIT_7 == databit)) {
		length_type = AG903_UART_WORDLEN7_STOP2;
	}
	else if((AG903_UART_STOPBIT_2 == stopbit) && (AG903_UART_DATBIT_8 == databit)) {
		length_type = AG903_UART_WORDLEN8_STOP2;
	}
	else {
		return -AG903_EINVAL;
	}

	AG903_UARTPrmSetWordLength(ch, parity_type, length_type);
	
	return AG903_ENONE;
}

/*
	FIFOݒ.
	chFUART`l.
	rx_trglFMFIFOgKx.
	tx_trglFMFIFOgKx.
	enableFFIFOL/.
*/
static int32_t UARTMgr_SetFifo(uint8_t ch, uint8_t rx_trgl, uint8_t tx_trgl, _Bool enable)
{

	if(AG903_UART_CH_NUM <= ch) {
		return -AG903_EINVAL;
	}

	AG903_UARTPrmDisableFifo(ch);

	if(true == enable) {
		AG903_UARTPrmSetFifoTrigger(ch, rx_trgl, tx_trgl);
		AG903_UARTPrmEnableFifo(ch);
	}

	return AG903_ENONE;
}

/*
	t[ݒ.
	chFUART`l.
	flowFt[.
*/
static int32_t UARTMgr_SetFlow(uint8_t ch, uint8_t flow)
{
	if(AG903_UART_CH_NUM <= ch) {
		return -AG903_EINVAL;
	}
	if((AG903_UART_FLOW_CTS != flow) && (AG903_UART_FLOW_NON != flow)) {
		return -AG903_EINVAL;
	}

	AG903_UARTPrmDisableFlowControl(ch, AG903_UART_FLOW_ALL_BIT);
	
	if(flow == AG903_UART_FLOW_CTS) {
		AG903_UARTPrmEnableFlowControl(ch, (AG903_UART_FLOW_RTS_BIT|AG903_UART_FLOW_CTS_BIT));
	}
	
	return AG903_ENONE;
}

/*
	݃nh(CH0).
*/
static void UARTMgr_Inthdr0(void)
{
	UARTMgr_IntProcess(0);	/* CH0 */
	return;
}

/*
	݃nh(CH1).
*/
static void UARTMgr_Inthdr1(void)
{
	UARTMgr_IntProcess(1);	/* CH1 */
	return;
}

/*
	݃nh(CH2).
*/
static void UARTMgr_Inthdr2(void)
{
	UARTMgr_IntProcess(2);	/* CH2 */
	return;
}

/*
	݃nh(CH3).
*/
static void UARTMgr_Inthdr3(void)
{
	UARTMgr_IntProcess(3);	/* CH3 */
	return;
}

/*
	ݏ.
	chFUART`l.
*/
static void UARTMgr_IntProcess(uint8_t ch)
{
	int32_t result = AG903_ENONE;
	uint32_t sndevent = 0;
	uint32_t rcvevent = 0;
	uint8_t	 inttype;
	
	AG903_UARTPrmGetIntType(ch, &inttype);
	
	if(AG903_UART_STAT_SEND == UartChStat[ch].snd.stat) {
		result = UARTMgr_IntSend(ch, inttype, &sndevent);
		if(AG903_ENONE != result) {
			/* Ȃ */
		}
	}
	if(AG903_UART_STAT_RECEIVE == UartChStat[ch].rcv.stat) {
		result = UARTMgr_IntReceive(ch, inttype, &rcvevent);
		if(AG903_ENONE != result) {
			/* Ȃ */
		}
	}

	if( (NULL != UartHandleStat[ch].clbk) &&
		((0 != sndevent) || (0 != rcvevent)) ) {
		UartHandleStat[ch].clbk((AG903_UARTMgrHandle*)&UartHandleStat[ch], (sndevent|rcvevent));
	}
	
	return;
}

/*
	f[^Mi݁j.
	chFUART`lԍ.
	inttypeFݎ.
	eventFCxg.
*/
static int32_t UARTMgr_IntSend(uint8_t ch, uint8_t inttype, uint32_t* event)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t sndsz;

	(*event) = 0;

	if( (AG903_UART_CH_NUM <= ch) ||
		(NULL == event) ) {
		return -AG903_EINVAL;
	}

	if(AG903_UART_INT_THREMPTY == inttype) {
		if(true == UartChStat[ch].snd.dma) {
			(*event) |= AG903_UART_EVENT_SNDEMPTY;
		}
		else {
			if(UartChStat[ch].snd.size <= UartChStat[ch].snd.cnt) {
				AG903_UARTPrmDisableInt(ch, AG903_UART_IER_THREMP_BIT);
				UartChStat[ch].snd.stat = AG903_UART_STAT_IDLE;	/* M */
				(*event) |= AG903_UART_EVENT_SNDEND;
			}
			else {
				sndsz = (UartChStat[ch].snd.size - UartChStat[ch].snd.cnt);
				if(AG903_UART_FIFO_SIZE < sndsz) {
					sndsz = AG903_UART_FIFO_SIZE;
				}
				AG903_UARTPrmSendData(ch, (uint8_t*)(UartChStat[ch].snd.buf+UartChStat[ch].snd.cnt), sndsz);
				UartChStat[ch].snd.cnt += sndsz;
			}
		}
	}

	return retval;
}

/*
	f[^Mi݁j.
	chFUART`lԍ.
	inttypeFݎ.
	eventFCxg.
*/
static int32_t UARTMgr_IntReceive(uint8_t ch, uint8_t inttype, uint32_t* event)
{
	int32_t	 retval = AG903_ENONE;
	int32_t  result;
	uint32_t rs485event;
	

	(*event) = 0;

	if( (AG903_UART_CH_NUM <= ch) ||
		(NULL == event) ) {
		return -AG903_EINVAL;
	}

	switch(inttype) {
		case AG903_UART_INT_LINESTATUS:
			result = UARTMgr_IntLineStatus(ch, event);
			if(AG903_ENONE != result) {
				/* Ȃ */
			}
			break;
		case AG903_UART_INT_DATAREADY:
			if(false == UartChStat[ch].rcv.dma) {
				(*event) |= AG903_UART_EVENT_DATARDY;
			}
			break;
		case AG903_UART_INT_RCVTIMEOUT:
			(*event) |= AG903_UART_EVENT_RCVTIMEOUT;
			break;
		default:
			/* Ȃ */
			break;
	}

	if(true == UartChStat[ch].rs485) {
		result = UARTMgr_IntRs485Status(ch, &rs485event);
		if(AG903_ENONE == result) {
			(*event) |= rs485event;
		}
	}

	return retval;
}

/*
	Line Stausݏ.
	chFUART`l.
	eventFCxg.
*/
static int32_t UARTMgr_IntLineStatus(uint8_t ch, uint32_t* event)
{
	int32_t	 retval = AG903_ENONE;
	uint8_t  linest;

	(*event) = 0;

	if( (AG903_UART_CH_NUM <= ch) ||
		(NULL == event) ) {
		return -AG903_EINVAL;
	}

	AG903_UARTPrmGetLineStatus(ch, &linest);	/* LSR Read */

	if(AG903_UART_LSR_FIFOERR_BIT & linest) {
		(*event) |= AG903_UART_EVENT_FIFOERR;
	}
	if(AG903_UART_LSR_BREAK_BIT & linest) {
		(*event) |= AG903_UART_EVENT_BREAK;
	}
	if(AG903_UART_LSR_FRAMING_BIT & linest) {
		(*event) |= AG903_UART_EVENT_FRAMINGERR;
	}
	if(AG903_UART_LSR_PARITY_BIT & linest) {
		(*event) |= AG903_UART_EVENT_PARITYERR;
	}
	if(AG903_UART_LSR_OVERRUN_BIT & linest) {
		(*event) |= AG903_UART_EVENT_OVERRUN;
	}
	if(AG903_UART_LSR_DATARDY_BIT & linest) {
		(*event) |= AG903_UART_EVENT_DATARDY;
	}
	
	return retval;
}

/*
	RS485Xe[^Xݏ.
	chFUART`l.
	eventFCxg.
*/
static int32_t UARTMgr_IntRs485Status(uint8_t ch, uint32_t* event)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t stat;

	(*event) = 0;

	if( (AG903_UART_CH_NUM <= ch) ||
		(NULL == event) ) {
		return -AG903_EINVAL;
	}
	
	AG903_UARTPrmGetRs485Status(ch, &stat);
	
	if(AG903_UART_DETECT_RTO_BIT & stat) {
		(*event) |= AG903_UART_EVENT_RS485_RTO;
	}
	if(AG903_UART_DETECT_CTO_BIT & stat) {
		(*event) |= AG903_UART_EVENT_RS485_CTO;
	}
	AG903_UARTPrmClerTimeoutStatus(ch, AG903_UART_DETECT_ALL_BIT);

	return retval;
}
