/** 
 * @brief           	uC3 OS Wrapper
 * @author		AXELL
 * @description	uC3 OS Wrapper Layer.
 * @note            	none
 * @history     2017_02_22  
 * @history     2017_10_26  Ver2.0
 * @history     2019_12_27  [SDK3.0] AG903_OSWprGetMplAlignŃACߑOAhX̕ۑ@ύX (#2430)
 * @history     2019_12_27  [SDK3.0] AG903_OSWprInitMemoryŃǗ̈ύXOɃLbV悤ɏC (#2740)
 * @history     2019_12_27  [SDK3.0] AG903_OSWprGetMpl̎AG903_OSWprGetMplAlignĂяo悤ɕύX (#2741)
 * @history     2019_12_27  [SDK3.0] AG903_OSWprGetMplAlignŃAC̃`FbNǉ (#2742)
 * @history     2019_12_27  [SDK3.0] AG903_OSWprCreMpfAG903_OSWprCreMplɃ̃`FbNǉ (#2743)
*/
/* DOM-IGNORE-BEGIN */
/*
 * This program was created by AXELL CORPORATION.
 * Copyright (C) 2017-2019 AXELL CORPORATION, all rights reserved.
 */
/* DOM-IGNORE-END */

#include "AG903_errno.h"
#include "osw/oswrapper.h"
#include "uc3wrapper.h"

typedef struct _AG903_OSWMemMap {
	uintptr_t	ttbr0;		/* TTBR0ϊe[uAhX */
	uint32_t	ttbr0sz;	/* TTBR0ϊe[uTCY   */
	uintptr_t	ttbr1;		/* TTBR1ϊe[uAhX */
	uint32_t	ttbr1sz;	/* TTBR1ϊe[uTCY   */
	uintptr_t	l2tbl;		/* L2ϊe[uAhX    */
	uint32_t	l2tblsz;	/* L2ϊe[uTCY      */
	uintptr_t	ramtop;		/* ǗΏۗ̈擪AhX  */
	uintptr_t	ramend;		/* ǗΏۗ̈TCY        */
} AG903_OSWMemMap;

typedef struct _AG903_OSWBlkInf {
	VP		topaddr;		/* ubN擪AhX */
}AG903_OSWBlkInf;
/* AG903_OSWBlkInfLAhXƃAv֕ԂACߍσAhX̗
 * get_mplԂAhX̋Eƈv悤ɂ邽߂̒
 */
#define BOUNDARY		8	/* get_mplԂAhX̋E */
#define OSWBLKINF_SIZE	(((sizeof(AG903_OSWBlkInf) + BOUNDARY-1)/BOUNDARY)*BOUNDARY)

static volatile ER OswLastErr=0;
static AG903_OSWMemMap  OswMemMap = {0};	/* }bv */
static uint16_t			OswIntDisableCnt[AG903_OSW_IRQ_NUM_MAX]={0};	/* ݋֎~lXgJEg */
static uint8_t			OswIntPriorityTbl[AG903_OSW_IRQ_NUM_MAX]={0};	/* ݗDxe[u */


static int32_t OSW_CreMpl(int32_t mplid, AG903_OSWCMpl* pk_cmpl, int32_t* getid);
static int32_t OSW_CreMpf(int32_t mpfid, AG903_OSWCMpf* pk_cmpf, int32_t* getid);
static void OSW_SetMemoryType(uintptr_t addr, uint32_t size, uint8_t type);
static void OSW_CleanDataCache(uintptr_t addr, uint32_t size);
static void OSW_InvalidDataCache(uintptr_t addr, uint32_t size);
static void OSW_SetLastErcd(ER ercd);
static int32_t OSW_ChgErcd(ER ercd);

extern void OSW_GetTtbInfo(uint32_t* ttbcr, uint32_t* ttbr0, uint32_t* ttbr1);


/**
 * @brief           VRAM}l[WǗVRAM̈RAM
 * @param           ramtop [in] VRAM̈̊JnAhX
 * @param           ramsize [in] VRAM̈TCY
 * @return          
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @description     w肳ꂽVRAM̈̃AhXϊe[uy[WTCY(4KB)Pʂ
 *                  ύXł悤ɎĉBɂȂĂꍇ͉Kv͂܂B
 * @note            ̊֐MMUƃAhXϊe[uꂽxĂяo܂B
 * @note            ramsizeramtop1MBPʂŎw肵ĉB
 * @note            VRAM̈̈ꕔ͊Ǘ̂߂Ɏgp܂B
*/
int32_t AG903_OSWprInitMemory(uintptr_t ramtop, uint32_t ramsize)
{

	uint32_t*	wp;
	uint32_t	ttbcr, ttbr0, ttbr1;
	uint32_t	loop, cnt;
	uint32_t	l1offset;
	uint32_t	setval;
	uint8_t		ttbcrn;
	
	if( (0>=ramtop) || (0>=ramsize) ||
		(0xFFFFF&ramtop) || (0xFFFFF&ramsize) ) { 		/* 1MBPʈȊONG */
		return -AG903_EINVAL; 
	}
	OSW_GetTtbInfo(&ttbcr, &ttbr0, &ttbr1);
	ttbcrn   = (uint8_t)(ttbcr&0x03);
	OswMemMap.ttbr0   = ttbr0&(0xFFFFFF80<<(7-ttbcrn));	/* TTB0 [31:x] (x=14-TTBCR.N) */
	OswMemMap.ttbr0sz = (0x4000>>ttbcrn);				/* TTB0 Size (16KB Max) */
	OswMemMap.ttbr1   = ttbr1&0xFFFFFC00;				/* TTB1 [31:14] */
	OswMemMap.ttbr1sz = 0x4000;							/* TTB0 Size (16KB Fixed) */
	OswMemMap.l2tblsz = 0x1000*((ramsize>>22)+1);		/* L2 Table Size (4KB align) */
	OswMemMap.l2tbl   = (uintptr_t)(ramtop+ramsize-OswMemMap.l2tblsz);
	OswMemMap.ramtop  = ramtop;
	OswMemMap.ramend  = OswMemMap.l2tbl-1;
	
	/* 2ndxfBXNv^VRAM̈ɐ */
	setval = (ramtop | AG903_OSW_MEMTYPE_NORMAL_CACHE_ON);
	cnt    = (ramsize>>12);								/* 4KB(Small page) P */
	wp     = (uint32_t*)OswMemMap.l2tbl;
	for(loop=0; loop<cnt; loop++) {
		(*wp)   = setval;
		setval += 0x1000;
		wp++;
	}
	
	/* 1stxfBXNv^OɃLbV̐Ƃ */
	OSW_CleanDataCache(OswMemMap.ttbr0, OswMemMap.ttbr0sz);
	OSW_InvalidDataCache(OswMemMap.ttbr0, OswMemMap.ttbr0sz);
	OSW_CleanDataCache(OswMemMap.ttbr1, OswMemMap.ttbr1sz);
	OSW_InvalidDataCache(OswMemMap.ttbr1, OswMemMap.ttbr1sz);
	OSW_CleanDataCache(OswMemMap.l2tbl, OswMemMap.l2tblsz);
	OSW_InvalidDataCache(OswMemMap.l2tbl, OswMemMap.l2tblsz);

	/* 1stxfBXNv^2ndxfBXNv^w悤ɕύX */
	setval   = (OswMemMap.l2tbl | 0x01);				/* Section */
	cnt      = (ramsize>>20);							/* 1MBP */
	l1offset = (ramtop>>18);							/* 1Entry = 4[byte] */
	if(OswMemMap.ttbr0sz > l1offset) {
		wp = (uint32_t*)(OswMemMap.ttbr0+l1offset);
	}
	else {
		wp = (uint32_t*)(OswMemMap.ttbr1+(l1offset-OswMemMap.ttbr0sz));
	}
	for(loop=0; loop<cnt; loop++) {
		(*wp)   = setval;
		setval += 0x400;
		if(++wp == (uint32_t*)(OswMemMap.ttbr0+OswMemMap.ttbr0sz)) {
			wp  = (uint32_t*)OswMemMap.ttbr1;
		}
	}
	/* AhXϊe[uXVɂ茻݂TLBU */
	_kernel_invalid_tlb();
	
	return AG903_ENONE;
}

/**
 * @brief           ϒv[̐
 * @param           mplid [in] v[IDԍ
 * @param           pk_cmpl [in] 
 * @return          ϒv[̐
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @retval          -AG903_EFAULT  s(s)
 * @retval          -AG903_EEXIST  ɐς
 * @retval          -AG903_ESYS    VXeG[(zOG[)
 * @description     ϒv[̐܂B
 *                  iITRON4.0Acre_mplgp܂Bj
 * @note            v[̃AhXÓIɎw肷ꍇÃAhX
 *                  y[WTCY(AG903_OSW_CACHE_PAGESIZE)ɃACĂȂ΂Ȃ܂B
 *                  ܂̃̈Ədꍇ͕̓słB
 * @note            v[̃AhXNULLƂꍇ͓IɊ蓖Ă܂B
 *                  ̂ƂACwyуLbVw͂ł܂B
 *
*/
int32_t AG903_OSWprCreMpl(int32_t mplid, AG903_OSWCMpl* pk_cmpl)
{
	int32_t retval;
	int32_t getid;
	
	retval = OSW_CreMpl(mplid, pk_cmpl, &getid);
	
	return retval;
}

/**
 * @brief           ϒv[̐iIDtj
 * @param           pk_cmpl [in] 
 * @return          ϒv[̐
 * @retval          v[IDԍ  I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @retval          -AG903_EFAULT  s(s)
 * @retval          -AG903_ENOID   ID
 * @retval          -AG903_ESYS    VXeG[(zOG[)
 * @description     ϒv[̐iIDtj܂B
 *                  iITRON4.0Aacre_mplgp܂Bj
 * @note            v[̃AhXÓIɎw肷ꍇÃAhX
 *                  y[WTCY(AG903_OSW_CACHE_PAGESIZE)ɃACĂȂ΂Ȃ܂B
 *                  ܂̃̈Ədꍇ͕̓słB
 * @note @@@@   v[̃AhXNULLƂꍇ͓IɊ蓖Ă܂B
*/
int32_t AG903_OSWprCreMplAuto(AG903_OSWCMpl* pk_cmpl)
{
	int32_t retval;
	int32_t getid = 0;
	
	retval = OSW_CreMpl(0, pk_cmpl, &getid);
	if( (AG903_ENONE==retval)&&(0<getid) ) {
		retval = getid;
	}
	
	return retval;
}

/**
 * @brief           ϒv[̍폜
 * @param           mplid [in] v[IDԍ
 * @return          ϒv[̍폜
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @description     ϒv[̍폜܂
 *                  iuITRON4.0Adel_mplgp܂Bj
 * @note            폜ςIDԍɑ΂č폜͋֎~܂B
*/
int32_t AG903_OSWprDelMpl(int32_t mplid)
{
	ER ercd;
	int32_t retval;

	ercd = del_mpl((ID)mplid);
	OSW_SetLastErcd(ercd);
	retval = OSW_ChgErcd(ercd);

	return retval;
}

/**
 * @brief           ϒubN̊l
 * @param           mplid [in] v[IDԍ
 * @param           blksz [in] lubNTCY[byte]
 * @param           p_blk [out] lubN̐擪Ԓn
 * @return          ϒubN̊l
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @retval          -AG903_EFAULT  s(s)
 * @retval          -AG903_ECANCELED  ҂
 * @description     ϒubN̊l܂B
 *                  iuITRON4.0Apget_mplgp܂Bj<p>
 * @note            G[ƂȂꍇp_blk̒l͕słBG[̓G[R[hŊmFĂB
 * @note            ϒubN̊mۂł͎ۂɓIɊmۂ郁TCY
 *                  Ǘ̗̈mۂ邽߂ɗvTCY傫Ȃ܂B
 * @note            l郁ubÑAhX͏ȂƂ4oCgACۏႵĉB
*/
int32_t AG903_OSWprGetMpl(int32_t mplid, uint32_t blksz, void** p_blk)
{
	return AG903_OSWprGetMplAlign(mplid, BOUNDARY, blksz, p_blk);
}

/**
 * @brief           ϒubN̊liACwj
 * @param           mplid [in] v[IDԍ
 * @param           align [in] ACTCY[byte] (2̗ݏ4ȏ)
 * @param           blksz [in] lubNTCY[byte]
 * @param           p_blk [out] lubN̐擪Ԓn
 * @return          ϒubN̊l
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @retval          -AG903_EFAULT  s(s)
 * @retval          -AG903_ECANCELED  ҂
 * @description     w肵ACŉϒubNl܂B
 * @note            G[ƂȂꍇp_blk̒l͕słBG[̓G[R[hŊmFĂB
 * @note            ϒubN̊mۂł͎ۂɓIɊmۂ郁TCY
 *                  Ǘ̗̈mۂ邽߂ɗvTCY傫Ȃ܂B
 * @note            |[eBOȉɎ܂B<p>
 *   * x[XƂuITRON4.0get_mplgp܂B<p>
 *   * ۂɊmۂ郁TCY̓AvvTCYƎ̗̈̍vɂȂ܂B<p>
 *   *   1 AC߂̂߂̃̈<p>
 *   *   2 ACߍσ̃AhX猳Xget_mplԂAhXm邽߂
 *         AG903_OSWBlkInfo\̂𖄂߂Ă߂̗̈<p>
 *   * Av֕ԂACߍσAhX́Aget_mplŊmۂAhXɏL12̃TCY
 *     vꂽACɂȂ悤ɏ߂AhXɂȂ܂B<p>
 *   * AG903_OSWBlkInfo͂ɂ̃AhX\̂̃TCY߂AhXɋL܂B<p>
 */
int32_t AG903_OSWprGetMplAlign(int32_t mplid, uint32_t align, uint32_t blksz, void** p_blk)
{
	ER ercd;
	VP getaddr;
	AG903_OSWBlkInf* blkinf;	/* ACߑOAhX̕ۑ */
	int32_t		retval;
	uint32_t	extsize;		/* ߑOAhXƃAC߂̂߂̃TCY */
	void		*blk;			/* ACߌAhX */
	int			bit_count;
	int	 	 	ii;

	if( (NULL == p_blk) ||
		(0 == blksz) ) {
		return -AG903_EINVAL;
	}

	if (align < 4) {
		return -AG903_EINVAL;
	}
	/* align2̗ݏł邱ƂmF(aligñrbgŕ\\Ȕ͈͂) */
	bit_count = 0;
	for (ii = 0; ii < (int)sizeof(align)*8; ii++) {
		if (align>>ii & 0x1) {
			bit_count += 1;
		}
	}
	if (bit_count != 1) {
		return -AG903_EINVAL;
	}

	extsize = OSWBLKINF_SIZE + align-1;

	ercd = pget_mpl((ID)mplid, (UINT)(blksz+extsize), (VP*)&getaddr);
	OSW_SetLastErcd(ercd);
	retval = OSW_ChgErcd(ercd);
	
	if(AG903_ENONE == retval) {
		blk = (void *)(((uintptr_t)getaddr + extsize) & ~(align-1));
		*p_blk = blk;

		blkinf = (AG903_OSWBlkInf*)((uintptr_t)blk - OSWBLKINF_SIZE);
		blkinf->topaddr = getaddr;
	} else {
		*p_blk = NULL;
	}
	
	return retval;
}

/**
 * @brief           ϒubN̕ԋp
 * @param           mplid [in] IDԍ
 * @param           blk [in] ԋp郁ubN̐擪Ԓn
 * @return          ϒubN̕ԋp
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @description     ϒubN̕ԋp܂B
 *                  iuITRON4.0Arel_mplgp܂Bj
 * @note            ԋpςIDԍɑ΂ĕԋp͋֎~܂B
*/
int32_t AG903_OSWprRelMpl(int32_t mplid, void* blk)
{
	ER ercd;
	AG903_OSWBlkInf* blkinf;
	int32_t retval;

	if(NULL == blk) {
		return -AG903_EINVAL;
	}
	
	blkinf = (AG903_OSWBlkInf*)((int8_t*)blk - OSWBLKINF_SIZE);
	blk = blkinf->topaddr;
	
	ercd = rel_mpl((ID)mplid, (VP)blk);
	OSW_SetLastErcd(ercd);
	retval = OSW_ChgErcd(ercd);

	return retval;
}

/**
 * @brief           ϒubN̏ԎQ
 * @param           mplid [in] IDԍ
 * @param           pk_rmpl [out] ԏԋpAhX
 * @return          ϒubN̏ԎQƌ
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @description     ϒubN̏ԎQƂ܂B
 *                  iuITRON4.0Aref_mplgp܂Bj
 * @note            ۂɊmۉ\ȃTCY͎擾郁̃AC߂Ǘ̈̂
 *                  ̊֐ԂTCYȂȂ܂B<p>
 *                  ȂȂTCY̖ڈ AG903_OSWprGetMplAlign QƂĉB
*/
int32_t AG903_OSWprRefMpl(int32_t mplid, AG903_OSWRMpl* pk_rmpl)
{
	ER ercd;
	int32_t retval;
	T_RMPL rmpl;

	if(NULL == pk_rmpl) {
		return -AG903_EINVAL;
	}
	
	ercd = ref_mpl((ID)mplid, &rmpl);
	OSW_SetLastErcd(ercd);
	retval = OSW_ChgErcd(ercd);
	
	if(AG903_ENONE == retval) {
		pk_rmpl->wtskid = rmpl.wtskid;
		pk_rmpl->fmplsz = rmpl.fmplsz;
		pk_rmpl->fblksz = (rmpl.fblksz > OSWBLKINF_SIZE) ? (rmpl.fblksz - OSWBLKINF_SIZE) : 0;
	}

	return retval;
}

/**
 * @brief           Œ蒷v[̐
 * @param           mpfid [in] v[IDԍ
 * @param           pk_cmpf [in] 
 * @return          Œ蒷v[̐
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @retval          -AG903_EFAULT  s(s)
 * @retval          -AG903_EEXIST  ɐς
 * @retval          -AG903_ESYS    VXeG[(zOG[)
 * @description     Œ蒷v[̐܂B
 *                  iuITRON4.0Acre_mpfgp܂Bj
 * @note            v[̃AhXÓIɎw肷ꍇÃAhX
 *                  y[WTCY(AG903_OSW_CACHE_PAGESIZE)ɃACĂȂ΂Ȃ܂B
 *                  ܂̃̈Ədꍇ͕̓słB
 * @note            v[̃AhXNULLƂꍇ͓IɊ蓖Ă܂B
 *                  ̂ƂACwyуLbVw͂ł܂B
*/
int32_t AG903_OSWprCreMpf(int32_t mpfid, AG903_OSWCMpf* pk_cmpf)
{
	int32_t retval;
	int32_t getid;
	
	retval = OSW_CreMpf(mpfid, pk_cmpf, &getid);
	
	return retval;
}

/**
 * @brief           Œ蒷v[̐iIDtj
 * @param           pk_cmpf [in] 
 * @return          Œ蒷v[̐
 * @retval          v[IDԍ
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @retval          -AG903_EFAULT  s(s)
 * @retval          -AG903_ENOID  ID
 * @retval          -AG903_ESYS    VXeG[(zOG[)
 * @description     Œ蒷v[̐iIDtjs܂B
 *                  iuITRON4.0Aacre_mpfgp܂Bj
*/
int32_t AG903_OSWprCreMpfAuto(AG903_OSWCMpf* pk_cmpf)
{
	int32_t retval;
	int32_t getid = 0;
	
	retval = OSW_CreMpf(0, pk_cmpf, &getid);
	if( (AG903_ENONE==retval)&&(0<getid) ) {
		retval = getid;
	}
	
	return retval;
}

/**
 * @brief           Œ蒷v[̍폜
 * @param           mpfid [in] v[IDԍ
 * @return          Œ蒷v[̍폜
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @description     Œ蒷v[̍폜܂B
 *                  iuITRON4.0Adel_mpfgp܂Bj
*/
int32_t AG903_OSWprDelMpf(int32_t mpfid)
{
	ER ercd;
	int32_t retval;

	ercd = del_mpf((ID)mpfid);
	OSW_SetLastErcd(ercd);
	retval = OSW_ChgErcd(ercd);

	return retval;
}

/**
 * @brief           Œ蒷ubN̊li|[Oj
 * @param           mpfid [in] v[IDԍ
 * @param           p_blk [out] lubN̐擪Ԓn
 * @return          Œ蒷ubN̊l
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @retval          -AG903_EFAULT  s(s)
 * @retval          -AG903_ECANCELED  ҂
 * @description     Œ蒷ubN̊li|[Oj܂B
 *                  iuITRON4.0Apget_mpfgp܂Bj
 * @note            G[ƂȂꍇp_blk̒l͕słBG[̓G[R[hŊmFĂB
*/
int32_t AG903_OSWprGetMpf(int32_t mpfid, void** p_blk)
{
	ER ercd;
	int32_t retval;

	if(NULL == p_blk) {
		return -AG903_EINVAL;
	}

	ercd = pget_mpf((ID)mpfid, (VP*)p_blk);
	OSW_SetLastErcd(ercd);
	retval = OSW_ChgErcd(ercd);

	return retval;
}

/**
 * @brief           Œ蒷ubN̕ԋp
 * @param           mpfid [in] v[IDԍ
 * @param           blk [in] ԋp郁ubN̐擪Ԓn
 * @return          Œ蒷ubN̕ԋp
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @description     Œ蒷ubN̕ԋp܂B
 *                  iuITRON4.0Arel_mpfgp܂Bj
*/
int32_t AG903_OSWprRelMpf(int32_t mpfid, void* blk)
{
	ER ercd;
	int32_t retval;

	if(NULL == blk) {
		return -AG903_EINVAL;
	}

	ercd = rel_mpf((ID)mpfid, (VP)blk);
	OSW_SetLastErcd(ercd);
	retval = OSW_ChgErcd(ercd);

	return retval;
}

/**
 * @brief           Œ蒷ubN̏ԎQ
 * @param           mpfid [in] IDԍ
 * @param           pk_rmpf [out] ԂԂpPbgւ̃|C^
 * @return          Œ蒷ubN̏ԎQƌ
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @description     Œ蒷ubN̏ԂQƂ܂B
 *                  iuITRON4.0Aref_mpfgp܂Bj
*/
int32_t AG903_OSWprRefMpf(int32_t mpfid, AG903_OSWRMpf* pk_rmpf)
{
	ER ercd;
	int32_t retval;
	T_RMPF	rmpf;

	if(NULL == pk_rmpf) {
		return -AG903_EINVAL;
	}
	
	ercd = ref_mpf((ID)mpfid, &rmpf);
	OSW_SetLastErcd(ercd);
	retval = OSW_ChgErcd(ercd);
	
	if(AG903_ENONE == retval) {
		pk_rmpf->wtskid  = rmpf.wtskid;
		pk_rmpf->fblkcnt = rmpf.fblkcnt;
	}

	return retval;
}

/**
 * @brief           ݏ
 * @param           none
 * @return          ݏ̌
 * @retval          AG903_ENONE    I
 * @description     ϐ̏s܂B
 * @note            INTMgr̎gpJnOɕKsĂB
 * @note            荞ݗDxϊe[u̒`Ȃǂs܂B
 * @note            OSW̊֐ł͊ݗDx0:High`31:LowłƋK肵Ă܂B
 *                  Dx̏l͕WłAG903_OSW_DEFAULT_INTRPRI(20)łB
*/
int32_t AG903_OSWprIniInt(void)
{
	uint32_t	loop;

	for(loop=0; loop<AG903_OSW_IRQ_NUM_MAX; loop++) {
		OswIntPriorityTbl[loop] = AG903_OSW_DEFAULT_INTRPRI;	/* Dx */
	}

	return AG903_ENONE;
}

/**
 * @brief           ݃T[rX[`̐iIDtj
 * @param           pk_cisr [in] 
 * @return          ݃T[rX[`̐
 * @retval          T[rX[`IDԍ
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @retval          -AG903_EFAULT  s(s)
 * @retval          -AG903_ENOID   ID
 * @retval          -AG903_ESYS    VXeG[(zOG[)
 * @description     ݃T[rX[`̐iIDtj܂B
 *                  iuITRON4.0Aacre_isrgp܂Bj
 * @note            pk_cisr->intnoŎw肷銄荞ݔԍ0-63͈̔͂IRQԍłB<p>
                    |[eBO̍ۂɂ͂IRQԍOS̑Ή銄荞ݔԍɕϊăT[rXR[ĂяoĂB
*/
int32_t AG903_OSWprCreIsrAuto(AG903_OSWCIsr* pk_cisr)
{
	ID isrid;
	int32_t retval;
	T_CISR	cisr;

	if(NULL == pk_cisr) {
		return -AG903_EINVAL;
	}
	if( (AG903_OSW_IRQ_NUM_MAX <= pk_cisr->intno) ||
		(AG903_OSW_HLNG    != pk_cisr->atr)   ||
		(NULL == pk_cisr->func) ) {
		return -AG903_EINVAL;
	}

	cisr.isratr = (ATR)(TA_HLNG|TA_FPU);
	cisr.exinf	= (VP_INT)pk_cisr->exinf;
	cisr.intno	= (INTNO)(AG903_OSW_IRQ_TOPID + pk_cisr->intno);	/* uC3GICID I/F */
	cisr.isr	= (FP)pk_cisr->func;
	cisr.imask  = (IMASK)(OswIntPriorityTbl[pk_cisr->intno]<<3); 	/* OSWł̊荞ݗDx(0-31)OSdl(0-255)ɕϊ */
	isrid = acre_isr(&cisr);
	OSW_SetLastErcd(isrid);
	if(0 > isrid) {
		retval = OSW_ChgErcd(isrid);
	}
	else if(0 == isrid) {
		retval = -AG903_ESYS;
	}
	else {
		retval = isrid;
	}

	return retval;
}

/**
 * @brief           ݃T[rX[`̍폜
 * @param           isrid [in] T[rX[`IDԍ
 * @return          ݃T[rX[`̍폜
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @description     ݃T[rX[`̍폜܂B
 *                  iuITRON4.0Adel_isrgp܂Bj
*/
int32_t AG903_OSWprDelIsr(int32_t isrid)
{
	ER ercd;
	int32_t retval;

	ercd = del_isr((ID)isrid);
	OSW_SetLastErcd(ercd);
	retval = OSW_ChgErcd(ercd);

	return retval;
}

/**
 * @brief           ݃T[rX[`̏ԎQ
 * @param           isrid [in] T[rX[`IDԍ
 * @param           pk_risr [out] ԏԋpAhX
 * @return          ݃T[rX[`̏ԎQƌ
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @retval          -AG903_EPERM   sgp(荞݂CALLȂ)
 * @description     ݃T[rX[`̏ԂQƂ܂B
 *                  iuITRON4.0Aref_isrgp܂Bj
*/
int32_t AG903_OSWprRefIsr(int32_t isrid, AG903_OSWRIsr* pk_risr)
{
	ER ercd;
	int32_t retval;
	T_RISR	risr;
	
	if(NULL == pk_risr) {
		return -AG903_EINVAL;
	}
	
	ercd = ref_isr((ID)isrid, &risr);
	OSW_SetLastErcd(ercd);
	retval = OSW_ChgErcd(ercd);
	
	if(AG903_ENONE == retval) {
		pk_risr->intno  = (risr.intno - AG903_OSW_IRQ_TOPID);	/* uC3GICID I/F */
		pk_risr->isr    = (VP)risr.isr;
	}

	return retval;
}

/**
 * @brief           ݂̋֎~
 * @param           intno [in] ݔԍ
 * @return          ݂̋֎~
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @description     ݂̋֎~܂B
 *                  iuITRON4.0Adis_intgp܂Bj<p>
 *                  dis_int R[AJE^CNg܂Bi݋֎~̃lXgΉj
*/
int32_t AG903_OSWprDisInt(uint32_t intno)
{
	ER ercd;
	int32_t retval;
	
	if(AG903_OSW_IRQ_NUM_MAX <= intno) {
		return -AG903_EINVAL;
	}
	
	ercd = dis_int((INTNO)(AG903_OSW_IRQ_TOPID+intno));	/* uC3GICID I/F */
	OSW_SetLastErcd(ercd);
	retval = OSW_ChgErcd(ercd);

	OswIntDisableCnt[intno]++;
	
	return retval;
}

/**
 * @brief           ݂̋
 * @param           intno [in] ݔԍ
 * @return          ݂̋
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @description     ݂̋܂B
 *                  iuITRON4.0Aena_intgp܂Bj<p>
 *                  JE^fNg܂BiJE^ AG903_OSWprDisInt ŃCNg܂j
 *                  fNgO܂̓fNg̃JE^l0ꍇ ena_int R[܂B
*/
int32_t AG903_OSWprEnaInt(uint32_t intno)
{
	ER ercd;
	int32_t retval = AG903_ENONE;
	
	if(AG903_OSW_IRQ_NUM_MAX <= intno) {
		return -AG903_EINVAL;
	}

	if(0 < OswIntDisableCnt[intno]) {
		OswIntDisableCnt[intno]--;
	}
	if(0 >= OswIntDisableCnt[intno]) {
		ercd = ena_int((INTNO)(AG903_OSW_IRQ_TOPID+intno));	/* uC3GICID I/F */
		OSW_SetLastErcd(ercd);
		retval = OSW_ChgErcd(ercd);
	}
	
	return retval;
}

/**
 * @brief           ݃}XN̕ύX
 * @param           imask [in] }XNl (1ȏFmask, 0Funmask)
 * @return          ݃}XN̕ύX
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @description     ݃}XN̕ύX܂B
 *                  iuITRON4.0Achg_imsgp܂Bj
*/
int32_t AG903_OSWprChgIms(uint32_t imask)
{
	ER ercd;
	int32_t retval;

	ercd = chg_ims((IMASK)imask);
	OSW_SetLastErcd(ercd);
	retval = OSW_ChgErcd(ercd);
	
	return retval;
}

/**
 * @brief           ݃}XN̎Q
 * @param           imask [out] }XNl (1ȏFmask, 0Funmask)
 * @return          ݃}XN̎Qƌ
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @description     ݃}XN̎QƂ܂B
 *                  iuITRON4.0Aget_imsgp܂Bj
*/
int32_t AG903_OSWprGetIms(uint32_t* imask)
{
	ER ercd;
	int32_t retval;
	
	if(NULL == imask) {
		return -AG903_EINVAL;
	}

	*imask = 0;
	ercd = get_ims((IMASK*)imask);
	OSW_SetLastErcd(ercd);
	retval = OSW_ChgErcd(ercd);
	
	return retval;
}

/**
 * @brief           ݗDx̐ݒ
 * @param           intno [in] ݔԍ
 * @param           intpri [in] vCIeBl(0:High?31:Low)
 * @return          ݗDx̐ݒ茋
 * @retval          AG903_ENONE    I
 * @retval          -AG903_EINVAL  ُ
 * @description     ݗDx̐ݒ܂B
 *                  ̃e[ul܂B
 * @note            T[rX[`o^OɎsĉB
 *                  T[rX[`o^A݋ɗDxf܂B
*/
int32_t AG903_OSWprSetIpri (uint32_t intno, uint8_t intpri)
{
	if( (AG903_OSW_IRQ_NUM_MAX <= intno) ||
		(AG903_OSW_IRQ_PRI_MAX <= intpri) ) {
		return -AG903_EINVAL;
	}
	OswIntPriorityTbl[intno] = intpri;
	
	return AG903_ENONE;
}

/*
 * @brief	ϒv[
 * @param	mplid [in] v[IDԍ
 * @param	pk_cmpl [in] 
 * @param	getid [out] v[IDԍ
 * @return	G[R[h
 * @note    v[̃AhXÓIɎw肷ꍇÃAhX
 *          y[WTCY(AG903_OSW_CACHE_PAGESIZE)ɃACĂȂ΂Ȃ܂B
 *          ܂̃̈Ədꍇ͕̓słB
 * @note    v[̃AhXNULLƂꍇ͕Kvȃ̈悪IɊmۂ܂B
*/
static int32_t OSW_CreMpl(int32_t mplid, AG903_OSWCMpl* pk_cmpl, int32_t* getid)
{
	ER 		ercd;
	int32_t retval = AG903_ENONE;
	int32_t result;
	uint32_t size;
	T_CMPL	cmpl;
	
	if(NULL == pk_cmpl) {
		return -AG903_EINVAL;
	}
	if(AG903_OSW_ATRNUM <= pk_cmpl->mplatr) {
		return -AG903_EINVAL;
	}
	switch (pk_cmpl->memtype) {
	case AG903_OSW_NORMAL_CACHE_OFF:
	case AG903_OSW_NORMAL_CACHE_ON:
	case AG903_OSW_STRONGLY_ORDERED:
		break;
	default:
		return -AG903_EINVAL;
	}
	if(NULL != pk_cmpl->mpl) {
		if((uint32_t)pk_cmpl->mpl % AG903_OSW_CACHE_PAGESIZE) {
			return -AG903_EINVAL;
		}
		size = ((pk_cmpl->mplsz+AG903_OSW_CACHE_PAGESIZE-1) / AG903_OSW_CACHE_PAGESIZE);
		size *= AG903_OSW_CACHE_PAGESIZE;	/* LbṼy[WTCYPʂŊm */
		if( (OswMemMap.ramtop > (uint32_t)pk_cmpl->mpl) ||
			(OswMemMap.ramend < ((uint32_t)pk_cmpl->mpl+size)) ) {
			return -AG903_EINVAL;
		}
	}
	else {
		/* NULLݒ莞uC3MPLMEMmۂ */
		/* ̏ꍇACwyуLbVw͕s */
		/* LbVMPLMEM̈̐ݒɈˑ */
		size = pk_cmpl->mplsz;
	}
	
	if(AG903_OSW_ATRPRI == pk_cmpl->mplatr) {
		cmpl.mplatr = (ATR)TA_TPRI;
	}
	else {
		cmpl.mplatr = (ATR)TA_TFIFO;
	}
	cmpl.mplsz  = (SIZE)size;
	cmpl.mpl    = (VP)pk_cmpl->mpl;
	cmpl.name   = NULL;
	if(0==mplid) {
		result = acre_mpl(&cmpl);
		OSW_SetLastErcd(result);
		if(0 > result) {
			retval = OSW_ChgErcd(result);
		}
		else if(0 == result) {
			retval = -AG903_ESYS;
		}
		else {
			(*getid) = result;
		}
	}
	else {
		ercd = cre_mpl((ID)mplid, &cmpl);
		OSW_SetLastErcd(ercd);
		if(E_OK == ercd) {
			(*getid) = mplid;
		}
		else {
			retval = OSW_ChgErcd(ercd);
		}
	}
	
	if(AG903_ENONE == retval) {
		if(NULL == pk_cmpl->mpl) {
			/* LbVݒ薳 */
		}
		else {
			OSW_SetMemoryType((uintptr_t)pk_cmpl->mpl, size, pk_cmpl->memtype);
		}
	}

	return retval;
}

/*
 * @brief	Œ蒷v[̐
 * @param	mpfid [in] v[IDԍ
 * @param	pk_cmpf [in] 
 * @param	getid [out] v[IDԍ
 * @return	G[R[h
*/
static int32_t OSW_CreMpf(int32_t mpfid, AG903_OSWCMpf* pk_cmpf, int32_t* getid)
{
	ER 		ercd;
	int32_t retval = AG903_ENONE;
	int32_t result;
	uint32_t size = 0;
	T_CMPF	cmpf;
	
	if(NULL == pk_cmpf) {
		return -AG903_EINVAL;
	}
	if(AG903_OSW_ATRNUM <= pk_cmpf->mpfatr) {
		return -AG903_EINVAL;
	}
	switch (pk_cmpf->memtype) {
	case AG903_OSW_NORMAL_CACHE_OFF:
	case AG903_OSW_NORMAL_CACHE_ON:
	case AG903_OSW_STRONGLY_ORDERED:
		break;
	default:
		return -AG903_EINVAL;
	}
	if(NULL != pk_cmpf->mpf) {
		if((uint32_t)pk_cmpf->mpf % AG903_OSW_CACHE_PAGESIZE) {
			return -AG903_EINVAL;
		}
		size = (pk_cmpf->blksz * pk_cmpf->blkcnt);
		if(size % AG903_OSW_CACHE_PAGESIZE) {
			return -AG903_EINVAL;
		}
		if( (OswMemMap.ramtop > (uint32_t)pk_cmpf->mpf) ||
			(OswMemMap.ramend < ((uint32_t)pk_cmpf->mpf+size)) ) {
			return -AG903_EINVAL;
		}
	}
	else {
		/* NULLݒ莞uC3MPLMEMmۂ */
		/* ̏ꍇACwyуLbVw͕s */
		/* LbVMPLMEM̈̐ݒɈˑ */
	}
	
	if(AG903_OSW_ATRPRI == pk_cmpf->mpfatr) {
		cmpf.mpfatr = (ATR)TA_TPRI;
	}
	else {
		cmpf.mpfatr = (ATR)TA_TFIFO;
	}
	cmpf.blkcnt	= (UINT)pk_cmpf->blkcnt;
	cmpf.blksz	= (UINT)pk_cmpf->blksz;
	cmpf.mpf	= (VP)pk_cmpf->mpf;
	cmpf.name	= NULL;
	if(0==mpfid) {
		result = acre_mpf(&cmpf);
		OSW_SetLastErcd(result);
		if(0 > result) {
			retval = OSW_ChgErcd(result);
		}
		else if(0 == result) {
			retval = -AG903_ESYS;
		}
		else {
			(*getid) = result;
		}
	}
	else {
		ercd = cre_mpf((ID)mpfid, &cmpf);
		OSW_SetLastErcd(ercd);
		if(E_OK == ercd) {
			(*getid) = mpfid;
		}
		else {
			retval = OSW_ChgErcd(ercd);
		}
	}
	
	if(AG903_ENONE == retval) {
		if(NULL == pk_cmpf->mpf) {
			/* LbVݒ薳 */
		}
		else {
			OSW_SetMemoryType((uintptr_t)pk_cmpf->mpf, size, pk_cmpf->memtype);
		}
	}
	
	return retval;
}

/*
 * @brief	^Cvݒ.
 * @param	addr [in] ΏۃAhX.
 * @param	size [in] TCY.
 * @param	type [in] ^Cv.
 * @note    type̎wenum _AG903_OswMemType gpĉB
 *          0܂͒`ɂȂlw肵ꍇAG903_OSW_NORMAL_CACHE_OFF(=0)ƂĈ܂B
*/
static void OSW_SetMemoryType(uintptr_t addr, uint32_t size, uint8_t type)
{
	uint32_t*	wp;
	uint32_t	loop, cnt;
	uint32_t	setval;
	uint32_t	offset;
	
	OSW_CleanDataCache(addr, size);		/* Clean Cache */
	OSW_InvalidDataCache(addr, size);	/* Invalid Cache */

	 switch(type) {
		case AG903_OSW_NORMAL_CACHE_ON:
			setval = (addr | AG903_OSW_MEMTYPE_NORMAL_CACHE_ON);
			break;
		case AG903_OSW_NORMAL_CACHE_OFF:
			setval = (addr | AG903_OSW_MEMTYPE_NORMAL_CACHE_OFF);
			break;
		case AG903_OSW_STRONGLY_ORDERED:
			setval = (addr | AG903_OSW_MEMTYPE_STRONGLY_ORDERED);
			break;
		default:
			/* `̏ꍇNORMAL_CACHE_OFF */
			setval = (addr | AG903_OSW_MEMTYPE_NORMAL_CACHE_OFF);
			break;
	}
	offset = (addr-OswMemMap.ramtop)>>10;			/* 4KB(Small page)1Entry */
	wp     = (uint32_t*)(OswMemMap.l2tbl+offset);
	cnt    = (size>>12); 							/* 4KB(Small page) P */
	for(loop=0; loop<cnt; loop++) {					/* L2e[u */
		(*wp)   = setval;
		setval += 0x1000;
		wp++;
	}
	OSW_CleanDataCache(OswMemMap.l2tbl+offset, (cnt<<2));	/* Clean Cache */
	OSW_InvalidDataCache(OswMemMap.l2tbl+offset, (cnt<<2));	/* Invalid Cache */
	_kernel_invalid_tlb();									/* TLB invalid */

	return;
}

/*
 * @brief	f[^LbVN[jO.
 * @param	addr [in] ΏۃAhX.
 * @param	size [in] TCY.
*/
static void OSW_CleanDataCache(uintptr_t addr, uint32_t size)
{
	_kernel_clean_data_cache((void*)addr, size);
	_kernel_clean_l2_cache((void*)addr, size);

	return;
}

/*
 * @brief	f[^LbV.
 * @param	addr [in] ΏۃAhX.
 * @param	size [in] TCY.
*/
static void OSW_InvalidDataCache(uintptr_t addr, uint32_t size)
{
	_kernel_invalid_data_cache((void*)addr, size);
	_kernel_invalid_l2_cache((void*)addr, size);

	return;
}

/*
 * @brief	XgG[̕ۑ.
 * @param	ercd [in] uC3G[R[h.
*/
static void OSW_SetLastErcd(ER ercd)
{
	OswLastErr = ercd;
	return;
}

/*
 * @brief	G[R[h̕ϊ.
 * @param	ercd [in] uC3G[R[h.
 * @return	ϊ̃G[R[h.
*/
static int32_t OSW_ChgErcd(ER ercd)
{
	int32_t retval;

	switch(ercd) {
		case E_OK:
			retval = AG903_ENONE;
			break;
		case E_SYS:
			retval = -AG903_ESYS;
			break;
		case E_NOSPT:
			retval = -AG903_ENOTSUP;
			break;
		case E_RSFN:
		case E_RSATR:
		case E_PAR:
		case E_ID:
		case E_NOEXS:
			retval = -AG903_EINVAL;
			break;
		case E_CTX:
		case E_MACV:
		case E_OACV:
		case E_ILUSE:
			retval = -AG903_EPERM;
			break;
		case E_NOMEM:
		case E_TMOUT:
			retval = -AG903_EFAULT;
			break;
		case E_NOID:
			retval = -AG903_ENOID;
			break;
		case E_OBJ:
			retval = -AG903_EEXIST;
			break;
		case E_QOVR:
			retval = -AG903_ENOBUFS;
			break;
		case E_RLWAI:
		case E_DLT:
		case E_CLS:
			retval = -AG903_ECANCELED;
			break;
		case E_BOVR:
			retval = -AG903_ENOBUFS;
			break;
		default:
			retval = -AG903_ESYS;
			break;
	}
	
	return retval;
}
