/**
 * @brief       UART Primitive
 * @author      AXELL CORPORATION
 * @description UART Primitive Layer
 * @note        none
 * @history     2017_02_22  
 * @history     2017_10_26  Ver2.0
 * @history     2019_12_27  [SDK3.0] UARTFCRWX^ł̃CgANZX̕sC (#2575)
 * @history     2019_12_27  [SDK3.0] AG903dlAX51903_DS06.pdf̏C𔽉f (#2633)
 * @history     2020_07_22  [SDK3.1] UARTDMA[h2gpsɕύX (#2900)
 */
/* DOM-IGNORE-BEGIN */
/*
* This program was created by AXELL CORPORATION.
* Copyright (C) 2017-2020 AXELL CORPORATION, all rights reserved.
*/
/* DOM-IGNORE-END */

#include "AG903_errno.h"
#include "AG903_common.h"
#include "uart/uartprm.h"
#include "register/AG903_uartreg.h"


#define	AG903_UART_READ_REG(CH,REG)			(AG903_UARTn(CH)->REG)
#define	AG903_UART_READ_FLD(CH,REG,POS,MSK)	(((AG903_UARTn(CH)->REG)&MSK) >> POS)
#define	AG903_UART_WRITE_REG(CH,REG,VAL)	(AG903_UARTn(CH)->REG  = (VAL))
#define	AG903_UART_CLR_REG(CH,REG,VAL)		(AG903_UARTn(CH)->REG &= (~VAL))
#define	AG903_UART_SET_REG(CH,REG,VAL)		(AG903_UARTn(CH)->REG |= (VAL))
#define AG903_UART_RMW_VAR(VAR,F,V)         do{(VAR) &= ~(F##_MSK); (VAR) |= ((V)<<F##_POS); }while(0)

static const uint8_t UartLCRSetTbl[AG903_UART_PARITY_TYPE_MAX][AG903_UART_WORDLEN_TYPE_MAX] = {
{0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07}, /* Bit[5:3]FParity, Bit2FStopBits, Bit[1:0]FWord Length */
{0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F}, /* Bit[5:3]FParity, Bit2FStopBits, Bit[1:0]FWord Length */
{0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F}, /* Bit[5:3]FParity, Bit2FStopBits, Bit[1:0]FWord Length */
};

static const uint8_t UartFifoTrglSetTbl[AG903_UART_RXTRGL_MAX][AG903_UART_TXTRGL_MAX] = {
{0x00, 0x10, 0x20, 0x30}, /* Bit[7:6]FRXFIFO_TRGL, Bit[5:4]FTXFIFO_TRGL */
{0x40, 0x50, 0x60, 0x70}, /* Bit[7:6]FRXFIFO_TRGL, Bit[5:4]FTXFIFO_TRGL */
{0x80, 0x90, 0xA0, 0xB0}, /* Bit[7:6]FRXFIFO_TRGL, Bit[5:4]FTXFIFO_TRGL */
{0xC0, 0xD0, 0xE0, 0xF0}, /* Bit[7:6]FRXFIFO_TRGL, Bit[5:4]FTXFIFO_TRGL */
};

static uint8_t fcr_reg[AG903_UART_CH_NUM] =
	{0x00,0x00,0x00,0x00}; /* FCR reset value is 0x00 */

/**
 * @brief           UARTnDLL/UARTnDLM,UARTnPSR̃WX^Write
 * @param           ch [in] UART`l
 * @param           pscale [in] Prescalerݒl
 * @param           div [in] Divisorݒl
 * @return          void
 * @description     UARTnDLL/UARTnDLM,UARTnPSR̃WX^Write
 * @note            UARTnLCRWX^DLABrbg𑀍삵܂B
*/
void AG903_UARTPrmSetDivisor(uint8_t ch, uint8_t pscale, uint16_t div)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	ASSERT(AG903_UART_PSR_MAXNUM >= pscale);
	
	AG903_UART_SET_REG(ch,LCR,(1<<AG903_UARTn_LCR_DLAB_POS));	/* Divisor Latch Access Enable */

	AG903_UART_WRITE_REG(ch,PSR,pscale);
	AG903_UART_WRITE_REG(ch,DLL,div);
	AG903_UART_WRITE_REG(ch,DLM,(div>>8));

	AG903_UART_CLR_REG(ch,LCR,(1<<AG903_UARTn_LCR_DLAB_POS));	/* Divisor Latch Access Disable */
	return;
}

/**
 * @brief           UARTnLCR̃WX^Write
 * @param           ch [in] UART`l
 * @param           parity_type [in] peB
 * @param           length_type [in] f[^
 * @return          void
 * @description     UARTnLCR̃WX^Write
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmSetWordLength(uint8_t ch, uint8_t parity_type, uint8_t length_type)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	ASSERT(AG903_UART_PARITY_TYPE_MAX > parity_type);
	ASSERT(AG903_UART_WORDLEN_TYPE_MAX > length_type);

	AG903_UART_CLR_REG(ch,LCR,(AG903_UARTn_LCR_WL0_MSK|AG903_UARTn_LCR_WL1_MSK|AG903_UARTn_LCR_Stop_Bits_MSK|
		 					   AG903_UARTn_LCR_Parity_Enable_MSK|AG903_UARTn_LCR_Even_Parity_MSK|AG903_UARTn_LCR_Stick_Parity_MSK));
	AG903_UART_SET_REG(ch,LCR,UartLCRSetTbl[parity_type][length_type]);
	return;
}

/**
 * @brief           UARTnFCRWX^FIFO_EnablerbgClear
 * @param           ch [in] UART`l
 * @return          void
 * @description     UARTnFCRWX^FIFO_EnablerbgClear
 * @note            FIFOFCRWX^ŃgKADMA[hݒɗLɂĉB
*/
void AG903_UARTPrmDisableFifo(uint8_t ch)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_RMW_VAR(fcr_reg[ch],AG903_UARTn_FCR_FIFO_Enable,0);
	AG903_UART_WRITE_REG(ch,FCR,fcr_reg[ch]);
	return;
}

/**
 * @brief           UARTnFCRWX^FIFO_EnablerbgSet
 * @param           ch [in] UART`l
 * @return          void
 * @description     UARTnFCRWX^FIFO_EnablerbgSet
 * @note            FIFOFCRWX^ŃgKADMA[hݒɗLɂĉB
*/
void AG903_UARTPrmEnableFifo(uint8_t ch)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_RMW_VAR(fcr_reg[ch],AG903_UARTn_FCR_FIFO_Enable,1);
	AG903_UART_WRITE_REG(ch,FCR,fcr_reg[ch]);
	return;
}

/**
 * @brief           UARTnFCRWX^RXFIFO_TRGLtB[hݒ
 * @param           ch [in] UART`l
 * @param           rx_trgl [in] MFIFOgKx
 * @param           tx_trgl [in] ()
 * @return          void
 * @description     UARTnFCRWX^TXFIFO_TRGL,RXFIFO_TRGLtB[hݒ
 * @note            FIFOFCRWX^ŃgKADMA[hݒɗLɂĉB
 * @note            MFIFOgKx͕ύXł܂Btx_trgl͖܂B
*/
void AG903_UARTPrmSetFifoTrigger(uint8_t ch, uint8_t rx_trgl, uint8_t tx_trgl)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	ASSERT(AG903_UART_RXTRGL_MAX > rx_trgl);
	(void)tx_trgl;
	
	fcr_reg[ch] &= ~(AG903_UARTn_FCR_RXFIFO_TRGL_MSK);
	fcr_reg[ch] |= UartFifoTrglSetTbl[rx_trgl][0] & AG903_UARTn_FCR_RXFIFO_TRGL_MSK;
	AG903_UART_WRITE_REG(ch,FCR,fcr_reg[ch]);
	return;
}

/**
 * @brief           UARTnFCRWX^TX_FIFO_Reset,RX_FIFO_ResettB[hݒ
 * @param           ch [in] UART`l
 * @param           bit [in] ZbgΏFIFO
 * @return          void
 * @description     UARTnFCRWX^TX_FIFO_Reset,RX_FIFO_ResettB[hݒ
 * @note            Xe[^X̓Zbg܂BvƂɃANZXăNAĉB
 * @note            FIFOFCRWX^ŃgKADMA[hݒɗLɂĉB
*/
void AG903_UARTPrmResetFifo(uint8_t ch, uint8_t bit)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	
	if(AG903_UART_RXFIFO_BIT & bit) {
		AG903_UART_RMW_VAR(fcr_reg[ch],AG903_UARTn_FCR_RX_FIFO_Reset,1);
		AG903_UART_WRITE_REG(ch,FCR,fcr_reg[ch]);
		/* WX^ł̓Zbg̃rbg͎I0ɖ߂ */
		AG903_UART_RMW_VAR(fcr_reg[ch],AG903_UARTn_FCR_RX_FIFO_Reset,0);
	}
	if(AG903_UART_TXFIFO_BIT & bit) {
		AG903_UART_RMW_VAR(fcr_reg[ch],AG903_UARTn_FCR_TX_FIFO_Reset,1);
		AG903_UART_WRITE_REG(ch,FCR,fcr_reg[ch]);
		/* WX^ł̓Zbg̃rbg͎I0ɖ߂ */
		AG903_UART_RMW_VAR(fcr_reg[ch],AG903_UARTn_FCR_TX_FIFO_Reset,0);
	}

	return;
}

/**
 * @brief           UARTnIERWX^DSREn,DTREn,CTSEn,RTSEnrbgClear
 * @param           ch [in] UART`l
 * @param           bit [in] DisableΏۃrbg
 * @return          void
 * @description     UARTnIERWX^DSREn,DTREn,CTSEn,RTSEnrbgClear
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmDisableFlowControl(uint8_t ch, uint8_t bit)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_CLR_REG(ch,IER,(bit&0xF0));
	return;
}

/**
 * @brief           UARTnIERWX^DSREn,DTREn,CTSEn,RTSEnrbgSet
 * @param           ch [in] UART`l
 * @param           bit [in] EnableΏۃrbg
 * @return          void
 * @description     UARTnIERWX^DSREn,DTREn,CTSEn,RTSEnrbgSet
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmEnableFlowControl(uint8_t ch, uint8_t bit)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_SET_REG(ch,IER,(bit&0xF0));
	return;
}

/**
 * @brief           UARTnFCRWX^DMA_ModerbgSet
 * @param           ch [in] UART`l
 * @param           mode [in] DMA[h
 * @return          void
 * @description     UARTnFCRWX^DMA_ModerbgSet
 * @note            FIFOFCRWX^ŃgKADMA[hݒɗLɂĉB
 * @note            UARTnMCRWX^DMAmode2rbg͏0ɂĉB܂DMA[h2͎gpł܂B
 * @note            DMA[h1ł̎gpRXIFO臒lDMACBeat킹ĂB

*/
void AG903_UARTPrmSetDmaMode(uint8_t ch, uint8_t mode)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	ASSERT(AG903_UART_DMA_MODE_MAX > mode);
	
	AG903_UART_CLR_REG(ch,MCR,(1<<AG903_UARTn_MCR_DMAmode2_POS));
	if(AG903_UART_DMA_MODE1 == mode) {
		AG903_UART_RMW_VAR(fcr_reg[ch],AG903_UARTn_FCR_DMA_Mode,1);
		AG903_UART_WRITE_REG(ch,FCR,fcr_reg[ch]);
	}
	else {
		AG903_UART_RMW_VAR(fcr_reg[ch],AG903_UARTn_FCR_DMA_Mode,0);
		AG903_UART_WRITE_REG(ch,FCR,fcr_reg[ch]);
	}
	return;
}

/**
 * @brief           UARTnLCRWX^Set_BreakrbgSet/Clear
 * @param           ch [in] UART`l
 * @param           enable [in] o or ~
 * @return          void
 * @description     UARTnLCRWX^Set_BreakrbgSet /Clear
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmSetBreak(uint8_t ch, _Bool enable)
{
	ASSERT(AG903_UART_CH_NUM > ch);

	if(true == enable) {
		AG903_UART_SET_REG(ch,LCR,(1<<AG903_UARTn_LCR_Set_Break_POS));
	}
	else {
		AG903_UART_CLR_REG(ch,LCR,(1<<AG903_UARTn_LCR_Set_Break_POS));
	}
	return;
}

/**
 * @brief           UARTnIERWX^MODEM_Status,Receiver_Line_Status,THR_Empty,Receiver_Data_AvailablerbgClear
 * @param           ch [in] UART`l
 * @param           bit [in] Ώۃrbg
 * @return          void
 * @description     UARTnIERWX^MODEM_Status,Receiver_Line_Status,THR_Empty,Receiver_Data_AvailablerbgClear
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmDisableInt(uint8_t ch, uint32_t bit)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_CLR_REG(ch,IER,(bit&0x0F));

	return;
}

/**
 * @brief           UARTnIERWX^MODEM_Status,Receiver_Line_Status,THR_Empty,Receiver_Data_AvailablerbgSet
 * @param           ch [in] UART`l
 * @param           bit [in] Ώۃrbg
 * @return          void
 * @description     UARTnIERWX^MODEM_Status,Receiver_Line_Status,THR_Empty,Receiver_Data_AvailablerbgSet
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmEnableInt(uint8_t ch, uint32_t bit)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_SET_REG(ch,IER,(bit&0x0F));
	return;
}

/**
 * @brief           UARTnTHR̃WX^Write
 * @param           ch [in] UART`l
 * @param           buf [in] Mf[^AhX
 * @param           sndsz [in] MTCY
 * @return          void
 * @description     UARTnTHR̃WX^Write
*/
void AG903_UARTPrmSendData(uint8_t ch, uint8_t* buf, uint32_t sndsz)
{
	uint32_t loop;
	
	ASSERT(AG903_UART_CH_NUM > ch);
	ASSERT(AG903_UART_FIFO_SIZE >= sndsz);
	ASSERT(NULL != buf);

	for(loop=0; loop<sndsz; loop++) {
		AG903_UART_WRITE_REG(ch,THR,(*buf));
		buf++;
	}

	return;
}

/**
 * @brief           UARTnRBR̃WX^Read
 * @param           ch [in] UART`l
 * @param           buf [out] Mf[^
 * @return          void
 * @description     UARTnRBR̃WX^Read
*/
void AG903_UARTPrmGetReceiveData(uint8_t ch, uint8_t* buf)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	(*buf) = AG903_UART_READ_REG(ch,RBR);
	return;
}

/**
 * @brief           UARTnIIR̃WX^Read
 * @param           ch [in] UART`l
 * @param           type [out] ݎ
 * @return          void
 * @description     UARTnIIR̃WX^Read
*/
void AG903_UARTPrmGetIntType(uint8_t ch, uint8_t* type)
{
	uint8_t	val;

	ASSERT(AG903_UART_CH_NUM > ch);
	val = AG903_UART_READ_REG(ch,IIR);
	(*type) = (val&0x0F);

	return;
}

/**
 * @brief           UARTnLSR̃WX^Read
 * @param           ch [in] UART`l
 * @param           status [out] Xe[^X
 * @return          void
 * @description     UARTnLSR̃WX^Read
*/
void AG903_UARTPrmGetLineStatus(uint8_t ch, uint8_t* status)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	(*status) = AG903_UART_READ_REG(ch,LSR);
	return;
}

/**
 * @brief           UARTnMODWX^FCrbgClear
 * @param           ch [in] UART`l
 * @return          void
 * @description     UARTnMODWX^FCrbgClear
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmDisableRS485(uint8_t ch)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_CLR_REG(ch,MOD,(1<<AG903_UARTn_MOD_FC_POS));
	return;
}

/**
 * @brief           UARTnMODWX^FCrbgSet
 * @param           ch [in] UART`l
 * @return          void
 * @description     UARTnMODWX^FCrbgSet
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmEnableRS485(uint8_t ch)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_SET_REG(ch,MOD,(1<<AG903_UARTn_MOD_FC_POS));
	return;
}

/**
 * @brief           UARTnMODWX^TDrbgClear
 * @param           ch [in] UART`l
 * @return          void
 * @description     UARTnMODWX^TDrbgClear
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmDisableTimeoutDetect(uint8_t ch)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_CLR_REG(ch,MOD,(1<<AG903_UARTn_MOD_TD_POS));
	return;
}

/**
 * @brief           UARTnMODWX^TDrbgSet
 * @param           ch [in] UART`l
 * @return          void
 * @description     UARTnMODWX^TDrbgSet
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmEnableTimeoutDetect(uint8_t ch)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_SET_REG(ch,MOD,(1<<AG903_UARTn_MOD_TD_POS));
	return;
}

/**
 * @brief           UARTnINTMASK̃WX^Write(wrbgClear)
 * @param           ch [in] UART`l
 * @param           bit [in] Ώۃrbg
 * @return          void
 * @description     UARTnINTMASK̃WX^Write(wrbgClear)
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmDisableTimeoutIntMask(uint8_t ch, uint32_t bit)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_CLR_REG(ch,INTMASK,bit);
	return;
}

/**
 * @brief           UARTnINTMASK̃WX^Write(wrbgSet)
 * @param           ch [in] UART`l
 * @param           bit [in] Ώۃrbg
 * @return          void
 * @description     UARTnINTMASK̃WX^Write(wrbgSet)
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmEnableTimeoutIntMask(uint8_t ch, uint32_t bit)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_SET_REG(ch,INTMASK,bit);
	return;
}

/**
 * @brief           UARTnMODWX^HMrbg,SMrbgSet/Clear
 * @param           ch [in] UART`l
 * @param           hold [in] z[hԎL
 * @param           setup [in] ZbgAbvԎL
 * @return          void
 * @description     UARTnMODWX^HMrbg,SMrbgSet /Clear
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmSetAutoMode(uint8_t ch, _Bool hold, _Bool setup)
{
	ASSERT(AG903_UART_CH_NUM > ch);

	if(true == hold) {
		AG903_UART_CLR_REG(ch,MOD,(1<<AG903_UARTn_MOD_HM_POS));
	}
	else {
		AG903_UART_SET_REG(ch,MOD,(1<<AG903_UARTn_MOD_HM_POS));
	}
	
	if(true == setup) {
		AG903_UART_CLR_REG(ch,MOD,(1<<AG903_UARTn_MOD_SM_POS));
	}
	else {
		AG903_UART_SET_REG(ch,MOD,(1<<AG903_UARTn_MOD_SM_POS));
	}
	return;
}

/**
 * @brief           UARTnSETUPTIMẼWX^Write
 * @param           ch [in] UART`l
 * @param           setup_time [in] ZbgAbv [clk]
 * @return          void
 * @description     UARTnSETUPTIMẼWX^Write
*/
void AG903_UARTPrmSetSetuptime(uint8_t ch, uint32_t setup_time)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_WRITE_REG(ch,SETUPTIME,setup_time);
	return;
}

/**
 * @brief           UARTnHOLDTIMẼWX^Write
 * @param           ch [in] UART`l
 * @param           hold_time [in] z[h [clk]
 * @return          void
 * @description     UARTnHOLDTIMẼWX^Write
*/
void AG903_UARTPrmSetHoldtime(uint8_t ch, uint32_t hold_time)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_WRITE_REG(ch,HOLDTIME,hold_time);
	return;
}

/**
 * @brief           UARTnRECEIVETIMẼWX^Write
 * @param           ch [in] UART`l
 * @param           rcv_time [in] M [clk]
 * @return          void
 * @description     UARTnRECEIVETIMẼWX^Write
*/
void AG903_UARTPrmSetReceivetime(uint8_t ch, uint32_t rcv_time)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_WRITE_REG(ch,RECEIVETIME,rcv_time);
	return;
}

/**
 * @brief           UARTnCHARINTERVAL̃WX^Write
 * @param           ch [in] UART`l
 * @param           chara_interval [in] LN^Ԋu [clk]
 * @return          void
 * @description     UARTnCHARINTERVAL̃WX^Write
*/
void AG903_UARTPrmSetCharainterval(uint8_t ch, uint32_t chara_interval)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_WRITE_REG(ch,CHARINTERVAL,chara_interval);
	return;
}

/**
 * @brief           UARTnTIMEOUT̃WX^Write
 * @param           ch [in] UART`l
 * @param           timeout [in] ^CAEg [clk]
 * @return          void
 * @description     UARTnTIMEOUT̃WX^Write
*/
void AG903_UARTPrmSetTimeout(uint8_t ch, uint32_t timeout)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_WRITE_REG(ch,TIMEOUT,timeout);
	return;
}

/**
 * @brief           UARTnCLR̃WX^Write(wrbgSet)
 * @param           ch [in] UART`l
 * @param           bit [in] Ώۃrbg
 * @return          void
 * @description     UARTnCLR̃WX^Write(wrbgSet)
 * @note            WX^[hfBt@CCg܂B
*/
void AG903_UARTPrmClerTimeoutStatus(uint8_t ch, uint32_t bit)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	AG903_UART_SET_REG(ch,CLR,bit);
	return;
}

/**
 * @brief           UARTnSTAT̃WX^Read
 * @param           ch [in] UART`l
 * @param           status [out]  Xe[^X
 * @return          void
 * @description     UARTnSTAT̃WX^Read
*/
void AG903_UARTPrmGetRs485Status(uint8_t ch, uint32_t* status)
{
	ASSERT(AG903_UART_CH_NUM > ch);
	(*status) = AG903_UART_READ_REG(ch,STAT);
	return;
}
