/**
 * @brief       PGP Primitive
 * @author      AXELL CORPORATION
 * @description PGP Primitive Layer.
 * @note        none
 * @history     2017_02_22  
 * @history     2017_10_26  Ver2.0
*/
/* DOM-IGNORE-BEGIN */
/*
 * This program was created by AXELL CORPORATION.
 * Copyright (C) 2017 AXELL CORPORATION, all rights reserved.
 */
/* DOM-IGNORE-END */

#include <stdio.h>
#include "pgp/pgpprm.h"
#include "register/AG903_pgpreg.h"
#include "AG903_common.h"

#define AG903_PGPnDGC(ch)     ((volatile AG903_PGP0DGC_Type *)(AG903_PGP0DGC_BASE + 0x800 * ch))
#define AG903_PGPnDGCF(ch)    ((volatile AG903_PGP0DGCF_Type *)(AG903_PGP0DGCF_BASE + 0x800 * ch))


/**
 * @brief         PGPnCMDWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         cmd [in] CMDtB[hݒl
 * @return        void
 * @description   PGPnCMDWX^ݒ 
*/
void AG903_PgpPrmSetPGPnCMD(uint32_t id, uint32_t cmd)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(cmd & ~(0xF)));

    val = (cmd << AG903_PGPn_CMD_CMD_POS);

    AG903_PGPn(id)->CMD = val;
}


/**
 * @brief         PGPnSTATEWX^擾
 * @param         id [in] pCvCԍ
 * @param         state [out] PGPnSTATEWX^擾li[
 * @return        void
 * @description   PGPnSTATEWX^擾
*/
void AG903_PgpPrmGetPGPnSTATE(uint32_t id, PGPPrmParamPGPnSTATE *state)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(state != NULL);

    val = AG903_PGPn(id)->STATE;

    state->blank  = (val & AG903_PGPn_STATE_BLANK_MSK)  >> AG903_PGPn_STATE_BLANK_POS;
    state->vrmacc = (val & AG903_PGPn_STATE_VRMACC_MSK) >> AG903_PGPn_STATE_VRMACC_POS;
    state->queue  = (val & AG903_PGPn_STATE_QUEUE_MSK)  >> AG903_PGPn_STATE_QUEUE_POS;
    state->sst    = (val & AG903_PGPn_STATE_SST_MSK)    >> AG903_PGPn_STATE_SST_POS;
    state->st     = (val & AG903_PGPn_STATE_ST_MSK)     >> AG903_PGPn_STATE_ST_POS;
}


/**
 * @brief         PGPnINTCTRLIWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         field [in] FIELDtB[hݒl
 * @param         fcnt [in] FCNTtB[hݒl
 * @param         line [in] LINEtB[hݒl
 * @return        void
 * @description   PGPnINTCTRLIWX^ݒ 
*/
void AG903_PgpPrmSetPGPnINTCTRLI(uint32_t id, uint32_t field, uint32_t fcnt, uint32_t line)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(field & ~(0x3)));
    ASSERT(!(fcnt & ~(0xF)));
    ASSERT(!(line & ~(0x1FFF)));

    val = (field << AG903_PGPn_INTCTRLI_FIELD_POS)
        | (fcnt << AG903_PGPn_INTCTRLI_FCNT_POS)
        | (line << AG903_PGPn_INTCTRLI_LINE_POS);

    AG903_PGPn(id)->INTCTRLI = val;
}


/**
 * @brief         PGPnINTCTRLIWX^擾
 * @param         id [in] pCvCԍ
 * @param         field [out] FIELDtB[h擾l
 * @param         fcnt [out] FCNTtB[h擾l
 * @param         line [out] LINEtB[h擾l
 * @return        void
 * @description   PGPnINTCTRLIWX^擾
*/
void AG903_PgpPrmGetPGPnINTCTRLI(uint32_t id, uint32_t *field, uint32_t *fcnt, uint32_t *line)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(field != NULL);
    ASSERT(fcnt != NULL);
    ASSERT(line != NULL);

    val = AG903_PGPn(id)->INTCTRLI;

    *field = (val & AG903_PGPn_INTCTRLI_FIELD_MSK) >> AG903_PGPn_INTCTRLI_FIELD_POS;
    *fcnt  = (val & AG903_PGPn_INTCTRLI_FCNT_MSK) >> AG903_PGPn_INTCTRLI_FCNT_POS;
    *line  = (val & AG903_PGPn_INTCTRLI_LINE_MSK) >> AG903_PGPn_INTCTRLI_LINE_POS;
}


/**
 * @brief         PGPnINTCTRLOWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         field [in] FIELDtB[hݒl
 * @param         fcnt [in] FCNTtB[hݒl
 * @param         line [in] LINEtB[hݒl
 * @return        void
 * @description   PGPnINTCTRLOWX^ݒ 
*/
void AG903_PgpPrmSetPGPnINTCTRLO(uint32_t id, uint32_t field, uint32_t fcnt, uint32_t line)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(field & ~(0x3)));
    ASSERT(!(fcnt & ~(0xF)));
    ASSERT(!(line & ~(0xFFF)));

    val = (field << AG903_PGPn_INTCTRLO_FIELD_POS)
        | (fcnt << AG903_PGPn_INTCTRLO_FCNT_POS)
        | (line << AG903_PGPn_INTCTRLO_LINE_POS);

    AG903_PGPn(id)->INTCTRLO = val;
}


/**
 * @brief         PGPnINTCTRLOWX^擾
 * @param         id [in] pCvCԍ
 * @param         field [out] FIELDtB[h擾l
 * @param         fcnt [out] FCNTtB[h擾l
 * @param         line [out] LINEtB[h擾l
 * @return        void
 * @description   PGPnINTCTRLOWX^擾 
*/
void AG903_PgpPrmGetPGPnINTCTRLO(uint32_t id, uint32_t *field, uint32_t *fcnt, uint32_t *line)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(field != NULL);
    ASSERT(fcnt != NULL);
    ASSERT(line != NULL);

    val = AG903_PGPn(id)->INTCTRLO;

    *field = (val & AG903_PGPn_INTCTRLO_FIELD_MSK) >> AG903_PGPn_INTCTRLO_FIELD_POS;
    *fcnt  = (val & AG903_PGPn_INTCTRLO_FCNT_MSK) >> AG903_PGPn_INTCTRLO_FCNT_POS;
    *line  = (val & AG903_PGPn_INTCTRLO_LINE_MSK) >> AG903_PGPn_INTCTRLO_LINE_POS;
}


/**
 * @brief         PGPnINTENWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         inten [in] PGPnINTENWX^ݒli[
 * @return        void
 * @description   PGPnINTENWX^ݒ 
*/
void AG903_PgpPrmSetPGPnINTEN(uint32_t id, PGPPrmParamPGPnINTEN *inten)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(inten != NULL);
    ASSERT(!(inten->nd      & ~(0x1)));
    ASSERT(!(inten->voicpl  & ~(0x1)));
    ASSERT(!(inten->vooflw  & ~(0x1)));
    ASSERT(!(inten->ifcofs  & ~(0x1)));
    ASSERT(!(inten->ifcofm  & ~(0x1)));
    ASSERT(!(inten->lblcrpt & ~(0x1)));
    ASSERT(!(inten->lblcpl  & ~(0x1)));
    ASSERT(!(inten->hgmicpl & ~(0x1)));
    ASSERT(!(inten->hgmcpl  & ~(0x1)));
    ASSERT(!(inten->vsynco  & ~(0x1)));
    ASSERT(!(inten->vlineo  & ~(0x1)));
    ASSERT(!(inten->vsynci  & ~(0x1)));
    ASSERT(!(inten->vlinei  & ~(0x1)));

    val = (inten->nd      << AG903_PGPn_INTEN_ND_POS)
        | (inten->voicpl  << AG903_PGPn_INTEN_VOICPL_POS)
        | (inten->vooflw  << AG903_PGPn_INTEN_VOOFLW_POS)
        | (inten->ifcofs  << AG903_PGPn_INTEN_IFCOFS_POS)
        | (inten->ifcofm  << AG903_PGPn_INTEN_IFCOFM_POS)
        | (inten->lblcrpt << AG903_PGPn_INTEN_LBLCRPT_POS)
        | (inten->lblcpl  << AG903_PGPn_INTEN_LBLCPL_POS)
        | (inten->hgmicpl << AG903_PGPn_INTEN_HGMICPL_POS)
        | (inten->hgmcpl  << AG903_PGPn_INTEN_HGMCPL_POS)
        | (inten->vsynco  << AG903_PGPn_INTEN_VSYNCO_POS)
        | (inten->vlineo  << AG903_PGPn_INTEN_VLINEO_POS)
        | (inten->vsynci  << AG903_PGPn_INTEN_VSYNCI_POS)
        | (inten->vlinei  << AG903_PGPn_INTEN_VLINEI_POS);

    AG903_PGPn(id)->INTEN = val;
}


/**
 * @brief         PGPnINTENWX^擾
 * @param         id [in] pCvCԍ
 * @param         inten [out] PGPnINTENWX^擾li[
 * @return        void
 * @description   PGPnINTENWX^擾 
*/
void AG903_PgpPrmGetPGPnINTEN(uint32_t id, PGPPrmParamPGPnINTEN *inten)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(inten != NULL);

    val = AG903_PGPn(id)->INTEN;

    inten->nd      = (val & AG903_PGPn_INTEN_ND_MSK)      >> AG903_PGPn_INTEN_ND_POS;
    inten->voicpl  = (val & AG903_PGPn_INTEN_VOICPL_MSK)  >> AG903_PGPn_INTEN_VOICPL_POS;
    inten->vooflw  = (val & AG903_PGPn_INTEN_VOOFLW_MSK)  >> AG903_PGPn_INTEN_VOOFLW_POS;
    inten->ifcofs  = (val & AG903_PGPn_INTEN_IFCOFS_MSK)  >> AG903_PGPn_INTEN_IFCOFS_POS;
    inten->ifcofm  = (val & AG903_PGPn_INTEN_IFCOFM_MSK)  >> AG903_PGPn_INTEN_IFCOFM_POS;
    inten->lblcrpt = (val & AG903_PGPn_INTEN_LBLCRPT_MSK) >> AG903_PGPn_INTEN_LBLCRPT_POS;
    inten->lblcpl  = (val & AG903_PGPn_INTEN_LBLCPL_MSK)  >> AG903_PGPn_INTEN_LBLCPL_POS;
    inten->hgmicpl = (val & AG903_PGPn_INTEN_HGMICPL_MSK) >> AG903_PGPn_INTEN_HGMICPL_POS;
    inten->hgmcpl  = (val & AG903_PGPn_INTEN_HGMCPL_MSK)  >> AG903_PGPn_INTEN_HGMCPL_POS;
    inten->vsynco  = (val & AG903_PGPn_INTEN_VSYNCO_MSK)  >> AG903_PGPn_INTEN_VSYNCO_POS;
    inten->vlineo  = (val & AG903_PGPn_INTEN_VLINEO_MSK)  >> AG903_PGPn_INTEN_VLINEO_POS;
    inten->vsynci  = (val & AG903_PGPn_INTEN_VSYNCI_MSK)  >> AG903_PGPn_INTEN_VSYNCI_POS;
    inten->vlinei  = (val & AG903_PGPn_INTEN_VLINEI_MSK)  >> AG903_PGPn_INTEN_VLINEI_POS;
}


/**
 * @brief         PGPnINTSTATWX^擾
 * @param         id [in] pCvCԍ
 * @param         stat [out] PGPnINTENWX^擾li[
 * @return        void
 * @description   PGPnINTSTATWX^擾 
*/
void AG903_PgpPrmGetPGPnINTSTAT(uint32_t id, PGPPrmParamPGPnINTSTAT *stat)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(stat != NULL);

    val = AG903_PGPn(id)->INTSTAT;

    stat->nd      = (val & AG903_PGPn_INTSTAT_ND_MSK)      >> AG903_PGPn_INTSTAT_ND_POS;
    stat->voicpl  = (val & AG903_PGPn_INTSTAT_VOICPL_MSK)  >> AG903_PGPn_INTSTAT_VOICPL_POS;
    stat->vooflw  = (val & AG903_PGPn_INTSTAT_VOOFLW_MSK)  >> AG903_PGPn_INTSTAT_VOOFLW_POS;
    stat->ifcofs  = (val & AG903_PGPn_INTSTAT_IFCOFS_MSK)  >> AG903_PGPn_INTSTAT_IFCOFS_POS;
    stat->ifcofm  = (val & AG903_PGPn_INTSTAT_IFCOFM_MSK)  >> AG903_PGPn_INTSTAT_IFCOFM_POS;
    stat->lblcrpt = (val & AG903_PGPn_INTSTAT_LBLCRPT_MSK) >> AG903_PGPn_INTSTAT_LBLCRPT_POS;
    stat->lblcpl  = (val & AG903_PGPn_INTSTAT_LBLCPL_MSK)  >> AG903_PGPn_INTSTAT_LBLCPL_POS;
    stat->hgmicpl = (val & AG903_PGPn_INTSTAT_HGMICPL_MSK) >> AG903_PGPn_INTSTAT_HGMICPL_POS;
    stat->hgmcpl  = (val & AG903_PGPn_INTSTAT_HGMCPL_MSK)  >> AG903_PGPn_INTSTAT_HGMCPL_POS;
    stat->vsynco  = (val & AG903_PGPn_INTSTAT_VSYNCO_MSK)  >> AG903_PGPn_INTSTAT_VSYNCO_POS;
    stat->vlineo  = (val & AG903_PGPn_INTSTAT_VLINEO_MSK)  >> AG903_PGPn_INTSTAT_VLINEO_POS;
    stat->vsynci  = (val & AG903_PGPn_INTSTAT_VSYNCI_MSK)  >> AG903_PGPn_INTSTAT_VSYNCI_POS;
    stat->vlinei  = (val & AG903_PGPn_INTSTAT_VLINEI_MSK)  >> AG903_PGPn_INTSTAT_VLINEI_POS;
}


/**
 * @brief         PGPnINTCLRWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         intclr [in] PGPnINTCLRWX^ݒli[
 * @return        void
 * @description   PGPnINTCLRWX^ݒ 
*/
void AG903_PgpPrmSetPGPnINTCLR(uint32_t id, PGPPrmParamPGPnINTCLR *intclr)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(intclr != NULL);
    ASSERT(!(intclr->nd      & ~(0x1)));
    ASSERT(!(intclr->voicpl  & ~(0x1)));
    ASSERT(!(intclr->vooflw  & ~(0x1)));
    ASSERT(!(intclr->ifcofs  & ~(0x1)));
    ASSERT(!(intclr->ifcofm  & ~(0x1)));
    ASSERT(!(intclr->lblcrpt & ~(0x1)));
    ASSERT(!(intclr->lblcpl  & ~(0x1)));
    ASSERT(!(intclr->hgmicpl & ~(0x1)));
    ASSERT(!(intclr->hgmcpl  & ~(0x1)));
    ASSERT(!(intclr->vsynco  & ~(0x1)));
    ASSERT(!(intclr->vlineo  & ~(0x1)));
    ASSERT(!(intclr->vsynci  & ~(0x1)));
    ASSERT(!(intclr->vlinei  & ~(0x1)));

    val = (intclr->nd      << AG903_PGPn_INTCLR_ND_POS)
        | (intclr->voicpl  << AG903_PGPn_INTCLR_VOICPL_POS)
        | (intclr->vooflw  << AG903_PGPn_INTCLR_VOOFLW_POS)
        | (intclr->ifcofs  << AG903_PGPn_INTCLR_IFCOFS_POS)
        | (intclr->ifcofm  << AG903_PGPn_INTCLR_IFCOFM_POS)
        | (intclr->lblcrpt << AG903_PGPn_INTCLR_LBLCRPT_POS)
        | (intclr->lblcpl  << AG903_PGPn_INTCLR_LBLCPL_POS)
        | (intclr->hgmicpl << AG903_PGPn_INTCLR_HGMICPL_POS)
        | (intclr->hgmcpl  << AG903_PGPn_INTCLR_HGMCPL_POS)
        | (intclr->vsynco  << AG903_PGPn_INTCLR_VSYNCO_POS)
        | (intclr->vlineo  << AG903_PGPn_INTCLR_VLINEO_POS)
        | (intclr->vsynci  << AG903_PGPn_INTCLR_VSYNCI_POS)
        | (intclr->vlinei  << AG903_PGPn_INTCLR_VLINEI_POS);

    AG903_PGPn(id)->INTCLR = val;
}


/**
 * @brief         PGPnTRGCTRLIWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         field [in] FIELDtB[hݒl
 * @param         fcnt [in] FCNTtB[hݒl
 * @param         line [in] LINEtB[hݒl
 * @return        void
 * @description   PGPnTRGCTRLIWX^ݒ 
*/
void AG903_PgpPrmSetPGPnTRGCTRLI(uint32_t id, uint32_t field, uint32_t fcnt, uint32_t line)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(field & ~(0x3)));
    ASSERT(!(fcnt & ~(0xF)));
    ASSERT(!(line & ~(0x1FFF)));

    val = (field << AG903_PGPn_TRGCTRLI_FIELD_POS)
        | (fcnt << AG903_PGPn_TRGCTRLI_FCNT_POS)
        | (line << AG903_PGPn_TRGCTRLI_LINE_POS);

    AG903_PGPn(id)->TRGCTRLI = val;
}


/**
 * @brief         PGPnTRGCTRLIWX^擾
 * @param         id [in] pCvCԍ
 * @param         field [out] FIELDtB[h擾l
 * @param         fcnt [out] FCNTtB[h擾l
 * @param         line [out] LINEtB[h擾l
 * @return        void
 * @description   PGPnTRGCTRLIWX^擾 
*/
void AG903_PgpPrmGetPGPnTRGCTRLI(uint32_t id, uint32_t *field, uint32_t *fcnt, uint32_t *line)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(field != NULL);
    ASSERT(fcnt != NULL);
    ASSERT(line != NULL);

    val = AG903_PGPn(id)->TRGCTRLI;

    *field = (val & AG903_PGPn_TRGCTRLI_FIELD_MSK) >> AG903_PGPn_TRGCTRLI_FIELD_POS;
    *fcnt  = (val & AG903_PGPn_TRGCTRLI_FCNT_MSK) >> AG903_PGPn_TRGCTRLI_FCNT_POS;
    *line  = (val & AG903_PGPn_TRGCTRLI_LINE_MSK) >> AG903_PGPn_TRGCTRLI_LINE_POS;
}


/**
 * @brief         PGPnTRGCTRLOWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         field [in] FIELDtB[hݒl
 * @param         fcnt [in] FCNTtB[hݒl
 * @param         line [in] LINEtB[hݒl
 * @return        void
 * @description   PGPnTRGCTRLOWX^ݒ 
*/
void AG903_PgpPrmSetPGPnTRGCTRLO(uint32_t id, uint32_t field, uint32_t fcnt, uint32_t line)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(field & ~(0x3)));
    ASSERT(!(fcnt & ~(0xF)));
    ASSERT(!(line & ~(0xFFF)));

    val = (field << AG903_PGPn_TRGCTRLO_FIELD_POS)
        | (fcnt << AG903_PGPn_TRGCTRLO_FCNT_POS)
        | (line << AG903_PGPn_TRGCTRLO_LINE_POS);

    AG903_PGPn(id)->TRGCTRLO = val;
}


/**
 * @brief         PGPnTRGCTRLOWX^擾
 * @param         id [in] pCvCԍ
 * @param         field [out] FIELDtB[h擾l
 * @param         fcnt [out] FCNTtB[h擾l
 * @param         line [out] LINEtB[h擾l
 * @return        void
 * @description   PGPnTRGCTRLOWX^擾 
*/
void AG903_PgpPrmGetPGPnTRGCTRLO(uint32_t id, uint32_t *field, uint32_t *fcnt, uint32_t *line)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(field != NULL);
    ASSERT(fcnt != NULL);
    ASSERT(line != NULL);

    val = AG903_PGPn(id)->TRGCTRLO;

    *field = (val & AG903_PGPn_TRGCTRLO_FIELD_MSK) >> AG903_PGPn_TRGCTRLO_FIELD_POS;
    *fcnt  = (val & AG903_PGPn_TRGCTRLO_FCNT_MSK) >> AG903_PGPn_TRGCTRLO_FCNT_POS;
    *line  = (val & AG903_PGPn_TRGCTRLO_LINE_MSK) >> AG903_PGPn_TRGCTRLO_LINE_POS;
}


/**
 * @brief         PGPnTRGENWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         trgen [in] PGPnTRGENWX^ݒli[
 * @return        void
 * @description   PGPnTRGENWX^ݒ 
*/
void AG903_PgpPrmSetPGPnTRGEN(uint32_t id, PGPPrmParamPGPnTRGEN *trgen)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(trgen != NULL);
    ASSERT(!(trgen->vsynco & ~(0x1)));
    ASSERT(!(trgen->vlineo & ~(0x1)));
    ASSERT(!(trgen->vsynci & ~(0x1)));
    ASSERT(!(trgen->vlinei & ~(0x1)));

    val = (trgen->vsynco << AG903_PGPn_TRGEN_VSYNCO_POS)
        | (trgen->vlineo << AG903_PGPn_TRGEN_VLINEO_POS)
        | (trgen->vsynci << AG903_PGPn_TRGEN_VSYNCI_POS)
        | (trgen->vlinei << AG903_PGPn_TRGEN_VLINEI_POS);

    AG903_PGPn(id)->TRGEN = val;
}


/**
 * @brief         PGPnTRGENWX^擾
 * @param         id [in] pCvCԍ
 * @param         trgen [out] PGPnTRGENWX^擾li[
 * @return        void
 * @description   PGPnTRGENWX^擾 
*/
void AG903_PgpPrmGetPGPnTRGEN(uint32_t id, PGPPrmParamPGPnTRGEN *trgen)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(trgen != NULL);

    val = AG903_PGPn(id)->TRGEN;

    trgen->vsynco = (val & AG903_PGPn_TRGEN_VSYNCO_MSK) >> AG903_PGPn_TRGEN_VSYNCO_POS;
    trgen->vlineo = (val & AG903_PGPn_TRGEN_VLINEO_MSK) >> AG903_PGPn_TRGEN_VLINEO_POS;
    trgen->vsynci = (val & AG903_PGPn_TRGEN_VSYNCI_MSK) >> AG903_PGPn_TRGEN_VSYNCI_POS;
    trgen->vlinei = (val & AG903_PGPn_TRGEN_VLINEI_MSK) >> AG903_PGPn_TRGEN_VLINEI_POS;
}


/**
 * @brief         PGPnINCTRLWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         inctrl [in] PGPnINCTRLWX^ݒli[
 * @return        void
 * @description   PGPnINCTRLWX^ݒ 
*/
void AG903_PgpPrmSetPGPnINCTRL(uint32_t id, PGPPrmParamPGPnINCTRL *inctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(inctrl != NULL);
    ASSERT(!(inctrl->hdly & ~(0x7)));
    ASSERT(!(inctrl->md   & ~(0x3)));
    ASSERT(!(inctrl->tick & ~(0x1)));
    ASSERT(!(inctrl->trg  & ~(0x7)));
    ASSERT(!(inctrl->sel  & ~(0xF)));
    ASSERT(!(inctrl->fmt  & ~(0x7)));

    val = (inctrl->hdly << AG903_PGPn_INCTRL_HDLY_POS)
        | (inctrl->md   << AG903_PGPn_INCTRL_MD_POS)
        | (inctrl->tick << AG903_PGPn_INCTRL_TICK_POS)
        | (inctrl->trg  << AG903_PGPn_INCTRL_TRG_POS)
        | (inctrl->sel  << AG903_PGPn_INCTRL_SEL_POS)
        | (inctrl->fmt  << AG903_PGPn_INCTRL_FMT_POS);

    AG903_PGPn(id)->INCTRL = val;
}


/**
 * @brief         PGPnINCTRLWX^擾
 * @param         id [in] pCvCԍ
 * @param         inctrl [out] PGPnINCTRLWX^擾li[
 * @return        void
 * @description   PGPnINCTRLWX^擾 
*/
void AG903_PgpPrmGetPGPnINCTRL(uint32_t id, PGPPrmParamPGPnINCTRL *inctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(inctrl != NULL);

    val = AG903_PGPn(id)->INCTRL;

    inctrl->hdly = (val & AG903_PGPn_INCTRL_HDLY_MSK) >> AG903_PGPn_INCTRL_HDLY_POS;
    inctrl->md   = (val & AG903_PGPn_INCTRL_MD_MSK)   >> AG903_PGPn_INCTRL_MD_POS;
    inctrl->tick = (val & AG903_PGPn_INCTRL_TICK_MSK) >> AG903_PGPn_INCTRL_TICK_POS;
    inctrl->trg  = (val & AG903_PGPn_INCTRL_TRG_MSK)  >> AG903_PGPn_INCTRL_TRG_POS;
    inctrl->sel  = (val & AG903_PGPn_INCTRL_SEL_MSK)  >> AG903_PGPn_INCTRL_SEL_POS;
    inctrl->fmt  = (val & AG903_PGPn_INCTRL_FMT_MSK)  >> AG903_PGPn_INCTRL_FMT_POS;
}


/**
 * @brief         PGPnFUNCCTRLWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         funcctrl [in] PGPnFUNCCTRLWX^ݒli[
 * @return        void
 * @description   PGPnFUNCCTRLWX^ݒ 
*/
void AG903_PgpPrmSetPGPnFUNCCTRL(uint32_t id, PGPPrmParamPGPnFUNCCTRL *funcctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(funcctrl != NULL);
    ASSERT(!(funcctrl->out    & ~(0x1)));
    ASSERT(!(funcctrl->ifcodr & ~(0x1)));
    ASSERT(!(funcctrl->ifc    & ~(0x1)));
    ASSERT(!(funcctrl->thr    & ~(0x1)));
    ASSERT(!(funcctrl->dgc    & ~(0x1)));
    ASSERT(!(funcctrl->hsv1   & ~(0x1)));
    ASSERT(!(funcctrl->spf    & ~(0x1)));
    ASSERT(!(funcctrl->scale  & ~(0x1)));
    ASSERT(!(funcctrl->csc1   & ~(0x1)));
    ASSERT(!(funcctrl->nrf    & ~(0x1)));

    val = (funcctrl->out    << AG903_PGPn_FUNCCTRL_OUT_POS)
        | (funcctrl->ifcodr << AG903_PGPn_FUNCCTRL_IFCODR_POS)
        | (funcctrl->ifc    << AG903_PGPn_FUNCCTRL_IFC_POS)
        | (funcctrl->thr    << AG903_PGPn_FUNCCTRL_THR_POS)
        | (funcctrl->dgc    << AG903_PGPn_FUNCCTRL_DGC_POS)
        | (funcctrl->hsv1   << AG903_PGPn_FUNCCTRL_HSV1_POS)
        | (funcctrl->spf    << AG903_PGPn_FUNCCTRL_SPF_POS)
        | (funcctrl->scale  << AG903_PGPn_FUNCCTRL_SCALE_POS)
        | (funcctrl->csc1   << AG903_PGPn_FUNCCTRL_CSC1_POS)
        | (funcctrl->nrf    << AG903_PGPn_FUNCCTRL_NRF_POS);

    AG903_PGPn(id)->FUNCCTRL = val;
}


/**
 * @brief         PGPnFUNCCTRLWX^擾
 * @param         id [in] pCvCԍ
 * @param         funcctrl [out] PGPnFUNCCTRLWX^擾li[
 * @return        void
 * @description   PGPnFUNCCTRLWX^擾 
*/
void AG903_PgpPrmGetPGPnFUNCCTRL(uint32_t id, PGPPrmParamPGPnFUNCCTRL *funcctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(funcctrl != NULL);

    val = AG903_PGPn(id)->FUNCCTRL;

    funcctrl->out    = (val & AG903_PGPn_FUNCCTRL_OUT_MSK)    >> AG903_PGPn_FUNCCTRL_OUT_POS;
    funcctrl->ifcodr = (val & AG903_PGPn_FUNCCTRL_IFCODR_MSK) >> AG903_PGPn_FUNCCTRL_IFCODR_POS;
    funcctrl->ifc    = (val & AG903_PGPn_FUNCCTRL_IFC_MSK)    >> AG903_PGPn_FUNCCTRL_IFC_POS;
    funcctrl->thr    = (val & AG903_PGPn_FUNCCTRL_THR_MSK)    >> AG903_PGPn_FUNCCTRL_THR_POS;
    funcctrl->dgc    = (val & AG903_PGPn_FUNCCTRL_DGC_MSK)    >> AG903_PGPn_FUNCCTRL_DGC_POS;
    funcctrl->hsv1   = (val & AG903_PGPn_FUNCCTRL_HSV1_MSK)   >> AG903_PGPn_FUNCCTRL_HSV1_POS;
    funcctrl->spf    = (val & AG903_PGPn_FUNCCTRL_SPF_MSK)    >> AG903_PGPn_FUNCCTRL_SPF_POS;
    funcctrl->scale  = (val & AG903_PGPn_FUNCCTRL_SCALE_MSK)  >> AG903_PGPn_FUNCCTRL_SCALE_POS;
    funcctrl->csc1   = (val & AG903_PGPn_FUNCCTRL_CSC1_MSK)   >> AG903_PGPn_FUNCCTRL_CSC1_POS;
    funcctrl->nrf    = (val & AG903_PGPn_FUNCCTRL_NRF_MSK)    >> AG903_PGPn_FUNCCTRL_NRF_POS;
}


/**
 * @brief         PGPnSIZEWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         vsize [in] VSIZEtB[hݒl
 * @param         hsize [in] HSIZEtB[hݒl
 * @return        void
 * @description   PGPnSIZEWX^ݒ 
*/
void AG903_PgpPrmSetPGPnSIZE(uint32_t id, uint32_t vsize, uint32_t hsize)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(vsize & ~(0xFFF)));
    ASSERT(!(hsize & ~(0xFFF)));

    val = (vsize << AG903_PGPn_SIZE_VSIZE_POS)
        | (hsize << AG903_PGPn_SIZE_HSIZE_POS);

    AG903_PGPn(id)->SIZE = val;
}


/**
 * @brief         PGPnSIZEWX^擾
 * @param         id [in] pCvCԍ
 * @param         vsize [out] VSIZEtB[h擾l
 * @param         hsize [out] HSIZEtB[h擾l
 * @return        void
 * @description   PGPnSIZEWX^擾 
*/
void AG903_PgpPrmGetPGPnSIZE(uint32_t id, uint32_t *vsize, uint32_t *hsize)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(vsize != NULL);
    ASSERT(hsize != NULL);

    val = AG903_PGPn(id)->SIZE;

    *vsize = (val & AG903_PGPn_SIZE_VSIZE_MSK) >> AG903_PGPn_SIZE_VSIZE_POS;
    *hsize = (val & AG903_PGPn_SIZE_HSIZE_MSK) >> AG903_PGPn_SIZE_HSIZE_POS;
}


/**
 * @brief         PGPnPOSWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         pos [in] PGPnPOSWX^ݒli[
 * @return        void
 * @description   PGPnPOSWX^ݒ 
*/
void AG903_PgpPrmSetPGPnPOS(uint32_t id, PGPPrmParamPGPnPOS *pos)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(pos != NULL);
    ASSERT(!(pos->epos & ~(0x1)));
    ASSERT(!(pos->opos & ~(0x1)));
    ASSERT(!(pos->scan & ~(0x1)));
    ASSERT(!(pos->vpos & ~(0x1FFF)));
    ASSERT(!(pos->hpos & ~(0x1FFF)));

    val = (pos->epos << AG903_PGPn_POS_EPOS_POS)
        | (pos->opos << AG903_PGPn_POS_OPOS_POS)
        | (pos->scan << AG903_PGPn_POS_SCAN_POS)
        | (pos->vpos << AG903_PGPn_POS_VPOS_POS)
        | (pos->hpos << AG903_PGPn_POS_HPOS_POS);

    AG903_PGPn(id)->POS = val;
}


/**
 * @brief         PGPnPOSWX^擾
 * @param         id [in] pCvCԍ
 * @param         pos [out] PGPnPOSWX^擾li[
 * @return        void
 * @description   PGPnPOSWX^擾 
*/
void AG903_PgpPrmGetPGPnPOS(uint32_t id, PGPPrmParamPGPnPOS *pos)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(pos != NULL);

    val = AG903_PGPn(id)->POS;

    pos->epos = (val & AG903_PGPn_POS_EPOS_MSK) >> AG903_PGPn_POS_EPOS_POS;
    pos->opos = (val & AG903_PGPn_POS_OPOS_MSK) >> AG903_PGPn_POS_OPOS_POS;
    pos->scan = (val & AG903_PGPn_POS_SCAN_MSK) >> AG903_PGPn_POS_SCAN_POS;
    pos->vpos = (val & AG903_PGPn_POS_VPOS_MSK) >> AG903_PGPn_POS_VPOS_POS;
    pos->hpos = (val & AG903_PGPn_POS_HPOS_MSK) >> AG903_PGPn_POS_HPOS_POS;
}


/**
 * @brief         PGPnOUTCTRL0WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         outctrl0 [in] PGPnOUTCTRL0WX^ݒli[
 * @return        void
 * @description   PGPnOUTCTRL0WX^ݒ 
*/
void AG903_PgpPrmSetPGPnOUTCTRL0(uint32_t id, PGPPrmParamPGPnOUTCTRL0 *outctrl0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(outctrl0 != NULL);
    ASSERT(!(outctrl0->pad   & ~(0xFF)));
    ASSERT(!(outctrl0->limit & ~(0x1)));
    ASSERT(!(outctrl0->fai   & ~(0x1)));
    ASSERT(!(outctrl0->dth   & ~(0x1)));
    ASSERT(!(outctrl0->bmu   & ~(0x3)));
    ASSERT(!(outctrl0->scan  & ~(0x1)));
    ASSERT(!(outctrl0->dim   & ~(0x1)));
    ASSERT(!(outctrl0->swapw & ~(0x1)));
    ASSERT(!(outctrl0->swaph & ~(0x1)));
    ASSERT(!(outctrl0->swap4 & ~(0x1)));
    ASSERT(!(outctrl0->swap1 & ~(0x1)));
    ASSERT(!(outctrl0->md    & ~(0x7)));
    ASSERT(!(outctrl0->fmt   & ~(0xF)));

    val = (outctrl0->pad   << AG903_PGPn_OUTCTRL0_PAD_POS)
        | (outctrl0->limit << AG903_PGPn_OUTCTRL0_LIMIT_POS)
        | (outctrl0->fai   << AG903_PGPn_OUTCTRL0_FAI_POS)
        | (outctrl0->dth   << AG903_PGPn_OUTCTRL0_DTH_POS)
        | (outctrl0->bmu   << AG903_PGPn_OUTCTRL0_BMU_POS)
        | (outctrl0->scan  << AG903_PGPn_OUTCTRL0_SCAN_POS)
        | (outctrl0->dim   << AG903_PGPn_OUTCTRL0_DIM_POS)
        | (outctrl0->swapw << AG903_PGPn_OUTCTRL0_SWAPW_POS)
        | (outctrl0->swaph << AG903_PGPn_OUTCTRL0_SWAPH_POS)
        | (outctrl0->swap4 << AG903_PGPn_OUTCTRL0_SWAP4_POS)
        | (outctrl0->swap1 << AG903_PGPn_OUTCTRL0_SWAP1_POS)
        | (outctrl0->md    << AG903_PGPn_OUTCTRL0_MD_POS)
        | (outctrl0->fmt   << AG903_PGPn_OUTCTRL0_FMT_POS);

    AG903_PGPn(id)->OUTCTRL0 = val;
}


/**
 * @brief         PGPnOUTCTRL0WX^擾
 * @param         id [in] pCvCԍ
 * @param         outctrl0 [out] PGPnOUTCTRL0WX^擾li[
 * @return        void
 * @description   PGPnOUTCTRL0WX^擾 
*/
void AG903_PgpPrmGetPGPnOUTCTRL0(uint32_t id, PGPPrmParamPGPnOUTCTRL0 *outctrl0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(outctrl0 != NULL);

    val = AG903_PGPn(id)->OUTCTRL0;

    outctrl0->pad   = (val & AG903_PGPn_OUTCTRL0_PAD_MSK)   >> AG903_PGPn_OUTCTRL0_PAD_POS;
    outctrl0->limit = (val & AG903_PGPn_OUTCTRL0_LIMIT_MSK) >> AG903_PGPn_OUTCTRL0_LIMIT_POS;
    outctrl0->fai   = (val & AG903_PGPn_OUTCTRL0_FAI_MSK)   >> AG903_PGPn_OUTCTRL0_FAI_POS;
    outctrl0->dth   = (val & AG903_PGPn_OUTCTRL0_DTH_MSK)   >> AG903_PGPn_OUTCTRL0_DTH_POS;
    outctrl0->bmu   = (val & AG903_PGPn_OUTCTRL0_BMU_MSK)   >> AG903_PGPn_OUTCTRL0_BMU_POS;
    outctrl0->scan  = (val & AG903_PGPn_OUTCTRL0_SCAN_MSK)  >> AG903_PGPn_OUTCTRL0_SCAN_POS;
    outctrl0->dim   = (val & AG903_PGPn_OUTCTRL0_DIM_MSK)   >> AG903_PGPn_OUTCTRL0_DIM_POS;
    outctrl0->swapw = (val & AG903_PGPn_OUTCTRL0_SWAPW_MSK) >> AG903_PGPn_OUTCTRL0_SWAPW_POS;
    outctrl0->swaph = (val & AG903_PGPn_OUTCTRL0_SWAPH_MSK) >> AG903_PGPn_OUTCTRL0_SWAPH_POS;
    outctrl0->swap4 = (val & AG903_PGPn_OUTCTRL0_SWAP4_MSK) >> AG903_PGPn_OUTCTRL0_SWAP4_POS;
    outctrl0->swap1 = (val & AG903_PGPn_OUTCTRL0_SWAP1_MSK) >> AG903_PGPn_OUTCTRL0_SWAP1_POS;
    outctrl0->md    = (val & AG903_PGPn_OUTCTRL0_MD_MSK)    >> AG903_PGPn_OUTCTRL0_MD_POS;
    outctrl0->fmt   = (val & AG903_PGPn_OUTCTRL0_FMT_MSK)   >> AG903_PGPn_OUTCTRL0_FMT_POS;
}


/**
 * @brief         PGPnOUTCTRL1WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         re [in] REtB[hݒl
 * @param         ge [in] GEtB[hݒl
 * @param         be [in] BEtB[hݒl
 * @return        void
 * @description   PGPnOUTCTRL1WX^ݒ 
*/
void AG903_PgpPrmSetPGPnOUTCTRL1(uint32_t id, uint32_t re, uint32_t ge, uint32_t be)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(re & ~(0x1)));
    ASSERT(!(ge & ~(0x1)));
    ASSERT(!(be & ~(0x1)));

    val = (re << AG903_PGPn_OUTCTRL1_RE_POS)
        | (ge << AG903_PGPn_OUTCTRL1_GE_POS)
        | (be << AG903_PGPn_OUTCTRL1_BE_POS);

    AG903_PGPn(id)->OUTCTRL1 = val;
}


/**
 * @brief         PGPnOUTCTRL1WX^擾
 * @param         id [in] pCvCԍ
 * @param         re [out] REtB[h擾l
 * @param         ge [out] GEtB[h擾l
 * @param         be [out] BEtB[h擾l
 * @return        void
 * @description   PGPnOUTCTRL1WX^擾 
*/
void AG903_PgpPrmGetPGPnOUTCTRL1(uint32_t id, uint32_t *re, uint32_t *ge, uint32_t *be)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(re != NULL);
    ASSERT(ge != NULL);
    ASSERT(be != NULL);

    val = AG903_PGPn(id)->OUTCTRL1;

    *re = (val & AG903_PGPn_OUTCTRL1_RE_MSK) >> AG903_PGPn_OUTCTRL1_RE_POS;
    *ge = (val & AG903_PGPn_OUTCTRL1_GE_MSK) >> AG903_PGPn_OUTCTRL1_GE_POS;
    *be = (val & AG903_PGPn_OUTCTRL1_BE_MSK) >> AG903_PGPn_OUTCTRL1_BE_POS;
}


/**
 * @brief         PGPnOUTBASEWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         adr [in] ADRtB[hݒl
 * @return        void
 * @description   PGPnOUTBASEWX^ݒ 
*/
void AG903_PgpPrmSetPGPnOUTBASE(uint32_t id, uint32_t adr)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(adr & ~(0x1FFFFFFF)));

    val = (adr << AG903_PGPn_OUTBASE_ADR_POS);

    AG903_PGPn(id)->OUTBASE = val;
}


/**
 * @brief         PGPnOUTBASEWX^擾
 * @param         id [in] pCvCԍ
 * @param         adr [out] ADRtB[h擾l
 * @return        void
 * @description   PGPnOUTBASEWX^擾 
*/
void AG903_PgpPrmGetPGPnOUTBASE(uint32_t id, uint32_t *adr)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(adr != NULL);

    val = AG903_PGPn(id)->OUTBASE;

    *adr = (val & AG903_PGPn_OUTBASE_ADR_MSK) >> AG903_PGPn_OUTBASE_ADR_POS;
}


/**
 * @brief         PGPnOUTHSIZEWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         hsize [in] HSIZEtB[hݒl
 * @return        void
 * @description   PGPnOUTHSIZEWX^ݒ 
*/
void AG903_PgpPrmSetPGPnOUTHSIZE(uint32_t id, uint32_t hsize)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(hsize & ~(0x3FFF)));

    val = (hsize << AG903_PGPn_OUTHSIZE_HSIZE_POS);

    AG903_PGPn(id)->OUTHSIZE = val;
}


/**
 * @brief         PGPnOUTHSIZEWX^擾
 * @param         id [in] pCvCԍ
 * @param         hsize [out] HSIZEtB[h擾l
 * @return        void
 * @description   PGPnOUTHSIZEWX^擾 
*/
void AG903_PgpPrmGetPGPnOUTHSIZE(uint32_t id, uint32_t *hsize)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(hsize != NULL);

    val = AG903_PGPn(id)->OUTHSIZE;

    *hsize = (val & AG903_PGPn_OUTHSIZE_HSIZE_MSK) >> AG903_PGPn_OUTHSIZE_HSIZE_POS;
}


/**
 * @brief         PGPnFAIVAL0WX^擾
 * @param         id [in] pCvCԍ
 * @param         tmstamp [out] TMSTAMPtB[h擾l
 * @return        void
 * @description   PGPnFAIVAL0WX^擾 
*/
void AG903_PgpPrmGetPGPnFAIVAL0(uint32_t id, uint32_t *tmstamp)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(tmstamp != NULL);

    val = AG903_PGPn(id)->FAIVAL0;

    *tmstamp = (val & AG903_PGPn_FAIVAL0_TMSTMP_MSK) >> AG903_PGPn_FAIVAL0_TMSTMP_POS;
}


/**
 * @brief         PGPnFAIVAL1WX^擾
 * @param         id [in] pCvCԍ
 * @param         field [out] FIELDtB[h擾l
 * @param         port [out] PORTtB[h擾l
 * @param         fnum [out] FNUMtB[h擾l
 * @return        void
 * @description   PGPnFAIVAL1WX^擾 
*/
void AG903_PgpPrmGetPGPnFAIVAL1(uint32_t id, uint32_t *field, uint32_t *port, uint32_t *fnum)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(field != NULL);
    ASSERT(port != NULL);
    ASSERT(fnum != NULL);

    val = AG903_PGPn(id)->FAIVAL1;

    *field = (val & AG903_PGPn_FAIVAL1_FIELD_MSK) >> AG903_PGPn_FAIVAL1_FIELD_POS;
    *port  = (val & AG903_PGPn_FAIVAL1_PORT_MSK) >> AG903_PGPn_FAIVAL1_PORT_POS;
    *fnum  = (val & AG903_PGPn_FAIVAL1_FNUM_MSK) >> AG903_PGPn_FAIVAL1_FNUM_POS;
}


/**
 * @brief         PGPnIPCPRMWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         limit [in] LIMITtB[hݒl
 * @param         blky [in] BLKYtB[hݒl
 * @param         blkc [in] BLKCtB[hݒl
 * @return        void
 * @description   PGPnIPCPRMWX^ݒ 
*/
void AG903_PgpPrmSetPGPnIPCPRM(uint32_t id, uint32_t limit, uint32_t blky, uint32_t blkc)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(limit & ~(0x1)));
    ASSERT(!(blky & ~(0xFF)));
    ASSERT(!(blkc & ~(0xFF)));

    val = (limit << AG903_PGPn_IPCPRM_LIMIT_POS)
        | (blky << AG903_PGPn_IPCPRM_BLKY_POS)
        | (blkc << AG903_PGPn_IPCPRM_BLKC_POS);

    AG903_PGPn(id)->IPCPRM = val;
}


/**
 * @brief         PGPnIPCPRMWX^擾
 * @param         id [in] pCvCԍ
 * @param         limit [out] LIMITtB[h擾l
 * @param         blky [out] BLKYtB[h擾l
 * @param         blkc [out] BLKCtB[h擾l
 * @return        void
 * @description   PGPnIPCPRMWX^擾 
*/
void AG903_PgpPrmGetPGPnIPCPRM(uint32_t id, uint32_t *limit, uint32_t *blky, uint32_t *blkc)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(limit != NULL);
    ASSERT(blky != NULL);
    ASSERT(blkc != NULL);

    val = AG903_PGPn(id)->IPCPRM;

    *limit = (val & AG903_PGPn_IPCPRM_LIMIT_MSK) >> AG903_PGPn_IPCPRM_LIMIT_POS;
    *blky  = (val & AG903_PGPn_IPCPRM_BLKY_MSK) >> AG903_PGPn_IPCPRM_BLKY_POS;
    *blkc  = (val & AG903_PGPn_IPCPRM_BLKC_MSK) >> AG903_PGPn_IPCPRM_BLKC_POS;
}


/**
 * @brief         PGPnCSC1CTRLWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         limit [in] LIMITtB[hݒl
 * @param         sft [in] SFTtB[hݒl
 * @return        void
 * @description   PGPnCSC1CTRLWX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC1CTRL(uint32_t id, uint32_t limit, uint32_t sft)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(limit & ~(0x1)));
    ASSERT(!(sft & ~(0xF)));

    val = (limit << AG903_PGPn_CSC1CTRL_LIMIT_POS)
        | (sft << AG903_PGPn_CSC1CTRL_SFT_POS);

    AG903_PGPn(id)->CSC1CTRL = val;
}


/**
 * @brief         PGPnCSC1CTRLWX^擾
 * @param         id [in] pCvCԍ
 * @param         limit [out] LIMITtB[h擾l
 * @param         sft [out] SFTtB[h擾l
 * @return        void
 * @description   PGPnCSC1CTRLWX^擾 
*/
void AG903_PgpPrmGetPGPnCSC1CTRL(uint32_t id, uint32_t *limit, uint32_t *sft)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(limit != NULL);
    ASSERT(sft != NULL);

    val = AG903_PGPn(id)->CSC1CTRL;

    *limit = (val & AG903_PGPn_CSC1CTRL_LIMIT_MSK) >> AG903_PGPn_CSC1CTRL_LIMIT_POS;
    *sft   = (val & AG903_PGPn_CSC1CTRL_SFT_MSK) >> AG903_PGPn_CSC1CTRL_SFT_POS;
}


/**
 * @brief         PGPnCSC1PRM0WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m12 [in] M12tB[hݒl
 * @param         m11 [in] M11tB[hݒl
 * @return        void
 * @description   PGPnCSC1PRM0WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC1PRM0(uint32_t id, uint32_t m12, uint32_t m11)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m12 & ~(0xFFF)));
    ASSERT(!(m11 & ~(0xFFF)));

    val = (m12 << AG903_PGPn_CSC1PRM0_M12_POS)
        | (m11 << AG903_PGPn_CSC1PRM0_M11_POS);

    AG903_PGPn(id)->CSC1PRM0 = val;
}


/**
 * @brief         PGPnCSC1PRM0WX^擾
 * @param         id [in] pCvCԍ
 * @param         m12 [out] M12tB[h擾l
 * @param         m11 [out] M11tB[h擾l
 * @return        void
 * @description   PGPnCSC1PRM0WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC1PRM0(uint32_t id, uint32_t *m12, uint32_t *m11)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m12 != NULL);
    ASSERT(m11 != NULL);

    val = AG903_PGPn(id)->CSC1PRM0;

    *m12 = (val & AG903_PGPn_CSC1PRM0_M12_MSK) >> AG903_PGPn_CSC1PRM0_M12_POS;
    *m11 = (val & AG903_PGPn_CSC1PRM0_M11_MSK) >> AG903_PGPn_CSC1PRM0_M11_POS;
}


/**
 * @brief         PGPnCSC1PRM1WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m13 [in] M13tB[hݒl
 * @return        void
 * @description   PGPnCSC1PRM1WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC1PRM1(uint32_t id, uint32_t m13)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m13 & ~(0xFFF)));

    val = (m13 << AG903_PGPn_CSC1PRM1_M13_POS);

    AG903_PGPn(id)->CSC1PRM1 = val;
}


/**
 * @brief         PGPnCSC1PRM1WX^擾
 * @param         id [in] pCvCԍ
 * @param         m13 [out] M13tB[h擾l
 * @return        void
 * @description   PGPnCSC1PRM1WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC1PRM1(uint32_t id, uint32_t *m13)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m13 != NULL);

    val = AG903_PGPn(id)->CSC1PRM1;

    *m13 = (val & AG903_PGPn_CSC1PRM1_M13_MSK) >> AG903_PGPn_CSC1PRM1_M13_POS;
}


/**
 * @brief         PGPnCSC1PRM2WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m14 [in] M14tB[hݒl
 * @return        void
 * @description   PGPnCSC1PRM2WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC1PRM2(uint32_t id, uint32_t m14)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m14 & ~(0xFFFFF)));

    val = (m14 << AG903_PGPn_CSC1PRM2_M14_POS);

    AG903_PGPn(id)->CSC1PRM2 = val;
}


/**
 * @brief         PGPnCSC1PRM2WX^擾
 * @param         id [in] pCvCԍ
 * @param         m14 [out] M14tB[h擾l
 * @return        void
 * @description   PGPnCSC1PRM2WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC1PRM2(uint32_t id, uint32_t *m14)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m14 != NULL);

    val = AG903_PGPn(id)->CSC1PRM2;

    *m14 = (val & AG903_PGPn_CSC1PRM2_M14_MSK) >> AG903_PGPn_CSC1PRM2_M14_POS;
}


/**
 * @brief         PGPnCSC1PRM3WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m22 [in] M22tB[hݒl
 * @param         m21 [in] M21tB[hݒl
 * @return        void
 * @description   PGPnCSC1PRM3WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC1PRM3(uint32_t id, uint32_t m22, uint32_t m21)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m22 & ~(0xFFF)));
    ASSERT(!(m21 & ~(0xFFF)));

    val = (m22 << AG903_PGPn_CSC1PRM3_M22_POS)
        | (m21 << AG903_PGPn_CSC1PRM3_M21_POS);

    AG903_PGPn(id)->CSC1PRM3 = val;
}


/**
 * @brief         PGPnCSC1PRM3WX^擾
 * @param         id [in] pCvCԍ
 * @param         m22 [out] M22tB[h擾l
 * @param         m21 [out] M21tB[h擾l
 * @return        void
 * @description   PGPnCSC1PRM3WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC1PRM3(uint32_t id, uint32_t *m22, uint32_t *m21)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m22 != NULL);
    ASSERT(m21 != NULL);

    val = AG903_PGPn(id)->CSC1PRM3;

    *m22 = (val & AG903_PGPn_CSC1PRM3_M22_MSK) >> AG903_PGPn_CSC1PRM3_M22_POS;
    *m21 = (val & AG903_PGPn_CSC1PRM3_M21_MSK) >> AG903_PGPn_CSC1PRM3_M21_POS;
}


/**
 * @brief         PGPnCSC1PRM4WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m23 [in] M23tB[hݒl
 * @return        void
 * @description   PGPnCSC1PRM4WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC1PRM4(uint32_t id, uint32_t m23)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m23 & ~(0xFFF)));

    val = (m23 << AG903_PGPn_CSC1PRM4_M23_POS);

    AG903_PGPn(id)->CSC1PRM4 = val;
}


/**
 * @brief         PGPnCSC1PRM4WX^擾
 * @param         id [in] pCvCԍ
 * @param         m23 [out] M23tB[h擾l
 * @return        void
 * @description   PGPnCSC1PRM4WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC1PRM4(uint32_t id, uint32_t *m23)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m23 != NULL);

    val = AG903_PGPn(id)->CSC1PRM4;

    *m23 = (val & AG903_PGPn_CSC1PRM4_M23_MSK) >> AG903_PGPn_CSC1PRM4_M23_POS;
}


/**
 * @brief         PGPnCSC1PRM5WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m24 [in] M24tB[hݒl
 * @return        void
 * @description   PGPnCSC1PRM5WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC1PRM5(uint32_t id, uint32_t m24)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m24 & ~(0xFFFFF)));

    val = (m24 << AG903_PGPn_CSC1PRM5_M24_POS);

    AG903_PGPn(id)->CSC1PRM5 = val;
}


/**
 * @brief         PGPnCSC1PRM5WX^擾
 * @param         id [in] pCvCԍ
 * @param         m24 [out] M24tB[h擾l
 * @return        void
 * @description   PGPnCSC1PRM5WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC1PRM5(uint32_t id, uint32_t *m24)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m24 != NULL);

    val = AG903_PGPn(id)->CSC1PRM5;

    *m24 = (val & AG903_PGPn_CSC1PRM5_M24_MSK) >> AG903_PGPn_CSC1PRM5_M24_POS;
}


/**
 * @brief         PGPnCSC1PRM6WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m32 [in] M32tB[hݒl
 * @param         m31 [in] M31tB[hݒl
 * @return        void
 * @description   PGPnCSC1PRM6WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC1PRM6(uint32_t id, uint32_t m32, uint32_t m31)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m32 & ~(0xFFF)));
    ASSERT(!(m31 & ~(0xFFF)));

    val = (m32 << AG903_PGPn_CSC1PRM6_M32_POS)
        | (m31 << AG903_PGPn_CSC1PRM6_M31_POS);

    AG903_PGPn(id)->CSC1PRM6 = val;
}


/**
 * @brief         PGPnCSC1PRM6WX^擾
 * @param         id [in] pCvCԍ
 * @param         m32 [out] M32tB[h擾l
 * @param         m31 [out] M31tB[h擾l
 * @return        void
 * @description   PGPnCSC1PRM6WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC1PRM6(uint32_t id, uint32_t *m32, uint32_t *m31)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m32 != NULL);
    ASSERT(m31 != NULL);

    val = AG903_PGPn(id)->CSC1PRM6;

    *m32 = (val & AG903_PGPn_CSC1PRM6_M32_MSK) >> AG903_PGPn_CSC1PRM6_M32_POS;
    *m31 = (val & AG903_PGPn_CSC1PRM6_M31_MSK) >> AG903_PGPn_CSC1PRM6_M31_POS;
}


/**
 * @brief         PGPnCSC1PRM7WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m33 [in] M33tB[hݒl
 * @return        void
 * @description   PGPnCSC1PRM7WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC1PRM7(uint32_t id, uint32_t m33)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m33 & ~(0xFFF)));

    val = (m33 << AG903_PGPn_CSC1PRM7_M33_POS);

    AG903_PGPn(id)->CSC1PRM7 = val;
}


/**
 * @brief         PGPnCSC1PRM7WX^擾
 * @param         id [in] pCvCԍ
 * @param         m33 [out] M33tB[h擾l
 * @return        void
 * @description   PGPnCSC1PRM7WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC1PRM7(uint32_t id, uint32_t *m33)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m33 != NULL);

    val = AG903_PGPn(id)->CSC1PRM7;

    *m33 = (val & AG903_PGPn_CSC1PRM7_M33_MSK) >> AG903_PGPn_CSC1PRM7_M33_POS;
}


/**
 * @brief         PGPnCSC1PRM8WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m34 [in] M34tB[hݒl
 * @return        void
 * @description   PGPnCSC1PRM8WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC1PRM8(uint32_t id, uint32_t m34)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m34 & ~(0xFFFFF)));

    val = (m34 << AG903_PGPn_CSC1PRM8_M34_POS);

    AG903_PGPn(id)->CSC1PRM8 = val;
}


/**
 * @brief         PGPnCSC1PRM8WX^擾
 * @param         id [in] pCvCԍ
 * @param         m34 [out] M34tB[h擾l
 * @return        void
 * @description   PGPnCSC1PRM8WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC1PRM8(uint32_t id, uint32_t *m34)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m34 != NULL);

    val = AG903_PGPn(id)->CSC1PRM8;

    *m34 = (val & AG903_PGPn_CSC1PRM8_M34_MSK) >> AG903_PGPn_CSC1PRM8_M34_POS;
}


/**
 * @brief         PGPnSCCTRLWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         scctrl [in] PGPnSCCTRLWX^ݒli[
 * @return        void
 * @description   PGPnSCCTRLWX^ݒ 
*/
void AG903_PgpPrmSetPGPnSCCTRL(uint32_t id, PGPPrmParamPGPnSCCTRL *scctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(scctrl != NULL);
    ASSERT(!(scctrl->limit & ~(0x1)));
    ASSERT(!(scctrl->mtd   & ~(0x1)));
    ASSERT(!(scctrl->nmrv  & ~(0xF)));
    ASSERT(!(scctrl->dnmv  & ~(0xF)));
    ASSERT(!(scctrl->nmrh  & ~(0xF)));
    ASSERT(!(scctrl->dnmh  & ~(0xF)));

    val = (scctrl->limit << AG903_PGPn_SCCTRL_LIMIT_POS)
        | (scctrl->mtd   << AG903_PGPn_SCCTRL_MTD_POS)
        | (scctrl->nmrv  << AG903_PGPn_SCCTRL_NMRV_POS)
        | (scctrl->dnmv  << AG903_PGPn_SCCTRL_DNMV_POS)
        | (scctrl->nmrh  << AG903_PGPn_SCCTRL_NMRH_POS)
        | (scctrl->dnmh  << AG903_PGPn_SCCTRL_DNMH_POS);

    AG903_PGPn(id)->SCCTRL = val;
}


/**
 * @brief         PGPnSCCTRLWX^擾
 * @param         id [in] pCvCԍ
 * @param         scctrl [out] PGPnSCCTRLWX^擾li[
 * @return        void
 * @description   PGPnSCCTRLWX^擾 
*/
void AG903_PgpPrmGetPGPnSCCTRL(uint32_t id, PGPPrmParamPGPnSCCTRL *scctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(scctrl != NULL);

    val = AG903_PGPn(id)->SCCTRL;

    scctrl->limit = (val & AG903_PGPn_SCCTRL_LIMIT_MSK) >> AG903_PGPn_SCCTRL_LIMIT_POS;
    scctrl->mtd   = (val & AG903_PGPn_SCCTRL_MTD_MSK)   >> AG903_PGPn_SCCTRL_MTD_POS;
    scctrl->nmrv  = (val & AG903_PGPn_SCCTRL_NMRV_MSK)  >> AG903_PGPn_SCCTRL_NMRV_POS;
    scctrl->dnmv  = (val & AG903_PGPn_SCCTRL_DNMV_MSK)  >> AG903_PGPn_SCCTRL_DNMV_POS;
    scctrl->nmrh  = (val & AG903_PGPn_SCCTRL_NMRH_MSK)  >> AG903_PGPn_SCCTRL_NMRH_POS;
    scctrl->dnmh  = (val & AG903_PGPn_SCCTRL_DNMH_MSK)  >> AG903_PGPn_SCCTRL_DNMH_POS;
}


/**
 * @brief         PGPnSPFCTRLWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         spfctrl [in] PGPnSPFCTRLWX^ݒli[
 * @return        void
 * @description   PGPnSPFCTRLWX^ݒ 
*/
void AG903_PgpPrmSetPGPnSPFCTRL(uint32_t id, PGPPrmParamPGPnSPFCTRL *spfctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(spfctrl != NULL);
    ASSERT(!(spfctrl->sft & ~(0xF)));
    ASSERT(!(spfctrl->bc  & ~(0x1)));
    ASSERT(!(spfctrl->r   & ~(0xFF)));
    ASSERT(!(spfctrl->g   & ~(0xFF)));
    ASSERT(!(spfctrl->b   & ~(0xFF)));

    val = (spfctrl->sft << AG903_PGPn_SPFCTRL_SFT_POS)
        | (spfctrl->bc  << AG903_PGPn_SPFCTRL_BC_POS)
        | (spfctrl->r   << AG903_PGPn_SPFCTRL_R_POS)
        | (spfctrl->g   << AG903_PGPn_SPFCTRL_G_POS)
        | (spfctrl->b   << AG903_PGPn_SPFCTRL_B_POS);

    AG903_PGPn(id)->SPFCTRL = val;
}


/**
 * @brief         PGPnSPFCTRLWX^擾
 * @param         id [in] pCvCԍ
 * @param         spfctrl [out] PGPnSPFCTRLWX^擾li[
 * @return        void
 * @description   PGPnSPFCTRLWX^擾
*/
void AG903_PgpPrmGetPGPnSPFCTRL(uint32_t id, PGPPrmParamPGPnSPFCTRL *spfctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(spfctrl != NULL);

    val = AG903_PGPn(id)->SPFCTRL;

    spfctrl->sft = (val & AG903_PGPn_SPFCTRL_SFT_MSK) >> AG903_PGPn_SPFCTRL_SFT_POS;
    spfctrl->bc  = (val & AG903_PGPn_SPFCTRL_BC_MSK)  >> AG903_PGPn_SPFCTRL_BC_POS;
    spfctrl->r   = (val & AG903_PGPn_SPFCTRL_R_MSK)   >> AG903_PGPn_SPFCTRL_R_POS;
    spfctrl->g   = (val & AG903_PGPn_SPFCTRL_G_MSK)   >> AG903_PGPn_SPFCTRL_G_POS;
    spfctrl->b   = (val & AG903_PGPn_SPFCTRL_B_MSK)   >> AG903_PGPn_SPFCTRL_B_POS;
}


/**
 * @brief         PGPnSPFPRM0WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         h1m [in] H1MtB[hݒl
 * @param         h0m [in] H0MtB[hݒl
 * @param         hmm [in] HMMtB[hݒl
 * @return        void
 * @description   PGPnSPFPRM0WX^ݒ 
*/
void AG903_PgpPrmSetPGPnSPFPRM0(uint32_t id, uint32_t h1m, uint32_t h0m, uint32_t hmm)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(h1m & ~(0xFF)));
    ASSERT(!(h0m & ~(0xFF)));
    ASSERT(!(hmm & ~(0xFF)));

    val = (h1m << AG903_PGPn_SPFPRM0_H1M_POS)
        | (h0m << AG903_PGPn_SPFPRM0_H0M_POS)
        | (hmm << AG903_PGPn_SPFPRM0_HMM_POS);

    AG903_PGPn(id)->SPFPRM0 = val;
}


/**
 * @brief         PGPnSPFPRM0WX^擾
 * @param         id [in] pCvCԍ
 * @param         h1m [out] H1MtB[h擾l
 * @param         h0m [out] H0MtB[h擾l
 * @param         hmm [out] HMMtB[h擾l
 * @return        void
 * @description   PGPnSPFPRM0WX^擾 
*/
void AG903_PgpPrmGetPGPnSPFPRM0(uint32_t id, uint32_t *h1m, uint32_t *h0m, uint32_t *hmm)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(h1m != NULL);
    ASSERT(h0m != NULL);
    ASSERT(hmm != NULL);

    val = AG903_PGPn(id)->SPFPRM0;

    *h1m = (val & AG903_PGPn_SPFPRM0_H1M_MSK) >> AG903_PGPn_SPFPRM0_H1M_POS;
    *h0m = (val & AG903_PGPn_SPFPRM0_H0M_MSK) >> AG903_PGPn_SPFPRM0_H0M_POS;
    *hmm = (val & AG903_PGPn_SPFPRM0_HMM_MSK) >> AG903_PGPn_SPFPRM0_HMM_POS;
}


/**
 * @brief         PGPnSPFPRM1WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         h10 [in] H10tB[hݒl
 * @param         h00 [in] H00tB[hݒl
 * @param         hm0 [in] HM0tB[hݒl
 * @return        void
 * @description   PGPnSPFPRM1WX^ݒ 
*/
void AG903_PgpPrmSetPGPnSPFPRM1(uint32_t id, uint32_t h10, uint32_t h00, uint32_t hm0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(h10 & ~(0xFF)));
    ASSERT(!(h00 & ~(0xFF)));
    ASSERT(!(hm0 & ~(0xFF)));

    val = (h10 << AG903_PGPn_SPFPRM1_H10_POS)
        | (h00 << AG903_PGPn_SPFPRM1_H00_POS)
        | (hm0 << AG903_PGPn_SPFPRM1_HM0_POS);

    AG903_PGPn(id)->SPFPRM1 = val;
}


/**
 * @brief         PGPnSPFPRM1WX^擾
 * @param         id [in] pCvCԍ
 * @param         h10 [out] H10tB[h擾l
 * @param         h00 [out] H00tB[h擾l
 * @param         hm0 [out] HM0tB[h擾l
 * @return        void
 * @description   PGPnSPFPRM1WX^擾 
*/
void AG903_PgpPrmGetPGPnSPFPRM1(uint32_t id, uint32_t *h10, uint32_t *h00, uint32_t *hm0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(h10 != NULL);
    ASSERT(h00 != NULL);
    ASSERT(hm0 != NULL);

    val = AG903_PGPn(id)->SPFPRM1;

    *h10 = (val & AG903_PGPn_SPFPRM1_H10_MSK) >> AG903_PGPn_SPFPRM1_H10_POS;
    *h00 = (val & AG903_PGPn_SPFPRM1_H00_MSK) >> AG903_PGPn_SPFPRM1_H00_POS;
    *hm0 = (val & AG903_PGPn_SPFPRM1_HM0_MSK) >> AG903_PGPn_SPFPRM1_HM0_POS;
}


/**
 * @brief         PGPnSPFPRM2WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         h11 [in] H11tB[hݒl
 * @param         h01 [in] H01tB[hݒl
 * @param         hm1 [in] HM1tB[hݒl
 * @return        void
 * @description   PGPnSPFPRM2WX^ݒ 
*/
void AG903_PgpPrmSetPGPnSPFPRM2(uint32_t id, uint32_t h11, uint32_t h01, uint32_t hm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(h11 & ~(0xFF)));
    ASSERT(!(h01 & ~(0xFF)));
    ASSERT(!(hm1 & ~(0xFF)));

    val = (h11 << AG903_PGPn_SPFPRM2_H11_POS)
        | (h01 << AG903_PGPn_SPFPRM2_H01_POS)
        | (hm1 << AG903_PGPn_SPFPRM2_HM1_POS);

    AG903_PGPn(id)->SPFPRM2 = val;
}


/**
 * @brief         PGPnSPFPRM2WX^擾
 * @param         id [in] pCvCԍ
 * @param         h11 [out] H11tB[h擾l
 * @param         h01 [out] H01tB[h擾l
 * @param         hm1 [out] HM1tB[h擾l
 * @return        void
 * @description   PGPnSPFPRM2WX^擾 
*/
void AG903_PgpPrmGetPGPnSPFPRM2(uint32_t id, uint32_t *h11, uint32_t *h01, uint32_t *hm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(h11 != NULL);
    ASSERT(h01 != NULL);
    ASSERT(hm1 != NULL);

    val = AG903_PGPn(id)->SPFPRM2;

    *h11 = (val & AG903_PGPn_SPFPRM2_H11_MSK) >> AG903_PGPn_SPFPRM2_H11_POS;
    *h01 = (val & AG903_PGPn_SPFPRM2_H01_MSK) >> AG903_PGPn_SPFPRM2_H01_POS;
    *hm1 = (val & AG903_PGPn_SPFPRM2_HM1_MSK) >> AG903_PGPn_SPFPRM2_HM1_POS;
}


/**
 * @brief         PGPnSPFPRM3WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         ofs [in] OFStB[hݒl
 * @return        void
 * @description   PGPnSPFPRM3WX^ݒ 
*/
void AG903_PgpPrmSetPGPnSPFPRM3(uint32_t id, uint32_t ofs)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(ofs & ~(0x1FF)));

    val = (ofs << AG903_PGPn_SPFPRM3_OFS_POS);

    AG903_PGPn(id)->SPFPRM3 = val;
}


/**
 * @brief         PGPnSPFPRM3WX^擾
 * @param         id [in] pCvCԍ
 * @param         ofs [out] OFStB[h擾l
 * @return        void
 * @description   PGPnSPFPRM3WX^擾 
*/
void AG903_PgpPrmGetPGPnSPFPRM3(uint32_t id, uint32_t *ofs)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(ofs != NULL);

    val = AG903_PGPn(id)->SPFPRM3;

    *ofs = (val & AG903_PGPn_SPFPRM3_OFS_MSK) >> AG903_PGPn_SPFPRM3_OFS_POS;
}


/**
 * @brief         PGPnHSV1CTRLWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         fmt [in] FMTtB[hݒl
 * @return        void
 * @description   PGPnHSV1CTRLWX^ݒ 
*/
void AG903_PgpPrmSetPGPnHSV1CTRL(uint32_t id, uint32_t fmt)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(fmt & ~(0x1)));

    val = (fmt << AG903_PGPn_HSV1CTRL_FMT_POS);

    AG903_PGPn(id)->HSV1CTRL = val;
}


/**
 * @brief         PGPnHSV1CTRLWX^擾
 * @param         id [in] pCvCԍ
 * @param         fmt [out] FMTtB[h擾l
 * @return        void
 * @description   PGPnHSV1CTRLWX^擾 
*/
void AG903_PgpPrmGetPGPnHSV1CTRL(uint32_t id, uint32_t *fmt)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(fmt != NULL);

    val = AG903_PGPn(id)->HSV1CTRL;

    *fmt = (val & AG903_PGPn_HSV1CTRL_FMT_MSK) >> AG903_PGPn_HSV1CTRL_FMT_POS;
}


/**
 * @brief         PGPnDGCSTATLWX^擾
 * @param         id [in] pCvCԍ
 * @param         queue [out] QUEUEtB[h擾l
 * @return        void
 * @description   PGPnDGCSTATLWX^擾 
*/
void AG903_PgpPrmGetPGPnDGCSTAT(uint32_t id, uint32_t *queue)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(queue != NULL);

    val = AG903_PGPn(id)->DGCSTAT;

    *queue = (val & AG903_PGPn_DGCSTAT_QUEUE_MSK) >> AG903_PGPn_DGCSTAT_QUEUE_POS;
}


/**
 * @brief         PGPnHGMCMDWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         cmd [in] CMDtB[hݒl
 * @return        void
 * @description   PGPnHGMCMDWX^ݒ 
*/
void AG903_PgpPrmSetPGPnHGMCMD(uint32_t id, uint32_t cmd)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(cmd & ~(0x7)));

    val = (cmd << AG903_PGPn_HGMCMD_CMD_POS);

    AG903_PGPn(id)->HGMCMD = val;
}


/**
 * @brief         PGPnHGMCTRLWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         hgmctrl [in] PGPnHGMCTRLWX^ݒli[
 * @return        void
 * @description   PGPnHGMCTRLWX^ݒ 
*/
void AG903_PgpPrmSetPGPnHGMCTRL(uint32_t id, PGPPrmParamPGPnHGMCTRL *hgmctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(hgmctrl != NULL);
    ASSERT(!(hgmctrl->inv  & ~(0xF)));
    ASSERT(!(hgmctrl->bmu  & ~(0x1)));
    ASSERT(!(hgmctrl->exc  & ~(0x3)));
    ASSERT(!(hgmctrl->thr  & ~(0xF)));
    ASSERT(!(hgmctrl->sft  & ~(0xF)));
    ASSERT(!(hgmctrl->vinc & ~(0xF)));
    ASSERT(!(hgmctrl->hinc & ~(0xF)));

    val = (hgmctrl->inv  << AG903_PGPn_HGMCTRL_INV_POS)
        | (hgmctrl->bmu  << AG903_PGPn_HGMCTRL_BMU_POS)
        | (hgmctrl->exc  << AG903_PGPn_HGMCTRL_EXC_POS)
        | (hgmctrl->thr  << AG903_PGPn_HGMCTRL_THR_POS)
        | (hgmctrl->sft  << AG903_PGPn_HGMCTRL_SFT_POS)
        | (hgmctrl->vinc << AG903_PGPn_HGMCTRL_VINC_POS)
        | (hgmctrl->hinc << AG903_PGPn_HGMCTRL_HINC_POS);

    AG903_PGPn(id)->HGMCTRL = val;
}


/**
 * @brief         PGPnHGMCTRLWX^擾
 * @param         id [in] pCvCԍ
 * @param         hgmctrl [out] PGPnHGMCTRLWX^擾li[
 * @return        void
 * @description   PGPnHGMCTRLWX^擾 
*/
void AG903_PgpPrmGetPGPnHGMCTRL(uint32_t id, PGPPrmParamPGPnHGMCTRL *hgmctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(hgmctrl != NULL);

    val = AG903_PGPn(id)->HGMCTRL;

    hgmctrl->inv  = (val & AG903_PGPn_HGMCTRL_INV_MSK)  >> AG903_PGPn_HGMCTRL_INV_POS;
    hgmctrl->bmu  = (val & AG903_PGPn_HGMCTRL_BMU_MSK)  >> AG903_PGPn_HGMCTRL_BMU_POS;
    hgmctrl->exc  = (val & AG903_PGPn_HGMCTRL_EXC_MSK)  >> AG903_PGPn_HGMCTRL_EXC_POS;
    hgmctrl->thr  = (val & AG903_PGPn_HGMCTRL_THR_MSK)  >> AG903_PGPn_HGMCTRL_THR_POS;
    hgmctrl->sft  = (val & AG903_PGPn_HGMCTRL_SFT_MSK)  >> AG903_PGPn_HGMCTRL_SFT_POS;
    hgmctrl->vinc = (val & AG903_PGPn_HGMCTRL_VINC_MSK) >> AG903_PGPn_HGMCTRL_VINC_POS;
    hgmctrl->hinc = (val & AG903_PGPn_HGMCTRL_HINC_MSK) >> AG903_PGPn_HGMCTRL_HINC_POS;
}


/**
 * @brief         PGPnHGMSIZEWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         vsize [in] VSIZEtB[hݒl
 * @param         hsize [in] HSIZEtB[hݒl
 * @return        void
 * @description   PGPnHGMSIZEWX^ݒ 
*/
void AG903_PgpPrmSetPGPnHGMSIZE(uint32_t id, uint32_t vsize, uint32_t hsize)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(vsize & ~(0x3FF)));
    ASSERT(!(hsize & ~(0x3FF)));

    val = (vsize << AG903_PGPn_HGMSIZE_VSIZE_POS)
        | (hsize << AG903_PGPn_HGMSIZE_HSIZE_POS);

    AG903_PGPn(id)->HGMSIZE = val;
}


/**
 * @brief         PGPnHGMSIZEWX^擾
 * @param         id [in] pCvCԍ
 * @param         vsize [out] VSIZEtB[h擾l
 * @param         hsize [out] HSIZEtB[h擾l
 * @return        void
 * @description   PGPnHGMSIZEWX^擾 
*/
void AG903_PgpPrmGetPGPnHGMSIZE(uint32_t id, uint32_t *vsize, uint32_t *hsize)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(vsize != NULL);
    ASSERT(hsize != NULL);

    val = AG903_PGPn(id)->HGMSIZE;

    *vsize = (val & AG903_PGPn_HGMSIZE_VSIZE_MSK) >> AG903_PGPn_HGMSIZE_VSIZE_POS;
    *hsize = (val & AG903_PGPn_HGMSIZE_HSIZE_MSK) >> AG903_PGPn_HGMSIZE_HSIZE_POS;
}


/**
 * @brief         PGPnHGMPOSWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         vpos [in] VPOStB[hݒl
 * @param         hpos [in] HPOStB[hݒl
 * @return        void
 * @description   PGPnHGMPOSWX^ݒ 
*/
void AG903_PgpPrmSetPGPnHGMPOS(uint32_t id, uint32_t vpos, uint32_t hpos)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(vpos & ~(0xFFF)));
    ASSERT(!(hpos & ~(0xFFF)));

    val = (vpos << AG903_PGPn_HGMPOS_VPOS_POS)
        | (hpos << AG903_PGPn_HGMPOS_HPOS_POS);

    AG903_PGPn(id)->HGMPOS = val;
}


/**
 * @brief         PGPnHGMPOSWX^擾
 * @param         id [in] pCvCԍ
 * @param         vpos [out] VPOStB[h擾l
 * @param         hpos [out] HPOStB[h擾l
 * @return        void
 * @description   PGPnHGMPOSWX^擾 
*/
void AG903_PgpPrmGetPGPnHGMPOS(uint32_t id, uint32_t *vpos, uint32_t *hpos)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(vpos != NULL);
    ASSERT(hpos != NULL);

    val = AG903_PGPn(id)->HGMPOS;

    *vpos = (val & AG903_PGPn_HGMPOS_VPOS_MSK) >> AG903_PGPn_HGMPOS_VPOS_POS;
    *hpos = (val & AG903_PGPn_HGMPOS_HPOS_MSK) >> AG903_PGPn_HGMPOS_HPOS_POS;
}


/**
 * @brief         PGPnHGMDSTWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         adr [in] ADRtB[hݒl
 * @return        void
 * @description   PGPnHGMDSTWX^ݒ 
*/
void AG903_PgpPrmSetPGPnHGMDST(uint32_t id, uint32_t adr)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(adr & ~(0x1FFFFFFF)));

    val = (adr << AG903_PGPn_HGMDST_ADR_POS);

    AG903_PGPn(id)->HGMDST = val;
}


/**
 * @brief         PGPnHGMDSTWX^擾
 * @param         id [in] pCvCԍ
 * @param         adr [out] ADRtB[h擾l
 * @return        void
 * @description   PGPnHGMDSTWX^擾 
*/
void AG903_PgpPrmGetPGPnHGMDST(uint32_t id, uint32_t *adr)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(adr != NULL);

    val = AG903_PGPn(id)->HGMDST;

    *adr = (val & AG903_PGPn_HGMDST_ADR_MSK) >> AG903_PGPn_HGMDST_ADR_POS;
}


/**
 * @brief         PGPnHGMMOD0WX^擾
 * @param         id [in] pCvCԍ
 * @param         bw [out] BWtB[h擾l
 * @param         val [out] VALtB[h擾l
 * @param         cnt [out] CNTtB[h擾l
 * @return        void
 * @description   PGPnHGMOD0WX^擾 
*/
void AG903_PgpPrmGetPGPnHGMMOD0(uint32_t id, uint32_t *bw, uint32_t *val, uint32_t *cnt)
{
    uint32_t                    tmp;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(bw != NULL);
    ASSERT(val != NULL);
    ASSERT(cnt != NULL);

    tmp = AG903_PGPn(id)->HGMMOD0;

    *bw  = (tmp & AG903_PGPn_HGMMOD0_BW_MSK) >> AG903_PGPn_HGMMOD0_BW_POS;
    *val = (tmp & AG903_PGPn_HGMMOD0_VAL_MSK) >> AG903_PGPn_HGMMOD0_VAL_POS;
    *cnt = (tmp & AG903_PGPn_HGMMOD0_CNT_MSK) >> AG903_PGPn_HGMMOD0_CNT_POS;
}


/**
 * @brief         PGPnHGMRNG0WX^擾
 * @param         id [in] pCvCԍ
 * @param         maxval [out] MAXVALtB[h擾l
 * @param         minval [out] MINVALtB[h擾l
 * @return        void
 * @description   PGPnHGMRNG0WX^擾 
*/
void AG903_PgpPrmGetPGPnHGMRNG0(uint32_t id, uint32_t *maxval, uint32_t *minval)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(maxval != NULL);
    ASSERT(minval != NULL);

    val = AG903_PGPn(id)->HGMRNG0;

    *maxval = (val & AG903_PGPn_HGMRNG0_MAXVAL_MSK) >> AG903_PGPn_HGMRNG0_MAXVAL_POS;
    *minval = (val & AG903_PGPn_HGMRNG0_MINVAL_MSK) >> AG903_PGPn_HGMRNG0_MINVAL_POS;
}


/**
 * @brief         PGPnHGMMOD1WX^擾
 * @param         id [in] pCvCԍ
 * @param         bw [out] BWtB[h擾l
 * @param         val [out] VALtB[h擾l
 * @param         cnt [out] CNTtB[h擾l
 * @return        void
 * @description   PGPnHGMMOD1WX^擾 
*/
void AG903_PgpPrmGetPGPnHGMMOD1(uint32_t id, uint32_t *bw, uint32_t *val, uint32_t *cnt)
{
    uint32_t                    dat;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(bw != NULL);
    ASSERT(val != NULL);
    ASSERT(cnt != NULL);

    dat = AG903_PGPn(id)->HGMMOD1;

    *bw  = (dat & AG903_PGPn_HGMMOD1_BW_MSK) >> AG903_PGPn_HGMMOD1_BW_POS;
    *val = (dat & AG903_PGPn_HGMMOD1_VAL_MSK) >> AG903_PGPn_HGMMOD1_VAL_POS;
    *cnt = (dat & AG903_PGPn_HGMMOD1_CNT_MSK) >> AG903_PGPn_HGMMOD1_CNT_POS;
}


/**
 * @brief         PGPnHGMRNG1WX^擾
 * @param         id [in] pCvCԍ
 * @param         maxval [out] MAXVALtB[h擾l
 * @param         minval [out] MINVALtB[h擾l
 * @return        void
 * @description   PGPnHGMRNG1WX^擾 
*/
void AG903_PgpPrmGetPGPnHGMRNG1(uint32_t id, uint32_t *maxval, uint32_t *minval)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(maxval != NULL);
    ASSERT(minval != NULL);

    val = AG903_PGPn(id)->HGMRNG1;

    *maxval = (val & AG903_PGPn_HGMRNG1_MAXVAL_MSK) >> AG903_PGPn_HGMRNG1_MAXVAL_POS;
    *minval = (val & AG903_PGPn_HGMRNG1_MINVAL_MSK) >> AG903_PGPn_HGMRNG1_MINVAL_POS;
}


/**
 * @brief         PGPnHGMMOD2WX^擾
 * @param         id [in] pCvCԍ
 * @param         bw [out] BWtB[h擾l
 * @param         val [out] VALtB[h擾l
 * @param         cnt [out] CNTtB[h擾l
 * @return        void
 * @description   PGPnHGMMOD2WX^擾 
*/
void AG903_PgpPrmGetPGPnHGMMOD2(uint32_t id, uint32_t *bw, uint32_t *val, uint32_t *cnt)
{
    uint32_t                    dat;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(bw != NULL);
    ASSERT(val != NULL);
    ASSERT(cnt != NULL);

    dat = AG903_PGPn(id)->HGMMOD2;

    *bw  = (dat & AG903_PGPn_HGMMOD2_BW_MSK) >> AG903_PGPn_HGMMOD2_BW_POS;
    *val = (dat & AG903_PGPn_HGMMOD2_VAL_MSK) >> AG903_PGPn_HGMMOD2_VAL_POS;
    *cnt = (dat & AG903_PGPn_HGMMOD2_CNT_MSK) >> AG903_PGPn_HGMMOD2_CNT_POS;
}


/**
 * @brief         PGPnHGMRNG2WX^擾
 * @param         id [in] pCvCԍ
 * @param         maxval [out] MAXVALtB[h擾l
 * @param         minval [out] MINVALtB[h擾l
 * @return        void
 * @description   PGPnHGMRNG2WX^擾 
*/
void AG903_PgpPrmGetPGPnHGMRNG2(uint32_t id, uint32_t *maxval, uint32_t *minval)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(maxval != NULL);
    ASSERT(minval != NULL);

    val = AG903_PGPn(id)->HGMRNG2;

    *maxval = (val & AG903_PGPn_HGMRNG2_MAXVAL_MSK) >> AG903_PGPn_HGMRNG2_MAXVAL_POS;
    *minval = (val & AG903_PGPn_HGMRNG2_MINVAL_MSK) >> AG903_PGPn_HGMRNG2_MINVAL_POS;
}


/**
 * @brief         PGPnHGMSTATWX^擾
 * @param         id [in] pCvCԍ
 * @param         stat [out] PGPnHGMSTATWX^擾li[
 * @return        void
 * @description   PGPnHGMSTATWX^擾 
*/
void AG903_PgpPrmGetPGPnHGMSTAT(uint32_t id, PGPPrmParamPGPnHGMSTAT *stat)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(stat != NULL);

    val = AG903_PGPn(id)->HGMSTAT;

    stat->init   = (val & AG903_PGPn_HGMSTAT_INIT_MSK)   >> AG903_PGPn_HGMSTAT_INIT_POS;
    stat->vrmacc = (val & AG903_PGPn_HGMSTAT_VRMACC_MSK) >> AG903_PGPn_HGMSTAT_VRMACC_POS;
    stat->act    = (val & AG903_PGPn_HGMSTAT_ACT_MSK)    >> AG903_PGPn_HGMSTAT_ACT_POS;
    stat->queue  = (val & AG903_PGPn_HGMSTAT_QUEUE_MSK)  >> AG903_PGPn_HGMSTAT_QUEUE_POS;
    stat->cmd    = (val & AG903_PGPn_HGMSTAT_CMD_MSK)    >> AG903_PGPn_HGMSTAT_CMD_POS;
}


/**
 * @brief         PGPnTHRCTRLWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         thrctrl [in] PGPnTHRCTRLWX^ݒli[
 * @return        void
 * @description   PGPnTHRCTRLWX^ݒ 
*/
void AG903_PgpPrmSetPGPnTHRCTRL(uint32_t id, PGPPrmParamPGPnTHRCTRL *thrctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(thrctrl != NULL);
    ASSERT(!(thrctrl->limit  & ~(0x1)));
    ASSERT(!(thrctrl->spf1   & ~(0x1)));
    ASSERT(!(thrctrl->msk    & ~(0x1)));
    ASSERT(!(thrctrl->csc2   & ~(0x1)));
    ASSERT(!(thrctrl->hsv2   & ~(0x1)));
    ASSERT(!(thrctrl->thr8   & ~(0x1)));
    ASSERT(!(thrctrl->outsel & ~(0x3)));
    ASSERT(!(thrctrl->c8sel  & ~(0x3)));

    val = (thrctrl->limit  << AG903_PGPn_THRCTRL_LIMIT_POS)
        | (thrctrl->spf1   << AG903_PGPn_THRCTRL_SPF1_POS)
        | (thrctrl->msk    << AG903_PGPn_THRCTRL_MSK_POS)
        | (thrctrl->csc2   << AG903_PGPn_THRCTRL_CSC2_POS)
        | (thrctrl->hsv2   << AG903_PGPn_THRCTRL_HSV2_POS)
        | (thrctrl->thr8   << AG903_PGPn_THRCTRL_THR8_POS)
        | (thrctrl->outsel << AG903_PGPn_THRCTRL_OUTSEL_POS)
        | (thrctrl->c8sel  << AG903_PGPn_THRCTRL_C8SEL_POS);

    AG903_PGPn(id)->THRCTRL = val;
}


/**
 * @brief         PGPnTHRCTRLWX^擾
 * @param         id [in] pCvCԍ
 * @param         thrctrl [out] PGPnTHRCTRLWX^擾li[
 * @return        void
 * @description   PGPnTHRCTRLWX^擾 
*/
void AG903_PgpPrmGetPGPnTHRCTRL(uint32_t id, PGPPrmParamPGPnTHRCTRL *thrctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(thrctrl != NULL);

    val = AG903_PGPn(id)->THRCTRL;

    thrctrl->limit  = (val & AG903_PGPn_THRCTRL_LIMIT_MSK)  >> AG903_PGPn_THRCTRL_LIMIT_POS;
    thrctrl->spf1   = (val & AG903_PGPn_THRCTRL_SPF1_MSK)   >> AG903_PGPn_THRCTRL_SPF1_POS;
    thrctrl->msk    = (val & AG903_PGPn_THRCTRL_MSK_MSK)    >> AG903_PGPn_THRCTRL_MSK_POS;
    thrctrl->csc2   = (val & AG903_PGPn_THRCTRL_CSC2_MSK)   >> AG903_PGPn_THRCTRL_CSC2_POS;
    thrctrl->hsv2   = (val & AG903_PGPn_THRCTRL_HSV2_MSK)   >> AG903_PGPn_THRCTRL_HSV2_POS;
    thrctrl->thr8   = (val & AG903_PGPn_THRCTRL_THR8_MSK)   >> AG903_PGPn_THRCTRL_THR8_POS;
    thrctrl->outsel = (val & AG903_PGPn_THRCTRL_OUTSEL_MSK) >> AG903_PGPn_THRCTRL_OUTSEL_POS;
    thrctrl->c8sel  = (val & AG903_PGPn_THRCTRL_C8SEL_MSK)  >> AG903_PGPn_THRCTRL_C8SEL_POS;
}


/**
 * @brief         PGPnHSV2CTRLWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         fmt [in] FMTtB[hݒl
 * @return        void
 * @description   PGPnHSV2CTRLWX^ݒ 
*/
void AG903_PgpPrmSetPGPnHSV2CTRL(uint32_t id, uint32_t fmt)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(fmt & ~(0x1)));

    val = (fmt << AG903_PGPn_HSV2CTRL_FMT_POS);

    AG903_PGPn(id)->HSV2CTRL = val;
}


/**
 * @brief         PGPnHSV2CTRLWX^擾
 * @param         id [in] pCvCԍ
 * @param         fmt [out] FMTtB[h擾l
 * @return        void
 * @description   PGPnHSV2CTRLWX^擾 
*/
void AG903_PgpPrmGetPGPnHSV2CTRL(uint32_t id, uint32_t *fmt)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(fmt != NULL);

    val = AG903_PGPn(id)->HSV2CTRL;

    *fmt = (val & AG903_PGPn_HSV2CTRL_FMT_MSK) >> AG903_PGPn_HSV2CTRL_FMT_POS;
}


/**
 * @brief         PGPnCSC2CTRLWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         limit [in] LIMITtB[hݒl
 * @param         sft [in] SFTtB[hݒl
 * @return        void
 * @description   PGPnCSC2CTRLWX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC2CTRL(uint32_t id, uint32_t limit, uint32_t sft)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(limit & ~(0x1)));
    ASSERT(!(sft & ~(0xF)));

    val = (limit << AG903_PGPn_CSC2CTRL_LIMIT_POS)
        | (sft << AG903_PGPn_CSC2CTRL_SFT_POS);

    AG903_PGPn(id)->CSC2CTRL = val;
}


/**
 * @brief         PGPnCSC2CTRLWX^擾
 * @param         id [in] pCvCԍ
 * @param         limit [out] LIMITtB[h擾l
 * @param         sft [out] SFTtB[h擾l
 * @return        void
 * @description   PGPnCSC2CTRLWX^擾 
*/
void AG903_PgpPrmGetPGPnCSC2CTRL(uint32_t id, uint32_t *limit, uint32_t *sft)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(limit != NULL);
    ASSERT(sft != NULL);

    val = AG903_PGPn(id)->CSC2CTRL;

    *limit = (val & AG903_PGPn_CSC2CTRL_LIMIT_MSK) >> AG903_PGPn_CSC2CTRL_LIMIT_POS;
    *sft   = (val & AG903_PGPn_CSC2CTRL_SFT_MSK) >> AG903_PGPn_CSC2CTRL_SFT_POS;
}


/**
 * @brief         PGPnCSC2PRM0WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m12 [in] M12tB[hݒl
 * @param         m11 [in] M11tB[hݒl
 * @return        void
 * @description   PGPnCSC2PRM0WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC2PRM0(uint32_t id, uint32_t m12, uint32_t m11)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m12 & ~(0xFFF)));
    ASSERT(!(m11 & ~(0xFFF)));

    val = (m12 << AG903_PGPn_CSC2PRM0_M12_POS)
        | (m11 << AG903_PGPn_CSC2PRM0_M11_POS);

    AG903_PGPn(id)->CSC2PRM0 = val;
}


/**
 * @brief         PGPnCSC2PRM0WX^擾
 * @param         id [in] pCvCԍ
 * @param         m12 [out] M12tB[h擾l
 * @param         m11 [out] M11tB[h擾l
 * @return        void
 * @description   PGPnCSC2PRM0WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC2PRM0(uint32_t id, uint32_t *m12, uint32_t *m11)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m12 != NULL);
    ASSERT(m11 != NULL);

    val = AG903_PGPn(id)->CSC2PRM0;

    *m12 = (val & AG903_PGPn_CSC2PRM0_M12_MSK) >> AG903_PGPn_CSC2PRM0_M12_POS;
    *m11 = (val & AG903_PGPn_CSC2PRM0_M11_MSK) >> AG903_PGPn_CSC2PRM0_M11_POS;
}


/**
 * @brief         PGPnCSC2PRM1WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m13 [in] M13tB[hݒl
 * @return        void
 * @description   PGPnCSC2PRM1WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC2PRM1(uint32_t id, uint32_t m13)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m13 & ~(0xFFF)));

    val = (m13 << AG903_PGPn_CSC2PRM1_M13_POS);

    AG903_PGPn(id)->CSC2PRM1 = val;
}


/**
 * @brief         PGPnCSC2PRM1WX^擾
 * @param         id [in] pCvCԍ
 * @param         m13 [out] M13tB[h擾l
 * @return        void
 * @description   PGPnCSC2PRM1WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC2PRM1(uint32_t id, uint32_t *m13)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m13 != NULL);

    val = AG903_PGPn(id)->CSC2PRM1;

    *m13 = (val & AG903_PGPn_CSC2PRM1_M13_MSK) >> AG903_PGPn_CSC2PRM1_M13_POS;
}


/**
 * @brief         PGPnCSC2PRM2WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m14 [in] M14tB[hݒl
 * @return        void
 * @description   PGPnCSC2PRM2WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC2PRM2(uint32_t id, uint32_t m14)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m14 & ~(0xFFFFF)));

    val = (m14 << AG903_PGPn_CSC2PRM2_M14_POS);

    AG903_PGPn(id)->CSC2PRM2 = val;
}


/**
 * @brief         PGPnCSC2PRM2WX^擾
 * @param         id [in] pCvCԍ
 * @param         m14 [out] M14tB[h擾l
 * @return        void
 * @description   PGPnCMDWX^擾 
*/
void AG903_PgpPrmGetPGPnCSC2PRM2(uint32_t id, uint32_t *m14)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m14 != NULL);

    val = AG903_PGPn(id)->CSC2PRM2;

    *m14 = (val & AG903_PGPn_CSC2PRM2_M14_MSK) >> AG903_PGPn_CSC2PRM2_M14_POS;
}


/**
 * @brief         PGPnCSC2PRM3WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m22 [in] M22tB[hݒl
 * @param         m21 [in] M21tB[hݒl
 * @return        void
 * @description   PGPnCSC2PRM3WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC2PRM3(uint32_t id, uint32_t m22, uint32_t m21)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m22 & ~(0xFFF)));
    ASSERT(!(m21 & ~(0xFFF)));

    val = (m22 << AG903_PGPn_CSC2PRM3_M22_POS)
        | (m21 << AG903_PGPn_CSC2PRM3_M21_POS);

    AG903_PGPn(id)->CSC2PRM3 = val;
}


/**
 * @brief         PGPnCSC2PRM3WX^擾
 * @param         id [in] pCvCԍ
 * @param         m22 [out] M22tB[h擾l
 * @param         m21 [out] M21tB[h擾l
 * @return        void
 * @description   PGPnCSC2PRM3WX^ݒ 
*/
void AG903_PgpPrmGetPGPnCSC2PRM3(uint32_t id, uint32_t *m22, uint32_t *m21)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m22 != NULL);
    ASSERT(m21 != NULL);

    val = AG903_PGPn(id)->CSC2PRM3;

    *m22 = (val & AG903_PGPn_CSC2PRM3_M22_MSK) >> AG903_PGPn_CSC2PRM3_M22_POS;
    *m21 = (val & AG903_PGPn_CSC2PRM3_M21_MSK) >> AG903_PGPn_CSC2PRM3_M21_POS;
}


/**
 * @brief         PGPnCSC2PRM4WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m23 [in] M23tB[hݒl
 * @return        void
 * @description   PGPnCSC2PRM4WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC2PRM4(uint32_t id, uint32_t m23)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m23 & ~(0xFFF)));

    val = (m23 << AG903_PGPn_CSC2PRM4_M23_POS);

    AG903_PGPn(id)->CSC2PRM4 = val;
}


/**
 * @brief         PGPnCSC2PRM4WX^擾
 * @param         id [in] pCvCԍ
 * @param         m23 [out] M23tB[h擾l
 * @return        void
 * @description   PGPnCSC2PRM4WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC2PRM4(uint32_t id, uint32_t *m23)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m23 != NULL);

    val = AG903_PGPn(id)->CSC2PRM4;

    *m23 = (val & AG903_PGPn_CSC2PRM4_M23_MSK) >> AG903_PGPn_CSC2PRM4_M23_POS;
}


/**
 * @brief         PGPnCSC2PRM5WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m24 [in] M24tB[hݒl
 * @return        void
 * @description   PGPnCSC2PRM5WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC2PRM5(uint32_t id, uint32_t m24)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m24 & ~(0xFFFFF)));

    val = (m24 << AG903_PGPn_CSC2PRM5_M24_POS);

    AG903_PGPn(id)->CSC2PRM5 = val;
}


/**
 * @brief         PGPnCSC2PRM5WX^擾
 * @param         id [in] pCvCԍ
 * @param         m24 [out] M24tB[h擾l
 * @return        void
 * @description   PGPnCSC2PRM5WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC2PRM5(uint32_t id, uint32_t *m24)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m24 != NULL);

    val = AG903_PGPn(id)->CSC2PRM5;

    *m24 = (val & AG903_PGPn_CSC2PRM5_M24_MSK) >> AG903_PGPn_CSC2PRM5_M24_POS;
}


/**
 * @brief         PGPnCSC2PRM6WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m32 [in] M32tB[hݒl
 * @param         m31 [in] M31tB[hݒl
 * @return        void
 * @description   PGPnCSC2PRM6WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC2PRM6(uint32_t id, uint32_t m32, uint32_t m31)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m32 & ~(0xFFF)));
    ASSERT(!(m31 & ~(0xFFF)));

    val = (m32 << AG903_PGPn_CSC2PRM6_M32_POS)
        | (m31 << AG903_PGPn_CSC2PRM6_M31_POS);

    AG903_PGPn(id)->CSC2PRM6 = val;
}


/**
 * @brief         PGPnCSC2PRM6WX^擾
 * @param         id [in] pCvCԍ
 * @param         m32 [out] M32tB[h擾l
 * @param         m31 [out] M31tB[h擾l
 * @return        void
 * @description   PGPnCSC2PRM6WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC2PRM6(uint32_t id, uint32_t *m32, uint32_t *m31)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m32 != NULL);
    ASSERT(m31 != NULL);

    val = AG903_PGPn(id)->CSC2PRM6;

    *m32 = (val & AG903_PGPn_CSC2PRM6_M32_MSK) >> AG903_PGPn_CSC2PRM6_M32_POS;
    *m31 = (val & AG903_PGPn_CSC2PRM6_M31_MSK) >> AG903_PGPn_CSC2PRM6_M31_POS;
}


/**
 * @brief         PGPnCSC2PRM7WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m33 [in] M33tB[hݒl
 * @return        void
 * @description   PGPnCSC2PRM7WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC2PRM7(uint32_t id, uint32_t m33)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m33 & ~(0xFFF)));

    val = (m33 << AG903_PGPn_CSC2PRM7_M33_POS);

    AG903_PGPn(id)->CSC2PRM7 = val;
}


/**
 * @brief         PGPnCSC2PRM7WX^擾
 * @param         id [in] pCvCԍ
 * @param         m33 [out] M33tB[h擾l
 * @return        void
 * @description   PGPnCSC2PRM7WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC2PRM7(uint32_t id, uint32_t *m33)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m33 != NULL);

    val = AG903_PGPn(id)->CSC2PRM7;

    *m33 = (val & AG903_PGPn_CSC2PRM7_M33_MSK) >> AG903_PGPn_CSC2PRM7_M33_POS;
}


/**
 * @brief         PGPnCSC2PRM8WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         m34 [in] M34tB[hݒl
 * @return        void
 * @description   PGPnCSC2PRM8WX^ݒ 
*/
void AG903_PgpPrmSetPGPnCSC2PRM8(uint32_t id, uint32_t m34)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(m34 & ~(0xFFFFF)));

    val = (m34 << AG903_PGPn_CSC2PRM8_M34_POS);

    AG903_PGPn(id)->CSC2PRM8 = val;
}


/**
 * @brief         PGPnCSC2PRM8WX^擾
 * @param         id [in] pCvCԍ
 * @param         m34 [out] M34tB[h擾l
 * @return        void
 * @description   PGPnCSC2PRM8WX^擾 
*/
void AG903_PgpPrmGetPGPnCSC2PRM8(uint32_t id, uint32_t *m34)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(m34 != NULL);

    val = AG903_PGPn(id)->CSC2PRM8;

    *m34 = (val & AG903_PGPn_CSC2PRM8_M34_MSK) >> AG903_PGPn_CSC2PRM8_M34_POS;
}


/**
 * @brief         PGPnMSK0PRM0WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         h [in] HtB[hݒl
 * @param         l [in] LtB[hݒl
 * @param         x [in] XtB[hݒl
 * @return        void
 * @description   PGPnMSK0PRM0WX^ݒ 
*/
void AG903_PgpPrmSetPGPnMSK0PRM0(uint32_t id, uint32_t h, uint32_t l, uint32_t x)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(h & ~(0x1)));
    ASSERT(!(l & ~(0x1)));
    ASSERT(!(x & ~(0x1)));

    val = (h << AG903_PGPn_MSK0PRM0_H_POS)
        | (l << AG903_PGPn_MSK0PRM0_L_POS)
        | (x << AG903_PGPn_MSK0PRM0_X_POS);

    AG903_PGPn(id)->MSK0PRM0 = val;
}


/**
 * @brief         PGPnMSK0PRM0WX^擾
 * @param         id [in] pCvCԍ
 * @param         h [out] HtB[h擾l
 * @param         l [out] LtB[h擾l
 * @param         x [out] XtB[h擾l
 * @return        void
 * @description   PGPnMSK0PRM0WX^擾 
*/
void AG903_PgpPrmGetPGPnMSK0PRM0(uint32_t id, uint32_t *h, uint32_t *l, uint32_t *x)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(h != NULL);
    ASSERT(l != NULL);
    ASSERT(x != NULL);

    val = AG903_PGPn(id)->MSK0PRM0;

    *h = (val & AG903_PGPn_MSK0PRM0_H_MSK) >> AG903_PGPn_MSK0PRM0_H_POS;
    *l = (val & AG903_PGPn_MSK0PRM0_L_MSK) >> AG903_PGPn_MSK0PRM0_L_POS;
    *x = (val & AG903_PGPn_MSK0PRM0_X_MSK) >> AG903_PGPn_MSK0PRM0_X_POS;
}


/**
 * @brief         PGPnMSK0PRM1WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         msk0prm1 [in] PGPnMSK0PRM1WX^ݒli[
 * @return        void
 * @description   PGPnMSK0PRM1WX^ݒ 
*/
void AG903_PgpPrmSetPGPnMSK0PRM1(uint32_t id, PGPPrmParamPGPnMSK0PRM1 *msk0prm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(msk0prm1 != NULL);
    ASSERT(!(msk0prm1->hmax & ~(0xFF)));
    ASSERT(!(msk0prm1->hmin & ~(0xFF)));
    ASSERT(!(msk0prm1->lmax & ~(0xFF)));
    ASSERT(!(msk0prm1->lmin & ~(0xFF)));

    val = (msk0prm1->hmax << AG903_PGPn_MSK0PRM1_HMAX_POS)
        | (msk0prm1->hmin << AG903_PGPn_MSK0PRM1_HMIN_POS)
        | (msk0prm1->lmax << AG903_PGPn_MSK0PRM1_LMAX_POS)
        | (msk0prm1->lmin << AG903_PGPn_MSK0PRM1_LMIN_POS);

    AG903_PGPn(id)->MSK0PRM1 = val;
}


/**
 * @brief         PGPnMSK0PRM1WX^擾
 * @param         id [in] pCvCԍ
 * @param         msk0prm1 [out] PGPnMSK0PRM1WX^擾li[
 * @return        void
 * @description   PGPnMSK0PRM1WX^擾 
*/
void AG903_PgpPrmGetPGPnMSK0PRM1(uint32_t id, PGPPrmParamPGPnMSK0PRM1 *msk0prm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(msk0prm1 != NULL);

    val = AG903_PGPn(id)->MSK0PRM1;

    msk0prm1->hmax = (val & AG903_PGPn_MSK0PRM1_HMAX_MSK) >> AG903_PGPn_MSK0PRM1_HMAX_POS;
    msk0prm1->hmin = (val & AG903_PGPn_MSK0PRM1_HMIN_MSK) >> AG903_PGPn_MSK0PRM1_HMIN_POS;
    msk0prm1->lmax = (val & AG903_PGPn_MSK0PRM1_LMAX_MSK) >> AG903_PGPn_MSK0PRM1_LMAX_POS;
    msk0prm1->lmin = (val & AG903_PGPn_MSK0PRM1_LMIN_MSK) >> AG903_PGPn_MSK0PRM1_LMIN_POS;
}


/**
 * @brief         PGPnMSK1PRM0WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         h [in] HtB[hݒl
 * @param         l [in] LtB[hݒl
 * @param         x [in] XtB[hݒl
 * @return        void
 * @description   PGPnMSK1PRM0WX^ݒ 
*/
void AG903_PgpPrmSetPGPnMSK1PRM0(uint32_t id, uint32_t h, uint32_t l, uint32_t x)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(h & ~(0x1)));
    ASSERT(!(l & ~(0x1)));
    ASSERT(!(x & ~(0x1)));

    val = (h << AG903_PGPn_MSK1PRM0_H_POS)
        | (l << AG903_PGPn_MSK1PRM0_L_POS)
        | (x << AG903_PGPn_MSK1PRM0_X_POS);

    AG903_PGPn(id)->MSK1PRM0 = val;
}


/**
 * @brief         PGPnMSK1PRM0WX^擾
 * @param         id [in] pCvCԍ
 * @param         h [out] HtB[h擾l
 * @param         l [out] LtB[h擾l
 * @param         x [out] XtB[h擾l
 * @return        void
 * @description   PGPnMSK1PRM0WX^擾 
*/
void AG903_PgpPrmGetPGPnMSK1PRM0(uint32_t id, uint32_t *h, uint32_t *l, uint32_t *x)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(h != NULL);
    ASSERT(l != NULL);
    ASSERT(x != NULL);

    val = AG903_PGPn(id)->MSK1PRM0;

    *h = (val & AG903_PGPn_MSK1PRM0_H_MSK) >> AG903_PGPn_MSK1PRM0_H_POS;
    *l = (val & AG903_PGPn_MSK1PRM0_L_MSK) >> AG903_PGPn_MSK1PRM0_L_POS;
    *x = (val & AG903_PGPn_MSK1PRM0_X_MSK) >> AG903_PGPn_MSK1PRM0_X_POS;
}


/**
 * @brief         PGPnMSK1PRM1WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         msk1prm1 [in] PGPnMSK1PRM1WX^ݒli[
 * @return        void
 * @description   PGPnMSK1PRM1WX^ݒ 
*/
void AG903_PgpPrmSetPGPnMSK1PRM1(uint32_t id, PGPPrmParamPGPnMSK1PRM1 *msk1prm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(msk1prm1 != NULL);
    ASSERT(!(msk1prm1->hmax & ~(0xFF)));
    ASSERT(!(msk1prm1->hmin & ~(0xFF)));
    ASSERT(!(msk1prm1->lmax & ~(0xFF)));
    ASSERT(!(msk1prm1->lmin & ~(0xFF)));

    val = (msk1prm1->hmax << AG903_PGPn_MSK1PRM1_HMAX_POS)
        | (msk1prm1->hmin << AG903_PGPn_MSK1PRM1_HMIN_POS)
        | (msk1prm1->lmax << AG903_PGPn_MSK1PRM1_LMAX_POS)
        | (msk1prm1->lmin << AG903_PGPn_MSK1PRM1_LMIN_POS);

    AG903_PGPn(id)->MSK1PRM1 = val;
}


/**
 * @brief         PGPnMSK1PRM1WX^擾
 * @param         id [in] pCvCԍ
 * @param         msk1prm1 [out] PGPnMSK1PRM1WX^擾li[
 * @return        void
 * @description   PGPnMSK1PRM1WX^擾 
*/
void AG903_PgpPrmGetPGPnMSK1PRM1(uint32_t id, PGPPrmParamPGPnMSK1PRM1 *msk1prm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(msk1prm1 != NULL);

    val = AG903_PGPn(id)->MSK1PRM1;

    msk1prm1->hmax = (val & AG903_PGPn_MSK1PRM1_HMAX_MSK) >> AG903_PGPn_MSK1PRM1_HMAX_POS;
    msk1prm1->hmin = (val & AG903_PGPn_MSK1PRM1_HMIN_MSK) >> AG903_PGPn_MSK1PRM1_HMIN_POS;
    msk1prm1->lmax = (val & AG903_PGPn_MSK1PRM1_LMAX_MSK) >> AG903_PGPn_MSK1PRM1_LMAX_POS;
    msk1prm1->lmin = (val & AG903_PGPn_MSK1PRM1_LMIN_MSK) >> AG903_PGPn_MSK1PRM1_LMIN_POS;
}


/**
 * @brief         PGPnMSK2PRM0WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         h [in] HtB[hݒl
 * @param         l [in] LtB[hݒl
 * @param         x [in] XtB[hݒl
 * @return        void
 * @description   PGPnMSK2PRM0WX^ݒ 
*/
void AG903_PgpPrmSetPGPnMSK2PRM0(uint32_t id, uint32_t h, uint32_t l, uint32_t x)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(h & ~(0x1)));
    ASSERT(!(l & ~(0x1)));
    ASSERT(!(x & ~(0x1)));

    val = (h << AG903_PGPn_MSK2PRM0_H_POS)
        | (l << AG903_PGPn_MSK2PRM0_L_POS)
        | (x << AG903_PGPn_MSK2PRM0_X_POS);

    AG903_PGPn(id)->MSK2PRM0 = val;
}


/**
 * @brief         PGPnMSK2PRM0WX^擾
 * @param         id [in] pCvCԍ
 * @param         h [out] HtB[h擾l
 * @param         l [out] LtB[h擾l
 * @param         x [out] XtB[h擾l
 * @return        void
 * @description   PGPnMSK2PRM0WX^擾 
*/
void AG903_PgpPrmGetPGPnMSK2PRM0(uint32_t id, uint32_t *h, uint32_t *l, uint32_t *x)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(h != NULL);
    ASSERT(l != NULL);
    ASSERT(x != NULL);

    val = AG903_PGPn(id)->MSK2PRM0;

    *h = (val & AG903_PGPn_MSK2PRM0_H_MSK) >> AG903_PGPn_MSK2PRM0_H_POS;
    *l = (val & AG903_PGPn_MSK2PRM0_L_MSK) >> AG903_PGPn_MSK2PRM0_L_POS;
    *x = (val & AG903_PGPn_MSK2PRM0_X_MSK) >> AG903_PGPn_MSK2PRM0_X_POS;
}


/**
 * @brief         PGPnMSK2PRM1WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         msk2prm1 [in] PGPnMSK2PRM1WX^ݒli[
 * @return        void
 * @description   PGPnMSK2PRM1WX^ݒ 
*/
void AG903_PgpPrmSetPGPnMSK2PRM1(uint32_t id, PGPPrmParamPGPnMSK2PRM1 *msk2prm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(msk2prm1 != NULL);
    ASSERT(!(msk2prm1->hmax & ~(0xFF)));
    ASSERT(!(msk2prm1->hmin & ~(0xFF)));
    ASSERT(!(msk2prm1->lmax & ~(0xFF)));
    ASSERT(!(msk2prm1->lmin & ~(0xFF)));

    val = (msk2prm1->hmax << AG903_PGPn_MSK2PRM1_HMAX_POS)
        | (msk2prm1->hmin << AG903_PGPn_MSK2PRM1_HMIN_POS)
        | (msk2prm1->lmax << AG903_PGPn_MSK2PRM1_LMAX_POS)
        | (msk2prm1->lmin << AG903_PGPn_MSK2PRM1_LMIN_POS);

    AG903_PGPn(id)->MSK2PRM1 = val;
}


/**
 * @brief         PGPnMSK2PRM1WX^擾
 * @param         id [in] pCvCԍ
 * @param         msk2prm1 [out] PGPnMSK2PRM1WX^擾li[
 * @return        void
 * @description   PGPnMSK2PRM1WX^擾 
*/
void AG903_PgpPrmGetPGPnMSK2PRM1(uint32_t id, PGPPrmParamPGPnMSK2PRM1 *msk2prm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(msk2prm1 != NULL);

    val = AG903_PGPn(id)->MSK2PRM1;

    msk2prm1->hmax = (val & AG903_PGPn_MSK2PRM1_HMAX_MSK) >> AG903_PGPn_MSK2PRM1_HMAX_POS;
    msk2prm1->hmin = (val & AG903_PGPn_MSK2PRM1_HMIN_MSK) >> AG903_PGPn_MSK2PRM1_HMIN_POS;
    msk2prm1->lmax = (val & AG903_PGPn_MSK2PRM1_LMAX_MSK) >> AG903_PGPn_MSK2PRM1_LMAX_POS;
    msk2prm1->lmin = (val & AG903_PGPn_MSK2PRM1_LMIN_MSK) >> AG903_PGPn_MSK2PRM1_LMIN_POS;
}


/**
 * @brief         PGPnTHR8PRM0WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         thr8prm0 [in] PGPnTHR8PRM0WX^ݒli[
 * @return        void
 * @description   PGPnTHR8PRM0WX^ݒ 
*/
void AG903_PgpPrmSetPGPnTHR8PRM0(uint32_t id, PGPPrmParamPGPnTHR8PRM0 *thr8prm0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(thr8prm0 != NULL);
    ASSERT(!(thr8prm0->type   & ~(0xF)));
    ASSERT(!(thr8prm0->maxval & ~(0xFF)));
    ASSERT(!(thr8prm0->maxthr & ~(0xFF)));
    ASSERT(!(thr8prm0->minthr & ~(0xFF)));

    val = (thr8prm0->type   << AG903_PGPn_THR8PRM0_TYPE_POS)
        | (thr8prm0->maxval << AG903_PGPn_THR8PRM0_MAXVAL_POS)
        | (thr8prm0->maxthr << AG903_PGPn_THR8PRM0_MAXTHR_POS)
        | (thr8prm0->minthr << AG903_PGPn_THR8PRM0_MINTHR_POS);

    AG903_PGPn(id)->THR8PRM0 = val;
}


/**
 * @brief         PGPnTHR8PRM0WX^擾
 * @param         id [in] pCvCԍ
 * @param         thr8prm0 [out] PGPnTHR8PRM0WX^擾li[
 * @return        void
 * @description   PGPnTHR8PRM0WX^擾 
*/
void AG903_PgpPrmGetPGPnTHR8PRM0(uint32_t id, PGPPrmParamPGPnTHR8PRM0 *thr8prm0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(thr8prm0 != NULL);

    val = AG903_PGPn(id)->THR8PRM0;

    thr8prm0->type   = (val & AG903_PGPn_THR8PRM0_TYPE_MSK)   >> AG903_PGPn_THR8PRM0_TYPE_POS;
    thr8prm0->maxval = (val & AG903_PGPn_THR8PRM0_MAXVAL_MSK) >> AG903_PGPn_THR8PRM0_MAXVAL_POS;
    thr8prm0->maxthr = (val & AG903_PGPn_THR8PRM0_MAXTHR_MSK) >> AG903_PGPn_THR8PRM0_MAXTHR_POS;
    thr8prm0->minthr = (val & AG903_PGPn_THR8PRM0_MINTHR_MSK) >> AG903_PGPn_THR8PRM0_MINTHR_POS;
}


/**
 * @brief         PGPnTHR8PRM1WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         thr8prm1 [in] PGPnTHR8PRM1WX^ݒli[
 * @return        void
 * @description   PGPnTHR8PRM1WX^ݒ 
*/
void AG903_PgpPrmSetPGPnTHR8PRM1(uint32_t id, PGPPrmParamPGPnTHR8PRM1 *thr8prm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(thr8prm1 != NULL);
    ASSERT(!(thr8prm1->type   & ~(0xF)));
    ASSERT(!(thr8prm1->maxval & ~(0xFF)));
    ASSERT(!(thr8prm1->maxthr & ~(0xFF)));
    ASSERT(!(thr8prm1->minthr & ~(0xFF)));

    val = (thr8prm1->type   << AG903_PGPn_THR8PRM1_TYPE_POS)
        | (thr8prm1->maxval << AG903_PGPn_THR8PRM1_MAXVAL_POS)
        | (thr8prm1->maxthr << AG903_PGPn_THR8PRM1_MAXTHR_POS)
        | (thr8prm1->minthr << AG903_PGPn_THR8PRM1_MINTHR_POS);

    AG903_PGPn(id)->THR8PRM1 = val;
}


/**
 * @brief         PGPnTHR8PRM1WX^擾
 * @param         id [in] pCvCԍ
 * @param         thr8prm1 [out] PGPnTHR8PRM1WX^擾li[
 * @return        void
 * @description   PGPnTHR8PRM1WX^擾 
*/
void AG903_PgpPrmGetPGPnTHR8PRM1(uint32_t id, PGPPrmParamPGPnTHR8PRM1 *thr8prm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(thr8prm1 != NULL);

    val = AG903_PGPn(id)->THR8PRM1;

    thr8prm1->type   = (val & AG903_PGPn_THR8PRM1_TYPE_MSK)   >> AG903_PGPn_THR8PRM1_TYPE_POS;
    thr8prm1->maxval = (val & AG903_PGPn_THR8PRM1_MAXVAL_MSK) >> AG903_PGPn_THR8PRM1_MAXVAL_POS;
    thr8prm1->maxthr = (val & AG903_PGPn_THR8PRM1_MAXTHR_MSK) >> AG903_PGPn_THR8PRM1_MAXTHR_POS;
    thr8prm1->minthr = (val & AG903_PGPn_THR8PRM1_MINTHR_MSK) >> AG903_PGPn_THR8PRM1_MINTHR_POS;
}


/**
 * @brief         PGPnTHR8PRM2WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         thr8prm2 [in] PGPnTHR8PRM2WX^ݒli[
 * @return        void
 * @description   PGPnTHR8PRM2WX^ݒ 
*/
void AG903_PgpPrmSetPGPnTHR8PRM2(uint32_t id, PGPPrmParamPGPnTHR8PRM2 *thr8prm2)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(thr8prm2 != NULL);
    ASSERT(!(thr8prm2->type   & ~(0xF)));
    ASSERT(!(thr8prm2->maxval & ~(0xFF)));
    ASSERT(!(thr8prm2->maxthr & ~(0xFF)));
    ASSERT(!(thr8prm2->minthr & ~(0xFF)));

    val = (thr8prm2->type   << AG903_PGPn_THR8PRM2_TYPE_POS)
        | (thr8prm2->maxval << AG903_PGPn_THR8PRM2_MAXVAL_POS)
        | (thr8prm2->maxthr << AG903_PGPn_THR8PRM2_MAXTHR_POS)
        | (thr8prm2->minthr << AG903_PGPn_THR8PRM2_MINTHR_POS);

    AG903_PGPn(id)->THR8PRM2 = val;
}


/**
 * @brief         PGPnTHR8PRM2WX^擾
 * @param         id [in] pCvCԍ
 * @param         thr8prm2 [out] PGPnTHR8PRM2WX^擾li[
 * @return        void
 * @description   PGPnTHR8PRM2WX^擾 
*/
void AG903_PgpPrmGetPGPnTHR8PRM2(uint32_t id, PGPPrmParamPGPnTHR8PRM2 *thr8prm2)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(thr8prm2 != NULL);

    val = AG903_PGPn(id)->THR8PRM2;

    thr8prm2->type   = (val & AG903_PGPn_THR8PRM2_TYPE_MSK)   >> AG903_PGPn_THR8PRM2_TYPE_POS;
    thr8prm2->maxval = (val & AG903_PGPn_THR8PRM2_MAXVAL_MSK) >> AG903_PGPn_THR8PRM2_MAXVAL_POS;
    thr8prm2->maxthr = (val & AG903_PGPn_THR8PRM2_MAXTHR_MSK) >> AG903_PGPn_THR8PRM2_MAXTHR_POS;
    thr8prm2->minthr = (val & AG903_PGPn_THR8PRM2_MINTHR_MSK) >> AG903_PGPn_THR8PRM2_MINTHR_POS;
}


/**
 * @brief         PGPnTHR1PRM0WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         h [in] HtB[hݒl
 * @param         l [in] LtB[hݒl
 * @param         x [in] XtB[hݒl
 * @return        void
 * @description   PGPnTHR1PRM0WX^ݒ 
*/
void AG903_PgpPrmSetPGPnTHR1PRM0(uint32_t id, uint32_t h, uint32_t l, uint32_t x)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(h & ~(0x1)));
    ASSERT(!(l & ~(0x1)));
    ASSERT(!(x & ~(0x1)));

    val = (h << AG903_PGPn_THR1PRM0_H_POS)
        | (l << AG903_PGPn_THR1PRM0_L_POS)
        | (x << AG903_PGPn_THR1PRM0_X_POS);

    AG903_PGPn(id)->THR1PRM0 = val;
}


/**
 * @brief         PGPnTHR1PRM0WX^擾
 * @param         id [in] pCvCԍ
 * @param         h [out] HtB[h擾l
 * @param         l [out] LtB[h擾l
 * @param         x [out] XtB[h擾l
 * @return        void
 * @description   PGPnTHR1PRM0WX^擾 
*/
void AG903_PgpPrmGetPGPnTHR1PRM0(uint32_t id, uint32_t *h, uint32_t *l, uint32_t *x)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(h != NULL);
    ASSERT(l != NULL);
    ASSERT(x != NULL);

    val = AG903_PGPn(id)->THR1PRM0;

    *h = (val & AG903_PGPn_THR1PRM0_H_MSK) >> AG903_PGPn_THR1PRM0_H_POS;
    *l = (val & AG903_PGPn_THR1PRM0_L_MSK) >> AG903_PGPn_THR1PRM0_L_POS;
    *x = (val & AG903_PGPn_THR1PRM0_X_MSK) >> AG903_PGPn_THR1PRM0_X_POS;
}


/**
 * @brief         PGPnTHR1PRM1WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         thr1prm1 [in] PGPnTHR1PRM1WX^ݒli[
 * @return        void
 * @description   PGPnTHR1PRM1WX^ݒ 
*/
void AG903_PgpPrmSetPGPnTHR1PRM1(uint32_t id, PGPPrmParamPGPnTHR1PRM1 *thr1prm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(thr1prm1 != NULL);
    ASSERT(!(thr1prm1->hmax & ~(0xFF)));
    ASSERT(!(thr1prm1->hmin & ~(0xFF)));
    ASSERT(!(thr1prm1->lmax & ~(0xFF)));
    ASSERT(!(thr1prm1->lmin & ~(0xFF)));

    val = (thr1prm1->hmax << AG903_PGPn_THR1PRM1_HMAX_POS)
        | (thr1prm1->hmin << AG903_PGPn_THR1PRM1_HMIN_POS)
        | (thr1prm1->lmax << AG903_PGPn_THR1PRM1_LMAX_POS)
        | (thr1prm1->lmin << AG903_PGPn_THR1PRM1_LMIN_POS);

    AG903_PGPn(id)->THR1PRM1 = val;
}


/**
 * @brief         PGPnTHR1PRM1WX^擾
 * @param         id [in] pCvCԍ
 * @param         thr1prm1 [out] PGPnTHR1PRM1WX^擾li[
 * @return        void
 * @description   PGPnTHR1PRM1WX^擾 
*/
void AG903_PgpPrmGetPGPnTHR1PRM1(uint32_t id, PGPPrmParamPGPnTHR1PRM1 *thr1prm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(thr1prm1 != NULL);

    val = AG903_PGPn(id)->THR1PRM1;

    thr1prm1->hmax = (val & AG903_PGPn_THR1PRM1_HMAX_MSK) >> AG903_PGPn_THR1PRM1_HMAX_POS;
    thr1prm1->hmin = (val & AG903_PGPn_THR1PRM1_HMIN_MSK) >> AG903_PGPn_THR1PRM1_HMIN_POS;
    thr1prm1->lmax = (val & AG903_PGPn_THR1PRM1_LMAX_MSK) >> AG903_PGPn_THR1PRM1_LMAX_POS;
    thr1prm1->lmin = (val & AG903_PGPn_THR1PRM1_LMIN_MSK) >> AG903_PGPn_THR1PRM1_LMIN_POS;
}


/**
 * @brief         PGPnSPF1PRM0WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         spf1prm0 [in] PGPnSPF1PRM0WX^ݒli[
 * @return        void
 * @description   PGPnSPF1PRM0WX^ݒ 
*/
void AG903_PgpPrmSetPGPnSPF1PRM0(uint32_t id, PGPPrmParamPGPnSPF1PRM0 *spf1prm0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(spf1prm0 !=  NULL);
    ASSERT(!(spf1prm0->h10 & ~(0x7)));
    ASSERT(!(spf1prm0->h00 & ~(0x7)));
    ASSERT(!(spf1prm0->hm0 & ~(0x7)));
    ASSERT(!(spf1prm0->h1m & ~(0x7)));
    ASSERT(!(spf1prm0->h0m & ~(0x7)));
    ASSERT(!(spf1prm0->hmm & ~(0x7)));

    val = (spf1prm0->h10 << AG903_PGPn_SPF1PRM0_H10_POS)
        | (spf1prm0->h00 << AG903_PGPn_SPF1PRM0_H00_POS)
        | (spf1prm0->hm0 << AG903_PGPn_SPF1PRM0_HM0_POS)
        | (spf1prm0->h1m << AG903_PGPn_SPF1PRM0_H1M_POS)
        | (spf1prm0->h0m << AG903_PGPn_SPF1PRM0_H0M_POS)
        | (spf1prm0->hmm << AG903_PGPn_SPF1PRM0_HMM_POS);

    AG903_PGPn(id)->SPF1PRM0 = val;
}


/**
 * @brief         PGPnSPF1PRM0WX^擾
 * @param         id [in] pCvCԍ
 * @param         spf1prm0 [out] PGPnSPF1PRM0WX^ݒli[
 * @return        void
 * @description   PGPnSPF1PRM0WX^擾 
*/
void AG903_PgpPrmGetPGPnSPF1PRM0(uint32_t id, PGPPrmParamPGPnSPF1PRM0 *spf1prm0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(spf1prm0 !=  NULL);

    val = AG903_PGPn(id)->SPF1PRM0;

    spf1prm0->h10 = (val & AG903_PGPn_SPF1PRM0_H10_MSK) >> AG903_PGPn_SPF1PRM0_H10_POS;
    spf1prm0->h00 = (val & AG903_PGPn_SPF1PRM0_H00_MSK) >> AG903_PGPn_SPF1PRM0_H00_POS;
    spf1prm0->hm0 = (val & AG903_PGPn_SPF1PRM0_HM0_MSK) >> AG903_PGPn_SPF1PRM0_HM0_POS;
    spf1prm0->h1m = (val & AG903_PGPn_SPF1PRM0_H1M_MSK) >> AG903_PGPn_SPF1PRM0_H1M_POS;
    spf1prm0->h0m = (val & AG903_PGPn_SPF1PRM0_H0M_MSK) >> AG903_PGPn_SPF1PRM0_H0M_POS;
    spf1prm0->hmm = (val & AG903_PGPn_SPF1PRM0_HMM_MSK) >> AG903_PGPn_SPF1PRM0_HMM_POS;
}


/**
 * @brief         PGPnSPF1PRM1WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         spf1prm1 [in] PGPnSPF1PRM1WX^ݒli[
 * @return        void
 * @description   PGPnSPF1PRM1WX^ݒ 
*/
void AG903_PgpPrmSetPGPnSPF1PRM1(uint32_t id, PGPPrmParamPGPnSPF1PRM1 *spf1prm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(spf1prm1 !=  NULL);
    ASSERT(!(spf1prm1->thr & ~(0x3F)));
    ASSERT(!(spf1prm1->h11 & ~(0x7)));
    ASSERT(!(spf1prm1->h01 & ~(0x7)));
    ASSERT(!(spf1prm1->hm1 & ~(0x7)));

    val = (spf1prm1->thr << AG903_PGPn_SPF1PRM1_THR_POS)
        | (spf1prm1->h11 << AG903_PGPn_SPF1PRM1_H11_POS)
        | (spf1prm1->h01 << AG903_PGPn_SPF1PRM1_H01_POS)
        | (spf1prm1->hm1 << AG903_PGPn_SPF1PRM1_HM1_POS);

    AG903_PGPn(id)->SPF1PRM1 = val;
}


/**
 * @brief         PGPnSPF1PRM1WX^擾
 * @param         id [in] pCvCԍ
 * @param         spf1prm1 [out] PGPnSPF1PRM1WX^擾li[
 * @return        void
 * @description   PGPnSPF1PRM1WX^擾 
*/
void AG903_PgpPrmGetPGPnSPF1PRM1(uint32_t id, PGPPrmParamPGPnSPF1PRM1 *spf1prm1)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(spf1prm1 !=  NULL);

    val = AG903_PGPn(id)->SPF1PRM1;

    spf1prm1->thr = (val & AG903_PGPn_SPF1PRM1_THR_MSK) >> AG903_PGPn_SPF1PRM1_THR_POS;
    spf1prm1->h11 = (val & AG903_PGPn_SPF1PRM1_H11_MSK) >> AG903_PGPn_SPF1PRM1_H11_POS;
    spf1prm1->h01 = (val & AG903_PGPn_SPF1PRM1_H01_MSK) >> AG903_PGPn_SPF1PRM1_H01_POS;
    spf1prm1->hm1 = (val & AG903_PGPn_SPF1PRM1_HM1_MSK) >> AG903_PGPn_SPF1PRM1_HM1_POS;
}


/**
 * @brief         PGPnTHR1CNTWX^擾
 * @param         id [in] pCvCԍ
 * @param         cnt [out] CNTtB[h擾l
 * @return        void
 * @description   PGPnTHR1CNTWX^擾 
*/
void AG903_PgpPrmGetPGPnTHR1CNT(uint32_t id, uint32_t *cnt)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(cnt != NULL);

    val = AG903_PGPn(id)->THR1CNT;

    *cnt = (val & AG903_PGPn_THR1CNT_CNT_MSK) >> AG903_PGPn_THR1CNT_CNT_POS;
}


/**
 * @brief         PGPnLBLCMDWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         cmd [in] CMDtB[hݒl
 * @return        void
 * @description   PGPnLBLCMDWX^ݒ 
*/
void AG903_PgpPrmSetPGPnLBLCMD(uint32_t id, uint32_t cmd)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(cmd & ~(0x7)));

    val = (cmd << AG903_PGPn_LBLCMD_CMD_POS);

    AG903_PGPn(id)->LBLCMD = val;
}


/**
 * @brief         PGPnLBLCTRLWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         lblctrl [in] PGPnLBLCTRLWX^ݒli[
 * @return        void
 * @description   PGPnLBLCTRLWX^ݒ 
*/
void AG903_PgpPrmSetPGPnLBLCTRL(uint32_t id, PGPPrmParamPGPnLBLCTRL *lblctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(lblctrl != NULL);
    ASSERT(!(lblctrl->vld   & ~(0xF)));
    ASSERT(!(lblctrl->inv   & ~(0xF)));
    ASSERT(!(lblctrl->cnct  & ~(0x1)));
    ASSERT(!(lblctrl->bmu   & ~(0x1)));
    ASSERT(!(lblctrl->flthr & ~(0xFF)));
    ASSERT(!(lblctrl->maxid & ~(0xFF)));

    val = (lblctrl->vld   << AG903_PGPn_LBLCTRL_VLD_POS)
        | (lblctrl->inv   << AG903_PGPn_LBLCTRL_INV_POS)
        | (lblctrl->cnct  << AG903_PGPn_LBLCTRL_CNCT_POS)
        | (lblctrl->bmu   << AG903_PGPn_LBLCTRL_BMU_POS)
        | (lblctrl->flthr << AG903_PGPn_LBLCTRL_FLTHR_POS)
        | (lblctrl->maxid << AG903_PGPn_LBLCTRL_MAXID_POS);

    AG903_PGPn(id)->LBLCTRL = val;
}


/**
 * @brief         PGPnLBLCTRLWX^擾
 * @param         id [in] pCvCԍ
 * @param         lblctrl [out] PGPnLBLCTRLWX^擾li[
 * @return        void
 * @description   PGPnLBLCTRLWX^擾 
*/
void AG903_PgpPrmGetPGPnLBLCTRL(uint32_t id, PGPPrmParamPGPnLBLCTRL *lblctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(lblctrl != NULL);

    val = AG903_PGPn(id)->LBLCTRL;

    lblctrl->vld   = (val & AG903_PGPn_LBLCTRL_VLD_MSK)   >> AG903_PGPn_LBLCTRL_VLD_POS;
    lblctrl->inv   = (val & AG903_PGPn_LBLCTRL_INV_MSK)   >> AG903_PGPn_LBLCTRL_INV_POS;
    lblctrl->cnct  = (val & AG903_PGPn_LBLCTRL_CNCT_MSK)  >> AG903_PGPn_LBLCTRL_CNCT_POS;
    lblctrl->bmu   = (val & AG903_PGPn_LBLCTRL_BMU_MSK)   >> AG903_PGPn_LBLCTRL_BMU_POS;
    lblctrl->flthr = (val & AG903_PGPn_LBLCTRL_FLTHR_MSK) >> AG903_PGPn_LBLCTRL_FLTHR_POS;
    lblctrl->maxid = (val & AG903_PGPn_LBLCTRL_MAXID_MSK) >> AG903_PGPn_LBLCTRL_MAXID_POS;
}


/**
 * @brief         PGPnLBLSIZEWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         vsize [in] VSIZEtB[hݒl
 * @param         hsize [in] HSIZEtB[hݒl
 * @return        void
 * @description   PGPnLBLSIZEWX^ݒ 
*/
void AG903_PgpPrmSetPGPnLBLSIZE(uint32_t id, uint32_t vsize, uint32_t hsize)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(vsize & ~(0x3FF)));
    ASSERT(!(hsize & ~(0x3FF)));

    val = (vsize << AG903_PGPn_LBLSIZE_VSIZE_POS)
        | (hsize << AG903_PGPn_LBLSIZE_HSIZE_POS);

    AG903_PGPn(id)->LBLSIZE = val;
}


/**
 * @brief         PGPnLBLSIZEWX^擾
 * @param         id [in] pCvCԍ
 * @param         vsize [out] VSIZEtB[h擾l
 * @param         hsize [out] HSIZEtB[h擾l
 * @return        void
 * @description   PGPnLBLSIZEWX^擾 
*/
void AG903_PgpPrmGetPGPnLBLSIZE(uint32_t id, uint32_t *vsize, uint32_t *hsize)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(vsize != NULL);
    ASSERT(hsize != NULL);

    val = AG903_PGPn(id)->LBLSIZE;

    *vsize = (val & AG903_PGPn_LBLSIZE_VSIZE_MSK) >> AG903_PGPn_LBLSIZE_VSIZE_POS;
    *hsize = (val & AG903_PGPn_LBLSIZE_HSIZE_MSK) >> AG903_PGPn_LBLSIZE_HSIZE_POS;
}


/**
 * @brief         PGPnLBLPOSWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         vpos [in] VPOStB[hݒl
 * @param         hpos [in] HPOStB[hݒl
 * @return        void
 * @description   PGPnLBLPOSWX^ݒ 
*/
void AG903_PgpPrmSetPGPnLBLPOS(uint32_t id, uint32_t vpos, uint32_t hpos)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(vpos & ~(0xFFF)));
    ASSERT(!(hpos & ~(0xFFF)));

    val = (vpos << AG903_PGPn_LBLPOS_VPOS_POS)
        | (hpos << AG903_PGPn_LBLPOS_HPOS_POS);

    AG903_PGPn(id)->LBLPOS = val;
}


/**
 * @brief         PGPnLBLPOSWX^擾
 * @param         id [in] pCvCԍ
 * @param         vpos [out] VPOStB[h擾l
 * @param         hpos [out] HPOStB[h擾l
 * @return        void
 * @description   PGPnLBLPOSWX^擾 
*/
void AG903_PgpPrmGetPGPnLBLPOS(uint32_t id, uint32_t *vpos, uint32_t *hpos)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(vpos != NULL);
    ASSERT(hpos != NULL);

    val = AG903_PGPn(id)->LBLPOS;

    *vpos = (val & AG903_PGPn_LBLPOS_VPOS_MSK) >> AG903_PGPn_LBLPOS_VPOS_POS;
    *hpos = (val & AG903_PGPn_LBLPOS_HPOS_MSK) >> AG903_PGPn_LBLPOS_HPOS_POS;
}


/**
 * @brief         PGPnLBLDSTWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         adr [in] ADRtB[hݒl
 * @return        void
 * @description   PGPnLBLDSTWX^ݒ 
*/
void AG903_PgpPrmSetPGPnLBLDST(uint32_t id, uint32_t adr)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(adr & ~(0x1FFFFFFF)));

    val = (adr << AG903_PGPn_LBLDST_ADR_POS);

    AG903_PGPn(id)->LBLDST = val;
}


/**
 * @brief         PGPnLBLDSTWX^擾
 * @param         id [in] pCvCԍ
 * @param         adr [out] ADRtB[h擾l
 * @return        void
 * @description   PGPnLBLDSTWX^擾 
*/
void AG903_PgpPrmGetPGPnLBLDST(uint32_t id, uint32_t *adr)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(adr != NULL);

    val = AG903_PGPn(id)->LBLDST;

    *adr = (val & AG903_PGPn_LBLDST_ADR_MSK) >> AG903_PGPn_LBLDST_ADR_POS;
}


/**
 * @brief         PGPnLBLSTATWX^擾
 * @param         id [in] pCvCԍ
 * @param         lblstat [out] PGPnLBLSTATWX^擾li[
 * @return        void
 * @description   PGPnLBLSTATWX^擾 
*/
void AG903_PgpPrmGetPGPnLBLSTAT(uint32_t id, PGPPrmParamPGPnLBLSTAT *lblstat)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(lblstat != NULL);

    val = AG903_PGPn(id)->LBLSTAT;

    lblstat->l_id   = (val & AG903_PGPn_LBLSTAT_ID_MSK)     >> AG903_PGPn_LBLSTAT_ID_POS;
    lblstat->vrmacc = (val & AG903_PGPn_LBLSTAT_VRMACC_MSK) >> AG903_PGPn_LBLSTAT_VRMACC_POS;
    lblstat->act    = (val & AG903_PGPn_LBLSTAT_ACT_MSK)    >> AG903_PGPn_LBLSTAT_ACT_POS;
    lblstat->queue  = (val & AG903_PGPn_LBLSTAT_QUEUE_MSK)  >> AG903_PGPn_LBLSTAT_QUEUE_POS;
    lblstat->cmd    = (val & AG903_PGPn_LBLSTAT_CMD_MSK)    >> AG903_PGPn_LBLSTAT_CMD_POS;
}


/**
 * @brief         PGPnIFCCMDWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         cmd [in] CMDtB[hݒl
 * @return        void
 * @description   PGPnIFCCMDWX^ݒ 
*/
void AG903_PgpPrmSetPGPnIFCCMD(uint32_t id, uint32_t cmd)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(cmd & ~(0x7)));

    val = (cmd << AG903_PGPn_IFCCMD_CMD_POS);

    AG903_PGPn(id)->IFCCMD = val;
}


/**
 * @brief         PGPnIFCCTRLWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         ifcctrl [in] PGPnIFCCTRLWX^ݒli[
 * @return        void
 * @description   PGPnIFCCTRLWX^ݒ 
*/
void AG903_PgpPrmSetPGPnIFCCTRL(uint32_t id, PGPPrmParamPGPnIFCCTRL *ifcctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(ifcctrl != NULL);
    ASSERT(!(ifcctrl->scan  & ~(0x1)));
    ASSERT(!(ifcctrl->limit & ~(0x3)));
    ASSERT(!(ifcctrl->src   & ~(0x3)));
    ASSERT(!(ifcctrl->op    & ~(0x7)));
    ASSERT(!(ifcctrl->sft1  & ~(0x7)));
    ASSERT(!(ifcctrl->sft0  & ~(0x7)));

    val = (ifcctrl->scan  << AG903_PGPn_IFCCTRL_SCAN_POS)
        | (ifcctrl->limit << AG903_PGPn_IFCCTRL_LIMIT_POS)
        | (ifcctrl->src   << AG903_PGPn_IFCCTRL_SRC_POS)
        | (ifcctrl->op    << AG903_PGPn_IFCCTRL_OP_POS)
        | (ifcctrl->sft1  << AG903_PGPn_IFCCTRL_SFT1_POS)
        | (ifcctrl->sft0  << AG903_PGPn_IFCCTRL_SFT0_POS);

    AG903_PGPn(id)->IFCCTRL = val;
}


/**
 * @brief         PGPnIFCCTRLWX^擾
 * @param         id [in] pCvCԍ
 * @param         ifcctrl [out] PGPnIFCCTRLWX^擾li[
 * @return        void
 * @description   PGPnIFCCTRLWX^擾 
*/
void AG903_PgpPrmGetPGPnIFCCTRL(uint32_t id, PGPPrmParamPGPnIFCCTRL *ifcctrl)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(ifcctrl != NULL);

    val = AG903_PGPn(id)->IFCCTRL;

    ifcctrl->scan  = (val & AG903_PGPn_IFCCTRL_SCAN_MSK)  >> AG903_PGPn_IFCCTRL_SCAN_POS;
    ifcctrl->limit = (val & AG903_PGPn_IFCCTRL_LIMIT_MSK) >> AG903_PGPn_IFCCTRL_LIMIT_POS;
    ifcctrl->src   = (val & AG903_PGPn_IFCCTRL_SRC_MSK)   >> AG903_PGPn_IFCCTRL_SRC_POS;
    ifcctrl->op    = (val & AG903_PGPn_IFCCTRL_OP_MSK)    >> AG903_PGPn_IFCCTRL_OP_POS;
    ifcctrl->sft1  = (val & AG903_PGPn_IFCCTRL_SFT1_MSK)  >> AG903_PGPn_IFCCTRL_SFT1_POS;
    ifcctrl->sft0  = (val & AG903_PGPn_IFCCTRL_SFT0_MSK)  >> AG903_PGPn_IFCCTRL_SFT0_POS;
}


/**
 * @brief         PGPnIFCPRMWX^ݒ
 * @param         id [in] pCvCԍ
 * @param         gain1 [in] GAIN1tB[hݒl
 * @param         gain0 [in] GAIN0tB[hݒl
 * @return        void
 * @description   PGPnIFCPRMWX^ݒ 
*/
void AG903_PgpPrmSetPGPnIFCPRM(uint32_t id, uint32_t gain1, uint32_t gain0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(!(gain1 & ~(0xFF)));
    ASSERT(!(gain0 & ~(0xFF)));

    val = (gain1 << AG903_PGPn_IFCPRM_GAIN1_POS)
        | (gain0 << AG903_PGPn_IFCPRM_GAIN0_POS);

    AG903_PGPn(id)->IFCPRM = val;
}


/**
 * @brief         PGPnIFCPRMWX^擾
 * @param         id [in] pCvCԍ
 * @param         gain1 [out] GAIN1tB[h擾l
 * @param         gain0 [out] GAIN0tB[h擾l
 * @return        void
 * @description   PGPnIFCPRMWX^擾 
*/
void AG903_PgpPrmGetPGPnIFCPRM(uint32_t id, uint32_t *gain1, uint32_t *gain0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_PIPELINE);
    ASSERT(gain1 != NULL);
    ASSERT(gain0 != NULL);

    val = AG903_PGPn(id)->IFCPRM;

    *gain1 = (val & AG903_PGPn_IFCPRM_GAIN1_MSK) >> AG903_PGPn_IFCPRM_GAIN1_POS;
    *gain0 = (val & AG903_PGPn_IFCPRM_GAIN0_MSK) >> AG903_PGPn_IFCPRM_GAIN0_POS;
}


/**
 * @brief         PGPJPGOUTSELWX^ݒ
 * @param         en [in] ENtB[hݒl
 * @param         sel [in] SELtB[hݒl
 * @return        void
 * @description   PGPJPGOUTSELWX^ݒ 
*/
void AG903_PgpPrmSetPGPJPGOUTSEL(uint32_t en, uint32_t sel)
{
    uint32_t                    val;

    ASSERT(!(en & ~(0x1)));
    ASSERT(!(sel & ~(0x7)));

    val = (en << AG903_PGP_JPGOUTSEL_EN_POS)
        | (sel << AG903_PGP_JPGOUTSEL_SEL_POS);

    AG903_PGP->JPGOUTSEL = val;
}


/**
 * @brief         PGPJPGOUTSELWX^擾
 * @param         en [out] ENtB[h擾l
 * @param         sel [out] SELtB[h擾l
 * @return        void
 * @description   PGPJPGOUTSELWX^擾 
*/
void AG903_PgpPrmGetPGPJPGOUTSEL(uint32_t *en, uint32_t *sel)
{
    uint32_t                    val;

    ASSERT(en != NULL);
    ASSERT(sel != NULL);

    val = AG903_PGP->JPGOUTSEL;

    *en  = (val & AG903_PGP_JPGOUTSEL_EN_MSK) >> AG903_PGP_JPGOUTSEL_EN_POS;
    *sel = (val & AG903_PGP_JPGOUTSEL_SEL_MSK) >> AG903_PGP_JPGOUTSEL_SEL_POS;
}


/**
 * @brief         PGPVInCMDWX^ݒ
 * @param         id [in] VRAM̓|[gԍ
 * @param         cmd [in] CMDtB[hݒl
 * @return        void
 * @description   PGPVInCMDWX^ݒ 
*/
void AG903_PgpPrmSetPGPVInCMD(uint32_t id, uint32_t cmd)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(!(cmd & ~(0xF)));

    val = (cmd << AG903_PGPVIn_CMD_CMD_POS);

    AG903_PGPVIn(id)->CMD = val;
}


/**
 * @brief         PGPVInSTATEWX^擾
 * @param         id [in] VRAM̓|[gԍ
 * @param         vrmacc [out] VRMACCtB[h擾l
 * @param         queue [out] QUEUEtB[h擾l
 * @param         st [out] STtB[h擾l
 * @return        void
 * @description   PGPVInSTATEWX^擾 
*/
void AG903_PgpPrmGetPGPVInSTATE(uint32_t id, uint32_t *vrmacc, uint32_t *queue, uint32_t *st)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(vrmacc != NULL);
    ASSERT(queue != NULL);
    ASSERT(st != NULL);

    val = AG903_PGPVIn(id)->STATE;

    *vrmacc = (val & AG903_PGPVIn_STATE_VRMACC_MSK) >> AG903_PGPVIn_STATE_VRMACC_POS;
    *queue  = (val & AG903_PGPVIn_STATE_QUEUE_MSK) >> AG903_PGPVIn_STATE_QUEUE_POS;
    *st     = (val & AG903_PGPVIn_STATE_ST_MSK) >> AG903_PGPVIn_STATE_ST_POS;
}


/**
 * @brief         PGPVInCTRL0WX^ݒ
 * @param         id [in] pCvCԍ
 * @param         ctrl0 [in] PGPVInCTRL0WX^ݒli[
 * @return        void
 * @description   PGPVInCTRL0WX^ݒ 
*/
void AG903_PgpPrmSetPGPVInCTRL0(uint32_t id, PGPPrmParamPGPVInCTRL0 *ctrl0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(ctrl0 != NULL);
    ASSERT(!(ctrl0->fai   & ~(0x1)));
    ASSERT(!(ctrl0->bmu   & ~(0x3)));
    ASSERT(!(ctrl0->scan  & ~(0x1)));
    ASSERT(!(ctrl0->dim   & ~(0x1)));
    ASSERT(!(ctrl0->swapw & ~(0x1)));
    ASSERT(!(ctrl0->swaph & ~(0x1)));
    ASSERT(!(ctrl0->swap4 & ~(0x1)));
    ASSERT(!(ctrl0->swap1 & ~(0x1)));
    ASSERT(!(ctrl0->md    & ~(0x7)));
    ASSERT(!(ctrl0->fmt   & ~(0xF)));

    val = (ctrl0->fai   << AG903_PGPVIn_CTRL0_FAI_POS)
        | (ctrl0->bmu   << AG903_PGPVIn_CTRL0_BMU_POS)
        | (ctrl0->scan  << AG903_PGPVIn_CTRL0_SCAN_POS)
        | (ctrl0->dim   << AG903_PGPVIn_CTRL0_DIM_POS)
        | (ctrl0->swapw << AG903_PGPVIn_CTRL0_SWAPW_POS)
        | (ctrl0->swaph << AG903_PGPVIn_CTRL0_SWAPH_POS)
        | (ctrl0->swap4 << AG903_PGPVIn_CTRL0_SWAP4_POS)
        | (ctrl0->swap1 << AG903_PGPVIn_CTRL0_SWAP1_POS)
        | (ctrl0->md    << AG903_PGPVIn_CTRL0_MD_POS)
        | (ctrl0->fmt   << AG903_PGPVIn_CTRL0_FMT_POS);

    AG903_PGPVIn(id)->CTRL0 = val;
}


/**
 * @brief         PGPVInCTRL0WX^擾
 * @param         id [in] pCvCԍ
 * @param         ctrl0 [out] PGPVInCTRL0WX^擾li[
 * @return        void
 * @description   PGPVInCTRL0WX^擾 
*/
void AG903_PgpPrmGetPGPVInCTRL0(uint32_t id, PGPPrmParamPGPVInCTRL0 *ctrl0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(ctrl0 != NULL);

    val = AG903_PGPVIn(id)->CTRL0;

    ctrl0->fai   = (val & AG903_PGPVIn_CTRL0_FAI_MSK)   >> AG903_PGPVIn_CTRL0_FAI_POS;
    ctrl0->bmu   = (val & AG903_PGPVIn_CTRL0_BMU_MSK)   >> AG903_PGPVIn_CTRL0_BMU_POS;
    ctrl0->scan  = (val & AG903_PGPVIn_CTRL0_SCAN_MSK)  >> AG903_PGPVIn_CTRL0_SCAN_POS;
    ctrl0->dim   = (val & AG903_PGPVIn_CTRL0_DIM_MSK)   >> AG903_PGPVIn_CTRL0_DIM_POS;
    ctrl0->swapw = (val & AG903_PGPVIn_CTRL0_SWAPW_MSK) >> AG903_PGPVIn_CTRL0_SWAPW_POS;
    ctrl0->swaph = (val & AG903_PGPVIn_CTRL0_SWAPH_MSK) >> AG903_PGPVIn_CTRL0_SWAPH_POS;
    ctrl0->swap4 = (val & AG903_PGPVIn_CTRL0_SWAP4_MSK) >> AG903_PGPVIn_CTRL0_SWAP4_POS;
    ctrl0->swap1 = (val & AG903_PGPVIn_CTRL0_SWAP1_MSK) >> AG903_PGPVIn_CTRL0_SWAP1_POS;
    ctrl0->md    = (val & AG903_PGPVIn_CTRL0_MD_MSK)    >> AG903_PGPVIn_CTRL0_MD_POS;
    ctrl0->fmt   = (val & AG903_PGPVIn_CTRL0_FMT_MSK)   >> AG903_PGPVIn_CTRL0_FMT_POS;
}


/**
 * @brief         PGPVInCTRL1WX^ݒ
 * @param         id [in] VRAM̓|[gԍ
 * @param         div [in] DIVtB[hݒl
 * @param         vld [in] VLDtB[hݒl
 * @param         inv [in] INVtB[hݒl
 * @return        void
 * @description   PGPVInCTRL1WX^ݒ 
*/
void AG903_PgpPrmSetPGPVInCTRL1(uint32_t id, uint32_t div, uint32_t vld, uint32_t inv)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(!(div & ~(0xFF)));
    ASSERT(!(vld & ~(0xF)));
    ASSERT(!(inv & ~(0xF)));

    val = (div << AG903_PGPVIn_CTRL1_DIV_POS)
        | (vld << AG903_PGPVIn_CTRL1_VLD_POS)
        | (inv << AG903_PGPVIn_CTRL1_INV_POS);

    AG903_PGPVIn(id)->CTRL1 = val;
}


/**
 * @brief         PGPVInCTRL1WX^擾
 * @param         id [in] VRAM̓|[gԍ
 * @param         div [out] DIVtB[h擾l
 * @param         vld [out] VLDtB[h擾l
 * @param         inv [out] INVtB[h擾l
 * @return        void
 * @description   PGPVInCTRL1WX^擾 
*/
void AG903_PgpPrmGetPGPVInCTRL1(uint32_t id, uint32_t *div, uint32_t *vld, uint32_t *inv)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(div != NULL);
    ASSERT(vld != NULL);
    ASSERT(inv != NULL);

    val = AG903_PGPVIn(id)->CTRL1;

    *div = (val & AG903_PGPVIn_CTRL1_DIV_MSK) >> AG903_PGPVIn_CTRL1_DIV_POS;
    *vld = (val & AG903_PGPVIn_CTRL1_VLD_MSK) >> AG903_PGPVIn_CTRL1_VLD_POS;
    *inv = (val & AG903_PGPVIn_CTRL1_INV_MSK) >> AG903_PGPVIn_CTRL1_INV_POS;
}


/**
 * @brief         PGPVInBASEWX^ݒ
 * @param         id [in] VRAM̓|[gԍ
 * @param         adr [in] ADRtB[hݒl
 * @return        void
 * @description   PGPVInBASEWX^ݒ 
*/
void AG903_PgpPrmSetPGPVInBASE(uint32_t id, uint32_t adr)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(!(adr & ~(0x1FFFFFFF)));

    val = (adr << AG903_PGPVIn_BASE_ADR_POS);

    AG903_PGPVIn(id)->BASE = val;
}


/**
 * @brief         PGPVInBASEWX^擾
 * @param         id [in] VRAM̓|[gԍ
 * @param         adr [out] ADRtB[h擾l
 * @return        void
 * @description   PGPVInBASEWX^擾 
*/
void AG903_PgpPrmGetPGPVInBASE(uint32_t id, uint32_t *adr)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(adr != NULL);

    val = AG903_PGPVIn(id)->BASE;

    *adr = (val & AG903_PGPVIn_BASE_ADR_MSK) >> AG903_PGPVIn_BASE_ADR_POS;
}


/**
 * @brief         PGPVInHSIZEWX^ݒ
 * @param         id [in] VRAM̓|[gԍ
 * @param         hsize [in] HSIZEtB[hݒl
 * @return        void
 * @description   PGPVInHSIZEWX^ݒ 
*/
void AG903_PgpPrmSetPGPVInHSIZE(uint32_t id, uint32_t hsize)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(!(hsize & ~(0x3FFF)));

    val = (hsize << AG903_PGPVIn_HSIZE_HSIZE_POS);

    AG903_PGPVIn(id)->HSIZE = val;
}


/**
 * @brief         PGPVInHSIZEWX^擾
 * @param         id [in] VRAM̓|[gԍ
 * @param         hsize [out] HSIZEtB[h擾l
 * @return        void
 * @description   PGPVInHSIZEWX^擾 
*/
void AG903_PgpPrmGetPGPVInHSIZE(uint32_t id, uint32_t *hsize)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(hsize != NULL);

    val = AG903_PGPVIn(id)->HSIZE;

    *hsize = (val & AG903_PGPVIn_HSIZE_HSIZE_MSK) >> AG903_PGPVIn_HSIZE_HSIZE_POS;
}


/**
 * @brief         PGPVInHPRM0WX^ݒ
 * @param         id [in] VRAM̓|[gԍ
 * @param         hpw [in] HPWtB[hݒl
 * @return        void
 * @description   PGPVInHPRM0WX^ݒ 
*/
void AG903_PgpPrmSetPGPVInHPRM0(uint32_t id, uint32_t hpw)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(!(hpw & ~(0xFFF)));

    val = (hpw << AG903_PGPVIn_HPRM0_HPW_POS);

    AG903_PGPVIn(id)->HPRM0 = val;
}


/**
 * @brief         PGPVInHPRM0WX^擾
 * @param         id [in] VRAM̓|[gԍ
 * @param         hpw [out] HPWtB[h擾l
 * @return        void
 * @description   PGPVInHPRM0WX^擾 
*/
void AG903_PgpPrmGetPGPVInHPRM0(uint32_t id, uint32_t *hpw)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(hpw != NULL);

    val = AG903_PGPVIn(id)->HPRM0;

    *hpw = (val & AG903_PGPVIn_HPRM0_HPW_MSK) >> AG903_PGPVIn_HPRM0_HPW_POS;
}


/**
 * @brief         PGPVInHPRM1WX^ݒ
 * @param         id [in] VRAM̓|[gԍ
 * @param         hfp [in] HFPtB[hݒl
 * @param         hbp [in] HBPtB[hݒl
 * @return        void
 * @description   PGPVInHPRM1WX^ݒ 
*/
void AG903_PgpPrmSetPGPVInHPRM1(uint32_t id, uint32_t hfp, uint32_t hbp)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(!(hfp & ~(0xFFF)));
    ASSERT(!(hbp & ~(0xFFF)));

    val = (hfp << AG903_PGPVIn_HPRM1_HFP_POS)
        | (hbp << AG903_PGPVIn_HPRM1_HBP_POS);

    AG903_PGPVIn(id)->HPRM1 = val;
}


/**
 * @brief         PGPVInHPRM1WX^擾
 * @param         id [in] VRAM̓|[gԍ
 * @param         hfp [out] HFPtB[h擾l
 * @param         hbp [out] HBPtB[h擾l
 * @return        void
 * @description   PGPVInHPRM1WX^擾 
*/
void AG903_PgpPrmGetPGPVInHPRM1(uint32_t id, uint32_t *hfp, uint32_t *hbp)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(hfp != NULL);
    ASSERT(hbp != NULL);

    val = AG903_PGPVIn(id)->HPRM1;

    *hfp = (val & AG903_PGPVIn_HPRM1_HFP_MSK) >> AG903_PGPVIn_HPRM1_HFP_POS;
    *hbp = (val & AG903_PGPVIn_HPRM1_HBP_MSK) >> AG903_PGPVIn_HPRM1_HBP_POS;
}


/**
 * @brief         PGPVInVPRM0WX^ݒ
 * @param         id [in] VRAM̓|[gԍ
 * @param         vprm0 [in] PGPVInVPRM0WX^ݒli[
 * @return        void
 * @description   PGPVInVPRM0WX^ݒ 
*/
void AG903_PgpPrmSetPGPVInVPRM0(uint32_t id, PGPPrmParamPGPVInVPRM0 *vprm0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(vprm0 != NULL);
    ASSERT(!(vprm0->ebp & ~(0x1)));
    ASSERT(!(vprm0->efp & ~(0x1)));
    ASSERT(!(vprm0->obp & ~(0x1)));
    ASSERT(!(vprm0->ofp & ~(0x1)));
    ASSERT(!(vprm0->vpw & ~(0xFFF)));

    val = (vprm0->ebp << AG903_PGPVIn_VPRM0_EBP_POS)
        | (vprm0->efp << AG903_PGPVIn_VPRM0_EFP_POS)
        | (vprm0->obp << AG903_PGPVIn_VPRM0_OBP_POS)
        | (vprm0->ofp << AG903_PGPVIn_VPRM0_OFP_POS)
        | (vprm0->vpw << AG903_PGPVIn_VPRM0_VPW_POS);

    AG903_PGPVIn(id)->VPRM0 = val;
}


/**
 * @brief         PGPVInVPRM0WX^擾
 * @param         id [in] VRAM̓|[gԍ
 * @param         vprm0 [out] PGPVInVPRM0WX^擾li[
 * @return        void
 * @description   PGPVInVPRM0WX^擾 
*/
void AG903_PgpPrmGetPGPVInVPRM0(uint32_t id, PGPPrmParamPGPVInVPRM0 *vprm0)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(vprm0 != NULL);

    val = AG903_PGPVIn(id)->VPRM0;

    vprm0->ebp = (val & AG903_PGPVIn_VPRM0_EBP_MSK) >> AG903_PGPVIn_VPRM0_EBP_POS;
    vprm0->efp = (val & AG903_PGPVIn_VPRM0_EFP_MSK) >> AG903_PGPVIn_VPRM0_EFP_POS;
    vprm0->obp = (val & AG903_PGPVIn_VPRM0_OBP_MSK) >> AG903_PGPVIn_VPRM0_OBP_POS;
    vprm0->ofp = (val & AG903_PGPVIn_VPRM0_OFP_MSK) >> AG903_PGPVIn_VPRM0_OFP_POS;
    vprm0->vpw = (val & AG903_PGPVIn_VPRM0_VPW_MSK) >> AG903_PGPVIn_VPRM0_VPW_POS;
}


/**
 * @brief         PGPVInVPRM1WX^ݒ
 * @param         id [in] VRAM̓|[gԍ
 * @param         vfp [in] VFPtB[hݒl
 * @param         vbp [in] VBPtB[hݒl
 * @return        void
 * @description   PGPVInVPRM1WX^ݒ 
*/
void AG903_PgpPrmSetPGPVInVPRM1(uint32_t id, uint32_t vfp, uint32_t vbp)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(!(vfp & ~(0xFFF)));
    ASSERT(!(vbp & ~(0xFFF)));

    val = (vfp << AG903_PGPVIn_VPRM1_VFP_POS)
        | (vbp << AG903_PGPVIn_VPRM1_VBP_POS);

    AG903_PGPVIn(id)->VPRM1 = val;
}


/**
 * @brief         PGPVInVPRM1WX^擾
 * @param         id [in] VRAM̓|[gԍ
 * @param         vfp [out] VFPtB[h擾l
 * @param         vbp [out] VBPtB[h擾l
 * @return        void
 * @description   PGPVInVPRM1WX^擾 
*/
void AG903_PgpPrmGetPGPVInVPRM1(uint32_t id, uint32_t *vfp, uint32_t *vbp)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(vfp != NULL);
    ASSERT(vbp != NULL);

    val = AG903_PGPVIn(id)->VPRM1;

    *vfp = (val & AG903_PGPVIn_VPRM1_VFP_MSK) >> AG903_PGPVIn_VPRM1_VFP_POS;
    *vbp = (val & AG903_PGPVIn_VPRM1_VBP_MSK) >> AG903_PGPVIn_VPRM1_VBP_POS;
}


/**
 * @brief         PGPVInSIZEWX^ݒ
 * @param         id [in] VRAM̓|[gԍ
 * @param         ovsize [in] OVSIZEtB[hݒl
 * @param         vsize [in] VSIZEtB[hݒl
 * @param         hsize [in] HSIZEtB[hݒl
 * @return        void
 * @description   PGPVInSIZEWX^ݒ 
*/
void AG903_PgpPrmSetPGPVInSIZE(uint32_t id, uint32_t ovsize, uint32_t vsize, uint32_t hsize)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(!(ovsize & ~(0x1)));
    ASSERT(!(vsize & ~(0xFFF)));
    ASSERT(!(hsize & ~(0xFFF)));

    val = (ovsize << AG903_PGPVIn_SIZE_OVSIZE_POS)
        | (vsize << AG903_PGPVIn_SIZE_VSIZE_POS)
        | (hsize << AG903_PGPVIn_SIZE_HSIZE_POS);

    AG903_PGPVIn(id)->SIZE = val;
}


/**
 * @brief         PGPVInSIZEWX^擾
 * @param         id [in] VRAM̓|[gԍ
 * @param         ovsize [out] OVSIZEtB[h擾l
 * @param         vsize [out] VSIZEtB[h擾l
 * @param         hsize [out] HSIZEtB[h擾l
 * @return        void
 * @description   PGPVInSIZEWX^擾 
*/
void AG903_PgpPrmGetPGPVInSIZE(uint32_t id, uint32_t *ovsize, uint32_t *vsize, uint32_t *hsize)
{
    uint32_t                    val;

    ASSERT(id < AG903_PGP_PRM_MAX_VRAMI);
    ASSERT(ovsize != NULL);
    ASSERT(vsize != NULL);
    ASSERT(hsize != NULL);

    val = AG903_PGPVIn(id)->SIZE;

    *ovsize = (val & AG903_PGPVIn_SIZE_OVSIZE_MSK) >> AG903_PGPVIn_SIZE_OVSIZE_POS;
    *vsize  = (val & AG903_PGPVIn_SIZE_VSIZE_MSK) >> AG903_PGPVIn_SIZE_VSIZE_POS;
    *hsize  = (val & AG903_PGPVIn_SIZE_HSIZE_MSK) >> AG903_PGPVIn_SIZE_HSIZE_POS;
}


/**
 * @brief         PGPnDGCLUTBmWX^ݒ
 * @param         id [in] Zxϊe[uԍ
 * @param         blue [in] e[uݒl
 * @return        void
 * @description   PGPnDGCLUTBmWX^ݒ 
*/
void AG903_PgpPrmSetPGPnDGCLUTB(uint32_t id, uint8_t blue[256])
{
    volatile AG903_PGP0DGC_Type     *dgc;
    uint32_t                        val;
    uint32_t                        i;

    ASSERT(id < AG903_PGP_PRM_MAX_DGCLUT);
    ASSERT(blue != NULL);

    dgc = AG903_PGPnDGC(id);

    for (i = 0; i < 64; i++) {
        val = (blue[i * 4 + 0] << AG903_PGP0DGC_LUTB_B0_POS) |
              (blue[i * 4 + 1] << AG903_PGP0DGC_LUTB_B1_POS) |
              (blue[i * 4 + 2] << AG903_PGP0DGC_LUTB_B2_POS) |
              (blue[i * 4 + 3] << AG903_PGP0DGC_LUTB_B3_POS);
        dgc->LUTB[i] = val;
    }
}


/**
 * @brief         PGPnDGCLUTBmWX^擾
 * @param         id [in] Zxϊe[uԍ
 * @param         blue [out] e[u擾l
 * @return        void
 * @description   PGPnDGCLUTBmWX^擾 
*/
void AG903_PgpPrmGetPGPnDGCLUTB(uint32_t id, uint8_t *blue)
{
    volatile AG903_PGP0DGC_Type *dgc;
    uint32_t                    val;
    int32_t                     i;

    ASSERT(id < AG903_PGP_PRM_MAX_DGCLUT);
    ASSERT(blue != NULL);

    dgc = AG903_PGPnDGC(id);

    for (i = 0; i < 64; i++) {
        val = dgc->LUTB[i];
        blue[i * 4 + 0] = (val & AG903_PGP0DGC_LUTB_B0_MSK) >> AG903_PGP0DGC_LUTB_B0_POS;
        blue[i * 4 + 1] = (val & AG903_PGP0DGC_LUTB_B1_MSK) >> AG903_PGP0DGC_LUTB_B1_POS;
        blue[i * 4 + 2] = (val & AG903_PGP0DGC_LUTB_B2_MSK) >> AG903_PGP0DGC_LUTB_B2_POS;
        blue[i * 4 + 3] = (val & AG903_PGP0DGC_LUTB_B3_MSK) >> AG903_PGP0DGC_LUTB_B3_POS;
    }
}

/**
 * @brief         PGPnDGCFLUTBmWX^ݒ
 * @param         id [in] Zxϊe[uԍ
 * @param         blue [in] e[uݒl
 * @return        void
 * @brief         PGPnDGCFLUTBmWX^ݒ 
*/
void AG903_PgpPrmSetPGPnDGCFLUTB(uint32_t id, uint8_t blue[256])
{
    volatile AG903_PGP0DGCF_Type    *dgc;
    uint32_t                        val;
    uint32_t                        i;

    ASSERT(id < AG903_PGP_PRM_MAX_DGCLUT);
    ASSERT(blue != NULL);

    dgc = AG903_PGPnDGCF(id);

    /* FIFÕAhXłFIFŐ󂫂`FbNȂl݂܂ */
    for (i = 0; i < 64; i++) {
        while (1) {
            AG903_PgpPrmGetPGPnDGCSTAT(id, &val);

            if (val > 0) {
                break;
            }
        }
        val = (blue[i * 4 + 0] << AG903_PGP0DGC_LUTB_B0_POS) |
              (blue[i * 4 + 1] << AG903_PGP0DGC_LUTB_B1_POS) |
              (blue[i * 4 + 2] << AG903_PGP0DGC_LUTB_B2_POS) |
              (blue[i * 4 + 3] << AG903_PGP0DGC_LUTB_B3_POS);
        dgc->LUTB[i] = val;
    }
}

/**
 * @brief         PGPnDGCLUTGmWX^ݒ
 * @param         id [in] Zxϊe[uԍ
 * @param         green [in] e[uݒl
 * @return        void
 * @brief         PGPnDGCLUTGmWX^ݒ 
*/
void AG903_PgpPrmSetPGPnDGCLUTG(uint32_t id, uint8_t green[256])
{
    volatile AG903_PGP0DGC_Type    *dgc;
    uint32_t                        val;
    uint32_t                        i;

    ASSERT(id < AG903_PGP_PRM_MAX_DGCLUT);
    ASSERT(green != NULL);

    dgc = AG903_PGPnDGC(id);

    for (i = 0; i < 64; i++) {
        val = (green[i * 4 + 0] << AG903_PGP0DGC_LUTG_G0_POS) |
              (green[i * 4 + 1] << AG903_PGP0DGC_LUTG_G1_POS) |
              (green[i * 4 + 2] << AG903_PGP0DGC_LUTG_G2_POS) |
              (green[i * 4 + 3] << AG903_PGP0DGC_LUTG_G3_POS);
        dgc->LUTG[i] = val;
    }
}


/**
 * @brief         PGPnDGCLUTGmWX^擾
 * @param         id [in] Zxϊe[uԍ
 * @param         green [out] e[u擾l
 * @return        void
 * @brief         PGPnDGCLUTGmWX^擾 
*/
void AG903_PgpPrmGetPGPnDGCLUTG(uint32_t id, uint8_t *green)
{
    volatile AG903_PGP0DGC_Type *dgc;
    uint32_t                    val;
    int32_t                     i;

    ASSERT(id < AG903_PGP_PRM_MAX_DGCLUT);
    ASSERT(green != NULL);

    dgc = AG903_PGPnDGC(id);

    for (i = 0; i < 64; i++) {
        val = dgc->LUTG[i];
        green[i * 4 + 0] = (val & AG903_PGP0DGC_LUTG_G0_MSK) >> AG903_PGP0DGC_LUTG_G0_POS;
        green[i * 4 + 1] = (val & AG903_PGP0DGC_LUTG_G1_MSK) >> AG903_PGP0DGC_LUTG_G1_POS;
        green[i * 4 + 2] = (val & AG903_PGP0DGC_LUTG_G2_MSK) >> AG903_PGP0DGC_LUTG_G2_POS;
        green[i * 4 + 3] = (val & AG903_PGP0DGC_LUTG_G3_MSK) >> AG903_PGP0DGC_LUTG_G3_POS;
    }
}


/**
 * @brief         PGPnDGCFLUTGmWX^ݒ
 * @param         id [in] Zxϊe[uԍ
 * @param         green [in] e[uݒl
 * @return        void
 * @brief         PGPnDGCFLUTGmWX^ݒ 
*/
void AG903_PgpPrmSetPGPnDGCFLUTG(uint32_t id, uint8_t green[256])
{
    volatile AG903_PGP0DGCF_Type    *dgc;
    uint32_t                        val;
    uint32_t                        i;

    ASSERT(id < AG903_PGP_PRM_MAX_DGCLUT);
    ASSERT(green != NULL);

    dgc = AG903_PGPnDGCF(id);

    /* FIFÕAhXłFIFŐ󂫂`FbNȂl݂܂ */
    for (i = 0; i < 64; i++) {
        while (1) {
            AG903_PgpPrmGetPGPnDGCSTAT(id, &val);

            if (val > 0) {
                break;
            }
        }
        val = (green[i * 4 + 0] << AG903_PGP0DGC_LUTG_G0_POS) |
              (green[i * 4 + 1] << AG903_PGP0DGC_LUTG_G1_POS) |
              (green[i * 4 + 2] << AG903_PGP0DGC_LUTG_G2_POS) |
              (green[i * 4 + 3] << AG903_PGP0DGC_LUTG_G3_POS);
        dgc->LUTG[i] = val;
    }
}


/**
 * @brief         PGPnDGCLUTRmWX^ݒ
 * @param         id [in] Zxϊe[uԍ
 * @param         red [in] e[uݒl
 * @return        void
 * @brief         PGPnDGCLUTRmWX^ݒ 
*/
void AG903_PgpPrmSetPGPnDGCLUTR(uint32_t id, uint8_t red[256])
{
    volatile AG903_PGP0DGC_Type    *dgc;
    uint32_t                        val;
    uint32_t                        i;

    ASSERT(id < AG903_PGP_PRM_MAX_DGCLUT);
    ASSERT(red != NULL);

    dgc = AG903_PGPnDGC(id);

    for (i = 0; i < 64; i++) {
        val = (red[i * 4 + 0] << AG903_PGP0DGC_LUTR_R0_POS) |
              (red[i * 4 + 1] << AG903_PGP0DGC_LUTR_R1_POS) |
              (red[i * 4 + 2] << AG903_PGP0DGC_LUTR_R2_POS) |
              (red[i * 4 + 3] << AG903_PGP0DGC_LUTR_R3_POS);
        dgc->LUTR[i] = val;
    }
}


/**
 * @brief         PGPnDGCLUTRmWX^擾
 * @param         id [in] Zxϊe[uԍ
 * @param         red [out] e[u擾l
 * @return        void
 * @brief         PGPnDGCLUTRmWX^擾 
*/
void AG903_PgpPrmGetPGPnDGCLUTR(uint32_t id, uint8_t *red)
{
    volatile AG903_PGP0DGC_Type *dgc;
    uint32_t                    val;
    int32_t                     i;

    ASSERT(id < AG903_PGP_PRM_MAX_DGCLUT);
    ASSERT(red != NULL);

    dgc = AG903_PGPnDGC(id);

    for (i = 0; i < 64; i++) {
        val = dgc->LUTR[i];
        red[i * 4 + 0] = (val & AG903_PGP0DGC_LUTR_R0_MSK) >> AG903_PGP0DGC_LUTR_R0_POS;
        red[i * 4 + 1] = (val & AG903_PGP0DGC_LUTR_R1_MSK) >> AG903_PGP0DGC_LUTR_R1_POS;
        red[i * 4 + 2] = (val & AG903_PGP0DGC_LUTR_R2_MSK) >> AG903_PGP0DGC_LUTR_R2_POS;
        red[i * 4 + 3] = (val & AG903_PGP0DGC_LUTR_R3_MSK) >> AG903_PGP0DGC_LUTR_R3_POS;
    }
}

/**
 * @brief         PGPnDGCFLUTRmWX^ݒ
 * @param         id [in] Zxϊe[uԍ
 * @param         red [in] e[uݒl
 * @return        void
 * @brief         PGPnDGCFLUTRmWX^ݒ 
*/
void AG903_PgpPrmSetPGPnDGCFLUTR(uint32_t id, uint8_t red[256])
{
    volatile AG903_PGP0DGCF_Type    *dgc;
    uint32_t                        val;
    uint32_t                        i;

    ASSERT(id < AG903_PGP_PRM_MAX_DGCLUT);
    ASSERT(red != NULL);

    dgc = AG903_PGPnDGCF(id);

    /* FIFÕAhXłFIFŐ󂫂`FbNȂl݂܂ */
    for (i = 0; i < 64; i++) {
        while (1) {
            AG903_PgpPrmGetPGPnDGCSTAT(id, &val);

            if (val > 0) {
                break;
            }
        }
        val = (red[i * 4 + 0] << AG903_PGP0DGC_LUTR_R0_POS) |
              (red[i * 4 + 1] << AG903_PGP0DGC_LUTR_R1_POS) |
              (red[i * 4 + 2] << AG903_PGP0DGC_LUTR_R2_POS) |
              (red[i * 4 + 3] << AG903_PGP0DGC_LUTR_R3_POS);
        dgc->LUTR[i] = val;
    }
}
