/**
 * @brief           DMAC Manager
 * @author          AXELL CORPORATION
 * @description     DMAC Manager Layer.
 * @note            none
 * @history         2017_02_22  
 * @history         2017_05_29  [#1348]DMACԂ̔ʕ@C
 * @history         2017_10_26  Ver2.0
 * @history         2019_03_08  [SDK2.2] fBXNv^ɂDMA]̕sC (#2446)
 * @history         2019_12_27  [SDK3.0] DMAC}l[W̊荞݃NAC (#2589)
 * @history         2025_03_06  [SDK3.7] DMACɊ荞݃R[obNobN֐ǉ (#5756)
 * @history         2025_03_06  [SDK3.7] DMAC}l[WŊǗO`l̊荞݂NAsC (#5763)
*/
/* DOM-IGNORE-BEGIN */
/*
 * This program was created by Axell Corporation.
 * Copyright (C) 2017-2025 Axell Corporation, all rights reserved.
 */
/* DOM-IGNORE-END */

#include "AG903_common.h"
#include "AG903_regmap.h"

#include "dmac/dmacprm.h"
#include "dmac/dmacmgr.h"
#include "int/intmgr.h"

/** fBXNv^AhX */
typedef struct _AG903_DMACMgrAddrTable
{
	uint32_t current;	/** ݒl */
	uint32_t next;		/** l */
} AG903_DMACMgrAddrTable;

/** nh */
typedef struct _AG903_DMACMgrHandleStat
{
	void *cb_term;	/** term݃R[obN */
	void *cb_abt;	/** abort݃R[obN */
	void *cb_wdt;	/** wdt݃R[obN */
	void *cb_error;	/** error݃R[obN */

	/** fBXNv^AhXe[uizQƗpj
	 * AG903_DMACMgrSetDescListŃfBXNv^]Ƃɂ̃e[u\z
	 */
	AG903_DMACMgrAddrTable term_addr_table[AG903_DMAC_DESC_LINK_MAX];
	uint32_t term_addr_table_num; /** fBXNv^AhXe[u */

	uint8_t	unit_id;	/** DMAC`l */
	uint8_t	term_count; /** ^[~iJEg */
	uint8_t	reserve[2]; /** (폜) */

} AG903_DMACMgrHandleStat;

static AG903_DMACMgrHandleStat handle_list[AG903_DMAC_UNIT_NUM];
static _Bool handle_used_flag[AG903_DMAC_UNIT_NUM] = {false};

#define LOCAL_DESC_MEM_ADDR   (0xFD100000)		/** fBXNv^̕ۑ̃AhX */

#define CH_REG_SIZE (0x20)						/** `lƂ̃WX^̃TCY */

static int8_t dmac_int_init_flag = false;		/** DMAC̊荞݃nho^tO */
												/** DMACManagerňx̂݊荞݃nho^ */

static int32_t dmac_int_dmactc_hdrid  = 0;		/** dmactcID */

static int8_t ldd_flag[AG903_DMAC_LDD_ENTRY_NUM] = {0};	/** Local DMAC Descriptor */

static int8_t rdd_flag[AG903_DMAC_RDD_ENTRY_NUM] = {0};	/** Remote DMAC Descriptor */
static uint8_t* rdd_data = NULL;

#define TOTAL_ENTRY_NUM (AG903_DMAC_LDD_ENTRY_NUM+AG903_DMAC_RDD_ENTRY_NUM)	/** g[^̃fBXNv^̐ */

#define AG903_DMAC_CONFIG_TCMASK	(1<<0)	/** ConfigWX^TCIintMaskrbg */

static _Bool initdesc = false;	/** LDMtO */

static void write_desc(AG903_DMACMgrDesc *from, AG903_DMACMgrDesc *next, void *to, int is_reg);
static int32_t remove_desc_entry(int seq_id);
static uint8_t DMACMgr_CheckEnable(uint32_t unit);
static void DMACMgrTermIntHandler(uint32_t term);
static void DMACMgrErrIntHandler(uint32_t abt, uint32_t err);

/*
	݃nh.
*/
static void DMACMgrDmactcIntHandler()
{
	uint32_t term, abt, wdt, err;

	AG903_DMACPrmGetTIMECOUNT_STATUS(&term);
	AG903_DMACPrmGetERR_ABORT_STATUS(&err, &wdt, &abt);

	if(0 != term) {
		DMACMgrTermIntHandler(term);
	}
	if((0 != abt) || (0 != err)) {
		DMACMgrErrIntHandler(abt, err);
	}
	if(0 != wdt) {
		/* WDTł݂͊ȂׁAȂ */
		/* KvɉăAvP[VAG903_DMACMgrGetIntStatus()Ń`FbN */
	}

	return;
}

/*
	Termݏ.
*/
static void DMACMgrTermIntHandler(uint32_t term)
{
	AG903_DMACMgrHandleStat *handle;
	AG903_DMACMgrDesc *desc = NULL;
	AG903_DMACMgrDesc *next = NULL;
	uint32_t desc_save_mem;
	int loop;
	uint8_t end = false;

	for(loop=0; loop<AG903_DMAC_UNIT_NUM; loop++) {
		if(0 == term) {
			break;
		}
		if(term & (0x1<<loop)) {
			if(true == handle_used_flag[loop]) {	/* nh쒆 */
				/* R[obNŐVȓ]Jnꍇ邽ߗ\ߊ荞݂NA */
				AG903_DMACPrmSetTIMECOUNT_INT_CLEAR(0x1<<loop);

				handle = &handle_list[loop];
				if(handle->cb_term != NULL) { 		/* Callbacko^ */
					void (*cb_term)(AG903_DMACMgrHandle*, AG903_DMACMgrDesc**, uint8_t*) =
						(void(*)(AG903_DMACMgrHandle*, AG903_DMACMgrDesc**, uint8_t*))handle->cb_term;
					if(0 >= handle->term_addr_table_num) {
						cb_term((AG903_DMACMgrHandle*)handle, NULL, NULL);	/* Callback (,O) */
					}
					else {
						cb_term((AG903_DMACMgrHandle*)handle, &desc, &end);	/* Callback (zQ) */
						if(NULL != desc) {
							if(true == end) {
								next = NULL; /* endw肳ꂽꍇ͏zI */
							}
							else {
								next = (AG903_DMACMgrDesc*)handle->term_addr_table[handle->term_count].next;
							}
							desc_save_mem = handle->term_addr_table[handle->term_count].current;
							write_desc(desc, next, (void*)desc_save_mem, 0);	/* Current */
						}
						else {
							/* zQƒ̏ꍇAfBXNv^̓e͍XVꂸɏzQƂ */
							if(true == end) {
								desc_save_mem = handle->term_addr_table[handle->term_count].current;
								*(uint32_t*)(((uint32_t)desc_save_mem)+0x8) = 0; /* endw肳ꂽꍇ͏zI */
							}
						}
						handle->term_count = ((handle->term_count+1) % handle->term_addr_table_num);
					}
				}
			}
			else {
				/* gĂȂnh̊݃Xe[^X͖ */
			}
			term &= ~(0x1<<loop);
		}

	}

	return;
}

/*
	Errݏ.
*/
static void DMACMgrErrIntHandler(uint32_t abt, uint32_t err)
{
	AG903_DMACMgrHandleStat *handle;
	int loop;

	/*
	荞݂̕pxȂƁAȉ̃[ṽRXg傫Ȃ\邽߁A
	傫error̒PʂŃ`FbNs
	*/
	for(loop=0; loop<AG903_DMAC_UNIT_NUM; loop++) {
		if(0 == abt) {
			break;
		}
		if(abt & (0x1<<loop)) {
			if(true == handle_used_flag[loop]) {
				handle = &handle_list[loop];
				if(NULL != handle_list[loop].cb_abt) {
					void (*cb_abt)(AG903_DMACMgrHandle*) = (void(*)(AG903_DMACMgrHandle*))handle->cb_abt;
					cb_abt((AG903_DMACMgrHandle*)handle); /* Callback */
				}
				AG903_DMACMgrClearInt((AG903_DMACMgrHandle*)handle, AG903_DMAC_INTSTS_ABORT);	/* ݃Xe[^XNA */
			}
			else {
				/* gĂȂnh̊݃Xe[^X͖ */
			}
			abt &= ~(0x1<<loop);
		}
	}

	for(loop=0; loop<AG903_DMAC_UNIT_NUM; loop++) {
		if(0 == err) {
			break;
		}
		if(err & (0x1<<loop)) {
			if(true == handle_used_flag[loop]) {
				handle = &handle_list[loop];
				if(NULL != handle_list[loop].cb_error) {
					void (*cb_error)(AG903_DMACMgrHandle*) = (void(*)(AG903_DMACMgrHandle*))handle->cb_error;
					cb_error((AG903_DMACMgrHandle*)handle); /* Callback */
				}
				AG903_DMACMgrClearInt((AG903_DMACMgrHandle*)handle, AG903_DMAC_INTSTS_ERROR);	/* ݃Xe[^XNA */
			}
			else {
				/* gĂȂnh̊݃Xe[^X͖ */
			}
			err &= ~(0x1<<loop);
		}
	}

	return;
}

/**
 * @brief           DMACnh擾
 * @param           ch [in] DMAC`lԍ
 * @param           handle [out] DMACnh
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @retval          -AG903_EBUSY  nhgp
 * @description     DMACnh̎擾 (chw)
*/
int32_t AG903_DMACMgrGetHandle(int ch, AG903_DMACMgrHandle **handle)
{
	if(ch < 0 || AG903_DMAC_UNIT_NUM <= ch || NULL == handle)
	{
		return -AG903_EINVAL;
	}

	if(handle_used_flag[ch] == true)
	{
		return -AG903_EBUSY;
	}

	/* ݂̐ݒ */
	uint32_t chpos = 1 << ch;
	AG903_DMACPrmSetTIMECOUNT_INT_CLEAR(chpos);
	AG903_DMACPrmSetERR_ABORT_INT_CLEAR(chpos, chpos, chpos);

	(*handle) = (AG903_DMACMgrHandle*)&handle_list[ch];
	handle_used_flag[ch] = true;

	handle_list[ch].unit_id = ch;

	handle_list[ch].cb_term  = NULL;
	handle_list[ch].cb_error = NULL;
	handle_list[ch].cb_abt  = NULL;
	handle_list[ch].cb_wdt   = NULL;

	return AG903_ENONE;
}

/**
 * @brief           DMACnh
 * @param           handle [in] DMACnh
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @retval          -AG903_EBUSY  s
 * @description     DMACnh̉
 * @note            ̊֐̓fBXNv^Xgg]sĂƂs֎~łB
 * @note            fBXNv^XgDMACMgrnhԂŔrIɃANZXĉB
 * @note            KvɉAvAG903_INTMgrDisableIntɂDMAC̊荞݂𖳌ɂĉB
*/
int32_t AG903_DMACMgrReleaseHandle(AG903_DMACMgrHandle *handle)
{
	uint32_t unit = ((AG903_DMACMgrHandleStat*)handle - handle_list);
	uint8_t	 enable;

	/* ȃnh */
	if(unit >= AG903_DMAC_UNIT_NUM || &handle_list[unit] != (AG903_DMACMgrHandleStat*)handle)
	{
		return -AG903_EINVAL;
	}

	/* s̏ꍇ */
	enable = DMACMgr_CheckEnable(unit);
	if(true == enable) {
		return -AG903_EBUSY;
	}

	remove_desc_entry(unit);
	handle_used_flag[unit] = false;

	return AG903_ENONE;
}

/**
 * @brief           荞݃Xe[^X擾
 * @param           handle [in] DMACnh
 * @param           status [out] 荞݃Xe[^X
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     荞݃Xe[^X擾
*/
int32_t AG903_DMACMgrGetIntStatus(AG903_DMACMgrHandle *handle, uint32_t *status)
{
	uint32_t unit = ((AG903_DMACMgrHandleStat*)handle - handle_list);
	uint32_t term, abt, wdt, error;

	/* ȃnh */
	if(unit >= AG903_DMAC_UNIT_NUM || &handle_list[unit] != (AG903_DMACMgrHandleStat*)handle || NULL == status)
	{
		return -AG903_EINVAL;
	}

	AG903_DMACPrmGetTIMECOUNT_STATUS(&term);
	AG903_DMACPrmGetERR_ABORT_STATUS(&error, &wdt, &abt);
	term  = (term  >> unit) & 0x01;
	error = (error >> unit) & 0x01;
	wdt   = (wdt   >> unit) & 0x01;
	abt   = (abt   >> unit) & 0x01;

	(*status)  = (uint32_t)term;
	(*status) |= (uint32_t)(abt<<1);
	(*status) |= (uint32_t)(wdt<<2);
	(*status) |= (uint32_t)(error<<3);

	return AG903_ENONE;
}

/**
 * @brief           荞݃Xe[^XNA
 * @param           handle [in] DMACnh
 * @param           clrbit [in] NAw (rbgw)
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     荞݃Xe[^XNA
*/
int32_t AG903_DMACMgrClearInt(AG903_DMACMgrHandle *handle, uint32_t clrbit)
{
	uint32_t unit = ((AG903_DMACMgrHandleStat*)handle - handle_list);
	uint8_t	 term, abt, wdt, error;

	/* ȃnh */
	if(unit >= AG903_DMAC_UNIT_NUM || &handle_list[unit] != (AG903_DMACMgrHandleStat*)handle)
	{
		return -AG903_EINVAL;
	}

	term  = 0;
	abt   = 0;
	wdt   = 0;
	error = 0;
	if(AG903_DMAC_INTSTS_TERM&clrbit) {
		term =1;
	}
	if(AG903_DMAC_INTSTS_ABORT&clrbit) {
		abt=1;
	}
	if(AG903_DMAC_INTSTS_WDT&clrbit) {
		wdt=1;
	}
	if(AG903_DMAC_INTSTS_ERROR&clrbit) {
		error=1;
	}
	AG903_DMACPrmSetTIMECOUNT_INT_CLEAR(term<<unit);
	AG903_DMACPrmSetERR_ABORT_INT_CLEAR(error<<unit, wdt<<unit, abt<<unit);

	return AG903_ENONE;
}

/**
 * @brief           R[obNo^
 * @param           handle [in] DMACnh
 * @param           cb_term [out] I荞݂̃R[obN֐iw肵ȂꍇNULLj
 * @param           cb_abt [out] Abt荞݂̃R[obN֐iw肵ȂꍇNULLj
 * @param           cb_error [out] G[荞݂̃R[obN֐iw肵ȂꍇNULLj
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @retval          -AG903_EFAULT 荞݃nho^s
 * @description     eR[obN̓o^<p>
 *                  ȉINTManager APIgp܂B<p>
 *                  EAG903_INTMgrSetHandler<p>
 *                  EAG903_INTMgrEnableInt
 * @note            ̊֐͓INTMgrgp܂B\INTMgrĉB
 * @note            荞݂̓R[obN̒ł̓NAȂŉB
 *                  R[obN߂KvȏIドCuŃNA܂B
 * @seealso         AG903_INTMgrInit
*/
int32_t AG903_DMACMgrSetIntCallback(AG903_DMACMgrHandle *handle,
 void (*cb_term)(AG903_DMACMgrHandle*, AG903_DMACMgrDesc**, uint8_t*),
 void (*cb_abt)(AG903_DMACMgrHandle*),
 void (*cb_error)(AG903_DMACMgrHandle*))
{
	uint32_t unit = ((AG903_DMACMgrHandleStat*)handle - handle_list);
	AG903_INTMgrHdrPrm inthdr;

	/* ȃnh */
	if(unit >= AG903_DMAC_UNIT_NUM || &handle_list[unit] != (AG903_DMACMgrHandleStat*)handle)
	{
		return -AG903_EINVAL;
	}

	/* nh̓o^ */
	if(dmac_int_init_flag == false)
	{
		inthdr.atr   = AG903_INT_HLNG;
		inthdr.intno = AG903_IRQ26_DMAC;
		inthdr.func  = (void*)DMACMgrDmactcIntHandler;
		dmac_int_dmactc_hdrid = AG903_INTMgrSetHandler(&inthdr);
		if(0 >= dmac_int_dmactc_hdrid)
		{
			return -AG903_EFAULT;
		}
		dmac_int_init_flag = true;
	}

	handle_list[unit].cb_term  = (void*)cb_term;
	handle_list[unit].cb_abt  = (void*)cb_abt;
	handle_list[unit].cb_error = (void*)cb_error;

	AG903_INTMgrEnableInt(AG903_IRQ26_DMAC);

	return AG903_ENONE;
}

/**
 * @brief           R[obN
 * @param           handle [in] DMACnh
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @retval          -AG903_EFAULT 荞݃nhs
 * @retval          -AG903_EPREM  荞݃nho^
 * @description     eR[obN̉<p>
 *                  ȉINTManager APIgp܂B<p>
 *                  EAG903_INTMgrDeleteHandler<p>
 * @note            ̊֐͓INTMgrgp܂B\INTMgrĉB
 * @note            DMAC荞(IRQ26)͋֎~ɂĂ܂B
 *                  Kvɉă~hEFA܂߂ׂĂDMAC`lgpƂȂ
 *                  AG903_INTMgrDisableIntŋ֎~ĂB
 * @seealso         AG903_INTMgrInit
 * @seealso         AG903_INTMgrDisableInt
*/
int32_t AG903_DMACMgrUnsetIntCallback(AG903_DMACMgrHandle *handle)
{
	uint32_t unit = ((AG903_DMACMgrHandleStat*)handle - handle_list);
	int32_t retval;

	/* ȃnh */
	if(unit >= AG903_DMAC_UNIT_NUM || &handle_list[unit] != (AG903_DMACMgrHandleStat*)handle)
	{
		return -AG903_EINVAL;
	}

	/* nh̖o^ */
	if(dmac_int_init_flag == false || dmac_int_dmactc_hdrid <= 0)
	{
		return -AG903_EPERM;
	}
		
	retval = AG903_INTMgrDeleteHandler(dmac_int_dmactc_hdrid);
	if(retval < 0)
	{
		return -AG903_EFAULT;
	}

	dmac_int_init_flag    = false;
	dmac_int_dmactc_hdrid = 0;

	return AG903_ENONE;
}

/**
 * @brief           DMAC]iJnj
 * @param           handle [in] DMACnh
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @retval          -AG903_EBUSY  s
 * @description     DMAC]iJnj
*/
int32_t AG903_DMACMgrEnable(AG903_DMACMgrHandle *handle)
{
	uint32_t unit = ((AG903_DMACMgrHandleStat*)handle - handle_list);
	uint8_t	 enable;

	/* ȃnh */
	if(unit >= AG903_DMAC_UNIT_NUM || &handle_list[unit] != (AG903_DMACMgrHandleStat*)handle)
	{
		return -AG903_EINVAL;
	}

	/* s̏ꍇ */
	enable = DMACMgr_CheckEnable(unit);
	if(true == enable) {
		return -AG903_EBUSY;
	}

	/*
	Enableɂ^C~ODMACLDD߂ǂݍƂ̂ŁA
	̃^C~OŃJE^Zbg
	*/
	handle_list[unit].term_count = 0;

	DMACPrmParamCTRL ctrl;
	AG903_DMACPrmGetCTRL_REG(unit, &ctrl);
	ctrl.ChEn = 1;
	AG903_DMACPrmSetCTRL_REG(unit, &ctrl);

	return AG903_ENONE;
}

/**
 * @brief           DMAC]Abort
 * @param           handle [in] DMACnh
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @retval          -AG903_EPREM  ~
 * @description     DMAC]Abort
*/
int32_t AG903_DMACMgrAbort(AG903_DMACMgrHandle *handle)
{
	uint32_t unit = ((AG903_DMACMgrHandleStat*)handle - handle_list);
	uint8_t	 enable;

	/* ȃnh */
	if(unit >= AG903_DMAC_UNIT_NUM || &handle_list[unit] != (AG903_DMACMgrHandleStat*)handle)
	{
		return -AG903_EINVAL;
	}

	/* ~̏ꍇ */
	enable = DMACMgr_CheckEnable(unit);
	if(false == enable) {
		return -AG903_EPERM;
	}

	AG903_DMACPrmSetCHANNEL_ENABLE(~(1 << unit));

	return AG903_ENONE;
}

/**
 * @brief           ytFݒ
 * @param           port [in] DMAC|[gԍ
 * @param           val [in] 0ŖA1ŗL
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     ytFݒ
*/
int32_t AG903_DMACMgrSetSyncPeripheral(int port, int val)
{
	/* ȃnh */
	if(0 > port || port >= AG903_DMAC_PORT_NUM)
	{
		return -AG903_EINVAL;
	}

	if(!(val == 0 || val == 1))
	{
		return -AG903_EINVAL;
	}

	uint32_t reg;
	AG903_DMACPrmGetSYNC_PERI_IF(&reg);
	if (val == 0)
		reg &= ~(1 << port);
	else /* if == 1 */
		reg |= (1 << port);
	AG903_DMACPrmSetSYNC_PERI_IF(reg);

	return AG903_ENONE;
}

/**
 * @brief           EHb`hbOݒ
 * @param           val [in] NbNiől0xFFFFj
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     EHb`hbOݒ
 * @note            ̐ݒׂ͂ĂDMAC`lɋʂłB
*/
int32_t AG903_DMACMgrSetWatchdog(int val)
{
	if(0 > val || val > 0xFFFF)
	{
		return -AG903_EINVAL;
	}

	AG903_DMACPrmSetWATCHDOG_TIMER(val);

	return AG903_ENONE;
}

/**
 * @brief           WriteOnlyModěŒlݒ
 * @param           val [in] ݒl
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @description     WriteOnlyModěŒlݒ
 * @note            ̐ݒׂ͂ĂDMAC`lɋʂłB
*/
int32_t AG903_DMACMgrSetConstantValue(uint32_t val)
{
	AG903_DMACPrmSetCONSTANT_VALUE_WRITE_ONLY(val);

	return AG903_ENONE;
}

/**
 * @brief           Cxgݒ
 * @param           id [in] CxgID
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     Cxgݒ
*/
int32_t AG903_DMACMgrSetEvent(int id)
{
	if(0 > id || AG903_DMAC_EVENT_NUM <= id)
	{
		return -AG903_EINVAL;
	}

	AG903_DMACPrmSetGLOBAL_EVENT(1 << id, 0);

	return AG903_ENONE;
}

/**
 * @brief           CxgNA
 * @param           id [in] CxgID
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     CxgNA
*/
int32_t AG903_DMACMgrClearEvent(int id)
{
	if(0 > id || AG903_DMAC_EVENT_NUM <= id)
	{
		return -AG903_EINVAL;
	}

	AG903_DMACPrmSetGLOBAL_EVENT(0, 1 << id);

	return AG903_ENONE;
}

/**
 * @brief           CxgԎ擾
 * @param           id [in] CxgID
 * @param           val [out] Cxgԁi1Fsetj
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     CxgԎ擾
*/
int32_t AG903_DMACMgrGetEvent(int id, uint8_t *val)
{
	if(0 > id || AG903_DMAC_EVENT_NUM <= id || NULL == val)
	{
		return -AG903_EINVAL;
	}

	uint32_t reg;
	AG903_DMACPrmGetGLOBAL_EVENT(&reg);
	*val = (reg >> id) & 0x01;

	return AG903_ENONE;
}

/**
 * @brief           DMAC̏Ԃ擾
 * @param           handle [in] DMACnh
 * @param           status [out] DMACԁitrueFEnablej
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     DMAC̏Ԃ擾
*/
int32_t AG903_DMACMgrGetStatus(AG903_DMACMgrHandle *handle, AG903_DMACMgrStatus *status)
{
	uint32_t unit = ((AG903_DMACMgrHandleStat*)handle - handle_list);

	if(unit >= AG903_DMAC_UNIT_NUM || &handle_list[unit] != (AG903_DMACMgrHandleStat*)handle || NULL == status)
	{
		return -AG903_EINVAL;
	}

	status->enable = DMACMgr_CheckEnable(unit);

	return AG903_ENONE;
}

/*
	EnableԎ擾
	unitFDMAC`lԍ
*/
static uint8_t DMACMgr_CheckEnable(uint32_t unit)
{
	uint8_t retval = false;
	DMACPrmParamCTRL	ctrl;

	AG903_DMACPrmGetCTRL_REG(unit, &ctrl);
	if(0 != ctrl.ChEn) {
		retval = true;
	}

	return retval;
}


/*	fBXNv^̍\z֘A  */

/*
	ldd_flag/rdd_flag󂫂̃Gg
	seq_idFGggpV[PXID
	returnFGgindex
			i-1j󂫂Ȃ
*/
static int get_free_desc_index(int seq_id)
{
	int i;

	/*
	0͉蓖ĂĂȂԂȂ̂ŁA
	O悤seq_id蓖Ă
	*/
	if(seq_id == 0)	seq_id = TOTAL_ENTRY_NUM;

	/* ŏLDDiLocalj */
	for(i = 0 ; i < AG903_DMAC_LDD_ENTRY_NUM ; i++)
	{
		if(ldd_flag[i] == 0)
		{
			ldd_flag[i] = seq_id;
			return i;
		}
	}

	/* RDDiRemotej */
	if(NULL != rdd_data)
	{
		for(i = AG903_DMAC_LDD_ENTRY_NUM ; i < AG903_DMAC_RDD_ENTRY_NUM+AG903_DMAC_LDD_ENTRY_NUM ; i++)
		{
			if(rdd_flag[i-AG903_DMAC_LDD_ENTRY_NUM] == 0)
			{
				rdd_flag[i-AG903_DMAC_LDD_ENTRY_NUM] = seq_id;
				return i;
			}
		}
	}

	/* Ȃ */
	return -1;
}

/*
	ldd_flag/rdd_flagGg폜
	seq_idFGggpĂV[PXID
*/
static int32_t remove_desc_entry(int seq_id)
{
	int i;

	if(seq_id == 0)	seq_id = TOTAL_ENTRY_NUM;

	/* ŏLDDiLocalj */
	for(i = 0 ; i < AG903_DMAC_LDD_ENTRY_NUM ; i++)
	{
		if(ldd_flag[i] == seq_id)
		{
			ldd_flag[i] = 0;
		}
	}

	/* RDDiRemotej */
	for(i = AG903_DMAC_LDD_ENTRY_NUM ; i < AG903_DMAC_RDD_ENTRY_NUM+AG903_DMAC_LDD_ENTRY_NUM ; i++)
	{
		if(rdd_flag[i-AG903_DMAC_LDD_ENTRY_NUM] == seq_id)
		{
			rdd_flag[i-AG903_DMAC_LDD_ENTRY_NUM] = 0;
		}
	}

	return AG903_ENONE;
}

/*
	DMACIndexAhXɕύX
	indexFefBXNv^indexl
	returnFall fBXNv^̃|C^
*/
static AG903_DMACMgrDesc* desc_index_to_addr(int index)
{
	if(index < AG903_DMAC_LDD_ENTRY_NUM)
	{
		return (AG903_DMACMgrDesc*)(LOCAL_DESC_MEM_ADDR + (index * CH_REG_SIZE));
	}else
	{
		return (AG903_DMACMgrDesc*)(rdd_data+((index-AG903_DMAC_LDD_ENTRY_NUM)*CH_REG_SIZE));
	}
}

/*
	fBXNv^̃WX^̈̃AhX擾
	unitFDMAC̃`lID
	returnFfBXNv^̃WX^̈̃AhX
*/
static void* get_desc_reg_addr(int unit)
{
	return (void*)(AG903_DMAC_BASE + 0x100 + (unit*AG903_DMAC_DESC_SIZE));
}

/*
	DMAC̋󂫂̃fBXNv^
	seq_idFGggpĂV[PXID
	returnFfBXNv^̃|C^
			iNULLj 󂫂Ȃꍇ
*/
static AG903_DMACMgrDesc* get_free_desc(int seq_id)
{
	int index = get_free_desc_index(seq_id);
	if(index == -1)
	{
		return NULL;
	}


	return desc_index_to_addr(index);
}

/*
	fBXNv^
	fromF݌̃AhX
	nextFރfBXNv^̎̃AhX
	toFݐ̃AhX
	is_regFݐ悪WX^(1)A(0)
 */
static void write_desc(AG903_DMACMgrDesc *from, AG903_DMACMgrDesc *next, void *to, int is_reg)
{
	if(is_reg == 1)
	{
		*(uint32_t*)(((uint32_t)to) + 0x0)  = from->Ctrl.val;
		*(uint32_t*)(((uint32_t)to) + 0x8)  = from->SrcAddr;
		*(uint32_t*)(((uint32_t)to) + 0xc)  = from->DstAddr;
		*(uint32_t*)(((uint32_t)to) + 0x10) = (uint32_t)next;
		*(uint32_t*)(((uint32_t)to) + 0x14) = from->Trns.val;
		*(uint32_t*)(((uint32_t)to) + 0x18) = from->Stride.val;
	}else
	{
		*(uint32_t*)(((uint32_t)to) + 0x0)  = from->SrcAddr;
		*(uint32_t*)(((uint32_t)to) + 0x4)  = from->DstAddr;
		*(uint32_t*)(((uint32_t)to) + 0x8)  = (uint32_t)next;
		*(uint32_t*)(((uint32_t)to) + 0xc)  = from->Ctrl.val;
		*(uint32_t*)(((uint32_t)to) + 0x10) = from->Trns.val;
		*(uint32_t*)(((uint32_t)to) + 0x14) = from->Stride.val;
	}
}


/*
	fBXNv^̃Ggݒ肷
	handleFDMACnh
	desc_listFfBXNv^AhX
	list_numFfBXNv^Xg
*/
static int set_desc_entry(AG903_DMACMgrHandleStat *handle, AG903_DMACMgrDesc *desc_list, int list_num)
{
	int i;
	uint32_t loop;
	DMACPrmParamCFG cfg;
	AG903_DMACMgrDesc *desc = desc_list;

	/*
	^ꂽdesc̃Xgldd/rddɃRs[
	Rs[邽߁ANXgzQƂĂꍇ
	ώGɂȂĂ邱Ƃɒӂ
	i⑫j
	 xRs[̂łAzQƂꂽꍇɂ͍ēxQƂ\B
	 ̂ƂAHashmapƂĂ݂āAL[̗vf̃|C^AlRs[
	 AhXƂϐdesc_copy_entrył
	*/

	struct
	{
		/* @brief Rs[̃AhX */
		AG903_DMACMgrDesc *src;

		/* @brief Rs[̃AhX */
		AG903_DMACMgrDesc *dst;
	} desc_copy_entry[TOTAL_ENTRY_NUM];
	int desc_copy_entry_num = 0;

	/* NXgzQƂĂ邩̃tO */
	uint8_t ll_circ = false;
	uint32_t unit = (handle - handle_list);

	/* fBXNv^̕ۑ̃AhX */
	AG903_DMACMgrDesc *desc_save_mem = get_free_desc(unit);
	if(desc_save_mem == NULL)
	{
		return -AG903_ENOBUFS;
	}

	/* ŏ̃WX^ɏ */
	AG903_DMACMgrDesc first_desc;

	first_desc.SrcAddr       = 0;
	first_desc.DstAddr       = 0;
	first_desc.next			 = 0;
	first_desc.Ctrl.val		 = 0;
	first_desc.Ctrl.st.TCMsk = 1;
	first_desc.Trns.val		 = 0;
	first_desc.Stride.val	 = 0;
	write_desc(&first_desc, desc_save_mem, get_desc_reg_addr(unit), 1);

	/* term_count͖񃊃Zbg */
	handle->term_count          = 0;
	handle->term_addr_table_num = 0;
	AG903_DMACPrmGetCFG_REG(unit, &cfg);

	for(loop=0; loop<AG903_DMAC_DESC_LINK_MAX; loop++)
	{
		/* next̃AhXm肳 */
		AG903_DMACMgrDesc *next_desc_save_mem = NULL;

		if(desc->next != NULL)
		{
			/* zQƂĂP[XmF */
			for(i = 0 ; i < desc_copy_entry_num ; i++)
			{
				if(desc_copy_entry[i].src == desc->next)
				{
					next_desc_save_mem = desc_copy_entry[i].dst;
					ll_circ = true;
				}
			}

			/* zQƂĂȂ΁AVGg肷 */
			if(ll_circ != true)
			{
				next_desc_save_mem = get_free_desc(unit);

				/* 󂫂̃AhXȂꍇ */
				if(next_desc_save_mem == NULL)
				{
					remove_desc_entry(unit);
					return -AG903_ENOBUFS;
				}
			}
		}

		/* ݂descɏ */
		write_desc(desc, next_desc_save_mem, (void*)desc_save_mem, 0);

		if(true == ll_circ)
		{
			handle->term_addr_table[handle->term_addr_table_num].current = (uint32_t)desc_save_mem;
			handle->term_addr_table[handle->term_addr_table_num].next    = (uint32_t)next_desc_save_mem;
			handle->term_addr_table_num++;
		}

		desc_copy_entry[desc_copy_entry_num].src = desc;
		desc_copy_entry[desc_copy_entry_num].dst = desc_save_mem;
		desc_save_mem = next_desc_save_mem;
		desc_copy_entry_num++;

		/*
		I[͈ȉ2p^[
		1. desc->nextNULL̏ꍇ
		2. desc->nextzQƂĂꍇ
		3. Xg̗vfׂďIƂ
		*/
		if(desc->next == NULL || ll_circ == true || list_num == 0)
		{
			break;
		}

		list_num--;
		desc = desc->next;
	}
	if(AG903_DMAC_DESC_LINK_MAX <= loop)
	{
		return -AG903_EINVAL;	/* ő僊N */
	}

	return AG903_ENONE;
}


/**
 * @brief           ȈՓ]pfBXNv^ݒ
 * @param           handle [in] DMACnh
 * @param           from [in] ]AhX
 * @param           from_size [in] ]TCY
 * @param           to [in] ]惁AhX
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @retval          -AG903_EBUSY  s
 * @description     ȈՓ]pfBXNv^ݒ<p>
 *                  DMA]OɍĐݒ肵ꍇ͌̓eŏ㏑܂B
*/
int32_t AG903_DMACMgrSetSimpleTrnsDesc(AG903_DMACMgrHandle *handle, void *from,
 uint32_t from_size, void *to)
{
	uint32_t unit = ((AG903_DMACMgrHandleStat*)handle - handle_list);
	AG903_DMACMgrDesc 	dmac_desc;
	DMACPrmParamCFG cfg = {0};
	uint8_t	 enable;

	if(unit >= AG903_DMAC_UNIT_NUM || &handle_list[unit] != (AG903_DMACMgrHandleStat*)handle ||
	   NULL == from || NULL == to || 0 == from_size || AG903_DMAC_1DTRANS_MAX < from_size)
	{
		return -AG903_EINVAL;
	}

	/* s̏ꍇ */
	enable = DMACMgr_CheckEnable(unit);
	if(true == enable) {
		return -AG903_EBUSY;
	}

	cfg.ChPri = 1;
	AG903_DMACPrmSetCFG_REG(unit, &cfg);

	dmac_desc.SrcAddr           = (uint32_t)from;
	dmac_desc.DstAddr           = (uint32_t)to;
	dmac_desc.next				= 0;
	dmac_desc.Ctrl.val			= 0;
	dmac_desc.Trns.val			= 0;
	dmac_desc.Trns.d1.TCnt      = from_size;
	dmac_desc.Stride.val		= 0;

	/* WX^֏ */
	write_desc(&dmac_desc, NULL, get_desc_reg_addr(unit), 1);

	return AG903_ENONE;
}

/**
 * @brief           fBXNv^ݒ
 * @param           handle [in] DMACnh
 * @param           config [in] RtBOݒ
 * @param           desc_list [in] fBXNv^Xg
 * @param           list_num [in] fBXNv^Xg
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @retval          -AG903_EBUSY  s
 * @retval          -AG903_ENOBUFS fBXNv^ۑs
 * @description     fBXNv^ݒ
 * @description     fBXNv^Xg̐ݒAG903dl(AX51903_DSxx.pdf)̃WX^҂QƂĉB
 *                  * DMACnCTRL_REG DMACnCTRL_REG DMACnSRC_ADDR DMACnDST_ADDR DMACnLINK_LIST_POINTER DMACnTRNS_SIZE_xD DMACnSTRIDE_SRC_DST_ADDR
 *                  * ݒ肷ׂp͈̂̃RtBOݒɂ܂B
 *                  * next͈ƂēnfBXNv^Xg̃AhXŎw肵ĉB
 * @note            fBXNv^Xg͓]ƂɍĐݒ肷Kv܂B
 * @note            ̊֐͊荞݃R[obN̒Ŏgp邱Ƃł܂B
 *                  ̏ꍇ̃fBXNv^XgstaticȕϐƂĉB
 *                  荞݃R[obN̒łȂΊ֐߂ɔj邱Ƃł܂B
 * @note            ̊֐̓fBXNv^Xgg]sĂƂs֎~łB
 * @note            fBXNv^XgDMACMgrnhԂŔrIɃANZXĉB
*/
int32_t AG903_DMACMgrSetDescList(AG903_DMACMgrHandle *handle, AG903_DMACMgrConfig *config,
	                                                          AG903_DMACMgrDesc *desc_list, int list_num)
{
	uint32_t unit = ((AG903_DMACMgrHandleStat*)handle - handle_list);
	uint8_t	 enable;

	if(unit >= AG903_DMAC_UNIT_NUM || &handle_list[unit] != (AG903_DMACMgrHandleStat*)handle ||
	   NULL == config || NULL == desc_list || 0 == list_num)
	{
		return -AG903_EINVAL;
	}

	/* s̏ꍇ */
	enable = DMACMgr_CheckEnable(unit);
	if(true == enable) {
		return -AG903_EBUSY;
	}

	if(false == initdesc) {
		initdesc = true;
		AG903_DMACPrmSetLOCAL_DESC_MEM_BASE(LOCAL_DESC_MEM_ADDR);
	}
	DMACPrmParamCFG cfg = {
		.TCIntMsk    = config->st.TCIntMsk,
		.ErrIntMsk	 = config->st.ErrIntMsk,
		.AbtIntMsk	 = config->st.AbtIntMsk,
		.SrcRS		 = config->st.SrcRS,
		.SrcHEn		 = config->st.SrcHEn,
		.DstRS		 = config->st.DstRS,
		.DstHEn		 = config->st.DstHEn,
		.LLPCnt		 = config->st.LLPCnt,
		.ChGntWin	 = config->st.ChGntWin,
		.ChPri		 = config->st.ChPri,
		.WOMode		 = config->st.WOMode,
		.UnalignMode = config->st.UnalignMode,
	};
	AG903_DMACPrmSetCFG_REG(unit, &cfg);

	return set_desc_entry((AG903_DMACMgrHandleStat*)handle, desc_list, list_num);
}

/**
 * @brief           fBXNv^j
 * @param           handle [in] DMACnh
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     fBXNv^j
 * @note            ̊֐̓fBXNv^Xgg]sĂƂs֎~łB
 * @note            fBXNv^XgDMACMgrnhԂŔrIɃANZXĉB
*/
int32_t AG903_DMACMgrRemoveDescList(AG903_DMACMgrHandle *handle)
{
	uint32_t unit = ((AG903_DMACMgrHandleStat*)handle - handle_list);

	/* ȃnh */
	if(unit >= AG903_DMAC_UNIT_NUM || &handle_list[unit] != (AG903_DMACMgrHandleStat*)handle)
	{
		return -AG903_EINVAL;
	}

	return remove_desc_entry(unit);
}

/**
 * @brief           [hfBXNv^̈AhXw
 * @param           addr [in] [hfBXNv^̈AhX
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @retval          -AG903_EBUSY  s
 * @retval          -AG903_EPERM  ݒ
 * @description     [hfBXNv^̈AhXw
 * @note            ̐ݒׂ͂ĂDMAC`lɋʂłB
 * @note            fBXNv^Xgg]JnOɐݒ肵ĉB
 * @note            [gfBXNv^̈ AG903_DMAC_LDD_ENTRY_NUM ~ 20oCgKvłB
 * @note            [gfBXNv^̈̃̓LbVsɂĉB
 * @note            ̊֐̓fBXNv^Xgg]sĂƂs֎~łB
 * @seealso         AG903_DMAC_LDD_ENTRY_NUM
*/
int32_t AG903_DMACMgrSetRemoteDescAddr(void *addr)
{
	int32_t  retval = AG903_ENONE;
	uint32_t loop;
	uint8_t	 enable;

	if(NULL == addr)
	{
		return -AG903_EINVAL;
	}
	if(NULL != rdd_data)
	{
		return -AG903_EPERM;
	}

	for(loop=0; loop<AG903_DMAC_UNIT_NUM; loop++)
	{
		enable = DMACMgr_CheckEnable(loop);
		if(true == enable) {
			retval = -AG903_EBUSY;
			break;
		}
	}
	if(AG903_ENONE == retval)
	{
		rdd_data = (uint8_t*)addr;
	}

	return retval;
}

/**
 * @brief           [hfBXNv^̈AhXNA
 * @param           none
 * @return          G[R[h
 * @retval          AG903_ENONE  I
 * @retval          -AG903_EBUSY  s
 * @description     [hfBXNv^̈AhXNA
 * @note            ̐ݒׂ͂ĂDMAC`lɋʂłB
 * @note            ̊֐̓fBXNv^Xgg]sĂƂs֎~łB
*/
int32_t AG903_DMACMgrClearRemoteDescAddr(void)
{
	int32_t  retval = AG903_ENONE;
	uint32_t loop;
	uint8_t	 enable;

	for(loop=0; loop<AG903_DMAC_UNIT_NUM; loop++)
	{
		enable = DMACMgr_CheckEnable(loop);
		if(true == enable) {
			retval = -AG903_EBUSY;
			break;
		}
	}
	if(AG903_ENONE == retval)
	{
		rdd_data = NULL;
	}

	return retval;
}
