/**
 * @brief           I2C Manager
 * @author          AXELL CORPORATION
 * @description     I2C Manager Layer
 * @note            none
 * @history         2017_02_22  
 * @history         2017_10_26  Ver2.0
 * @history         2019_03_08  [SDK2.2] I2Cł̌yȕϐ̏C (#2111)
 */
/* DOM-IGNORE-BEGIN */
/*
 * This program was created by AXELL CORPORATION.
 * Copyright (C) 2017-2019 AXELL CORPORATION, all rights reserved.
 */
/* DOM-IGNORE-END */

#include "AG903_errno.h"
#include "AG903_intno.h"
#include "i2c/i2cmgr.h"
#include "i2c/i2cprm.h"
#include "i2c.h"
#include "int/intmgr.h"


typedef void (*AG903_I2CMgrIntHdr)(void);	/* R[obN֐^ */

typedef union _AG903_I2CMgrArg{
	AG903_I2CMgrWriteParam	wr;	/** Writep[^ */
	AG903_I2CMgrReadParam	rd;	/** Readp[^ */
} AG903_I2CMgrArg;

typedef struct _AG903_I2CMgrQue{
	AG903_I2CMgrArg	arg;		/**  */
	uint32_t	proc;			/** vZX */
	uint32_t	hdlnum;			/** nhԍ */
} AG903_I2CMgrQue;

typedef struct _AG903_I2CMgrQueStat{
	AG903_I2CMgrQue	que[AG903_I2C_QUENUM]; /* Que */
	uint32_t	wp;				/** Write Pointer */
	uint32_t	rp;				/** Read Pointer */
} AG903_I2CMgrQueStat;

typedef struct _AG903_I2CMgrChStat{
	AG903_I2CMgrArg 	arg;	/** I2CʐMp^\ */
	uint32_t	hdlnum;			/** nhԍ */
	uint32_t	stat;			/** I2CʐM */
	uint32_t	sequence;		/** V[PXNo. */
	uint32_t	cnt;			/** f[^JEg */
	int32_t		hdrid;			/** nhID */
} AG903_I2CMgrChStat;

typedef struct _AG903_I2CMgrHandleStat{
	AG903_I2CMgrClbk	clbk;	/** R[obN */
	uint32_t	stat;			/** nh */
	uint8_t		lock;			/** bN */
	uint8_t		reserve[3];		/** \ */
} AG903_I2CMgrHandleStat;

static AG903_I2CMgrChStat		I2cChStat[AG903_I2C_CH_NUM];							/** ch */
static AG903_I2CMgrHandleStat	I2cHandleStat[AG903_I2C_CH_NUM][AG903_I2C_HANDLE_NUM];	/** handle */
static AG903_I2CMgrQueStat		I2cQue[AG903_I2C_CH_NUM];								/** Que */

static void I2CMgr_InitState(uint8_t ch);
static void I2CMgr_MainProcess(uint8_t ch);
static int32_t I2CMgr_ExecuteProcess(uint8_t ch, uint32_t hdlnum, uint32_t proc, AG903_I2CMgrArg* arg);
static int32_t I2CMgr_MasterWrite(uint8_t ch, uint32_t hdlnum, AG903_I2CMgrWriteParam* param);
static int32_t I2CMgr_MasterRead(uint8_t ch, uint32_t hdlnum, AG903_I2CMgrReadParam* param);
static int32_t I2CMgr_SlaveWrite(uint8_t ch, uint32_t hdlnum, AG903_I2CMgrWriteParam* param);
static int32_t I2CMgr_SlaveRead(uint8_t ch, uint32_t hdlnum, AG903_I2CMgrReadParam* param);
static int32_t I2CMgr_SetFrequency(uint8_t ch, uint32_t freq);
static int32_t I2CMgr_SetClock(uint8_t ch, uint32_t cdr, uint16_t tsr, uint8_t gsr);
static int32_t I2CMgr_CheckHandle(AG903_I2CMgrHandle* handle, uint8_t* ch, uint32_t* hdlnum);
static int32_t I2CMgr_SetQue(uint8_t ch, AG903_I2CMgrQue* que);
static int32_t I2CMgr_GetQue(uint8_t ch, AG903_I2CMgrQue* que);
static uint32_t I2CMgr_GetNextQuePoint(uint32_t current);
static int32_t I2CMgr_GetQueCount(uint8_t ch, uint32_t* count);
static void I2CMgr_Inthdr0(void);
static void I2CMgr_Inthdr1(void);
static void I2CMgr_IntProcess(uint8_t ch);
static int32_t I2CMgr_IntMasterWrite(uint8_t ch, uint32_t* event);
static int32_t I2CMgr_IntMasterRead(uint8_t ch, uint32_t* event);
static int32_t I2CMgr_IntSlaveWrite(uint8_t ch, uint32_t* event);
static int32_t I2CMgr_IntSlaveRead(uint8_t ch, uint32_t* event);


static const AG903_I2CMgrIntHdr I2cIntHdr[AG903_I2C_CH_NUM] =
{ I2CMgr_Inthdr0, I2CMgr_Inthdr1 };

/**
 * @brief           I2CW[
 * @param           ch [in] I2C`lԍi0`1j
 * @return          
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُAftHgݒlُ
 * @retval          -AG903_EFAULT si݃nho^sj
 * @description     Ԃ̏SCLNbNݒ܂BVXeNCALLĉB<p>
 *                  SCL͈ȉ̒`ɏ]Đݒ肳܂B<p>
 *                  AG903_I2C_DFLT_SCL , AG903_I2C_DFLT_GSR , AG903_I2C_DFLT_TSR
 * @note            gp𒆎~Ƃ͕KvɉAvAG903_INTMgrDisableIntɂI2C̊荞݂𖳌ɂĉB
*/
int32_t	AG903_I2CMgrInit(uint8_t ch)
{
	AG903_INTMgrHdrPrm inthdr;
	int32_t	 retval = AG903_ENONE;
	int32_t	 hdrid;

	if(AG903_I2C_CH_NUM <= ch) {
		return -AG903_EINVAL;
	}

	I2CMgr_InitState(ch);	/* ԏ */

	if(0 >= I2cChStat[ch].hdrid) {
		inthdr.atr   = AG903_INT_HLNG;
		inthdr.intno = AG903_IRQ44_IIC0+ch;
		inthdr.func  = (void*)I2cIntHdr[ch];
		hdrid = AG903_INTMgrSetHandler(&inthdr);
		if(0 >= hdrid) {
			return -AG903_EFAULT;
		}
		I2cChStat[ch].hdrid = hdrid;
	}
	
	AG903_INTMgrEnableInt(AG903_IRQ44_IIC0+ch);

	retval = I2CMgr_SetFrequency(ch, AG903_I2C_DFLT_SCL);

	return retval;
}

/**
 * @brief           SCLNbNݒ
 * @param           ch [in] I2C`lԍi0`1j
 * @param           param [in] p[^
 * @return          SCLNbNݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     SCLNbNݒ肵܂B<p>
 *                  CLK_FTP̎g AG903_I2C_CLK Œ`܂B
 * @note            p[^ɂẮwAX51903dlxQƂĉB
*/
int32_t AG903_I2CMgrSetClock(uint8_t ch, AG903_I2CMgrClkPrm* param)
{
	int32_t retval = AG903_ENONE;
	
	if( (AG903_I2C_CH_NUM <= ch) ||
		(NULL == param) ) {
		return -AG903_EINVAL;
	}
	
	retval = I2CMgr_SetClock(ch, param->cdr_val, param->tsr_val, param->gsr_val);

	return retval;
}

/**
 * @brief           X[uAhXݒ
 * @param           ch [in] I2C`lԍi0`1j
 * @param           saddr [in] X[uAhX
 * @return          X[uAhXݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     X[uAhX̐ݒs܂B
*/
int32_t AG903_I2CMgrSetSlaveAddress(uint8_t ch, uint16_t saddr)
{
	int32_t	 retval = AG903_ENONE;
	
	if( (AG903_I2C_CH_NUM <= ch) ||
		(0xFC00 & saddr) ){
		return -AG903_EINVAL;
	}
	
	AG903_I2CPrmSetSlaveAddr(ch, saddr);
	
	return retval;
}

/**
 * @brief           I2Cnh擾
 * @param           ch [in] I2C`lԍi0`1j
 * @param           handle [out] YI2Cnh
 * @return          nh擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgpi󂫕sj
 * @description     w肵CH̃nh擾܂B
 * @note            CH̃nhƏ̃L[CO͈ȉŒ`܂B<p>
 *                  AG903_I2C_HANDLE_NUM , AG903_I2C_QUENUM<p>
 *                  AG903_I2C_HANDLE_NUM<= AG903_I2C_QUENUMƂāAnhɑ΂ď\ȃL[ݒ肵ĉB
*/
int32_t	AG903_I2CMgrGetHandle(uint8_t ch,  AG903_I2CMgrHandle** handle)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t loop;
	
	if( (AG903_I2C_CH_NUM <= ch) ||
		(NULL == handle) ) {
		return -AG903_EINVAL;
	}
	
	for(loop=0; loop<AG903_I2C_HANDLE_NUM; loop++) {
		if(false == I2cHandleStat[ch][loop].lock) {
			I2cHandleStat[ch][loop].lock = true;
			break;
		}
	}
	if(AG903_I2C_HANDLE_NUM <= loop) {
		return -AG903_EBUSY;
	}
	
	I2cHandleStat[ch][loop].stat = AG903_I2C_HANDLE_IDLE;
	I2cHandleStat[ch][loop].clbk = NULL;

	(*handle) = (AG903_I2CMgrHandle*)&I2cHandleStat[ch][loop];

	return retval;
}

/**
 * @brief           I2Cnh
 * @param           handle [in] I2Cnh
 * @return          nh
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgpisj
 * @description     nh܂B<p>쒆Ƀnhꂽꍇ̓G[Ԃ܂B
*/
int32_t	AG903_I2CMgrReleaseHandle(AG903_I2CMgrHandle* handle)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t hdlnum;
	uint8_t	 ch;
	
	result = I2CMgr_CheckHandle(handle, &ch, &hdlnum);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(AG903_I2C_HANDLE_IDLE != I2cHandleStat[ch][hdlnum].stat) {
		return -AG903_EBUSY;
	}
	
	I2cHandleStat[ch][hdlnum].clbk = NULL;
	I2cHandleStat[ch][hdlnum].lock = false;
	
	return retval;
}

/**
 * @brief           R[obNo^
 * @param           handle [in] I2Cnh
 * @param           clbk [in] R[obN֐|C^
 * @return          R[obNo^
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     R[obN֐o^܂B
*/
int32_t AG903_I2CMgrSetCallback(AG903_I2CMgrHandle* handle, AG903_I2CMgrClbk clbk)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t hdlnum;
	uint8_t	 ch;

	result = I2CMgr_CheckHandle(handle, &ch, &hdlnum);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	I2cHandleStat[ch][hdlnum].clbk = clbk;
	
	return retval;
}

/**
 * @brief           Master Write
 * @param           handle [in] I2Cnh
 * @param           param [in] p[^
 * @return          Master Write
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgpiL[COMAXj
 * @description     Master Writes܂B<p>
 *                  wTCYWriteŃR[obN܂B<p>
 *                  Stop ConditionLƂꍇ́AWriteStop Condition𔭍s܂B<p>
 *                  Restart condition𔺂ReadsꍇStop ConditionƂāA<p>
 *                  f[^Write AG903_I2CmgrReadData ֐gpĉB
*/
int32_t AG903_I2CMgrMasterWrite(AG903_I2CMgrHandle* handle, AG903_I2CMgrWriteParam* param)
{
	AG903_I2CMgrQue que;
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t hdlnum;
	uint8_t	 ch;

	result = I2CMgr_CheckHandle(handle, &ch, &hdlnum);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == param) {
		return -AG903_EINVAL;
	}
	if( (NULL == param->buf) ||
		(0 == param->size) ) {
		return -AG903_EINVAL;
	}
	if(0xFF00 & param->addr) {
		if( (AG903_I2C_10BITADDR_SIG != (param->addr>>11)) ||
			(0x0100 & param->addr) ) {	/* bit8FR/W */
			return -AG903_EINVAL;
		}
	}
	else {
		if(0x0001 & param->addr) {      /* bit0FR/W */
			return -AG903_EINVAL;
		}
	}
	
	que.proc   = AG903_I2C_PROC_MASTER_WRITE;
	que.hdlnum = hdlnum;
	que.arg.wr = (*param);	/* \̃Rs[ */
	retval = I2CMgr_SetQue(ch, &que);
	if(AG903_ENONE == retval) {
		I2cHandleStat[ch][hdlnum].stat = AG903_I2C_HANDLE_QUEUING;
		I2CMgr_MainProcess(ch);
	}

	return retval;
}

/**
 * @brief           Master Read
 * @param           handle [in] I2Cnh
 * @param           param [in] p[^
 * @return          Master Read
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgpiL[COMAXj
 * @description     Master Reads܂B<p>
 *                  wTCYReadŃR[obN܂B
*/
int32_t AG903_I2CMgrMasterRead(AG903_I2CMgrHandle* handle, AG903_I2CMgrReadParam* param)
{
	AG903_I2CMgrQue que;
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t hdlnum;
	uint8_t	 ch;

	result = I2CMgr_CheckHandle(handle, &ch, &hdlnum);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == param) {
		return -AG903_EINVAL;
	}
	if( (NULL == param->buf) ||
		(0 == param->size) ) {
		return -AG903_EINVAL;
	}
	if(0xFF00 & param->addr) {
		if( (AG903_I2C_10BITADDR_SIG != (param->addr>>11)) ||
			(0x0100 & ~param->addr) ) {	/* bit8FR/W */
			return -AG903_EINVAL;
		}
	}
	else {
		if(0x0001 & ~param->addr) {     /* bit0FR/W */
			return -AG903_EINVAL;
		}
	}
	
	que.proc   = AG903_I2C_PROC_MASTER_READ;
	que.hdlnum = hdlnum;
	que.arg.rd = (*param);	/* \̃Rs[ */
	retval = I2CMgr_SetQue(ch, &que);
	if(AG903_ENONE == retval) {
		I2cHandleStat[ch][hdlnum].stat = AG903_I2C_HANDLE_QUEUING;
		I2CMgr_MainProcess(ch);
	}

	return retval;
}

/**
 * @brief           Slave Write
 * @param           handle [in] I2Cnh
 * @param           buf [in] Writef[^
 * @param           size [in] WriteTCY
 * @return          Slave Write
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgpiL[COMAXj
 * @description     Slave Writes܂B<p>
 *                  wTCYWriteŃR[obN܂B
*/
int32_t AG903_I2CMgrSlaveWrite(AG903_I2CMgrHandle* handle, uint8_t* buf, uint32_t size)
{
	AG903_I2CMgrQue que;
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t hdlnum;
	uint8_t	 ch;
	
	result = I2CMgr_CheckHandle(handle, &ch, &hdlnum);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if((NULL == buf) || (0 == size)) {
		return -AG903_EINVAL;
	}

	que.proc   = AG903_I2C_PROC_SLAVE_WRITE;
	que.hdlnum = hdlnum;
	que.arg.wr.buf  = buf;
	que.arg.wr.size = size;
	que.arg.wr.addr = 0;
	retval = I2CMgr_SetQue(ch, &que);
	if(AG903_ENONE == retval) {
		I2cHandleStat[ch][hdlnum].stat = AG903_I2C_HANDLE_QUEUING;
		I2CMgr_MainProcess(ch);
	}

	return retval;
}

/**
 * @brief           Slave Read
 * @param           handle [in] I2Cnh
 * @param           buf [in] Readf[^
 * @param           size [in] ReadTCY
 * @return          Slave Read
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @retval          -AG903_EBUSY  nhgpiL[COMAXj
 * @description     Slave Reads܂B<p>
 *                  wTCYReadŃR[obN܂B
*/
int32_t AG903_I2CMgrSlaveRead(AG903_I2CMgrHandle* handle, uint8_t* buf, uint32_t size)
{
	AG903_I2CMgrQue que;
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t hdlnum;
	uint8_t	 ch;
	
	result = I2CMgr_CheckHandle(handle, &ch, &hdlnum);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if((NULL == buf) || (0 == size)) {
		return -AG903_EINVAL;
	}

	que.proc   = AG903_I2C_PROC_SLAVE_READ;
	que.hdlnum = hdlnum;
	que.arg.wr.buf  = buf;
	que.arg.wr.size = size;
	que.arg.wr.addr = 0;
	retval = I2CMgr_SetQue(ch, &que);
	if(AG903_ENONE == retval) {
		I2cHandleStat[ch][hdlnum].stat = AG903_I2C_HANDLE_QUEUING;
		I2CMgr_MainProcess(ch);
	}

	return retval;
}

/**
 * @brief           Rg[ Zbg
 * @param           handle [in] I2Cnh
 * @return          Zbg
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُAftHgݒlُ
 * @description     Rg[ɑ΂ăZbgs܂Bisj<p>
 *                  L[̓eNA܂B
*/
int32_t AG903_I2CMgrReset(AG903_I2CMgrHandle* handle)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t hdlnum;
	uint32_t status;
	uint32_t loop;
	uint8_t	 ch;

	result = I2CMgr_CheckHandle(handle, &ch, &hdlnum);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	
	AG903_I2CPrmResetController(ch);

	I2cQue[ch].wp = 0;
	I2cQue[ch].rp = 0;	/* QueNA */

	I2cChStat[ch].stat  = AG903_I2C_CH_IDLE;	/* ԏ */
	for(loop=0; loop<AG903_I2C_HANDLE_NUM; loop++) {
		I2cHandleStat[ch][loop].stat = AG903_I2C_HANDLE_IDLE;
	}

	AG903_I2CPrmGetStatus(ch, &status);	/* Read clear */
	retval = I2CMgr_SetFrequency(ch, AG903_I2C_DFLT_SCL);

	return retval;
}


/**
 * @brief           L[JEg擾
 * @param           handle [in] I2Cnh
 * @param           count [out] L[JEgl
 * @return          L[JEg擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     L[ɓo^Ă鏈iJEgjԂ܂B
*/
int32_t AG903_I2CMgrGetQueCount(AG903_I2CMgrHandle* handle, uint32_t* count)
{
	int32_t	 retval = AG903_ENONE;
	int32_t	 result;
	uint32_t hdlnum;
	uint8_t	 ch;

	result = I2CMgr_CheckHandle(handle, &ch, &hdlnum);
	if(AG903_ENONE != result) {
		return -AG903_EINVAL;
	}
	if(NULL == count) {
		return -AG903_EINVAL;
	}
	
	retval = I2CMgr_GetQueCount(ch, count);
	
	return retval;
}


/*
	ԏ.
	chFI2C`lԍi0`1j.
*/
static void I2CMgr_InitState(uint8_t ch)
{
	uint32_t loop;

	I2cChStat[ch].hdlnum = 0;
	I2cChStat[ch].stat  = AG903_I2C_CH_IDLE;
	I2cChStat[ch].sequence = AG903_I2C_SEQ_NONE;
	I2cChStat[ch].cnt   = 0;
	I2cQue[ch].wp = 0;
	I2cQue[ch].rp = 0;
	for(loop=0; loop<AG903_I2C_HANDLE_NUM; loop++) {
		I2cHandleStat[ch][loop].clbk = NULL;
		I2cHandleStat[ch][loop].stat = AG903_I2C_HANDLE_IDLE;
		I2cHandleStat[ch][loop].lock = false;
	}

	return;
}

/*
	I2CC֐.
	chFI2C`lԍi0`1j.
*/
static void I2CMgr_MainProcess(uint8_t ch)
{
	AG903_I2CMgrQue que;
	int32_t	 result;
	
	if(AG903_I2C_CH_IDLE != I2cChStat[ch].stat) {
		return;	/*  */
	}
	
	result = I2CMgr_GetQue(ch, &que);
	if(AG903_ENONE != result) {
		return;	/* ҂Ȃ */
	}
	
	I2cHandleStat[ch][que.hdlnum].stat = AG903_I2C_HANDLE_EXECUTE;
	result = I2CMgr_ExecuteProcess(ch, que.hdlnum, que.proc, &que.arg);
	if(AG903_ENONE != result) {
		I2cHandleStat[ch][que.hdlnum].stat = AG903_I2C_HANDLE_IDLE;
	}
	
	return;
}

/*
	I2Cs֐.
	chFI2C`lԍi0`1j.
	hdlnumFnhԍ.
	procFvZX.
	argFp[^.
*/
static int32_t I2CMgr_ExecuteProcess(uint8_t ch, uint32_t hdlnum, uint32_t proc, AG903_I2CMgrArg* arg)
{
	int32_t	 retval = AG903_ENONE;

	switch(proc) {
		case AG903_I2C_PROC_MASTER_WRITE:
			retval = I2CMgr_MasterWrite(ch, hdlnum, &arg->wr);
			break;
		case AG903_I2C_PROC_MASTER_READ:
			retval = I2CMgr_MasterRead(ch, hdlnum, &arg->rd);
			break;
		case AG903_I2C_PROC_SLAVE_WRITE:
			retval = I2CMgr_SlaveWrite(ch, hdlnum, &arg->wr);
			break;
		case AG903_I2C_PROC_SLAVE_READ:
			retval = I2CMgr_SlaveRead(ch, hdlnum, &arg->rd);
			break;
		default:
			/* Ȃ */
			retval = -AG903_EINVAL;
			break;
	}
	
	return retval;
}

/*
	Master Write.
	chFI2C`lԍi0`1j.
	hdlnumFnhԍ.
	paramFp[^.
*/
static int32_t I2CMgr_MasterWrite(uint8_t ch, uint32_t hdlnum, AG903_I2CMgrWriteParam* param)
{
	int32_t	 retval = AG903_ENONE;

	if( (AG903_I2C_CH_NUM <= ch) ||
		(AG903_I2C_HANDLE_NUM <= hdlnum) ||
		(NULL == param) ) {
		return -AG903_EINVAL;
	}
	
	if(AG903_I2C_CH_IDLE != I2cChStat[ch].stat) {
		return -AG903_EBUSY;
	}
	
	I2cChStat[ch].stat = AG903_I2C_CH_MASTER_WRITE;
	I2cChStat[ch].arg.wr = (*param);	/* \̃Rs[ */
	I2cChStat[ch].cnt = 0;
	I2cChStat[ch].hdlnum = hdlnum;
	if(0xFF00 & param->addr) {
		I2cChStat[ch].sequence = AG903_I2C_SEQ_SND_SADDR_H;
		AG903_I2CPrmSetExtAddrMode(ch);
		AG903_I2CPrmWriteData(ch, (uint8_t)(I2cChStat[ch].arg.wr.addr>>8));
	}
	else {
		I2cChStat[ch].sequence = AG903_I2C_SEQ_SND_SADDR_L;
		AG903_I2CPrmClearExtAddrMode(ch);
		AG903_I2CPrmWriteData(ch, (uint8_t)(I2cChStat[ch].arg.wr.addr));
	}
	AG903_I2CPrmClearControl(ch, (AG903_I2C_CR_STOP_BIT|AG903_I2C_CR_ACKNACK_BIT));
	AG903_I2CPrmSetControl(ch, AG903_I2C_CR_ALLINT_BIT|AG903_I2C_CR_I2CEN_BIT|AG903_I2C_CR_SCLEN_BIT|
		                                               AG903_I2C_CR_START_BIT|AG903_I2C_CR_TBEN_BIT);
	
	return retval;
}

/*
	Master Read.
	chFI2C`lԍi0`1j.
	hdlnumFnhԍ.
	paramFp[^.
*/
static int32_t I2CMgr_MasterRead(uint8_t ch, uint32_t hdlnum, AG903_I2CMgrReadParam* param)
{
	int32_t	retval = AG903_ENONE;

	if( (AG903_I2C_CH_NUM <= ch) ||
		(AG903_I2C_HANDLE_NUM <= hdlnum) ||
		(NULL == param) ) {
		return -AG903_EINVAL;
	}
	
	if(AG903_I2C_CH_IDLE != I2cChStat[ch].stat) {
		return -AG903_EBUSY;
	}

	I2cChStat[ch].stat = AG903_I2C_CH_MASTER_READ;
	I2cChStat[ch].arg.rd = (*param);	/* \̃Rs[ */
	I2cChStat[ch].cnt = 0;
	I2cChStat[ch].hdlnum = hdlnum;
	if(0xFF00 & param->addr) {
		I2cChStat[ch].sequence = AG903_I2C_SEQ_SND_SADDR_H;
		AG903_I2CPrmSetExtAddrMode(ch);
		AG903_I2CPrmWriteData(ch, (uint8_t)(I2cChStat[ch].arg.wr.addr>>8));
	}
	else {
		I2cChStat[ch].sequence = AG903_I2C_SEQ_SND_SADDR_L;
		AG903_I2CPrmClearExtAddrMode(ch);
		AG903_I2CPrmWriteData(ch, (uint8_t)(I2cChStat[ch].arg.wr.addr));
	}
	AG903_I2CPrmClearControl(ch, AG903_I2C_CR_STOP_BIT|AG903_I2C_CR_ACKNACK_BIT);
	AG903_I2CPrmSetControl(ch, AG903_I2C_CR_ALLINT_BIT|AG903_I2C_CR_I2CEN_BIT|AG903_I2C_CR_SCLEN_BIT|
													   AG903_I2C_CR_START_BIT|AG903_I2C_CR_TBEN_BIT);

	return retval;
}

/*
	Slave Write.
	chFI2C`lԍi0`1j.
	hdlnumFnhԍ.
	paramFp[^.
*/
static int32_t I2CMgr_SlaveWrite(uint8_t ch, uint32_t hdlnum, AG903_I2CMgrWriteParam* param)
{
	int32_t	 retval = AG903_ENONE;

	if( (AG903_I2C_CH_NUM <= ch) ||
		(AG903_I2C_HANDLE_NUM <= hdlnum) ||
		(NULL == param) ) {
		return -AG903_EINVAL;
	}
	
	if(AG903_I2C_CH_IDLE != I2cChStat[ch].stat) {
		return -AG903_EBUSY;
	}
	
	I2cChStat[ch].stat = AG903_I2C_CH_SLAVE_WRITE;
	I2cChStat[ch].arg.wr = (*param);	/* \̃Rs[ */
	I2cChStat[ch].cnt = 0;
	I2cChStat[ch].hdlnum = hdlnum;
	I2cChStat[ch].sequence = AG903_I2C_SEQ_RCV_SADDR;
	
	AG903_I2CPrmClearControl(ch, AG903_I2C_CR_SCLEN_BIT|AG903_I2C_CR_ACKNACK_BIT);
	AG903_I2CPrmSetControl(ch, AG903_I2C_CR_ALLINT_BIT|AG903_I2C_CR_I2CEN_BIT);
	
	return retval;
}

/*
	Slave Read.
	chFI2C`lԍi0`1j.
	hdlnumFnhԍ.
	paramFp[^.
*/
static int32_t I2CMgr_SlaveRead(uint8_t ch, uint32_t hdlnum, AG903_I2CMgrReadParam* param)
{
	int32_t	retval = AG903_ENONE;

	if( (AG903_I2C_CH_NUM <= ch) ||
		(AG903_I2C_HANDLE_NUM <= hdlnum) ||
		(NULL == param) ) {
		return -AG903_EINVAL;
	}
	
	if(AG903_I2C_CH_IDLE != I2cChStat[ch].stat) {
		return -AG903_EBUSY;
	}

	I2cChStat[ch].stat = AG903_I2C_CH_SLAVE_READ;
	I2cChStat[ch].arg.rd = (*param);	/* \̃Rs[ */
	I2cChStat[ch].cnt = 0;
	I2cChStat[ch].hdlnum = hdlnum;
	I2cChStat[ch].sequence = AG903_I2C_SEQ_RCV_SADDR;

	AG903_I2CPrmClearControl(ch, AG903_I2C_CR_SCLEN_BIT);
	AG903_I2CPrmSetControl(ch, AG903_I2C_CR_ALLINT_BIT|AG903_I2C_CR_I2CEN_BIT|AG903_I2C_CR_ACKNACK_BIT);

	return retval;
}

/*
	I2CNbNݒ肷igwj.
	chFI2C`lԍi0`1j.
	freqFg[Hz].
*/
static int32_t I2CMgr_SetFrequency(uint8_t ch, uint32_t freq)
{
	int32_t	retval = AG903_ENONE;
	int32_t	val;
	int32_t	cdr;
	
	if(AG903_I2C_CH_NUM <= ch) {
		return -AG903_EINVAL;
	}
	
	val = (AG903_I2C_DFLT_GSR >> 1);
	cdr = ((AG903_I2C_CLK >> 1) / freq) - (2 + val);							/* cdr  = (PCLK/(2*freq))-2-GSR/2 */
	
	retval = I2CMgr_SetClock(ch, cdr, AG903_I2C_DFLT_TSR, AG903_I2C_DFLT_GSR);	/* freq = PCLK/(2*(cdr+2)+GSR) */
	
	return retval;
}

/*
	I2CNbNݒ肷iWX^ݒj.
	chFI2C`lԍi0`1j.
	cdrFCDRݒl.
	tsrFTSRݒl.
	gsrFGSRݒl.
*/
static int32_t I2CMgr_SetClock(uint8_t ch, uint32_t cdr, uint16_t tsr, uint8_t gsr)
{
	if(	(AG903_I2C_CH_NUM <= ch) ||
		(0 >= tsr) ||
		(0xFC00 & tsr) || /* max 10bit */
		(0xF0   & gsr) || /* max 4bit */
		(cdr <= (3U+gsr+tsr)) ) {	/* Rg[ */
		return -AG903_EINVAL;
	}

	AG903_I2CPrmSetClockDivision(ch, cdr);
	AG903_I2CPrmSetGlitchSuppression(ch, tsr, gsr);

	return AG903_ENONE;
}

/*
	nh`FbN.
	handleFI2Cnh.
	chFI2C`l.
	hdlnumFnhԍ.
*/
static int32_t I2CMgr_CheckHandle(AG903_I2CMgrHandle* handle, uint8_t* ch, uint32_t* hdlnum)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t val;
	uint32_t get_num;
	uint32_t get_ch;
	
	val = ((uint32_t)handle - (uint32_t)I2cHandleStat) / sizeof(AG903_I2CMgrHandleStat);
	get_ch  = val / AG903_I2C_HANDLE_NUM;
	get_num = val % AG903_I2C_HANDLE_NUM;
	
	if( (AG903_I2C_CH_NUM <= get_ch) ||
		(&I2cHandleStat[get_ch][get_num] != (AG903_I2CMgrHandleStat*)handle) ) {
		return -AG903_EINVAL;
	}
	
 	(*ch) = (uint8_t)get_ch;
 	(*hdlnum) = get_num;
	
	return retval;
}

/*
	L[ .
	chFI2C`l.
	queF݃L[.
*/
static int32_t I2CMgr_SetQue(uint8_t ch, AG903_I2CMgrQue* que)
{
	int32_t	 	retval = AG903_ENONE;
	uint32_t	next_wp;
	
	if( (AG903_I2C_CH_NUM <= ch) ||
		(NULL == que) ){
		return -AG903_EINVAL;
	}
	
	if(1 >= AG903_I2C_QUENUM) {	/* L[1 */
		if(0 < I2cQue[ch].wp) {
			retval = -AG903_EBUSY;	/* Ȃ */
		}
		else {
			I2cQue[ch].que[0] = (*que);	/* \̃Rs[ */
			I2cQue[ch].wp++;
		}
	}
	else {
		next_wp = I2CMgr_GetNextQuePoint(I2cQue[ch].wp);
		if(next_wp == I2cQue[ch].rp) {
			retval = -AG903_EBUSY;	/* Ȃ */
		}
		else {
			I2cQue[ch].que[next_wp] = (*que);	/* \̃Rs[ */
			I2cQue[ch].wp = next_wp;
		}
	}
	
	return retval;
}

/*
	L[ Ǐo.
	chFI2C`l.
	queFǏoL[.
*/
static int32_t I2CMgr_GetQue(uint8_t ch, AG903_I2CMgrQue* que)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t next_rp;
	
	if( (AG903_I2C_CH_NUM <= ch) ||
		(NULL == que) ){
		return -AG903_EINVAL;
	}
	
	if(1 >= AG903_I2C_QUENUM) {	/* L[1 */
		if(0 >= I2cQue[ch].wp) {
			retval = -AG903_ENOMSG;	/* L[ */
		}
		else {
			(*que)  = I2cQue[ch].que[0];	/* \̃Rs[ */
			I2cQue[ch].wp--;
		}
	}
	else {
		if(I2cQue[ch].wp == I2cQue[ch].rp) {
			retval = -AG903_ENOMSG;	/* L[ */
		}
		else {
			next_rp = I2CMgr_GetNextQuePoint(I2cQue[ch].rp);
			(*que)  = I2cQue[ch].que[next_rp];	/* \̃Rs[ */
			I2cQue[ch].rp = next_rp;
		}
	}
	
	return retval;
}

/*
	L[|C^擾.
	currentFݒl.
	nextFL[|C^.
*/
static uint32_t I2CMgr_GetNextQuePoint(uint32_t current)
{
	uint32_t next;
	
	next = current+1;
	if(AG903_I2C_QUENUM <= next) {
		next = 0;
	}
	
	return next;
}

/*
	L[JEg擾.
	chFI2C`l.
	countFL[JEgl.
*/
static int32_t I2CMgr_GetQueCount(uint8_t ch, uint32_t* count)
{
	int32_t	 retval = AG903_ENONE;
	
	if( (AG903_I2C_CH_NUM <= ch) ||
		(NULL == count) ) {
		return -AG903_EINVAL;
	}
	
	if(I2cQue[ch].rp <= I2cQue[ch].wp) {
		(*count) = I2cQue[ch].wp - I2cQue[ch].rp;
	}
	else {
		(*count) = AG903_I2C_QUENUM - I2cQue[ch].rp + I2cQue[ch].wp + 1;
	}

	return retval;
}


/*
	݃nh(CH0).
*/
static void I2CMgr_Inthdr0(void)
{
	I2CMgr_IntProcess(0);	/* CH0 */
	return;
}

/*
	݃nh(CH1).
*/
static void I2CMgr_Inthdr1(void)
{
	I2CMgr_IntProcess(1);	/* CH1 */
	return;
}

/*
	ݏ.
	chFI2C`l.
*/
static void I2CMgr_IntProcess(uint8_t ch)
{
	int32_t result = AG903_ENONE;
	uint32_t event = 0;
	
	switch(I2cChStat[ch].stat) {
		case AG903_I2C_CH_MASTER_WRITE:
			result = I2CMgr_IntMasterWrite(ch, &event);
			if(AG903_ENONE != result) {
				/* Ȃ */
			}
			break;
		case AG903_I2C_CH_MASTER_READ:
			result = I2CMgr_IntMasterRead(ch, &event);
			if(AG903_ENONE != result) {
				/* Ȃ */
			}
			break;
		case AG903_I2C_CH_SLAVE_WRITE:
			result = I2CMgr_IntSlaveWrite(ch, &event);
			if(AG903_ENONE != result) {
				/* Ȃ */
			}
			break;
		case AG903_I2C_CH_SLAVE_READ:
			result = I2CMgr_IntSlaveRead(ch, &event);
			if(AG903_ENONE != result) {
				/* Ȃ */
			}
			break;
		default:
			/* Ȃ */
			break;
	}
	
	if(0 != event) {
		I2cHandleStat[ch][I2cChStat[ch].hdlnum].stat = AG903_I2C_HANDLE_IDLE;
		if(NULL != I2cHandleStat[ch][I2cChStat[ch].hdlnum].clbk) {
			I2cHandleStat[ch][I2cChStat[ch].hdlnum].clbk
					((AG903_I2CMgrHandle*)&I2cHandleStat[ch][I2cChStat[ch].hdlnum], event);	/* R[obN */
		}
	}
	
	I2CMgr_MainProcess(ch);
	
	return;
}

/*
	Master Writei݁j.
	chFI2C`lԍi0`1j.
	eventFCxg.
*/
static int32_t I2CMgr_IntMasterWrite(uint8_t ch, uint32_t* event)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t status;
	uint32_t set_bit = 0;
	uint8_t  nextdata = 0;
	
	(*event) = 0;

	if( (AG903_I2C_CH_NUM <= ch) ||
		(NULL == event) ) {
		return -AG903_EINVAL;
	}
	
	AG903_I2CPrmGetStatus(ch, &status);
	
	if(status&AG903_I2C_SR_AL_BIT) {	/* arbitration */
		(*event) |= AG903_I2C_EVENT_LOSES_ARBITRATION;
	}
	if(status&AG903_I2C_SR_BERR_BIT) {	/* Slave non-ACK */
		(*event) |= AG903_I2C_EVENT_SEQUENCE_ERR;
	}
	
	if((0 == (*event)) && (status&AG903_I2C_SR_DT_BIT)) {
		switch(I2cChStat[ch].sequence) {
			case AG903_I2C_SEQ_SND_SADDR_H:
				nextdata = (uint8_t)I2cChStat[ch].arg.wr.addr;
				set_bit  = AG903_I2C_CR_TBEN_BIT;
				I2cChStat[ch].sequence = AG903_I2C_SEQ_SND_SADDR_L;
				break;
			case AG903_I2C_SEQ_SND_SADDR_L:
				nextdata = (*(I2cChStat[ch].arg.wr.buf+I2cChStat[ch].cnt));
				set_bit  = AG903_I2C_CR_TBEN_BIT;
				if(((I2cChStat[ch].cnt+1) >= I2cChStat[ch].arg.wr.size) && (true == I2cChStat[ch].arg.wr.stop)){
					set_bit |= AG903_I2C_CR_STOP_BIT;	/* Stop Condition */
				}
				I2cChStat[ch].sequence = AG903_I2C_SEQ_SND;
				break;
			case AG903_I2C_SEQ_SND:
				I2cChStat[ch].cnt++;
				if(I2cChStat[ch].arg.wr.size <= I2cChStat[ch].cnt) {
					(*event) |= AG903_I2C_EVENT_WRITE_END;	/* Write */
				}
				else {
					nextdata = (*(I2cChStat[ch].arg.wr.buf+I2cChStat[ch].cnt));
					set_bit  = AG903_I2C_CR_TBEN_BIT;
					if(((I2cChStat[ch].cnt+1) >= I2cChStat[ch].arg.wr.size) && (true == I2cChStat[ch].arg.wr.stop)){
						set_bit |= AG903_I2C_CR_STOP_BIT;	/* Stop Condition */
					}
				}
				break;
			default:
				(*event) |= AG903_I2C_EVENT_WRITE_ERR;
				break;
		}
		if(0 == (*event)) {
			AG903_I2CPrmWriteData(ch, nextdata);
			AG903_I2CPrmClearControl(ch, (AG903_I2C_CR_START_BIT|AG903_I2C_CR_STOP_BIT));
			AG903_I2CPrmSetControl(ch, set_bit);	/* f[^Write */
		}
	}
	
	if(0 != (*event)) {
		AG903_I2CPrmClearControl(ch, (AG903_I2C_CR_ALLINT_BIT|AG903_I2C_CR_ALLCTRL_BIT));
		I2cChStat[ch].stat = AG903_I2C_CH_IDLE;
	}

	return retval;
}

/*
	Master Readi݁j.
	chFI2C`lԍi0`1j.
	eventFCxg.
*/
static int32_t I2CMgr_IntMasterRead(uint8_t ch, uint32_t* event)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t status;
	
	(*event) = 0;

	if( (AG903_I2C_CH_NUM <= ch) ||
		(NULL == event) ) {
		return -AG903_EINVAL;
	}
	
	AG903_I2CPrmGetStatus(ch, &status);

	if(status&AG903_I2C_SR_AL_BIT) {	/* arbitration */
		(*event) |= AG903_I2C_EVENT_LOSES_ARBITRATION;
	}

	switch(I2cChStat[ch].sequence) {
		case AG903_I2C_SEQ_SND_SADDR_H:
			if(status&AG903_I2C_SR_BERR_BIT) {	/* Slave non-ACK */
				(*event) |= AG903_I2C_EVENT_SEQUENCE_ERR;
			}
			if((0 == (*event)) && (status&AG903_I2C_SR_DT_BIT)) {
				AG903_I2CPrmWriteData(ch, (uint8_t)I2cChStat[ch].arg.rd.addr);
				AG903_I2CPrmClearControl(ch, (AG903_I2C_CR_START_BIT|AG903_I2C_CR_STOP_BIT));
				AG903_I2CPrmSetControl(ch, AG903_I2C_CR_TBEN_BIT);
				I2cChStat[ch].sequence = AG903_I2C_SEQ_SND_SADDR_L;
			}
			break;
		case AG903_I2C_SEQ_SND_SADDR_L:
			if(status&AG903_I2C_SR_BERR_BIT) {	/* Slave non-ACK */
				(*event) |= AG903_I2C_EVENT_SEQUENCE_ERR;
			}
			if((0 == (*event)) && (status&AG903_I2C_SR_DT_BIT)) {
				AG903_I2CPrmClearControl(ch, (AG903_I2C_CR_START_BIT|AG903_I2C_CR_STOP_BIT));
				if(1 >= I2cChStat[ch].arg.rd.size) {
					AG903_I2CPrmSetControl(ch, (AG903_I2C_CR_TBEN_BIT|AG903_I2C_CR_ACKNACK_BIT|AG903_I2C_CR_STOP_BIT));
					I2cChStat[ch].sequence = AG903_I2C_SEQ_RCV_LAST;	/* Last */
				}
				else {
					AG903_I2CPrmSetControl(ch, AG903_I2C_CR_TBEN_BIT);
					I2cChStat[ch].sequence = AG903_I2C_SEQ_RCV;
				}
			}
			break;
		case AG903_I2C_SEQ_RCV:
			if((0 == (*event)) && (status&AG903_I2C_SR_DR_BIT)) {
				AG903_I2CPrmReadData(ch, (uint8_t*)(I2cChStat[ch].arg.rd.buf+I2cChStat[ch].cnt));
				I2cChStat[ch].cnt++;
				AG903_I2CPrmClearControl(ch, (AG903_I2C_CR_START_BIT|AG903_I2C_CR_STOP_BIT));
				if((I2cChStat[ch].arg.rd.size-1) <= I2cChStat[ch].cnt) {
					AG903_I2CPrmSetControl(ch, (AG903_I2C_CR_TBEN_BIT|AG903_I2C_CR_ACKNACK_BIT|AG903_I2C_CR_STOP_BIT));
					I2cChStat[ch].sequence = AG903_I2C_SEQ_RCV_LAST;	/* Last */
				}
				else {
					AG903_I2CPrmSetControl(ch, AG903_I2C_CR_TBEN_BIT);
					I2cChStat[ch].sequence = AG903_I2C_SEQ_RCV;
				}
			}
			break;
		case AG903_I2C_SEQ_RCV_LAST:
			if(0 == (status&AG903_I2C_SR_ACK_BIT)) {	/* ACKM */
				(*event) |= AG903_I2C_EVENT_SEQUENCE_ERR;
			}
			if((0 == (*event)) && (status&AG903_I2C_SR_DR_BIT)) {
				AG903_I2CPrmReadData(ch, (uint8_t*)(I2cChStat[ch].arg.rd.buf+I2cChStat[ch].cnt));
				I2cChStat[ch].cnt++;
				(*event) |= AG903_I2C_EVENT_READ_END;
			}
			break;
		default:
			(*event) |= AG903_I2C_EVENT_READ_ERR;
			break;
	}

	if(0 != (*event)) {
		AG903_I2CPrmClearControl(ch, (AG903_I2C_CR_ALLINT_BIT|AG903_I2C_CR_ALLCTRL_BIT));
		I2cChStat[ch].stat = AG903_I2C_CH_IDLE;
	}

	return retval;
}

/*
	Slave Writei݁j.
	chFI2C`lԍi0`1j.
	eventFCxg.
*/
static int32_t I2CMgr_IntSlaveWrite(uint8_t ch, uint32_t* event)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t status;
	
	(*event) = 0;

	if( (AG903_I2C_CH_NUM <= ch) ||
		(NULL == event) ) {
		return -AG903_EINVAL;
	}
	
	AG903_I2CPrmGetStatus(ch, &status);
	
	switch(I2cChStat[ch].sequence) {
		case AG903_I2C_SEQ_RCV_SADDR:
			if((status&AG903_I2C_SR_SAM_BIT) && (status&AG903_I2C_SR_DR_BIT) && (status&AG903_I2C_SR_I2CB_BIT)) {
				AG903_I2CPrmWriteData(ch, (uint8_t)(*(I2cChStat[ch].arg.wr.buf+I2cChStat[ch].cnt)));
				AG903_I2CPrmSetControl(ch, AG903_I2C_CR_TBEN_BIT);
				I2cChStat[ch].sequence = AG903_I2C_SEQ_SND;
			}
			break;
		case AG903_I2C_SEQ_SND:
			if(status&AG903_I2C_SR_DT_BIT) {
				I2cChStat[ch].cnt++;
				if(I2cChStat[ch].arg.wr.size <= I2cChStat[ch].cnt) {
					(*event) |= AG903_I2C_EVENT_WRITE_END;	/* Write */
				}
				else {
					if((status&AG903_I2C_SR_ACK_BIT) || (status&AG903_I2C_SR_STOP_BIT)) {
						I2cChStat[ch].sequence = AG903_I2C_SEQ_RCV_SADDR;
					}
					else {
						AG903_I2CPrmWriteData(ch, (uint8_t)(*(I2cChStat[ch].arg.wr.buf+I2cChStat[ch].cnt)));
						AG903_I2CPrmSetControl(ch, AG903_I2C_CR_TBEN_BIT);
					}
				}
			}
			break;
		default:
			(*event) |= AG903_I2C_EVENT_WRITE_ERR;
			break;
	}
	
	if(0 != (*event)) {
		AG903_I2CPrmClearControl(ch, (AG903_I2C_CR_ALLINT_BIT|AG903_I2C_CR_ALLCTRL_BIT));
		I2cChStat[ch].stat = AG903_I2C_CH_IDLE;
	}

	return retval;
}

/*
	Slave Readi݁j.
	chFI2C`lԍi0`1j.
	eventFCxg.
*/
static int32_t I2CMgr_IntSlaveRead(uint8_t ch, uint32_t* event)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t status;
	
	(*event) = 0;

	if( (AG903_I2C_CH_NUM <= ch) ||
		(NULL == event) ) {
		return -AG903_EINVAL;
	}
	
	AG903_I2CPrmGetStatus(ch, &status);

	switch(I2cChStat[ch].sequence) {
		case AG903_I2C_SEQ_RCV_SADDR:
			if((status&AG903_I2C_SR_SAM_BIT) && (status&AG903_I2C_SR_DR_BIT) && (status&AG903_I2C_SR_I2CB_BIT)) {
				AG903_I2CPrmSetControl(ch, AG903_I2C_CR_TBEN_BIT);
				I2cChStat[ch].sequence = AG903_I2C_SEQ_RCV;
			}
			break;
		case AG903_I2C_SEQ_RCV:
			if (status&AG903_I2C_SR_DR_BIT) {
				AG903_I2CPrmReadData(ch, (uint8_t*)(I2cChStat[ch].arg.rd.buf+I2cChStat[ch].cnt));
				I2cChStat[ch].cnt++;
				if((I2cChStat[ch].arg.rd.size) <= I2cChStat[ch].cnt) {
					(*event) |= AG903_I2C_EVENT_READ_END;
				}
				if(status&AG903_I2C_SR_STOP_BIT) {
					I2cChStat[ch].sequence = AG903_I2C_SEQ_RCV_SADDR;
				}
				if((0==(*event)) && (AG903_I2C_SEQ_RCV==I2cChStat[ch].sequence)) {
					AG903_I2CPrmSetControl(ch, AG903_I2C_CR_TBEN_BIT);
				}
			}
			break;
		default:
			(*event) |= AG903_I2C_EVENT_WRITE_ERR;
			break;
	}
	
	if(0 != (*event)) {
		AG903_I2CPrmClearControl(ch, (AG903_I2C_CR_ALLINT_BIT|AG903_I2C_CR_ALLCTRL_BIT));
		I2cChStat[ch].stat = AG903_I2C_CH_IDLE;
	}

	return retval;
}
