/*
 * This program was created by AXELL CORPORATION.
 * Copyright (C) 2017-2019 AXELL CORPORATION, all rights reserved.
 *************************************************************************************
 * @brief		JPG decode code
 * @file		decode.c
 * @auther		AXELL
 * @date
 * @description
 * @note
 * @history		2019_12_27  [SDK3.0] TṽbZ[Wo͂ASCIIR[ĥ݂ɕύX (#2728)
*/
#include "jpg_sample.h"
#include "int/intmgr.h"
#include "dsp/dspmgr.h"
#include "jpg/jpg.h"
#include "bmu/bmumgr.h"
#include "sys/spcprm.h"
#include "sys/sscprm.h"
#include "../dsp/dsp_common.h"

/* Param */
#define	VOD_WIDTH				(1280)					/* 𑜓xE			*/
#define	VOD_HEIGHT				(1024)					/* 𑜓xEc			*/
#define VOD_SSAMP_NONSUPPORT 0
#define VOD_SSAMP_422        1
#define VOD_SSAMP_420        2

/* f[^ */
static AG903_JPGHooks		gHook[AG903_JPG_HOOK_MAX];
static AG903_DSPMgrHandle		*gDspHdl;
static AG903_BMUMgrHandle		*gBmuHdl;
static int32_t					gIntHdlID;
static uint32_t					gJpgFlag=0;

/* Macro */
#define	GETSEQ()			(gJpgFlag)
#define	SETSEQ(SEQ)			(GETSEQ() |= (SEQ))
#define	CLRSEQ(SEQ)			(GETSEQ() &= ~(SEQ))
#define	WAITSEQ(SEQ, T)		do { sys_dlytsk(T); } while (0 == (gJpgFlag & (SEQ)))

/* vg^Cv */
static int32_t jpg_create_YUVfile(void);
static int32_t jpg_display_image(void);
static int32_t jpg_decode_repeat(void);
static int32_t jpg_read_jpg(char* name, uint8_t* buf, uint32_t bufsize, uint32_t* size, uint32_t* width, uint32_t* height);
static int32_t jpg_read_mov(char* name, uint8_t* buf, uint32_t bufsize, uint32_t* size);
static int32_t jpg_write_yuv(char* name, uint8_t* buf, uint32_t width, uint32_t height);
static int32_t jpg_is_decodable(uint8_t* buf, uint32_t size);
static void* jpg_search_marker(uint8_t* buf, uint8_t marker, uint32_t size);
static int32_t jpg_get_pixcel_size(uint8_t *buf, uint32_t size, uint32_t *width, uint32_t *height);
static int32_t jpg_setup_bmu(void* buf, uint32_t bufsize);
static int32_t jpg_getaddr_bmu(void* srcaddr, void* snkaddr);
static int32_t jpg_term_bmu(void);
static int32_t jpg_setup_dsp(uint8_t* wbase, uint8_t* buf);
static int32_t jpg_enable_dsp(void);
static int32_t jpg_term_dsp(void);
static int32_t jpg_setup_dsp_display_sync(void);
static int32_t jpg_setup_dsp_option(void);
static int32_t jpg_setup_dsp_window(uint8_t* wbase, uint8_t* buf);
static int32_t jpg_set_dsp_window(uint32_t width, uint32_t height);
static void jpg_set_callback(AG903_JPGHandle* handle);
static void jpg_del_callback(AG903_JPGHandle* handle);
static void jpg_clbk(AG903_JPGHandle *handle, uint32_t param);

extern AG903_SPCPrmPllnParam	DSP_SAMPLE_PLL[];
extern AG903_DSPMgrSyncParam	DSP_SAMPLE_SYNCPARAM[];

/**
 * @brief	fR[hC
 * @param	mode  [in] [h
 * @param	param [in] ʃp[^
 * @return	none
 * @note	none
*/
void decode_main(uint8_t mode, uint32_t param)
{
	((void)param);

	CLRSEQ(JPG_FLAG_ALL);

	switch (mode) {
		case 0:	/* Decode(JPEG to YUV) */
			jpg_create_YUVfile();
			break;
		case 1:	/* Decode(Display image) */
			jpg_display_image();
			break;
		case 2:	/* Decode(JPEG to YUV Repeat & Display image) */
			jpg_decode_repeat();
			break;
		default:
			break;
	}

	return;
}

/**
 * @brief	fR[h (YUVt@C쐬)
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_create_YUVfile(void)
{
	AG903_JPGHandle* 	handle;
	AG903_JPGCtrlParam	ini_param;
	AG903_JPGCodecParam	set_param;
	int32_t		ret = AG903_ENONE;
	uint32_t	width, height, size;
	uint8_t*	inbuf  = NULL;
	uint8_t*	outbuf = NULL;
	uint8_t		ccomp;


	ret = AG903_JPGGetHandle(&handle);
	if(AG903_ENONE != ret) {
		JPG_ErrPrintf("ERROR: Handle get");
		return -AG903_EFAULT;
	}
	jpg_set_callback(handle);

	/* m */
	inbuf  = (uint8_t *)sys_memalign(JPG_JPGBUF_SIZE, 4);
	outbuf = (uint8_t *)sys_memalign(JPG_YUVBUF_SIZE, 4);
	if((NULL == inbuf) || (NULL == outbuf)) {
		JPG_ErrPrintf("ERROR: Memory alloc");
		return -AG903_EFAULT;
	}

	do {
		/* JPEGt@CRead */
		ret = jpg_read_jpg((char*)JPG_INPUT_DEC_PATH, inbuf, JPG_JPGBUF_SIZE, &size, &width, &height);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: File read");
			break;
		}

		/* wb_̏ɂfR[hۃ`FbN */
		ret = jpg_is_decodable(inbuf, size);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Non-support JPEG format");
			break;
		}

		/* R[fbN */
		ini_param.operation = AG903_JPG_ENDEC_DECODE;
		ini_param.interval  = 8;
		ini_param.SOSstop   = false;
		ret =  AG903_JPGInitCodec(handle, &ini_param);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Codec init");
			break;
		}
		WAITSEQ(JPG_FLAG_INIT_CODEC|JPG_FLAG_ERR_CODEC, 0);
		if(JPG_FLAG_ERR_CODEC & GETSEQ()) {
			JPG_ErrPrintf("ERROR: Codec init wait");
			break;
		}

		/* R[fbN */
		set_param.capture    = false;
		set_param.buf_flow   = false;
		set_param.flame_skip = false;
		set_param.h_size     = 0;
		set_param.v_size     = 0;
		set_param.in_buf     = (void*)(inbuf);
		set_param.out_buf    = (void*)(outbuf);
		set_param.strm_size  = size;
		set_param.stride     = width;
		ret = AG903_JPGStartCodec(handle, &set_param);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Codec start");
			break;
		}
		WAITSEQ(JPG_FLAG_END_CODEC|JPG_FLAG_ERR_CODEC, 0);
		if(JPG_FLAG_ERR_CODEC & GETSEQ()) {
			JPG_ErrPrintf("ERROR: Codec end wait");
			break;
		}

		/* R[fbNI */
		ret = AG903_JPGEndCodec(handle);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Codec end");
			break;
		}

		/* JPGfR[h擾 */
		ret = AG903_JPGGetColorComponent(handle, &ccomp);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Color component get");
			break;
		}
		JPG_Printf(" Format : %s\r\n", (ccomp == 0)?"YUV420":"YUV422");

		/* YUVt@CWrite */
		ret = jpg_write_yuv((char*)JPG_OUTPUT_YUV_PATH, outbuf, width, height);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: File write");
			break;
		}
	}while(0);

	jpg_del_callback(handle);

	/* nh */
	AG903_JPGReleaseHandle(handle);

	/*  */
	sys_free(inbuf);
	sys_free(outbuf);

	return ret;
}

/**
 * @brief	fR[h (ʕ\)
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_display_image(void)
{
	AG903_JPGHandle* 	handle;
	AG903_JPGCtrlParam	ini_param;
	AG903_JPGCodecParam	set_param;
	int32_t		ret = AG903_ENONE;
	uint32_t	width, height, size;
	uint8_t*	inbuf  = NULL;
	uint8_t*	outbuf = NULL;
	uint8_t*	wbase  = NULL;
	uint8_t		ccomp;

	ret = AG903_JPGGetHandle(&handle);
	if(AG903_ENONE != ret) {
		JPG_ErrPrintf("ERROR: Handle get");
		return -AG903_EFAULT;
	}
	jpg_set_callback(handle);

	/* m */
	inbuf  = (uint8_t *)sys_memalign(JPG_JPGBUF_SIZE, 4);
	outbuf = (uint8_t *)sys_memalign(JPG_YUVBUF_SIZE, 4);
	wbase  = (uint8_t *)sys_memalign(JPG_WINATTR_SIZE, 8);
	if((NULL == inbuf) || (NULL == outbuf) || (NULL == wbase)) {
		JPG_ErrPrintf("ERROR: Memory alloc");
		return -AG903_EFAULT;
	}

	do {
		/* JPEGt@CRead */
		ret = jpg_read_jpg((char*)JPG_INPUT_DEC_PATH, inbuf, JPG_JPGBUF_SIZE, &size, &width, &height);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: File read");
			break;
		}

		/* wb_̏ɂfR[hۃ`FbN */
		ret = jpg_is_decodable(inbuf, size);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Non-support JPEG format");
			break;
		}

		/* \ݒ */
		ret = jpg_setup_dsp(wbase, outbuf);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Display setup");
			break;
		}

		/* R[fbN */
		ini_param.operation = AG903_JPG_ENDEC_DECODE;
		ini_param.interval  = 8;
		ini_param.SOSstop   = false;
		ret =  AG903_JPGInitCodec(handle, &ini_param);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Codec init");
			break;
		}
		WAITSEQ(JPG_FLAG_INIT_CODEC|JPG_FLAG_ERR_CODEC, 0);
		if(JPG_FLAG_ERR_CODEC & GETSEQ()) {
			JPG_ErrPrintf("ERROR: Codec init wait");
			break;
		}

		/* R[fbN */
		set_param.capture    = false;
		set_param.buf_flow   = false;
		set_param.flame_skip = false;
		set_param.h_size     = 0;
		set_param.v_size     = 0;
		set_param.in_buf     = (void*)(inbuf);
		set_param.out_buf    = (void*)(outbuf);
		set_param.strm_size  = size;
		set_param.stride     = width;
		ret = AG903_JPGStartCodec(handle, &set_param);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Codec start");
			break;
		}
		WAITSEQ(JPG_FLAG_END_CODEC|JPG_FLAG_ERR_CODEC, 0);
		if(JPG_FLAG_ERR_CODEC & GETSEQ()) {
			JPG_ErrPrintf("ERROR: Codec end wait");
			break;
		}

		/* R[fbNI */
		ret = AG903_JPGEndCodec(handle);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Codec end");
			break;
		}

		/* JPGfR[h擾 */
		ret = AG903_JPGGetColorComponent(handle, &ccomp);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Color component get");
			break;
		}
		JPG_Printf(" Format : %s\r\n", (ccomp == 0)?"YUV420":"YUV422");

		/* EChEݒ */
		ret = jpg_set_dsp_window(width, height);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Display window set");
			break;
		}

		/* \ON */
		ret = jpg_enable_dsp();
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Display enable");
			break;
		}

		JPG_Printf(" Press enter key to exit: ");
		JPG_wait();	/* CRLF͑҂ */

		/* \I */
		ret = jpg_term_dsp();
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Display term");
			break;
		}

	}while(0);

	jpg_del_callback(handle);

	/* nh */
	AG903_JPGReleaseHandle(handle);

	/*  */
	sys_free(inbuf);
	sys_free(outbuf);
	sys_free(wbase);

	return ret;
}

/**
 * @brief	fR[h (JԂ)
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_decode_repeat(void)
{
	AG903_JPGHandle* 	handle;
	AG903_JPGCtrlParam	ini_param;
	AG903_JPGCodecParam	set_param;
	int32_t		ret = AG903_ENONE;
	uint32_t	width, height, size, fsize, jsize;
	uintptr_t	srcaddr, snkaddr;
	uint8_t*	inbuf   = NULL;
	uint8_t*	bmubuf  = NULL;
	uint8_t*	workbuf = NULL;
	uint8_t*	wbase   = NULL;
	uint8_t*	straddr = NULL;
	uint8_t*	endaddr = NULL;
	uint8_t*	bufptr  = NULL;

	ret = AG903_JPGGetHandle(&handle);
	if(AG903_ENONE != ret) {
		JPG_ErrPrintf("ERROR: Handle get");
		return -AG903_EFAULT;
	}
	jpg_set_callback(handle);

	/* m */
	inbuf  = (uint8_t *)sys_memalign(JPG_JPGBUF_SIZE, 4);
	bmubuf = (uint8_t *)sys_memalign(JPG_BMUBUF_SIZE, 4096);
	workbuf= (uint8_t *)sys_memalign(JPG_WORKBUF_SIZE, 32);
	wbase  = (uint8_t *)sys_memalign(JPG_WINATTR_SIZE, 8);
	if((NULL == inbuf) || (NULL == bmubuf) || (NULL == wbase)) {
		JPG_ErrPrintf("ERROR: Memory alloc");
		return -AG903_EFAULT;
	}
	sys_memset(bmubuf, 0, JPG_YUVBUF_SIZE);

	do {
		/* t@CRead */
		ret = jpg_read_mov((char*)JPG_INPUT_MOV_PATH, workbuf, JPG_WORKBUF_SIZE, &fsize);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: File read");
			break;
		}

		/* BMUݒ */
		ret = jpg_setup_bmu((void*)bmubuf, JPG_BMUBUF_SIZE/JPG_BMUFIFO_NUM);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: BMU setup");
			break;
		}
		ret = jpg_getaddr_bmu(&srcaddr, &snkaddr);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: BMU get address");
			break;
		}

		/* \ݒ */
		ret = jpg_setup_dsp(wbase, (uint8_t*)snkaddr);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Display setup");
			break;
		}

		/* R[fbN */
		ini_param.operation = AG903_JPG_ENDEC_DECODE;
		ini_param.interval  = 8;
		ini_param.SOSstop   = false;
		ret =  AG903_JPGInitCodec(handle, &ini_param);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Codec init");
			break;
		}
		WAITSEQ(JPG_FLAG_INIT_CODEC|JPG_FLAG_ERR_CODEC, 0);
		if(JPG_FLAG_ERR_CODEC & GETSEQ()) {
			JPG_ErrPrintf("ERROR: Codec init wait");
			break;
		}

		/* \ON */
		ret = jpg_enable_dsp();
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Display enable");
			break;
		}

		bufptr = workbuf;
		for(size=0; size<fsize; ) {
			/* JPEG؏o */
			straddr = (uint8_t*)jpg_search_marker(bufptr, 0xD8, fsize-size);	/* SOI */
			endaddr = (uint8_t*)jpg_search_marker(bufptr, 0xD9, fsize-size);	/* EOI */
			if((NULL == straddr) || (NULL == endaddr)){
				break;
			}
            endaddr += 2;
			bufptr = endaddr;
			jsize  = endaddr - straddr;
			size  += jsize;
			sys_memcpy(inbuf, straddr, jsize);

			/* PixcelTCY擾 */
			ret = jpg_get_pixcel_size(inbuf, jsize, &width, &height);
			if(AG903_ENONE != ret) {
				return -AG903_EFAULT;
			}
			inbuf[jsize+0] = 0xFF;	/* 4oCg̃_~[f[^t */
			inbuf[jsize+1] = 0xFF;
			inbuf[jsize+2] = 0xFF;
			inbuf[jsize+3] = 0xFF;
			jsize += 4;

			/* R[fbN */
			set_param.capture    = false;
			set_param.buf_flow   = true;
			set_param.flame_skip = false;
			set_param.h_size     = 0;
			set_param.v_size     = 0;
			set_param.in_buf     = (void*)(inbuf);
			set_param.out_buf    = (void*)(srcaddr);
			set_param.strm_size  = jsize;
			set_param.stride     = width;
			ret = AG903_JPGStartCodec(handle, &set_param);
			if(AG903_ENONE != ret) {
				JPG_ErrPrintf("ERROR: Codec start");
				break;
			}
			WAITSEQ(JPG_FLAG_END_CODEC|JPG_FLAG_ERR_CODEC, 0);
			if(JPG_FLAG_ERR_CODEC & GETSEQ()) {
				JPG_ErrPrintf("ERROR: Codec end wait");
				break;
			}
			CLRSEQ(JPG_FLAG_END_CODEC);

			/* EChEݒ */
			ret = jpg_set_dsp_window(width, height);
			if(AG903_ENONE != ret) {
				JPG_ErrPrintf("ERROR: Display window set");
				break;
			}
		}
		if(AG903_ENONE != ret) {
			break;
		}

		/* R[fbNI */
		ret = AG903_JPGEndCodec(handle);
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Codec end");
			break;
		}

		/* BMUI */
		jpg_term_bmu();
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: BMU term");
			break;
		}

		/* \I */
		ret = jpg_term_dsp();
		if(AG903_ENONE != ret) {
			JPG_ErrPrintf("ERROR: Display term");
			break;
		}

	}while(0);

	jpg_del_callback(handle);

	/* nh */
	AG903_JPGReleaseHandle(handle);

	/*  */
	sys_free(inbuf);
	sys_free(bmubuf);
	sys_free(workbuf);
	sys_free(wbase);

	return ret;
}

/**
 * @brief	JPEGt@C[h
 * @param	name	[in]  t@C
 * @param	buf		[out] [hf[^
 * @param	bufsize	[in]  obt@TCY
 * @param	size	[out] f[^TCY
 * @param	width	[out] 
 * @param	height	[out] 
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_read_jpg(char* name, uint8_t* buf, uint32_t bufsize, uint32_t* size, uint32_t* width, uint32_t* height)
{
	int32_t		ret;
	uint32_t	fsize;

	/* t@CTCY擾 */
	ret = jpg_GetFileSize(name, &fsize);
	if(AG903_ENONE != ret) {
		return -AG903_EFAULT;
	}
	if(fsize+4 > bufsize) {
		return -AG903_EFAULT;
	}

	/* t@C[h */
	ret = jpg_OpenRead(name, buf, fsize);
	if(AG903_ENONE != ret) {
		return -AG903_EFAULT;
	}
	buf[fsize+0] = 0xFF;	/* 4oCg̃_~[f[^t */
	buf[fsize+1] = 0xFF;
	buf[fsize+2] = 0xFF;
	buf[fsize+3] = 0xFF;

	/* PixcelTCY擾 */
	ret = jpg_get_pixcel_size(buf, fsize, width, height);
	if(AG903_ENONE != ret) {
		return -AG903_EFAULT;
	}
	(*size) = fsize+4;

	JPG_Printf(" Read JPEG data.\r\n width  :%8d\r\n", (unsigned int)(*width));
	JPG_Printf(" height :%8d\r\n length :%8d\r\n",(unsigned int)(*height), (unsigned int)(*size));

	return AG903_ENONE;
}

/**
 * @brief	MOVt@C[h
 * @param	name	[in]  t@C
 * @param	buf		[out] [hf[^
 * @param	bufsize	[in]  obt@TCY
 * @param	size	[out] t@CTCY
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_read_mov(char* name, uint8_t* buf, uint32_t bufsize, uint32_t* size)
{
	int32_t		ret;
	uint32_t	fsize;

	/* t@CTCY擾 */
	ret = jpg_GetFileSize(name, &fsize);
	if(AG903_ENONE != ret) {
		return -AG903_EFAULT;
	}
	if(fsize > bufsize) {
		return -AG903_EFAULT;
	}

	/* t@C[h */
	ret = jpg_OpenRead(name, buf, fsize);
	if(AG903_ENONE != ret) {
		return -AG903_EFAULT;
	}
	(*size) = fsize;

	JPG_Printf(" Read MOV file.\r\n size  :%8d\r\n", (unsigned int)(*size));

	return AG903_ENONE;
}

/**
 * @brief	YUVt@CCg
 * @param	name	[in] t@C
 * @param	buf		[in] Cgf[^
 * @param	width	[in] 
 * @param	height	[in] 
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_write_yuv(char* name, uint8_t* buf, uint32_t width, uint32_t height)
{
	int32_t		rc;
	uint32_t	wsize;

	wsize = (width*height)<<1;

	rc = jpg_WriteYUV(name, buf, wsize, width, height);
	if(AG903_ENONE != rc) {
		return -AG903_EFAULT;
	}

	JPG_Printf(" Write YUV file.\r\n");

	return AG903_ENONE;
}

/*
 * @brief	JPEGwb_̃fR[h
 * @param	buf		[in] JPEGf[^
 * @param   width   [out] 摜̕
 * @param   height  [out] 摜̍
 * @param   subsamp [out] ʓx̃TuTvO
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	Ή̃wb_
 * @note	t[SOF0̏ꍇ̂ݐɃfR[h\łB
 *          ɃfR[hłȂƂŎ󂯓n߂l͕słB
*/
static int32_t jpg_decompress_header(uint8_t* buf, uint32_t size, uint32_t *width, uint32_t *height, uint32_t *subsamp)
{
	uint32_t ii;
	for (ii = 0; ii+1 < size; ii++) {
		/* SOF0T */
		if (buf[ii]==0xFF && buf[ii+1]==0xC0) {

			/* SOF0ZOgmF */
			int lf = buf[ii+2]<<8 | buf[ii+3];
			if (lf != 17) {
				return -AG903_EFAULT;
			}

			/* 摜TCY擾 */
			int xf = buf[ii+7]<<8 | buf[ii+8]; /* XTCY */
			int yf = buf[ii+5]<<8 | buf[ii+6]; /* YTCY */
			if (height != NULL) {
				*height = xf;
			}
			if (width != NULL) {
				*width = yf;
			}

			/* FmF */
			int nf = buf[ii+9];
			if (nf != 3) {
				return -AG903_EFAULT; /* `FbNΉ̊i[ */
			}

			/* F̃TuTvO擾 */
			int ycn = buf[ii+10];
			int yhv = buf[ii+11];
			int ucn = buf[ii+13];
			int uhv = buf[ii+14];
			int vcn = buf[ii+16];
			int vhv = buf[ii+17];
			if (ycn==0x01 && yhv==0x21) {
				if ((ucn==0x02 && uhv==0x11) && (vcn==0x03 && vhv==0x11)) {
					*subsamp = VOD_SSAMP_422;
				} else {
					*subsamp = VOD_SSAMP_NONSUPPORT;
					return -AG903_EFAULT;
				}
			} else if (ycn==0x01 && yhv==0x22) {
				if ((ucn==0x02 && uhv==0x11) && (vcn==0x03 && vhv==0x11)) {
					*subsamp = VOD_SSAMP_420;
				} else {
					*subsamp = VOD_SSAMP_NONSUPPORT;
					return -AG903_EFAULT;
				}
			} else {
				*subsamp = VOD_SSAMP_NONSUPPORT;
				return -AG903_EFAULT;
			}

			return AG903_ENONE;
		}
	}

	return -AG903_EFAULT;
}


/*
 * @brief	JPEGf[^̃fR[hۊmF
 * @param	buf		[in] JPEGf[^
 * @param   width   [out] 摜̕
 * @param   height  [out] 摜̍
 * @param   subsamp [out] ʓx̃TuTvO
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	Ή̃wb_
 * @note	none
*/
static int32_t jpg_is_decodable(uint8_t* buf, uint32_t size)
{
	uint32_t width   = 0;
	uint32_t height  = 0;
	uint32_t subsamp = 0;
	int32_t ret;

	ret = jpg_decompress_header(buf, size, &width, &height, &subsamp);
	if (ret != AG903_ENONE) {
		return -AG903_EFAULT;
	}

	if (subsamp!=VOD_SSAMP_422 && subsamp!=VOD_SSAMP_420) {
		return -AG903_EFAULT;
	}

	if (subsamp==VOD_SSAMP_422 && height>4096) {
		return -AG903_EFAULT;
	}

	if (subsamp==VOD_SSAMP_420 && height>8192) {
		return -AG903_EFAULT;
	}

	if (width<8 || height<8) {
		return -AG903_EFAULT;
	}

	return AG903_ENONE;
}


/**
 * @brief	}[J[
 * @param	buf		[in] JPEGf[^
 * @param	marker	[in] }[J[
 * @param	size	[in] TCY
 * @return	pbuf	Ώۃ}[J[̃AhX
 * @return	NULL	Ώۃ}[J[
 * @note	none
 */
static void* jpg_search_marker(uint8_t* buf, uint8_t marker, uint32_t size)
{
	uint8_t*	pbuf;

	for(pbuf=buf; (uint32_t)(pbuf-buf)<size; ) {

		if(0xff != pbuf[0]) {	/* }[Jʎq(0xFF) */
			pbuf++;
			continue;
		}
		pbuf++;

		if (marker == pbuf[0]) {
			pbuf--;
			break;	/*  */
		}

		if((0xE0 == (pbuf[0]&0xF0)) ||	/* JFIFʎq */
		   (0xC0 == pbuf[0])        ||	/* SOF0 */
		   (0xC2 == pbuf[0])        ||	/* SOF2 */
		   (0xDB == pbuf[0])        ||	/* DQT */
		   (0xC4 == pbuf[0])        ||	/* DHT */
		   (0xDA == pbuf[0])) {			/* SOS */
			pbuf += (pbuf[1]<<8) + pbuf[2];
			continue;
		}
		pbuf++;
	}
	if((uint32_t)(pbuf-buf)>=size) {
		pbuf = NULL;
	}

	return (void*)pbuf;
}

/**
 * @brief	JPEGsNZTCY擾
 * @param	buf    [in]  JPEGf[^
 * @param	size   [in]  JPEGf[^TCY
 * @param	width  [out] rbg}bvf[^TCY
 * @param	height [out] obt@TCY
 * @return	AG903_ENONE   I
 * @return	-AG903_EFAULT JPEGf[^ُ
 * @note
*/
static int32_t jpg_get_pixcel_size(uint8_t *buf, uint32_t size, uint32_t *width, uint32_t *height)
{
	int32_t		rc = AG903_ENONE;
	uint8_t*	pbuf;

	(*width)  = 0;
	(*height) = 0;

	for(pbuf=buf; (uint32_t)(pbuf-buf)<size; ) {

		if(0xff != (pbuf[0])) {	/* }[Jʎq(0xFF) */
			pbuf++;
			continue;
		}
		pbuf++;

		if((0xE0 == ((pbuf[0])&0xF0)) ||	/* JFIFʎq */
		   (0xDB == (pbuf[0]))        ||	/* DQT */
		   (0xC4 == (pbuf[0]))        ||	/* DHT */
		   (0xDA == (pbuf[0]))) {			/* SOS */
			pbuf += (pbuf[1]<<8) + pbuf[2];
			continue;
		}

		if((0xC0 == pbuf[0]) ||	/* SOF0 */
		   (0xC2 == pbuf[0])) {	/* SOF2 */
		   break;
		}

	}
	if((uint32_t)(pbuf-buf) < size) {
		(*height) = (pbuf[4] << 8) + pbuf[5];	/*  */
		if((*height) % 16) {
			(*height) += 16 - (*height % 16);
		}
		(*width) = (pbuf[6] << 8) + pbuf[7];	/*  */
		if((*width) % 16) {
			(*width) += 16 - (*width % 16);
		}
	}
	else {
		rc = -AG903_EFAULT;
	}

	return rc;
}

/**
 * @brief	BMUݒ
 * @param	buf		[in] BMUobt@x[XAhX
 * @param	bufsize	[in] BMUobt@TCY
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_setup_bmu(void* buf, uint32_t bufsize)
{
	int32_t		rc = AG903_ENONE;

	if (rc == AG903_ENONE) { rc = AG903_BMUMgrGetHandle(&gBmuHdl); }
	if (rc == AG903_ENONE) { rc = AG903_BMUMgrSetMode(gBmuHdl, AG903_BMU_SINK_WAIT_DISABLE, AG903_BMU_BUF_MGR_MODE1); }
	if (rc == AG903_ENONE) { rc = AG903_BMUMgrSetSrcModule(gBmuHdl, AG903_BMU_SRC_JPG0); }
	if (rc == AG903_ENONE) { rc = AG903_BMUMgrAddSinkModule(gBmuHdl, AG903_BMU_SINK_DSP0); }
	if (rc == AG903_ENONE) { rc = AG903_BMUMgrSetBufferConfig(gBmuHdl, buf, bufsize, JPG_BMUFIFO_NUM); }
	if (rc == AG903_ENONE) { rc = AG903_BMUMgrEnable(gBmuHdl); }

	return rc;
}

/**
 * @brief	BMUAhX擾
 * @param	srcaddr	[out] \[Xobt@AhX
 * @param	snkaddr	[out] VNobt@AhX
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_getaddr_bmu(void* srcaddr, void* snkaddr)
{
	int32_t		rc = AG903_ENONE;

	if (rc == AG903_ENONE) { rc = AG903_BMUMgrGetBMUSrcAddress(gBmuHdl, (uint32_t*)srcaddr);  }
	if (rc == AG903_ENONE) { rc = AG903_BMUMgrGetBMUSinkAddress(gBmuHdl, (uint32_t*)snkaddr); }

	return rc;
}

/**
 * @brief	BMUI
 * @param	none
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_term_bmu(void)
{
	int32_t		rc = AG903_ENONE;
	AG903_BMUMgrStatus	bmustat;

	while (1) {
		if (rc != AG903_ENONE) { break; }
		rc = AG903_BMUMgrGetStatus(gBmuHdl, &bmustat);
		if (bmustat.is_empty != 0) { break; }
	}

	if (rc == AG903_ENONE) { rc = AG903_BMUMgrReleaseHandle(gBmuHdl); }

	return rc;
}

/**
 * @brief	\ݒ
 * @param	wbase	[in] EChEx[XAhX
 * @param	buf		[in] t[obt@AhX
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_setup_dsp(uint8_t* wbase, uint8_t* buf)
{
	int32_t	rc = AG903_ENONE;

	if (rc == AG903_ENONE) { rc = AG903_DSPMgrInit(); }
	if (rc == AG903_ENONE) { rc = AG903_DSPMgrGetHandle(0, &gDspHdl); }
	if (rc == AG903_ENONE) { rc = jpg_setup_dsp_display_sync(); }
	if (rc == AG903_ENONE) { rc = jpg_setup_dsp_option(); }
	if (rc == AG903_ENONE) { rc = jpg_setup_dsp_window(wbase, buf); }

	return rc;
}

/**
 * @brief	\ON
 * @param	none
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_enable_dsp(void)
{
	int32_t	rc = AG903_ENONE;

	if (rc == AG903_ENONE) { rc = AG903_DSPMgrEnable(gDspHdl); }
	Board_SelectDviTX();

	return rc;
}

/**
 * @brief	\I
 * @param	none
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_term_dsp(void)
{
	int32_t	rc = AG903_ENONE;

	if (rc == AG903_ENONE) { rc = AG903_DSPMgrDisable(gDspHdl); }

	while(AG903_DSPMgrCheckStopped(gDspHdl));

	if (gIntHdlID) {
		if (rc == AG903_ENONE) { rc = AG903_INTMgrDisableInt(AG903_IRQ16_DSP0 + 0); }
		if (rc == AG903_ENONE) { rc = AG903_INTMgrDeleteHandler(gIntHdlID); }
	}

	if (rc == AG903_ENONE) { rc = AG903_DSPMgrReleaseHandle(gDspHdl); }
	gDspHdl = NULL;

	return rc;
}

/**
 * @brief	ʏo͐ݒ
 * @param	none
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_setup_dsp_display_sync(void)
{
	int32_t						rc  = AG903_ENONE;
	AG903_DSPMgrCtrlParam		ctrl = {0};

	/* PLL */
	AG903_SPCPrmPllnParam pll = {0};
	AG903_SPCPrmSetPll1Ctrl(&pll);
	sys_delayus(50);
	AG903_SPCPrmSetPll1Ctrl(&DSP_SAMPLE_PLL[DSP_SAMPLE_RES_SXGA]);
	sys_delayus(50);

	/* \NbN\[X:PLL1 */
	AG903_SPCPrmClkSel clksrc;
	AG903_SPCPrmGetClkSrc(&clksrc);
	clksrc.dt0 = AG903_SPC_DT0_CLKSRC_PLL1;
	AG903_SPCPrmSetClkSrc(&clksrc);

	AG903_DSPMgrCMOSParam		cmos = {0};

	/*--- CMOSo̓p[^ ---*/

	cmos.rgbde_polarity       = AG903_DSP_POLARITY_POSI;						/* RGBDE̋ɐ					*/
	cmos.vsync_polarity       = AG903_DSP_POLARITY_NEGA;						/* VSYNC̋ɐ(ModeLineݒ)*/
	cmos.hsync_polarity       = AG903_DSP_POLARITY_NEGA;						/* HSYNC̋ɐ(ModeLineݒ)*/
	cmos.field_polarity       = AG903_DSP_POLARITY_NEGA;						/* tB[hM̋ɐ			*/
	cmos.colordetect_polarity = AG903_DSP_POLARITY_NEGA;						/* FoM̋ɐ				*/
	cmos.rgbde_en             = true;											/* RGBDE̗LE			*/
	cmos.vsync_en             = true;											/* VSYNC̗LE			*/
	cmos.hsync_en             = true;											/* HSYNC̗LE			*/
	cmos.field_en             = false;											/* tB[hM̗LE	*/
	cmos.colordetect_en       = false;											/* FoM̗LE		*/
	cmos.pixeldata_en         = true;											/* sNZf[^̗LE	*/
	cmos.dotclk_polarity      = AG903_DSP_POLARITY_NEGA;						/* hbgNbN̋ɐ			*/
	cmos.code_en              = false;											/* f[^SAVAEAVR[h߂ݗL	*/
	cmos.yuv_mode             = 0;												/* RGBYUVϊ@			*/

	/*--- \Hp[^ ---*/

	ctrl.ip_sel        = AG903_DSP_VMODE_NONINTERLACE;							/* rfI[h					*/
	ctrl.hrz_framesize = VOD_WIDTH;
	ctrl.vt_framesize  = VOD_HEIGHT;
    ctrl.syncparam     = &DSP_SAMPLE_SYNCPARAM[DSP_SAMPLE_RES_SXGA];

	rc = AG903_DSPMgrSetPortSel(gDspHdl, AG903_VOD0_PORTSEL_LVCMOS24);

	if (rc == AG903_ENONE) {
		rc = AG903_DSPMgrSetCMOSParam(gDspHdl, &ctrl, &cmos);
	}

	return rc;
}

/**
 * @brief	IvVݒ
 * @param	none
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_setup_dsp_option(void)
{
	int32_t						rc  = AG903_ENONE;
	AG903_DSPMgrExSyncParam		exsync;

	/*--- Op[^ ---*/
	if (rc == AG903_ENONE) {
		exsync.sync_sel = 0;
		rc = AG903_DSPMgrSetExSyncParam(gDspHdl, &exsync);
	}

	/*--- F␳p[^ ---*/
	if (rc == AG903_ENONE) {
		rc = AG903_DSPMgrSetLutParam(gDspHdl, false, NULL);
	}

	/*--- Fp[^ ---*/
	if (rc == AG903_ENONE) {
		rc = AG903_DSPMgrSetDithParam(gDspHdl, false, NULL);
	}

	/*--- Fop[^ ---*/
	if (rc == AG903_ENONE) {
		rc = AG903_DSPMgrSetCDParam(gDspHdl, NULL);
	}

	return rc;
}

/**
 * @brief	EBhEݒ
 * @param	wbase [in] EChEx[XAhX
 * @param	buf   [in] t[obt@AhX
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_setup_dsp_window(uint8_t* wbase, uint8_t* buf)
{
	int32_t						rc  = AG903_ENONE;
	AG903_DSPMgrWindowParam		win;
	AG903_DSPMgrWinAttribute	*attr;

	/*--- EBhEp[^ ---*/
	win.update_timing      = AG903_DSP_ATTR_END_OF_VSYNC;	/* EBhEAgr[gǏo^C~O	*/
	win.background         = 0xFF000000;					/* ʔwi						*/
	win.num_config         = 1;								/* 2ʍv16EBhE		*/
	win.num_attr           = 1;								/* 1EBhE=1Agr[g	*/
	win.window_attr_base   = (uint32_t)wbase;				/* EBhEAgr[gx[XAhX	*/
	win.window_attr_update = true;							/* EBhEAgr[gǏoݒ	*/
	win.palette_update     = false;							/* pbgǂݏoȂ			*/
	if (rc == AG903_ENONE) { rc = AG903_DSPMgrSetWindowParam(gDspHdl, &win); }

	/*--- EBhEAgr[g ---*/
	if (rc == AG903_ENONE) { rc = AG903_DSPMgrGetAttribute(gDspHdl, 0, &attr); }
	if (rc == AG903_ENONE) {
		AG903_DSPMgrWinAttribute zattr = {0};
		zattr.framebuffer_base        = (uint32_t)buf;
		zattr.conf.valid              = true;
		zattr.conf.default_alpha      = 0xFF;
		zattr.conf.framebuffer_format = AG903_DSP_FFMT_YUV422_BT601_FULL;
		*attr = zattr;
		rc = AG903_DSPMgrSetAttribute(gDspHdl, 0);
	}

	return rc;
}

/**
 * @brief	EBhEݒ
 * @param	width [in] 
 * @param	height[in] 
 * @return	AG903_ENONE		I
 * @return	AG903_ENONEȊO	G[R[h
 * @note	none
 */
static int32_t jpg_set_dsp_window(uint32_t width, uint32_t height)
{
	AG903_DSPMgrWinAttribute	*attr;
	int32_t		rc  = AG903_ENONE;
	uint32_t	hrzsize;

	if (height % 16) { height += 16 - (height % 16); }
	if (width % 16) { width += 16 - (width % 16); }
	hrzsize/*byte*/ = (width/*px*/ * 16/*bit*/) >> 3;	/* AG903_DSP_FFMT_YUV422_BT601_FULL̏ꍇ	*/

	if (rc == AG903_ENONE) { rc = AG903_DSPMgrGetAttribute(gDspHdl, 0, &attr); }
	if (rc == AG903_ENONE) {
		attr->destination_width  = width;	/* \ʏE		*/
		attr->destination_height = height;	/* \ʏEc		*/
		attr->source_width       = width;	/* VRAMEf[^	*/
		attr->source_height      = height;	/* VRAMEf[^c	*/
		attr->hrz_size           = hrzsize;	/* VRAMEt[obt@	*/
		rc = AG903_DSPMgrSetAttribute(gDspHdl, 0);
	}

	return rc;
}

/**
 * @brief	R[obNݒ
 * @param	handle [in] JPGnh
 * @return	none
 * @note	none
 */
static void jpg_set_callback(AG903_JPGHandle* handle)
{
	int32_t	loop;

	for (loop = 0;loop < AG903_JPG_HOOK_MAX;loop++) {
		gHook[loop].param = loop;
		gHook[loop].clbk  = (AG903_JPGClbk)jpg_clbk;
	}
	AG903_JPGSetHooks(handle, &gHook[0]);

	return;
}

/**
 * @brief	R[obN폜
 * @param	handle [in] JPGnh
 * @return	none
 * @note	none
 */
static void jpg_del_callback(AG903_JPGHandle* handle)
{
	int32_t	loop;

	for (loop = AG903_JPG_HOOK_CMP;loop < AG903_JPG_HOOK_MAX;loop++) {
		gHook[loop].param = 0;
		gHook[loop].clbk  = (AG903_JPGClbk)NULL;
	}
	AG903_JPGSetHooks(handle, &gHook[0]);

	return;
}

/**
 * @brief	R[obN֐
 * @param	handle [in] JPGnh
 * @param	param [in] ėpp[^
 * @return	none
 * @note	() ̃G[͑ΉR[obN֐o^ȂΌo܂B
 */
static void jpg_clbk(AG903_JPGHandle *handle, uint32_t param)
{
	((void)handle);

	switch (param) {
	case AG903_JPG_HOOK_INITF:	// 12:CODEC
		SETSEQ(JPG_FLAG_INIT_CODEC);
		break;
	case AG903_JPG_HOOK_CMP:	// 0:oX]
		SETSEQ(JPG_FLAG_END_CODEC);
		break;
	case AG903_JPG_HOOK_STRE:	// 1:Xg[f[^]G[
	case AG903_JPG_HOOK_IMGE:	// 2:摜f[^]G[
	case AG903_JPG_HOOK_CAPE:	// 3:Lv`[I[o[t[
	case AG903_JPG_HOOK_ESTE:	// 4:GR[hJnG[
	case AG903_JPG_HOOK_NG:		// 9:R[fbNG[
		SETSEQ(JPG_FLAG_ERR_CODEC);
		break;
	case AG903_JPG_HOOK_EOI:	// 5:EOIo
	case AG903_JPG_HOOK_SOI:	// 6:SOI/Picturewb_o
	case AG903_JPG_HOOK_EOP:	// 7:CODEC
	case AG903_JPG_HOOK_SOP:	// 8:CODECJn
	case AG903_JPG_HOOK_SOS:	// 10:SOSo
	case AG903_JPG_HOOK_DCTRI:	// 11:DCTǂݍ
	case AG903_JPG_HOOK_SRSTF:	// 13:CODEC\tgZbg
	default:
		break;
	}
}
