/**
 * @brief       GPIO Manager
 * @author      AXELL CORPORATION
 * @description GPIO Manager Layer
 * @note        none
 * @history     2017_02_22  
 * @history     2017_10_26  Ver2.0
*/
/* DOM-IGNORE-BEGIN */
/*
 * This program was created by AXELL CORPORATION.
 * Copyright (C) 2017-2019 AXELL CORPORATION, all rights reserved.
 */
/* DOM-IGNORE-END */

#include "AG903_errno.h"
#include "AG903_intno.h"
#include "gpio/gpiomgr.h"
#include "gpio/gpioprm.h"
#include "int/intmgr.h"

typedef void (*AG903_GPIOMgrIntHdr)(void);	/* R[obN֐^ */

typedef struct _AG903_GPIOMgrIntParamt{
	AG903_GPIOMgrClbk 	clbk;	/* R[obN */
	uint32_t	target;			/* R[obNΏ */
} AG903_GPIOMgrIntParam;

typedef struct _AG903_GPIOMgrChStat{
	AG903_GPIOMgrIntParam	intr[AG903_GPIO_CALLBACK_NUM];
	int32_t		hdrid;			/* nhID */
} AG903_GPIOMgrChStat;


static AG903_GPIOMgrChStat	GpioChStat[AG903_GPIO_CH_NUM];	/* Xe[^X */

static void GPIOMgr_Init(uint8_t ch);
static int32_t GPIOMgr_DeleteCallback(uint8_t ch, uint32_t num);
static void GPIOMgr_Inthdr0(void);
static void GPIOMgr_Inthdr1(void);
static void GPIOMgr_Inthdr2(void);
static void GPIOMgr_Inthdr3(void);
static void GPIOMgr_IntProcess(uint8_t ch);

static const AG903_GPIOMgrIntHdr GpioIntHdr[AG903_GPIO_CH_NUM] =
{ GPIOMgr_Inthdr0, GPIOMgr_Inthdr1, GPIOMgr_Inthdr2, GPIOMgr_Inthdr3 };


/**
 * @brief           GPIO Manager
 * @param           none
 * @return          
 * @retval          AG903_ENONE I
 * @description     ϐ܂BVXeNCALLĉB
 * @note            gp𒆎~Ƃ͕KvɉAvAG903_INTMgrDisableIntɂGPIO̊荞݂𖳌ɂĉB
*/
int32_t AG903_GPIOMgrInit(void)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t loop;
	
	for(loop=0; loop<AG903_GPIO_CH_NUM; loop++) {
		GPIOMgr_Init(loop);
	}
	
	return retval;
}

/**
 * @brief           Portf[^݁i8rbgPʁj
 * @param           blknum [in] PortubNԍi0`15j
 * @param           data [in] 8rbgf[^
 * @return          Portf[^݁i8rbgPʁj
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     ProtubN(8bit)PʂŃf[^݂܂B<p>
 *                  ProtubNԍ = GPIOԍ/8
*/
int32_t AG903_GPIOMgrSetByteData(uint8_t blknum, uint8_t data)
{
	int32_t retval = AG903_ENONE;
	uint8_t ch;
	uint8_t offset;
	
	if(AG903_GPIO_PORTBLK_TORTAL <= blknum) {
		return -AG903_EINVAL;
	}
	ch		= (blknum / 4);	/* 8rbgP */
	offset 	= (blknum % 4);	/* 8rbgP */
	AG903_GPIOPrmSetByteData(ch, offset, data);

	return retval;
}

/**
 * @brief           Portf[^
 * @param           portch [in] Port`lԍi0`3j
 * @param           data [in] Portf[^i32rbgj
 * @param           target [in] ^[Qbgwi0:ύX, 1: data𔽉fj
 * @return          Portf[^݌
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂŃf[^݂܂B
 *                  target 1w肵bit̂ data ̓ef܂B
*/
int32_t AG903_GPIOMgrSetPortData(uint8_t portch, uint32_t data, uint32_t target)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t set;
	uint32_t clr;

	if(AG903_GPIO_CH_NUM <= portch) {
		return -AG903_EINVAL;
	}
	
	set = (data&target);
	clr = (~data&target);
	
#if AG903_GPIO_USE_LDREX
	AG903_GPIOPrmModifyBit(portch, set, clr, true);
#else
	AG903_GPIOPrmModifyBit(portch, set, clr, false);
#endif

	return retval;
}

/**
 * @brief           Portf[^擾i8rbgPʁj
 * @param           blknum [in] Port ubNԍi0`15j
 * @param           data [out] Portf[^i8rbgj
 * @return          Portf[^擾i8rbgPʁj
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     ProtubN(8bit)PʂŃf[^擾܂B<p>
 *                  ProtubNԍ = GPIOԍ/8
*/
int32_t AG903_GPIOMgrGetByteData(uint8_t blknum, uint8_t* data)
{
	int32_t retval = AG903_ENONE;
	uint8_t ch;
	uint8_t offset;
	
	if( (AG903_GPIO_PORTBLK_TORTAL <= blknum) ||
		(NULL == data) ){
		return -AG903_EINVAL;
	}
	ch		= (blknum / 4);	/* 8rbgP */
	offset 	= (blknum % 4);	/* 8rbgP */
	AG903_GPIOPrmGetByteData(ch, offset, data);

	return retval;
}

/**
 * @brief           Portf[^擾
 * @param           portch [in] Port`lԍi0`3j
 * @param           data [out] Portf[^i32rbgj
 * @return          Portf[^擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂŃf[^擾܂B
*/
int32_t AG903_GPIOMgrGetPortData(uint8_t portch, uint32_t* data)
{
	int32_t	 retval = AG903_ENONE;
	
	if( (AG903_GPIO_CH_NUM <= portch) ||
		(NULL == data) ){
		return -AG903_EINVAL;
	}
	AG903_GPIOPrmGetPortData(portch, data);
	
	return retval;
}

/**
 * @brief           o͐ݒ
 * @param           portch [in] Port`lԍi0`3j
 * @param           direction [in] o͎wi0:, 1:ój
 * @return          o͐ݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂœo͂ݒ肵܂B
*/
int32_t AG903_GPIOMgrSetDirection(uint8_t portch, uint32_t direction)
{
	int32_t	 retval = AG903_ENONE;

	if(AG903_GPIO_CH_NUM <= portch) {
		return -AG903_EINVAL;
	}
	AG903_GPIOPrmSetDirection(portch, direction);

	return retval;
}

/**
 * @brief           o͎擾
 * @param           portch [in] Port`lԍi0`3j
 * @param           direction [out] ói0:, 1:ój
 * @return          o͎擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂœo͂̐ݒԂ擾܂B
*/
int32_t AG903_GPIOMgrGetDirection(uint8_t portch, uint32_t* direction)
{
	int32_t	 retval = AG903_ENONE;

	if( (AG903_GPIO_CH_NUM <= portch) ||
		(NULL == direction) ){
		return -AG903_EINVAL;
	}
	AG903_GPIOPrmGetDirection(portch, direction);

	return retval;
}

/**
 * @brief           [q^Cvݒ
 * @param           portch [in] Port`lԍi0`3j
 * @param           type [in] [q^Cvi0:ʏ, 1:I[vhCj
 * @return          [q^Cvݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂŒ[q^Cviʏ or I[vhCjݒ肵܂B
*/
int32_t AG903_GPIOMgrSetType(uint8_t portch, uint32_t type)
{
	int32_t	 retval = AG903_ENONE;

	if(AG903_GPIO_CH_NUM <= portch) {
		return -AG903_EINVAL;
	}
	AG903_GPIOPrmSetType(portch, type);

	return retval;
}

/**
 * @brief           [q^Cv擾
 * @param           portch [in] Port`lԍi0`3j
 * @param           type [out] [q^Cvi0:ʏ, 1:I[vhCj
 * @return          [q^Cv擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂŒ[q^Cviʏ or I[vhCj̐ݒԂ擾܂B
*/
int32_t AG903_GPIOMgrGetType(uint8_t portch, uint32_t* type)
{
	int32_t	 retval = AG903_ENONE;

	if( (AG903_GPIO_CH_NUM <= portch) ||
		(NULL == type) ){
		return -AG903_EINVAL;
	}
	AG903_GPIOPrmGetType(portch, type);

	return retval;
}

/**
 * @brief           オGbWoݒ
 * @param           portch [in] Port`lԍi0`3j
 * @param           enable [in] oݒi0:, 1:Lj
 * @return          オGbWoݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂŗオGbWo̗L/ݒ肵܂B
*/
int32_t AG903_GPIOMgrSetPositiveDetect(uint8_t portch, uint32_t enable)
{
	int32_t	 retval = AG903_ENONE;

	if(AG903_GPIO_CH_NUM <= portch) {
		return -AG903_EINVAL;
	}
	AG903_GPIOPrmSetPosEdgeDetect(portch, enable);

	return retval;
}

/**
 * @brief           オGbWoݒ擾
 * @param           portch [in] Port`lԍi0`3j
 * @param           enable [out] oݒi0:, 1:Lj
 * @return          オGbWoݒ擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂŗオGbWo̗L/ݒԂ擾܂B
*/
int32_t AG903_GPIOMgrGetPositiveDetect(uint8_t portch, uint32_t* enable)
{
	int32_t	 retval = AG903_ENONE;

	if( (AG903_GPIO_CH_NUM <= portch) ||
		(NULL == enable) ){
		return -AG903_EINVAL;
	}
	AG903_GPIOPrmGetPosEdgeDetect(portch, enable);

	return retval;
}

/**
 * @brief           GbWoݒ
 * @param           portch [in] Port`lԍi0`3j
 * @param           enable [in] oݒi0:, 1:Lj
 * @return          GbWoݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂŗGbWo̗L/ݒ肵܂B
*/
int32_t AG903_GPIOMgrSetNegativeDetect(uint8_t portch, uint32_t enable)
{
	int32_t	 retval = AG903_ENONE;

	if(AG903_GPIO_CH_NUM <= portch) {
		return -AG903_EINVAL;
	}
	AG903_GPIOPrmSetNegEdgeDetect(portch, enable);

	return retval;
}

/**
 * @brief           GbWoݒ擾
 * @param           portch [in] Port`lԍi0`3j
 * @param           enable [out] oݒi0:, 1:Lj
 * @return          GbWoݒ擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂŗGbWo̗L/ݒԂ擾܂B
*/
int32_t AG903_GPIOMgrGetNegativeDetect(uint8_t portch, uint32_t* enable)
{
	int32_t	 retval = AG903_ENONE;

	if( (AG903_GPIO_CH_NUM <= portch) ||
		(NULL == enable) ){
		return -AG903_EINVAL;
	}
	AG903_GPIOPrmGetNegEdgeDetect(portch, enable);

	return retval;
}

/**
 * @brief           GbWo}XNݒ
 * @param           portch [in] Port`lԍi0`3j
 * @param           mask [in] }XNݒi0:, 1:Lj
 * @return          GbWo}XNݒ茋
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂŃGbWo}XN̗L/ݒ肵܂B
*/
int32_t AG903_GPIOMgrSetMask(uint8_t portch, uint32_t mask)
{
	int32_t	 retval = AG903_ENONE;

	if(AG903_GPIO_CH_NUM <= portch) {
		return -AG903_EINVAL;
	}
	AG903_GPIOPrmSetMask(portch, mask);

	return retval;
}

/**
 * @brief           GbWo}XNݒ擾
 * @param           portch [in] Port`lԍi0`3j
 * @param           mask [out] }XNݒi0:, 1:Lj
 * @return          GbWo}XNݒ擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂŃGbWo}XN̗L/ݒԂ擾܂B
*/
int32_t AG903_GPIOMgrGetMask(uint8_t portch, uint32_t* mask)
{
	int32_t	 retval = AG903_ENONE;

	if( (AG903_GPIO_CH_NUM <= portch) ||
		(NULL == mask) ){
		return -AG903_EINVAL;
	}
	AG903_GPIOPrmGetMask(portch, mask);

	return retval;
}

/**
 * @brief           GbWoXe[^X擾
 * @param           portch [in] Port`l
 * @param           status [out] GbWoXe[^X
 * @return          GbWoXe[^X擾
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂŃGbWoXe[^X擾܂Bi0:o, 1:oj
*/
int32_t AG903_GPIOMgrGetEdgeDetect(uint8_t portch, uint32_t* status)
{
	int32_t	 retval = AG903_ENONE;

	if( (AG903_GPIO_CH_NUM <= portch) ||
		(NULL == status) ){
		return -AG903_EINVAL;
	}
	AG903_GPIOPrmGetEdgeDetect(portch, status);

	return retval;
}

/**
 * @brief           GbWoXe[^XNA
 * @param           portch [in] Port`l
 * @param           clrbit [in] Ώہirbgwj
 * @return          GbWoXe[^XNA
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂŃGbWoXe[^XNA܂B<p>
 *                  clrbit 1ݒ肵bit̂݃NA܂B
*/
int32_t AG903_GPIOMgrClearEdgeDetect(uint8_t portch, uint32_t clrbit)
{
	int32_t	 retval = AG903_ENONE;

	if( AG903_GPIO_CH_NUM <= portch) {
		return -AG903_EINVAL;
	}
	AG903_GPIOPrmClearEdgeDetect(portch, clrbit);

	return retval;
}

/**
 * @brief           GbWoR[obNݒ
 * @param           portch [in] Port`lԍi0`3j
 * @param           target [in] Ώۃ^[Qbgirbgwj
 * @param           clbk [in] R[obN֐|C^
 * @return          GbWoR[obNݒ茋
 * @retval          AG903_ENONE     I
 * @retval          -AG903_EINVAL   ُ
 * @retval          -AG903_ENOBUFS  o^I[o[
 * @retval          -AG903_EFAULT   si݃nho^s)
 * @description     Port`li32bitjPʂŃGbWoR[obN֐ݒ肵܂B<p>
 *                  target 1ݒ肵bitɑ΂Đݒ肳܂B
*/
int32_t AG903_GPIOMgrSetCallback(uint8_t portch, uint32_t target, AG903_GPIOMgrClbk clbk)
{
	int32_t	 retval = AG903_ENONE;
	AG903_INTMgrHdrPrm inthdr;
	int32_t	 hdrid;
	uint32_t loop;

	if( (AG903_GPIO_CH_NUM <= portch) ||
		(NULL == clbk) ){
		return -AG903_EINVAL;
	}
	
	if(0 == GpioChStat[portch].hdrid) {
		inthdr.atr   = AG903_INT_HLNG;
		inthdr.intno = AG903_IRQ40_GPIO0+portch;
		inthdr.func  = (void*)GpioIntHdr[portch];
		hdrid = AG903_INTMgrSetHandler(&inthdr);
		if(0 >= hdrid) {
			return -AG903_EFAULT;
		}
		GpioChStat[portch].hdrid = hdrid;	/* ݃nho^ */
		AG903_INTMgrEnableInt(AG903_IRQ40_GPIO0+portch);
	}
	
	do{
		if(AG903_ENONE != retval) {
			break;
		}
		
		for(loop=0; loop<AG903_GPIO_CALLBACK_NUM; loop++) {
			if(clbk == GpioChStat[portch].intr[loop].clbk) {
				GpioChStat[portch].intr[loop].target |= target;	/* ǉo^ */
				break;
			}
		}
		if(AG903_GPIO_CALLBACK_NUM > loop) {
			break;
		}
		
		for(loop=0; loop<AG903_GPIO_CALLBACK_NUM; loop++) {
			if(NULL == GpioChStat[portch].intr[loop].clbk) {
				GpioChStat[portch].intr[loop].clbk = clbk;
				GpioChStat[portch].intr[loop].target = target;	/* VKo^ */
				break;
			}
		}
		if(AG903_GPIO_CALLBACK_NUM <= loop) {
			retval = -AG903_ENOBUFS;	/* ȏo^oȂ */
		}
	}while(0);

	return retval;
}

/**
 * @brief           GbWoR[obNݒNA.
 * @param           portch [in] Port`lԍi0`3j
 * @param           target [in] Ώۃ^[Qbgirbgwj
 * @param           clbk [in] R[obN֐|C^
 * @return          GbWoR[obNݒNA
 * @retval          AG903_ENONE   I
 * @retval          -AG903_EINVAL ُ
 * @description     Port`li32bitjPʂŃGbWoR[obN֐ݒ肵܂B<p>
 *                  target 1ݒ肵bitɑ΂Đݒ肳܂B
*/
int32_t AG903_GPIOMgrClearCallback(uint8_t portch, uint32_t target, AG903_GPIOMgrClbk clbk)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t loop;

	if( (AG903_GPIO_CH_NUM <= portch) ||
		(NULL == clbk) ){
		return -AG903_EINVAL;
	}

	for(loop=0; loop<AG903_GPIO_CALLBACK_NUM; loop++) {
		if(clbk == GpioChStat[portch].intr[loop].clbk) {
			GpioChStat[portch].intr[loop].target &= ~target;
			if(0 == target) {
				GPIOMgr_DeleteCallback(portch, loop);
			}
			break;
		}
	}
	if(AG903_GPIO_CALLBACK_NUM <= loop) {
		retval = -AG903_EINVAL;	/* clbko^ */
	}
	
	return retval;
}

/*
 	ԏ
	chFPort`lԍi0`3j
*/
static void GPIOMgr_Init(uint8_t ch)
{
	int32_t loop;
	for(loop=0; loop<AG903_GPIO_CALLBACK_NUM; loop++) {
		GpioChStat[ch].intr[loop].clbk   = NULL;
		GpioChStat[ch].intr[loop].target = 0;
	}
	GpioChStat[ch].hdrid = 0;

	return;
}

/*
	GbWoR[obNo^폜
	chFPort`lԍi0`3j
	numFR[obNo^ԍ
*/
static int32_t GPIOMgr_DeleteCallback(uint8_t ch, uint32_t num)
{
	int32_t	 retval = AG903_ENONE;
	uint32_t loop;

	if( (AG903_GPIO_CH_NUM <= ch) ||
		(AG903_GPIO_CALLBACK_NUM <= num) ){
		return -AG903_EINVAL;
	}
	
	if((AG903_GPIO_CALLBACK_NUM-1) <= num) {
		GpioChStat[ch].intr[num].clbk = NULL;
	}
	else {
		for(loop=num; loop<(AG903_GPIO_CALLBACK_NUM-1); loop++) {
			GpioChStat[ch].intr[loop] = GpioChStat[ch].intr[loop+1];	/* \̃Rs[ */
			if(NULL == GpioChStat[ch].intr[loop].clbk) {
				break;
			}
		}
	}
	
	return retval;
}


/*
	݃nh(CH0)
*/
static void GPIOMgr_Inthdr0(void)
{
	GPIOMgr_IntProcess(0);	/* CH0 */
	return;
}

/*
	݃nh(CH1)
*/
static void GPIOMgr_Inthdr1(void)
{
	GPIOMgr_IntProcess(1);	/* CH1 */
	return;
}

/*
	݃nh(CH2).
*/
static void GPIOMgr_Inthdr2(void)
{
	GPIOMgr_IntProcess(2);	/* CH2 */
	return;
}

/*
	݃nh(CH3).
*/
static void GPIOMgr_Inthdr3(void)
{
	GPIOMgr_IntProcess(3);	/* CH3 */
	return;
}

/*
	ݏ.
*/
static void GPIOMgr_IntProcess(uint8_t ch)
{
	uint32_t status = 0;
	uint32_t val;
	uint32_t loop;
	
	AG903_GPIOPrmGetEdgeDetect(ch, &status);
	AG903_GPIOPrmClearEdgeDetect(ch, status);	/* oNA */
	
	for(loop=0; loop<AG903_GPIO_CALLBACK_NUM; loop++) {
		val = (status & GpioChStat[ch].intr[loop].target);
		if(0 != val) {
			GpioChStat[ch].intr[loop].clbk(ch, val); /* R[obN */
		}
		if(NULL == GpioChStat[ch].intr[loop].clbk) {
			break;
		}
	}
	
	return;
}
