/**
 * @brief       OSP Manager
 * @author      AXELL CORPORATION
 * @description OSP Manager Layer.
 * @note        none
 * @history     2017_02_22  
 * @history     2017_10_26  Ver2.0
 * @history     2019_03_08  [SDK2.2] AG903_OSPMgrInitłׂẴCxgNA悤ɏC (#2194)
 * @history     2019_03_08  [SDK2.2] OSPł̃R}hobt@̕sC (#2200)
*/
/* DOM-IGNORE-BEGIN */
/*
 * This program was created by AXELL CORPORATION.
 * Copyright (C) 2017-2019 AXELL CORPORATION, all rights reserved.
 */
/* DOM-IGNORE-END */

#include "AG903_common.h"

#include "osp/ospprm.h"
#include "osp/ospmgr.h"
#include "osp.h"


/* 	Build command ID
	ID:command ID(max length is 4bit) */
#define	OSP_CMDID(ID)				((ID) << 28)


/*	BAS(Bus Access Single)
	LEN:number of address and data pair(max length is 63(7bit)) */
#define	OSP_CMD_BAS(LEN)			(OSP_CMDID(OSP_CMDID_BAS) | (((LEN) & AG903_OSP_CMD_BAS_SIZE) << 0))

/* 	BAB(Buss Access Burst)
	LEN:number of address and data pair(max length is 63(7bit))
	F:Address 0:Fixed 1:Increment */
#define	OSP_CMD_BAB(LEN,F)			(OSP_CMDID(OSP_CMDID_BAB) | (((F) & 0x1) << 27) | (((LEN) & AG903_OSP_CMD_BAB_SIZE) << 0))

/*	Set address for Bus Access Command */
#define	OSP_CMD_ADDR(ADR)			((ADR) & 0xFFFFFFFC)

/*	Set data for Bus Access Command */
#define	OSP_CMD_DATA(DAT)			(DAT)

/*	BAW(Bus Access Wait) */
#define	OSP_CMD_BAW()				(OSP_CMDID(OSP_CMDID_BAW))

/*	INTR(Wait Interrupt)
	ID:number of IR */
#define	OSP_CMD_INTR(ID)			(OSP_CMDID(OSP_CMDID_INTR) | (((ID) & 0x3F) << 0))

/*	EVENT(Wait Event)
	ID:number of event */
#define	OSP_CMD_EVENT(ID)			(OSP_CMDID(OSP_CMDID_EVENT) | (((ID) & 0x3F) << 0))

/*	EVENTD(Event Decrement)
	ID:number of event */
#define	OSP_CMD_EVENTD(ID)			(OSP_CMDID(OSP_CMDID_EVENTD) | (((ID) & 0x3F) << 0))

/*	EVENTC(Event Clear)
	ID:number of event */
#define	OSP_CMD_EVENTC(ID)			(OSP_CMDID(OSP_CMDID_EVENTC) | (((ID) & 0x3F) << 0))

/*	SETF(Set Flag)
	FLAG:flag for wait control */
#define	OSP_CMD_SETF(FLAG)			(OSP_CMDID(OSP_CMDID_SETF) | (((FLAG) & 0x0FFFFFFF) << 0))

/*	CLRF(Clear Flag)
	FLAG:flag for wait control */
#define	OSP_CMD_CLRF(FLAG)			(OSP_CMDID(OSP_CMDID_CLRF) | (((FLAG) & 0x0FFFFFFF) << 0))

/*	NOP(No Operation)
	NOP is undefined operation */
#define	OSP_CMD_NOP()				(OSP_CMDID(OSP_CMDID_NOP))


/** R}hobt@wb_\ */
typedef struct _OSPCmdDesc {
	uint32_t	*Buf;			/** obt@|C^ */
	uint32_t	BufLen;			/** obt@e(R}h) */
	uint32_t	CurLen;			/** R}ho^ */
	uint32_t	*WrBuf;			/** ݃obt@|C^ */
	uint32_t	*RdBuf;			/** Ǎ݃obt@|C^ */
	uint32_t	BuffMode;		/** obt@[h (0:FIFO 1:RING) */
	uint32_t	Reserved[2];	/** Iɂ\̂32oCgɕۂ߂̗\̈ */
} OSPCmdDesc;

/** `lԕێ\ */
typedef struct _AG903_OSPMgrChStat{
	void*		cmdbuf;		/** R}hobt@AhX */
} AG903_OSPMgrChStat;

/** nhԕێ\ */
typedef struct _AG903_OSPMgrHandleStat{
	uint8_t		lock;		/** bN */
	uint8_t		reserve[3];	/** \ */
} AG903_OSPMgrHandleStat;

static AG903_OSPMgrChStat		OspChStat[AG903_OSP_CH_NUM];		/* ch */
static AG903_OSPMgrHandleStat	OspHandleStat[AG903_OSP_CH_NUM];	/* handle */


static int32_t OSPMgrCheckHandle(AG903_OSPMgrHandle *handle,uint8_t *ch);
static int32_t OSPMgrCheckOverflow(uint8_t ch);
static int32_t OSPMgrSetCommand(uint8_t ch, uint32_t c);
#if AG903_OSP_CMD_PARSE
static int32_t OSPMgrParseCommand(OSPCmdDesc *desc);
static uint16_t OSPMgrParseCommandCRC(uint8_t mode, uint8_t* data, uint32_t s_adr, uint32_t e_adr);
#endif/* if AG903_OSP_CMD_PARSE */


/**
 * @brief           OSPManager
 * @param           void
 * @return          G[R[h
 * @retval          AG903_ENONE I
 * @description     W[ϐ̏, уWX^̏s܂.
 * @note            OSPW[gpOɕKsĂ.
 *                  ̏OSP̃`lɈˑȂʂ̏ł.
 */
int32_t AG903_OSPMgrInit(void)
{
	int32_t ch, evt;

	/* OSPe`l */
	for (ch = 0;ch < AG903_OSP_CH_NUM;ch++) {
		AG903_OSPPrmFIFOCTRL(ch);
		AG903_OSPPrmPRIOR(ch, 0);
	}

	/* CxgNA */
	for (evt=0; evt<AG903_OSP_EVENT_NUM; evt++)
		AG903_OSPPrmEnableEvent(evt, 0, 0);

	/* tONA */
	AG903_OSPPrmFLAGCLR(0x0FFFFFFF);

	for (ch = 0;ch < AG903_OSP_CH_NUM;ch++) {
		OspChStat[ch].cmdbuf   = NULL;
		OspHandleStat[ch].lock = false;
	}

	return AG903_ENONE;
}

/**
 * @brief           OSPnh擾
 * @param           handle [out] OSPnh
 * @return          G[R[h
 * @retval          AG903_ENONE I
 * @retval          -AG903_EINVAL p[^ُ
 * @retval          -AG903_EBUSY nhgp
 * @description     OSP̃nh擾܂.<p>
 *                  擾nhch͎IɊU܂.
 */
int32_t AG903_OSPMgrGetHandle(AG903_OSPMgrHandle **handle)
{
	int32_t	rc = AG903_ENONE;
	int32_t	loop;

	if(NULL == handle) {
		rc = -AG903_EINVAL;
	}

	for(loop=0; loop<AG903_OSP_CH_NUM; loop++) {
		if(false == OspHandleStat[loop].lock) {
			OspHandleStat[loop].lock = true;
			break;
		}
	}
	if(AG903_OSP_CH_NUM <= loop) {
		/* nhgp */
		return -AG903_EBUSY;
	}

	(*handle) = (AG903_OSPMgrHandle*)&OspHandleStat[loop];

	return rc;
}

/**
 * @brief           OSPnh
 * @param           handle [in] OSPnh
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @retval          -AG903_EBUSY  s
 * @description     OSP̃nhԋp܂.<p>
 *                  nhɊĂꂽOSPs̏ꍇ͕ԋpł܂.
 */
int32_t AG903_OSPMgrReleaseHandle(AG903_OSPMgrHandle *handle)
{
	int32_t		rc = AG903_ENONE;
	uint8_t		ch;
	uint8_t		stat, cnt;

	if(NULL == handle) {
		rc = -AG903_EINVAL;
	}

	rc = OSPMgrCheckHandle(handle, &ch);

	if (rc == AG903_ENONE) {
		rc = AG903_OSPMgrGetStat(handle, &stat, &cnt);
	}

	if (rc == AG903_ENONE) {
		if (stat != 0) {
			/* oX]s */
			rc = -AG903_EBUSY;
		}
	}

	if (rc == AG903_ENONE) {
		AG903_OSPMgrDisable(handle);
		OspHandleStat[ch].lock = false;
	}

	return rc;
}

/**
 * @brief           R}hobt@ݒ
 * @param           handle [in] OSPnh
 * @param           param [in] R}hobt@p[^
 * @param           init [in] R}hobt@witrueFj
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     R}hobt@o^܂.
 * @note            R}hKvɉR}hobt@AvP[VŊmۂĉ.
 * @note            ̃R}hobt@̓nhJ܂Ŋmۂĉ.
 *                  ܂ڃANZXȂŉ.
 * @note            R}hobt@̓R}ĥ߂̃TCY(1R}h32oCg)+32oCgȏmۂĉ.
 * @note            paramsize͂SoCg1PʂƂĎw肵ĉ.
 */
int32_t AG903_OSPMgrSetCommandBuf(AG903_OSPMgrHandle *handle, AG903_OSPCmdBuf *param, bool init)
{
	int32_t				rc = AG903_ENONE;
	uint8_t				ch;
	OSPCmdDesc			*desc;

	if(NULL == handle || NULL == param) {
		rc = -AG903_EINVAL;
	}

	rc = OSPMgrCheckHandle(handle, &ch);

	if (rc == AG903_ENONE) {
		if ((AG903_OSP_BUFMODE_NUM <= param->mode) || (NULL == param->addr) || (0 == param->size)) {
			/* p[^ُ */
			rc = -AG903_EINVAL;
		}
	}

	if (rc == AG903_ENONE) {
		/*
		 * Commad Buffer:
		 * +--------------------+
		 * | Command Head       | sizeof(OSPCmdDesc)
		 * +--------------------+
		 * | Command Buff       | R}h~4Bytes
		 * |  +-----------------+
		 * |  | Commad 1        |
		 * |  +-----------------+
		 * |  | Commad 2        |
		 * |  +-----------------+
		 * |  |      :          |
		 * |  +-----------------+
		 * |  | Commad n        |
		 * +--------------------+
		 */
		OspChStat[ch].cmdbuf = (void*)param->addr;
		desc = (OSPCmdDesc*)param->addr;
		desc->Buf      = param->addr + sizeof(OSPCmdDesc) / sizeof(uint32_t);
		desc->BufLen   = (param->size - sizeof(OSPCmdDesc)) / sizeof(uint32_t);
		desc->CurLen   = 0;
		desc->WrBuf    = desc->Buf;
		desc->RdBuf    = desc->Buf;		
		desc->BuffMode = param->mode;
		if(true == init) {
			AG903_OSPMgrClearCommand(handle);
		}
	}

	return rc;
}

/**
 * @brief           R}hNA
 * @param           handle [in] OSPnh
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     R}hobt@ɓo^R}hj܂.<p>
 *                  FIFOɃZbgĂR}h̓NAꂸɏ܂.
 */
int32_t AG903_OSPMgrClearCommand(AG903_OSPMgrHandle *handle)
{
	int32_t				rc = AG903_ENONE;
	uint8_t				ch;
	OSPCmdDesc			*desc;							/* R}hfXNv^		*/

	if(NULL == handle) {
		rc = -AG903_EINVAL;
	}

	rc = OSPMgrCheckHandle(handle, &ch);

	if (rc == AG903_ENONE) {
		desc = (OSPCmdDesc*)OspChStat[ch].cmdbuf;

		desc->CurLen = 0;
		desc->RdBuf  = desc->Buf;
		desc->WrBuf  = desc->Buf;
	}

	return rc;
}

/**
 * @brief           R}ho^擾
 * @param           handle [in] OSPnh
 * @param           cnt [out] o^R}h
 * @return          G[R[h
 * @retval           AG903_ENONE   I
 * @retval          -AG903_EINVAL  p[^ُ
 * @retval          -AG903_ENODATA R}hobt@ُ
 * @description     R}hobt@ɓo^ĂR}hԂ܂.
 */
int32_t AG903_OSPMgrGetCommandCnt(AG903_OSPMgrHandle *handle, uint32_t *cnt)
{
	int32_t				rc = AG903_ENONE;
	uint8_t				ch;
	OSPCmdDesc			*desc;							/* R}hfXNv^		*/

	if(NULL == handle) {
		rc = -AG903_EINVAL;
	}

	rc = OSPMgrCheckHandle(handle, &ch);

	if (rc == AG903_ENONE) {
		if (cnt == NULL) {
			/* p[^ُ */
			rc = -AG903_EINVAL;
		}
	}
	if (rc == AG903_ENONE) {
		if(NULL == OspChStat[ch].cmdbuf) {
			/* R}hobt@ُ */
			rc = -AG903_ENODATA;
		}
	}

	if (rc == AG903_ENONE) {
		desc   = (OSPCmdDesc*)OspChStat[ch].cmdbuf;
		(*cnt) = desc->CurLen;
	}

	return rc;
}

/**
 * @brief           Dxݒ
 * @param           handle [in] OSPnh
 * @param           priority [in] Dxi0`3j
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     `lʂɗDxݒ肵܂.<p>
 *                  `lœɃoXANZXꍇ, Dxɏ]ăR}hs܂.<p>
 *                  Dxꍇ, Ehrŏ܂.
 * @seealso         _AG903_OSPMgrPri
 */
int32_t AG903_OSPMgrSetPriority(AG903_OSPMgrHandle *handle, uint8_t priority)
{
	int32_t		rc = AG903_ENONE;
	uint8_t		ch;

	if(NULL == handle) {
		rc = -AG903_EINVAL;
	}

	rc = OSPMgrCheckHandle(handle, &ch);

	if (rc == AG903_ENONE) {
		if (priority >= AG903_OSP_PRI_NUM) {
			/* Dxs */
			rc = -AG903_EINVAL;
		}
	}

	if (rc == AG903_ENONE) {
		AG903_OSPPrmPRIOR(ch, priority);
	}

	return rc;
}

/**
 * @brief           OSPJn
 * @param           handle [in] OSPnh
 * @return          G[R[h
 * @return           AG903_ENONE  I
 * @return          -AG903_EINVAL p[^ُ
 * @description     FIFOɃZbgꂽR}hs܂.
 */
int32_t AG903_OSPMgrEnable(AG903_OSPMgrHandle *handle)
{
	int32_t		rc = AG903_ENONE;
	uint8_t		ch;

	if(NULL == handle) {
		rc = -AG903_EINVAL;
	}

	rc = OSPMgrCheckHandle(handle, &ch);

	if (rc == AG903_ENONE) {
		AG903_OSPPrmCTRL(ch, true);
	}

	return rc;
}

/**
 * @brief           OSP~
 * @param           handle [in] OSPnh
 * @return          G[R[h
 * @return           AG903_ENONE  I
 * @return          -AG903_EINVAL p[^ُ
 * @description     handleɊĂꂽ`lOSP~܂.
 */
int32_t AG903_OSPMgrDisable(AG903_OSPMgrHandle *handle)
{
	int32_t		rc = AG903_ENONE;
	uint8_t		ch;

	if(NULL == handle) {
		rc = -AG903_EINVAL;
	}

	rc = OSPMgrCheckHandle(handle, &ch);

	if (rc == AG903_ENONE) {
		AG903_OSPPrmCTRL(ch, false);
	}

	return rc;
}

/**
 * @brief           Ԏ擾
 * @param           handle [in] OSPnh
 * @param           stat [out] OSPԁi0F~, 0ȊOF쒆j
 * @param           cnt [out] ݎsFIFOJEg
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     OSP̓, ьݎsFIFOJEgԂ܂.
 */
int32_t AG903_OSPMgrGetStat(AG903_OSPMgrHandle *handle, uint8_t *stat, uint8_t *cnt)
{
	int32_t		rc = AG903_ENONE;
	uint8_t		ch;

	if(NULL == handle) {
		rc = -AG903_EINVAL;
	}

	rc = OSPMgrCheckHandle(handle, &ch);

	if (rc == AG903_ENONE) {
		if ((stat == NULL)
		||  (cnt  == NULL)) {
			/* p[^s */
			rc = -AG903_EINVAL;
		}
	}

	if (rc == AG903_ENONE) {
		AG903_OSPPrmSTAT(ch, stat, cnt);
	}

	return rc;
}

/**
 * @brief           FIFOԎ擾
 * @param           handle [in] OSPnh
 * @param           ovf [out] I[o[t[ (0:Ȃ 1:)
 * @param           siz [out] FIFŐ󂫗eʂ擾ϐւ̃|C^ (0`AG903_OSP_FIFO_NUM)
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     nhɊ蓖Ăꂽ`lFIFOԂ擾܂.
 */
int32_t AG903_OSPMgrGetFIFOStat(AG903_OSPMgrHandle *handle, uint8_t *ovf, uint8_t *siz)
{
	int32_t		rc = AG903_ENONE;
	uint8_t		ch;

	if(NULL == handle) {
		rc = -AG903_EINVAL;
	}

	rc = OSPMgrCheckHandle(handle, &ch);

	if (rc == AG903_ENONE) {
		if ((ovf == NULL)
		||  (siz == NULL)) {
			/* p[^ُ */
			rc = -AG903_EINVAL;
		}
	}

	if (rc == AG903_ENONE) {
		AG903_OSPPrmFIFOSTAT(ch, ovf, siz);
	}

	return rc;
}

/**
 * @brief           tOݒ
 * @param           flag [in] tO (erbg)
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     EFCgp̃tOZbg܂.<p>
 *                  ̃tOw\ł.
 * @note            tO͑S`lŋʂł.
 */
int32_t AG903_OSPMgrSetFlag(uint32_t flag)
{
	int32_t		rc = AG903_ENONE;

	if ((flag & 0xF0000000)) {
		/* p[^ُ */
		rc = -AG903_EINVAL;
	}

	if (rc == AG903_ENONE) {
		AG903_OSPPrmFLAGSET(flag);
	}

	return rc;
}

/**
 * @brief           tONA
 * @param           flag [in] tO (erbg)
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     EFCgp̃tONA܂.<p>
 *                  ̃tOw\ł.
 * @note            tO͑S`lŋʂł.
 */
int32_t AG903_OSPMgrClearFlag(uint32_t flag)
{
	int32_t		rc = AG903_ENONE;

	if ((flag & 0xF0000000)) {
		/* p[^ُ */
		rc = -AG903_EINVAL;
	}

	if (rc == AG903_ENONE) {
		AG903_OSPPrmFLAGCLR(flag);
	}

	return rc;
}

/**
 * @brief           tOԎ擾
 * @param           flag [out] tO (erbg)
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     EFCgp̃tȌԂ擾܂.
 * @note            tO͑S`lŋʂł.
 */
int32_t AG903_OSPMgrGetFlagStat(uint32_t *flag)
{
	int32_t		rc = AG903_ENONE;

	if (flag == NULL) {
		/* p[^ُ */
		rc = -AG903_EINVAL;
	}

	if (rc == AG903_ENONE) {
		AG903_OSPPrmFLAGSTAT(flag);
	}

	return rc;
}

/**
 * @brief           CxgJEgLEݒ
 * @param           evt [in] Cxgԍi0`63j
 * @param           enable [in] LEݒi0: 1:Lj
 * @param           cnt [in] CxgJE^i0`15j
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     CxgJEg̗LEݒ肵܂.<p>
 *                  CxgJEg̍őliI[o[t[ojݒ肵܂.<p>
 *                  CxgJE^Lɂꍇ, CxgɂJE^CNg܂.<p>
 *                  CNgꂽʃI[o[t[ꍇ, ̏Ԃێ܂.
 * @note            tO͑S`lŋʂł.<p>
 *                  CxgԍF0`63<p>
 *                  LEݒF0: 1:L<p>
 *                  CxgJEgF0`15
 */
int32_t AG903_OSPMgrSetEventCntEnable(uint32_t evt, uint8_t enable, uint8_t cnt)
{
	int32_t	rc = AG903_ENONE;

	if (evt >= AG903_OSP_EVENT_NUM) {
		/* Cxgԍُ */
		rc = -AG903_EINVAL;
	}

	if (rc == AG903_ENONE) {
		if ((enable & ~0x01)
		||  (cnt    & ~0x0F)) {
			/* p[^ُ */
			rc = -AG903_EINVAL;
		}
	}

	if (rc == AG903_ENONE) {
		AG903_OSPPrmEnableEvent(evt, enable, cnt);
	}

	return rc;
}

/**
 * @brief           CxgԎ擾
 * @param           evt [in] Cxgԍi0`63j
 * @param           ovf [out] I[o[t[Ԋi[obt@
 * @param           cnt [out] CxgJE^i[obt@
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     wCxgԍ̃I[o[t[, ь݂̃JEglԂ܂.
 * @note            tO͑S`lŋʂł.<p>
 *                  CxgԍF0`63<p>
 *                  I[o[t[ԁF0:Ȃ 1:<p>
 *                  CxgJEgF0`15
 */
int32_t AG903_OSPMgrGetEventStat(uint32_t evt, uint8_t *ovf, uint8_t *cnt)
{
	int32_t	rc = AG903_ENONE;

	if (evt >= AG903_OSP_EVENT_NUM) {
		/* Cxgԍُ */
		rc = -AG903_EINVAL;
	}

	if (rc == AG903_ENONE) {
		if ((ovf == NULL)
		||  (cnt == NULL)) {
			/* p[^ُ */
			rc = -AG903_EINVAL;
		}
	}

	if (rc == AG903_ENONE) {
		AG903_OSPPrmGetEventStat(evt, ovf, cnt);
	}

	return rc;
}

/**
 * @brief           FIFOւ̃R}h̃Zbg
 * @param           handle [in] OSPnh
 * @param           setcnt [in] ZbgR}h
 * @return          G[R[h
 * @retval           AG903_ENONE     I
 * @retval          -AG903_EINVAL    p[^ُ
 * @retval          -AG903_ENODATA   R}hobt@ُ
 * @retval          -AG903_EOVERFLOW I[o[t[
 * @retval          -AG903_EBUSY
 * @description     R}hobt@setcnt̃R}hFIFOɃZbg܂.<p>
 *                  R}hobt@̏ꍇ̓G[Ԃ܂(-AG903_ENODATA).<p>
 *                  setcntFIFŐ󂫒Ăꍇ̓G[Ԃ܂(-AG903_EOVERFLOW).<p>
 *                  setcnto^σR}h𒴂Ăꍇ̓Nbv܂.
 * @note            o^F1`R}hobt@o^iő吔FAG903_OSP_FIFO_NUMj
 * @note            setcntR}hFIFOɃZbg܂Ŗ{API畜A܂.
 * @note            setcnt͂ЂƂ܂Ƃ܂̃R}h𕪒f邱ƂȂ悤ɑIŉB
 */
int32_t AG903_OSPMgrSetFIFO(AG903_OSPMgrHandle *handle, uint32_t setcnt)
{
	int32_t				rc = AG903_ENONE;
	uint8_t				ch;
	OSPCmdDesc			*desc;							/* R}hfXNv^		*/
	uint8_t				ovf = 0;
	uint8_t				siz = 0;
	uint32_t			MaxLen;

	if(NULL == handle) {
		rc = -AG903_EINVAL;
	}

	rc = OSPMgrCheckHandle(handle, &ch);
	if(AG903_ENONE != rc) {
		/* handleُ */
		return -AG903_EINVAL;
	}
	if(NULL == OspChStat[ch].cmdbuf) {
		/* R}hobt@ُ */
		return -AG903_ENODATA;
	}

	desc = (OSPCmdDesc*)OspChStat[ch].cmdbuf;
	if(0 == desc->CurLen) {
		/* R}ho^ */
		return -AG903_ENODATA;
	}

	if (setcnt > desc->CurLen) {
		/* R}ho^ȏ̃ZbgR}hw肷邱Ƃ͗p@ԈĂ */
		return -AG903_EINVAL;
	}

	AG903_OSPPrmFIFOSTAT(ch, &ovf, &siz);

	if (ovf == true) {
		/* FIFOI[o[t[ */
		return -AG903_EOVERFLOW;
	}
	if ((siz == 0) || (siz < setcnt)) {
		/* FIFOZbg(=setcnt)̋󂫂FIFOɂȂꍇG[ */
		return -AG903_EBUSY;
	}

#if AG903_OSP_CMD_PARSE
	if (rc == AG903_ENONE) {
		rc = OSPMgrParseCommand(desc);
	}
#endif/* if AG903_OSP_CMD_PARSE */

	if (AG903_OSP_BUFMODE_RING == desc->BuffMode) {
		MaxLen = desc->BufLen;	/* RINGobt@[h */
	}
	else {
		MaxLen = desc->CurLen;	/* FIFOobt@[h */
	}

	if ((desc->RdBuf + setcnt) > (desc->Buf + MaxLen)) {
		uint32_t	Len;
		/*
		 * +-----------+ <- Buf
		 * | 0         |
		 * +-----------+
		 * | :         |
		 * +-----------+ <- RdBuf
		 * | :         |
		 * +-----------+ <- (Buf + MaxLen)
		 * |           |
		 * +-----------+ <- (RdBuf + setcnt)
		 * | :         |
		 * +-----------+
		 */
		Len = (desc->Buf + MaxLen) - desc->RdBuf;
		AG903_OSPPrmFIFODT(ch, desc->RdBuf, Len);

		desc->RdBuf = desc->Buf;
		setcnt     -= Len;

		if (AG903_OSP_BUFMODE_RING == desc->BuffMode) {
			desc->CurLen -= Len;
		}
	}

	AG903_OSPPrmFIFODT(ch, desc->RdBuf, setcnt);
	desc->RdBuf += setcnt;

	if (AG903_OSP_BUFMODE_RING == desc->BuffMode) {
		desc->CurLen -= setcnt;
	}

	if (desc->RdBuf >= (desc->Buf + MaxLen)) {
		desc->RdBuf  = desc->Buf;
	}

	return rc;
}

/**
 * @brief           FIFONA
 * @param           handle [in] OSPnh
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     FIFOɃZbgR}hNA܂.<p>
 *                  FIFŐ󂫗e, I[o[t[Ԃ܂.
 * @note            R}hɎsꍇ, ̃f[^͔j܂.
 */
int32_t AG903_OSPMgrClearFIFO(AG903_OSPMgrHandle *handle)
{
	int32_t		rc = AG903_ENONE;
	uint8_t		ch;

	if(NULL == handle) {
		rc = -AG903_EINVAL;
	}

	rc = OSPMgrCheckHandle(handle, &ch);

	if (rc == AG903_ENONE) {
		AG903_OSPPrmFIFOCTRL(ch);
	}

	return rc;
}

/**
 * @brief           R}ho^
 * @param           handle [in] OSPnh
 * @param           cmd [in] R}h
 * @param           param [in] p[^iR}hʁj
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @retval          -AG903_EOVERFLOW I[o[t[
 * @description     R}ho^܂.
 */
int32_t AG903_OSPMgrSetCommand(AG903_OSPMgrHandle *handle, AG903_OSPMgrCmd cmd, uint32_t param)
{
	int32_t		rc = AG903_ENONE;
	uint32_t	cmddt;
	uint8_t		ch = -1;

	if(NULL == handle) {
		rc = -AG903_EINVAL;
	}

	rc = OSPMgrCheckHandle(handle, &ch);

	if (rc == AG903_ENONE) {
		rc = OSPMgrCheckOverflow(ch);
	}

	switch(cmd) {
		case AG903_OSP_CMD_BUS_SINGLE:	/* BAS */
			if( (0==param) || (AG903_OSP_CMD_BAS_SIZE<param) ) {
				/* o^f[^ُ */
				rc = -AG903_EINVAL;
			}
			else {
				cmddt = OSP_CMD_BAS(param);
			}
			break;
		case AG903_OSP_CMD_BUS_BURST_FIX:	/* BAB [F=1] */
			if( (0==param) || (AG903_OSP_CMD_BAB_SIZE<param) ) {
				/* o^f[^ُ */
				rc = -AG903_EINVAL;
			}
			else {
				cmddt = OSP_CMD_BAB(param, 1);
			}
			break;
		case AG903_OSP_CMD_BUS_BURST_INC:	/* BAB [F=0] */
			if( (0==param) || (AG903_OSP_CMD_BAB_SIZE<param) ) {
				/* o^f[^ُ */
				rc = -AG903_EINVAL;
			}
			else {
				cmddt = OSP_CMD_BAB(param, 0);
			}
			break;
		case AG903_OSP_CMD_BUS_ADDR:	/* ADDR */
			if( (0==param) || (param%4) ) {
				/* oXAhXُ */
				rc = -AG903_EINVAL;
			}
			else {
				cmddt = OSP_CMD_ADDR(param);
			}
			break;
		case AG903_OSP_CMD_BUS_DATA:	/* DATA */
			cmddt = OSP_CMD_DATA(param);
			break;
		case AG903_OSP_CMD_BUS_WAIT:	/* BAW */
			cmddt = OSP_CMD_BAW();
			break;
		case AG903_OSP_CMD_INTR_WAIT:	/* INTR */
			cmddt = OSP_CMD_INTR(param);
			break;
		case AG903_OSP_CMD_EVNT_WAIT:	/* EVENT */
			if(AG903_OSP_EVENT_NUM<=param) {
				/* CxgIDُ */
				rc = -AG903_EINVAL;
			}
			else {
				cmddt = OSP_CMD_EVENT(param);
			}
			break;
		case AG903_OSP_CMD_EVNT_DEC:	/* EVENTD */
			if(AG903_OSP_EVENT_NUM<=param) {
				/* CxgIDُ */
				rc = -AG903_EINVAL;
			}
			else {
				cmddt = OSP_CMD_EVENTD(param);
			}
			break;
		case AG903_OSP_CMD_EVNT_CLR:	/* EVENTC */
			if(AG903_OSP_EVENT_NUM<=param) {
				/* CxgIDُ */
				rc = -AG903_EINVAL;
			}
			else {
				cmddt = OSP_CMD_EVENTC(param);
			}
			break;
		case AG903_OSP_CMD_FLAG_SET:	/* SETF */
			cmddt = OSP_CMD_SETF(param);
			break;
		case AG903_OSP_CMD_FLAG_CLR:	/* CLRF */
			cmddt = OSP_CMD_CLRF(param);
			break;
		case AG903_OSP_CMD_NOP:
			cmddt = OSP_CMD_NOP();
			break;
		default:
			/* R}hُ */
			rc = -AG903_EINVAL;
			break;
	}

	if (rc == AG903_ENONE) {
		rc = OSPMgrSetCommand(ch, cmddt);
	}

	return rc;

}

/**
 * @brief           nh`FbN
 * @param           handle [in] OSPnh
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @description     nhl, `l̐`FbN܂.
 */
static int32_t OSPMgrCheckHandle(AG903_OSPMgrHandle *handle, uint8_t *ch)
{
	int32_t		rc = AG903_ENONE;
	uint32_t	channel;


	channel = ((uint32_t)handle - (uint32_t)OspHandleStat) / sizeof(AG903_OSPMgrHandleStat);

	if( (AG903_OSP_CH_NUM <= channel) ||
		(&OspHandleStat[channel] != (AG903_OSPMgrHandleStat*)handle) ) {
		/* nhُ */
		rc = -AG903_EINVAL;
	}
	else {
		(*ch) = (uint8_t)channel;
	}

	return rc;
}

/**
 * @brief           I[o[t[`FbN
 * @param           handle [in] OSPnh
 * @return          G[R[h
 * @return           AG903_ENONE  I
 * @retval          -AG903_EINVAL p[^ُ
 * @retval          -AG903_EOVERFLOW I[o[t[
 * @description     I[o[t[̔L`FbN܂.<p>
 *                  handleɓnnh OSPMgrCheckHandle ɂă`FbNς݂ł邱Ƃz肵Ă܂.
 */
static int32_t OSPMgrCheckOverflow(uint8_t ch)
{
	int32_t				rc = AG903_ENONE;
	OSPCmdDesc			*desc;							/* R}hfXNv^		*/


	desc = (OSPCmdDesc*)OspChStat[ch].cmdbuf;

	if ((desc->Buf == (uint32_t *)NULL                     )
	||  (desc->BufLen == 0                                 )
	||  (desc->CurLen > desc->BufLen                       )
	||	(desc->RdBuf < desc->Buf                           )
	||	(desc->Buf + desc->BufLen <= desc->RdBuf           )
	||	(desc->WrBuf < desc->Buf                           )
	||	(desc->Buf + desc->BufLen <= desc->WrBuf           )) {
		/* R}hobt@ُ */
		rc = -AG903_EINVAL;
	}

	if (rc == AG903_ENONE) {
		if ((desc->CurLen + 1) > desc->BufLen) {
			/* R}hobt@I[o[t[ */
			rc = -AG903_EOVERFLOW;
		}
	}

	return rc;
}

/**
 * @brief           R}hݒ
 * @param           handle [in] OSPnh
 * @param           cmd [in] R}h
 * @return          G[R[h
 * @retval          AG903_ENONE I
 * @description     R}hݒp[`ł.
 */
static int32_t OSPMgrSetCommand(uint8_t ch, uint32_t c)
{
	int32_t				rc = AG903_ENONE;
	OSPCmdDesc			*desc;						/* R}hfXNv^		*/


	desc = (OSPCmdDesc*)OspChStat[ch].cmdbuf;

	*(desc->WrBuf) = c;
	desc->WrBuf++;
	desc->CurLen++;

	if (desc->BuffMode) {							/* RINGobt@[h		*/
		if (desc->WrBuf >= (desc->Buf + desc->BufLen)) {
			desc->WrBuf = desc->Buf;
		}
	}

	return rc;
}

#if AG903_OSP_CMD_PARSE
/**
 * @brief           R}h
 * @param           desc [in] R}hfXNv^
 * @return          G[R[h
 * @retval           AG903_ENONE  I
 * @retval          -AG903_EFAULT R}hُ
 * @description     o^ꂽR}h͂܂.
 */
static int32_t OSPMgrParseCommand(OSPCmdDesc *desc)
{
	int32_t				rc = AG903_ENONE;
	uint32_t			*buf, *endbuf;
	uint8_t				cmdid;
	uint32_t			len;
	uint32_t			ii;


	buf    = desc->RdBuf;								/* 擪R}hobt@		*/
	endbuf = desc->Buf + desc->BufLen;					/* I[R}hobt@		*/

	for (ii = 0;ii < desc->CurLen;ii++) {
		cmdid = (*buf >> 28);							/* R}hID擾			*/

		switch (cmdid) {
			case OSP_CMDID_BAS:
				len  = ((*buf & AG903_OSP_CMD_BAS_SIZE) << 1) + 1/*Bus Address*/;
				buf += len;
				ii  += len;
				break;

			case OSP_CMDID_BAB:
				len  = (*buf & AG903_OSP_CMD_BAB_SIZE) + 1/*Bus Address*/;
				buf += len;
				ii  += len;
				break;

			case OSP_CMDID_BAW:
			case OSP_CMDID_INTR:
			case OSP_CMDID_EVENT:
			case OSP_CMDID_EVENTD:
			case OSP_CMDID_EVENTC:
			case OSP_CMDID_SETF:
			case OSP_CMDID_CLRF:
			case OSP_CMDID_NOP:
				break;

			default:
				/* R}hobt@AhXُ */
				rc = -AG903_EFAULT;
				goto err;
		}

		buf++;
		if (buf > endbuf) { buf = desc->Buf; }
	}

err:
	return rc;
}

/**
 * @brief           CRC^`FbN
 * @param           mode [in] CRC[h(0:CRC 1:CRC`FbN)
 * @param           data [in] f[^̃x[XAhX
 * @param           s_adr [in] CRCvZ̊JnItZbg
 * @param           e_adr [in] CRCvZ̏IItZbg
 * @return          [hɑ΂錋
 * @retval          >0 CRCvZ(CRC[h)
 * @retval          >=0 `FbN(CRC`FbN[h)
 * @description     modeɂāACRC̐ ܂ CRC̃`FbNs܂.
 */
static uint16_t OSPMgrParseCommandCRC(uint8_t mode, uint8_t* data, uint32_t s_adr, uint32_t e_adr)
{
	uint32_t	crc;												/* CRCvZϐ			*/
	uint8_t		*d_adr;												/* vZf[^ւ̃|C^		*/
	uint32_t	d_len;												/* vZf[^̒			*/
	uint16_t	ret;												/* ߂li[ϐ			*/

	/*
	 * f[^̈撼2oCgCRC݈ʒuƂ
	 * +--------------------+
	 * | f[^̈			|
	 * |			+---+---+
	 * |			|  CRC	|
	 * +---+---+----+---+---+
	 */

	/*  */
	crc	  = 0;														/* CRC̏					*/
	d_adr = (data + s_adr);											/* vZJnʒuւ̃|C^	*/
	d_len = e_adr - s_adr + 2;										/* vZf[^̐+2(CRCݗp)	*/

	if (mode == 0) {												/* CRC[h				*/
		/* CRC݈ʒȕ */
		*(data + e_adr + 0) = 0;
		*(data + e_adr + 1) = 0;
	}
	else {															/* CRC`FbN[h			*/
	}

	for (uint32_t ii = 0;ii < d_len;ii++) {							/* CRC݈ʒu܂߃[v	*/
		crc = crc | *(d_adr++);										/* 1Byteǂ݂				*/

		/* CRC1BytevZ */
		for (int8_t jj = 0;jj < 8;jj++) {
			crc = crc << 1;											/* CRCƃf[^xɃVtg	*/

			if ((crc & 0x01000000) != 0) {							/* CRCvZϐVtgŌӂꂵmF	*/
				crc = crc ^ 0x00102100;								/* XOR				*/
			}
		}
	}

	ret = (uint16_t)((crc & 0x00ffff00) >> 8);

	if (mode == 0) {												/* CRC[h				*/
		*(d_adr - 2) = (uint8_t)((ret >> 8) & 0x00ff);				/* CRC̏ʏ			*/
		*(d_adr - 1) = (uint8_t)((ret >> 0) & 0x00ff);				/* CRC̉ʏ			*/
	}
	else {															/* CRC`FbN[h			*/
		/* ret = 0Ȃ琳 */
	}

	return ret;
}
#endif/* if AG903_OSP_CMD_PARSE */
