/**
 * @brief       SSC Primitive
 * @author      AXELL CORPORATION
 * @description SSC Primitive Layer
 * @note        none
 * @history     2017_02_22  
 * @history     2017_10_26  Ver2.0
 * @history     2019_03_08  [SDK2.2] SYSł̌yȕsC (#2443)
*/
/* DOM-IGNORE-BEGIN */
/*
 * This program was created by AXELL CORPORATION.
 * Copyright (C) 2017-2019 AXELL CORPORATION, all rights reserved.
 */
/* DOM-IGNORE-END */

#include "AG903_errno.h"
#include "AG903_common.h"
#include "sys/sscprm.h"
#include "ssp/sspprm.h"
#include "register/AG903_sscreg.h"

/**
 * @brief           SSCREVISIOÑWX^Read
 * @param           rev [out] ݒp[^
 * @return          void
 * @description     SSCREVISIOÑWX^Read
*/
void AG903_SSCPrmGetRevesion(uint32_t* rev)
{
	ASSERT(NULL != rev);
	(*rev) = AG903_SSC->REVISION;
	return;
}

/**
 * @brief           SSCMODE_STATUS̃WX^Read
 * @param           btmode [out] MODE[q
 * @param           bttest [out] TEST[q
 * @param           status [out] Nv
 * @return          void
 * @description     SSCMODE_STATUS̃WX^Read
*/
void AG903_SSCPrmGetModeStatus(uint8_t* btmode, uint8_t* bttest, uint8_t* status)
{
	uint32_t	val;

	ASSERT(NULL != btmode);
	ASSERT(NULL != bttest);
	ASSERT(NULL != status);

	val = AG903_SSC->MODE_STATUS;
	(*btmode) = (uint8_t)((val & AG903_SSC_MODE_STATUS_BOOTMODE_MSK) >> AG903_SSC_MODE_STATUS_BOOTMODE_POS);
	(*bttest) = (uint8_t)((val & AG903_SSC_MODE_STATUS_BOOTTEST_MSK) >> AG903_SSC_MODE_STATUS_BOOTTEST_POS);
	(*status) = (uint8_t)((val & (AG903_SSC_MODE_STATUS_WDTCA5_MSK|AG903_SSC_MODE_STATUS_WDTGPP_MSK|AG903_SSC_MODE_STATUS_WDTPIN_MSK|
		                         AG903_SSC_MODE_STATUS_INITPIN_MSK)) >> AG903_SSC_MODE_STATUS_WDTCA5_POS);
	return;
}

/**
 * @brief           SSCMODE_STATUS̃WX^Write(wrbgSetŌʂ0NA)
 * @param           clr [in] NAΏۃrbg (clr̃rbg[3:0]SSCMODE_STATUS[11:8]ɑΉ܂j
 * @return          void
 * @description     SSCMODE_STATUS̃WX^Write(wrbgSetŌʂ0NA)
 * @note            ۂɃNAłSSCMODE_STATUSWX^̃rbg
 *                  [11:8]ɑΉINITPIN,WDTPIN,WDTGPP,WDTCA5łB[7:0]͉e󂯂܂B
*/
void AG903_SSCPrmClrModeStatus(uint8_t clr)
{
	uint32_t	val;

	val  = (uint32_t)(clr << AG903_SSC_MODE_STATUS_WDTCA5_POS);
	val &= (AG903_SSC_MODE_STATUS_WDTCA5_MSK|AG903_SSC_MODE_STATUS_WDTGPP_MSK|AG903_SSC_MODE_STATUS_WDTPIN_MSK|AG903_SSC_MODE_STATUS_INITPIN_MSK);

	AG903_SSC->MODE_STATUS = val;

	return;
}

/**
 * @brief           SSCIRQ_STATUS_LOWER,SSCIRQ_STATUS_UPPER̃WX^Read
 * @param           lower [out] ݃Xe[^XiIRQ 0`31j
 * @param           upper [out] ݃Xe[^XiIRQ31`63j
 * @return          void
 * @description     SSCIRQ_STATUS_LOWER,SSCIRQ_STATUS_UPPER̃WX^Read
*/
void AG903_SSCPrmGetIrqStatus(uint32_t* lower, uint32_t* upper)
{
	ASSERT(NULL != lower);
	ASSERT(NULL != upper);
	
	(*lower) = AG903_SSC->IRQ_STATUS_LOWER;
	(*upper) = AG903_SSC->IRQ_STATUS_UPPER;
	return;
}

/**
 * @brief           SSCIRQ_BUS_STATUS̃WX^Read
 * @param           status [out] oXݗvXe[^X
 * @return          void
 * @description     SSCIRQ_BUS_STATUS̃WX^Read
*/
void AG903_SSCPrmGetIntBusStatus(uint8_t* status)
{
	ASSERT(NULL != status);
	(*status) = (uint8_t)(AG903_SSC->IRQ_BUS_STATUS & AG903_SSC_IRQ_BUS_STATUS_IRQBUSSTATUS_MSK);
	return;
}

/**
 * @brief           SSCIRQ_BUS_ENABLẼWX^Write
 * @param           enable [in] oX݃Cl[uݒirbgwj
 * @return          void
 * @description     SSCIRQ_BUS_ENABLẼWX^Write
 * @note            lWX^ɏ݂܂B<p>
 *                  lORꍇAG903_SSCPrmGetIntBusEnable()gpʑwORĉB
*/
void AG903_SSCPrmSetIntBusEnable(uint8_t enable)
{
	AG903_SSC->IRQ_BUS_ENABLE = (uint32_t)(enable & AG903_SSC_IRQ_BUS_ENABLE_IRQBUSENABLE_MSK);
	return;
}

/**
 * @brief           SSCIRQ_BUS_ENABLẼWX^Read
 * @param           status [out] oX݃Cl[uݒ
 * @return          void
 * @description     SSCIRQ_BUS_ENABLẼWX^Read
*/
void AG903_SSCPrmGetIntBusEnable(uint8_t* status)
{
	ASSERT(NULL != status);
	(*status) = (uint8_t)(AG903_SSC->IRQ_BUS_ENABLE & AG903_SSC_IRQ_BUS_ENABLE_IRQBUSENABLE_MSK);
	return;
}

/**
 * @brief           SSCINT_STATUS_LOWER,SSCINT_STATUS_MIDDLE,SSCINT_STATUS_UPPER̃WX^Read(Xe[^X\)
 * @param           lower  [out] ݃Xe[^Xi 0`31j
 * @param           middle [out] ݃Xe[^Xi31`63j
 * @param           upper  [out] ݃Xe[^Xi64`95j
 * @return          void
 * @description     SSCINT_STATUS_LOWER,SSCINT_STATUS_MIDDLE,SSCINT_STATUS_UPPER̃WX^Read(Xe[^X\)
*/
void AG903_SSCPrmGetSysIntStatus(uint32_t* lower, uint32_t* middle, uint32_t* upper)
{
	ASSERT(NULL != lower);
	ASSERT(NULL != middle);
	ASSERT(NULL != upper);

	(*lower)  = AG903_SSC->INT_STATUS_LOWER;
	(*middle) = AG903_SSC->INT_STATUS_MIDDLE;
	(*upper)  = AG903_SSC->INT_STATUS_UPPER;
	return;
}

/**
 * @brief           SSCINT_STATUS_LOWER,SSCINT_STATUS_MIDDLE,SSCINT_STATUS_UPPER̃WX^Write(Cl[uݒ)
 * @param           ch     [in] ݃`li0`3j
 * @param           lower  [in] ݃Cl[uݒi 0`31j
 * @param           middle [in] ݃Cl[uݒi31`63j
 * @param           upper  [in] ݃Cl[uݒi64`95j
 * @return          void
 * @description     SSCINT_STATUS_LOWER,SSCINT_STATUS_MIDDLE,SSCINT_STATUS_UPPER̃WX^Write(Cl[uݒ)
 * @note            lWX^ɏ݂܂B
 *                  lORꍇAG903_SSCPrmGetSysIntEnable()gpʑwORĉB
*/
void AG903_SSCPrmSetSysIntEnable(uint8_t ch, uint32_t lower, uint32_t middle, uint32_t upper)
{
	ASSERT(AG903_SSC_SYSINT_CH_NUM > ch);
	
	switch(ch) {
		case 0:
			AG903_SSC->INT0_ENABLE_LOWER  = lower;
			AG903_SSC->INT0_ENABLE_MIDDLE = middle;
			AG903_SSC->INT0_ENABLE_UPPER  = upper;
			break;
		case 1:
			AG903_SSC->INT1_ENABLE_LOWER  = lower;
			AG903_SSC->INT1_ENABLE_MIDDLE = middle;
			AG903_SSC->INT1_ENABLE_UPPER  = upper;
			break;
		case 2:
			AG903_SSC->INT2_ENABLE_LOWER  = lower;
			AG903_SSC->INT2_ENABLE_MIDDLE = middle;
			AG903_SSC->INT2_ENABLE_UPPER  = upper;
			break;
		case 3:
			AG903_SSC->INT3_ENABLE_LOWER  = lower;
			AG903_SSC->INT3_ENABLE_MIDDLE = middle;
			AG903_SSC->INT3_ENABLE_UPPER  = upper;
			break;
		default:
			/* ASSERTŃ`FbNĂ̂łɂ͗ȂB */
			break;
	}
	
	return;
}

/**
 * @brief           SSCINT_STATUS_LOWER,SSCINT_STATUS_MIDDLE,SSCINT_STATUS_UPPER̃WX^Read(Cl[uݒ\)
 * @param           ch     [in] ݃`li0`3j
 * @param           lower  [out] ݃Cl[uݒԁi 0`31j
 * @param           middle [out] ݃Cl[uݒԁi31`63j
 * @param           upper  [out] ݃Cl[uݒԁi64`95j
 * @return          void
 * @description     SSCINT_STATUS_LOWER,SSCINT_STATUS_MIDDLE,SSCINT_STATUS_UPPER̃WX^Read(Cl[uݒ\)
*/
void AG903_SSCPrmGetSysIntEnable(uint8_t ch, uint32_t* lower, uint32_t* middle, uint32_t* upper)
{
	ASSERT(AG903_SSC_SYSINT_CH_NUM > ch);
	ASSERT(NULL != lower);
	ASSERT(NULL != middle);
	ASSERT(NULL != upper);
	
	switch(ch) {
		case 0:
			(*lower)  = AG903_SSC->INT0_ENABLE_LOWER;
			(*middle) = AG903_SSC->INT0_ENABLE_MIDDLE;
			(*upper)  = AG903_SSC->INT0_ENABLE_UPPER;
			break;
		case 1:
			(*lower)  = AG903_SSC->INT1_ENABLE_LOWER;
			(*middle) = AG903_SSC->INT1_ENABLE_MIDDLE;
			(*upper)  = AG903_SSC->INT1_ENABLE_UPPER;
			break;
		case 2:
			(*lower)  = AG903_SSC->INT2_ENABLE_LOWER;
			(*middle) = AG903_SSC->INT2_ENABLE_MIDDLE;
			(*upper)  = AG903_SSC->INT2_ENABLE_UPPER;
			break;
		case 3:
			(*lower)  = AG903_SSC->INT3_ENABLE_LOWER;
			(*middle) = AG903_SSC->INT3_ENABLE_MIDDLE;
			(*upper)  = AG903_SSC->INT3_ENABLE_UPPER;
			break;
		default:
			/* ASSERTŃ`FbNĂ̂łɂ͗ȂB */
			break;
	}
	
	return;
}

/**
 * @brief           SSCCLKDUTY_GFX̃WX^Write
 * @param           clk  [in] NbNݒitrue/falsej
 * @param           duty [in] dutyiduty/128j
 * @return          void
 * @description     SSCCLKDUTY_GFX̃WX^Write
*/
void AG903_SSCPrmSetGfxClkDuty(uint8_t clk, uint8_t duty)
{
	uint32_t val;

	val = (uint32_t)(duty & AG903_SSC_CLKDUTY_GFX_DUTY_MSK);
	if(true == clk) {
		val |= (1<<AG903_SSC_CLKDUTY_GFX_VSON_POS);
	}
	AG903_SSC->CLKDUTY_GFX = val;

	return;
}

/**
 * @brief           SSCCLKDUTY_GFX̃WX^Read
 * @param           clk  [out] NbNݒitrue/falsej
 * @param           duty [out] dutyiduty/128j
 * @return          void
 * @description     SSCCLKDUTY_GFX̃WX^Read
*/
void AG903_SSCPrmGetGfxClkDuty(uint8_t* clk, uint8_t* duty)
{
	uint32_t val;

	ASSERT(NULL != clk);
	ASSERT(NULL != duty);

	val = AG903_SSC->CLKDUTY_GFX;
	(*duty) = (uint8_t)((val & AG903_SSC_CLKDUTY_GFX_DUTY_MSK) >> AG903_SSC_CLKDUTY_GFX_DUTY_POS);
	if(AG903_SSC_CLKDUTY_GFX_VSON_MSK & val) {
		(*clk) = true;
	}
	else {
		(*clk) = false;
	}

	return;
}

/**
 * @brief           SSCCLKDUTY_GVD̃WX^Write
 * @param           duty [in] dutyiduty/128j
 * @return          void
 * @description     SSCCLKDUTY_GVD̃WX^Write
*/
void AG903_SSCPrmSetGvdClkDuty(uint8_t duty)
{
	AG903_SSC->CLKDUTY_GVD = (uint32_t)(duty & AG903_SSC_CLKDUTY_GVD_DUTY_MSK);
	return;
}

/**
 * @brief           SSCCLKDUTY_GVD̃WX^Read
 * @param           duty [out] dutyiduty/128j
 * @return          void
 * @description     SSCCLKDUTY_GVD̃WX^Read
*/
void AG903_SSCPrmGetGvdClkDuty(uint8_t* duty)
{
	ASSERT(NULL != duty);

	(*duty) = (uint8_t)((AG903_SSC->CLKDUTY_GVD & AG903_SSC_CLKDUTY_GVD_DUTY_MSK) >> AG903_SSC_CLKDUTY_GVD_DUTY_POS);
	return;
}

/**
 * @brief           SSCWDTOUT_INTMODẼWX^Write
 * @param           type [in] ݐݒix/pXj
 * @return          void
 * @description     SSCWDTOUT_INTMODẼWX^Write
*/
void AG903_SSCPrmSetWdtIntMode(uint8_t mode)
{
	AG903_SSC->WDTOUT_INTMODE = (uint32_t)(mode & AG903_SSC_WDTOUT_INTMODE_LEVEL_MSK);
	return;
}

/**
 * @brief           SSCWDTOUT_INTMODẼWX^Read
 * @param           type [out] ݐݒԁix/pXj
 * @return          void
 * @description     SSCWDTOUT_INTMODẼWX^Read
*/
void AG903_SSCPrmGetWdtIntMode(uint8_t* mode)
{
	uint32_t	val;

	ASSERT(NULL != mode);

	val = AG903_SSC->WDTOUT_INTMODE;
	if(AG903_SSC_WDTOUT_INTMODE_LEVEL_MSK & val) {
		(*mode) = AG903_SSC_WDTINT_LEVEL;
	}
	else {
		(*mode) = AG903_SSC_WDTINT_PULSE;
	}

	return;
}

/**
 * @brief           SSCWDTOUT_INTCLEAR̃WX^Write
 * @param           none
 * @return          void
 * @description     SSCWDTOUT_INTCLEAR̃WX^Write
*/
void AG903_SSCPrmClrWdtInt(void)
{
	AG903_SSC->WDTOUT_INTCLEAR = (1 << AG903_SSC_WDTOUT_INTCLEAR_CLEAR_POS);
	return;
}

/**
 * @brief           SSCWDTOUT_SETUP̃WX^Write
 * @param           rst_ini	[in] INIT[qɂ郊Zbgi0:֎~ 1:)
 * @param           rst_wdt	[in] WDT[qɂ郊Zbgi0:֎~ 1:)
 * @param           wdt_ena	[in] WDT[q̏ói0:֎~ 1:)
 * @param           wdt_od	[in] WDT[q̏o͕@i0:ʏ 1:[I[vhC)
 * @return          void
 * @description     SSCWDTOUT_SETUP̃WX^Write
*/
void AG903_SSCPrmSetWdtOutputMode(uint8_t rst_ini, uint8_t rst_wdt, uint8_t wdt_ena, uint8_t wdt_od)
{
	uint32_t val = 0;
	
	if(true == rst_ini) {
		val |= (1 << AG903_SSC_WDTOUT_SETUP_INIT_POS);
	}
	if(true == rst_wdt) {
		val |= (1 << AG903_SSC_WDTOUT_SETUP_WDT_POS);
	}
	if(true == wdt_ena) {
		val |= (1 << AG903_SSC_WDTOUT_SETUP_OUT_POS);
	}
	if(true == wdt_od) {
		val |= (1 << AG903_SSC_WDTOUT_SETUP_OD_POS);
	}
	AG903_SSC->WDTOUT_SETUP = val;
	
	return;
}

/**
 * @brief           SSCWDTOUT_SETUP̃WX^Read
 * @param           rst_ini	[out] INIT[qZbgi/֎~j
 * @param           rst_wdt	[out] WDT[qZbgi/֎~j
 * @param           wdt_ena	[out] WDT[qói/֎~j
 * @param           wdt_od	[out] WDT[q[I[vhCói/֎~j
 * @return          void
 * @description     SSCWDTOUT_SETUP̃WX^Read
*/
void AG903_SSCPrmGetWdtOutputMode(uint8_t* rst_ini, uint8_t* rst_wdt, uint8_t* wdt_ena, uint8_t* wdt_od)
{
	uint32_t val;

	ASSERT(NULL != rst_ini);
	ASSERT(NULL != rst_wdt);
	ASSERT(NULL != wdt_ena);
	ASSERT(NULL != wdt_od);

	val = AG903_SSC->WDTOUT_SETUP;
	if(AG903_SSC_WDTOUT_SETUP_INIT_MSK & val) {
		(*rst_ini) = true;
	}
	else {
		(*rst_ini) = false;
	}
	if(AG903_SSC_WDTOUT_SETUP_WDT_MSK & val) {
		(*rst_wdt) = true;
	}
	else {
		(*rst_wdt) = false;
	}
	if(AG903_SSC_WDTOUT_SETUP_OUT_MSK & val) {
		(*wdt_ena) = true;
	}
	else {
		(*wdt_ena) = false;
	}
	if(AG903_SSC_WDTOUT_SETUP_OD_MSK & val) {
		(*wdt_od) = true;
	}
	else {
		(*wdt_od) = false;
	}

	return;
}

/**
 * @brief           SSCPIN_FUNC0,SSCPIN_FUNC1̃WX^Write
 * @param           reg_no [in] WX^No.i0`1j
 * @param           val    [in] ݒl
 * @return          void
 * @description     SSCPIN_FUNC0,SSCPIN_FUNC1̃WX^Write
*/
void AG903_SSCPrmSetPinFunction(uint8_t reg_no, uint32_t val)
{
	ASSERT(AG903_SSC_PINFUNC_REG_NUM > reg_no);

	switch(reg_no) {
		case 0:
			AG903_SSC->PIN_FUNC0 = val;
			break;
		case 1:
			AG903_SSC->PIN_FUNC1 = val;
			break;
		default:
			/* ASSERTŃ`FbNĂ̂łɂ͗ȂB */
			break;
	}
	
	return;
}

/**
 * @brief           SSCPIN_FUNC0,SSCPIN_FUNC1̃WX^Read
 * @param           reg_no [in] WX^No.i0`1j
 * @param           val    [out] ݒl
 * @return          void
 * @description     SSCPIN_FUNC0,SSCPIN_FUNC1̃WX^Read
*/
void AG903_SSCPrmGetPinFunction(uint8_t reg_no, uint32_t* val)
{
	ASSERT(NULL != val);
	ASSERT(AG903_SSC_PINFUNC_REG_NUM > reg_no);

	switch(reg_no) {
		case 0:
			(*val) = AG903_SSC->PIN_FUNC0;
			break;
		case 1:
			(*val) = AG903_SSC->PIN_FUNC1;
			break;
		default:
			/* ASSERTŃ`FbNĂ̂łɂ͗ȂB */
			break;
	}
	
	return;
}

/**
 * @brief           SSCPIN_SETUP_DATÃWX^Write
 * @param           tbl_no [in] Pin Setup Table No.i0`136j
 * @param           setup  [in] ݒe
 * @return          void
 * @description     SSCPIN_SETUP_DATÃWX^Write
*/
void AG903_SSCPrmSetPinSetup(uint8_t tbl_no, AG903_SSCPrmPinSetup* setup)
{
	uint32_t	val = 0;

	ASSERT(NULL != setup);
	ASSERT(AG903_SSC_PIN_SETUP_INDEX_MAX >= tbl_no);

	if(true == setup->io3v) {
		val |= (1 << AG903_SSC_PIN_SETUP_LS3V_POS);
	}

	val |= (setup->drive << AG903_SSC_PIN_SETUP_DV_POS) & AG903_SSC_PIN_SETUP_DV_MSK;

	if(true == setup->sr_slow) {
		val |= (1 << AG903_SSC_PIN_SETUP_SR_POS);
	}

	if(true == setup->schmitt) {
		val |= (1 << AG903_SSC_PIN_SETUP_SMT_POS);
	}

	val |= (setup->lv_set << AG903_SSC_PIN_SETUP_LVST_POS) & AG903_SSC_PIN_SETUP_LVST_MSK;

	AG903_SSC->PIN_SETUP_INDEX = (uint32_t)(tbl_no & AG903_SSC_PIN_SETUP_INDEX_INDEX_MSK); /* Ώێw */
	AG903_SSC->PIN_SETUP_DATA  = val;
	
	return;
}

/**
 * @brief           SSCPIN_SETUP_DATÃWX^Read
 * @param           tbl_no [in] Pin Setup Table No.i0`136j
 * @param           setup  [out] ݒe
 * @return          void
 * @description     SSCPIN_SETUP_DATÃWX^Read
*/
void AG903_SSCPrmGetPinSetup(uint8_t tbl_no, AG903_SSCPrmPinSetup* setup)
{
	uint32_t	val;

	ASSERT(NULL != setup);
	ASSERT(AG903_SSC_PIN_SETUP_INDEX_MAX >= tbl_no);

	AG903_SSC->PIN_SETUP_INDEX = (uint32_t)(tbl_no & AG903_SSC_PIN_SETUP_INDEX_INDEX_MSK); /* Ώێw */
	val = AG903_SSC->PIN_SETUP_DATA;

	if(AG903_SSC_PIN_SETUP_LS3V_MSK & val) {
		setup->io3v = true;
	}
	else {
		setup->io3v = false;
	}
	
	setup->drive = (uint8_t)((val & AG903_SSC_PIN_SETUP_DV_MSK) >> AG903_SSC_PIN_SETUP_DV_POS);
	
	if(AG903_SSC_PIN_SETUP_SR_MSK & val) {
		setup->sr_slow = true;
	}
	else {
		setup->sr_slow = false;
	}
	
	if(AG903_SSC_PIN_SETUP_SMT_MSK & val) {
		setup->schmitt = true;
	}
	else {
		setup->schmitt = false;
	}
	
	setup->lv_set = (uint8_t)((val & AG903_SSC_PIN_SETUP_LVST_MSK) >> AG903_SSC_PIN_SETUP_LVST_POS);
	
	return;
}

/**
 * @brief           SSCPIN_GPIO_ENABLEñWX^Write
 * @param           reg_no [in] WX^No.i0`3j
 * @param           val    [in] ݒl
 * @return          void
 * @description     SSCPIN_GPIO_ENABLEñWX^Write
*/
void AG903_SSCPrmSetPinGpioEnable(uint8_t reg_no, uint32_t val)
{
	ASSERT(AG903_SSC_GPIOEN_REG_NUM > reg_no);
	AG903_SSC->PIN_GPIO_ENABLE[reg_no] = val;
	return;
}

/**
 * @brief           SSCPIN_GPIO_ENABLEñWX^Read
 * @param           reg_no [in] WX^No.i0`3j
 * @param           val    [out] ݒl
 * @return          void
 * @description     SSCPIN_GPIO_ENABLEñWX^Read
*/
void AG903_SSCPrmGetPinGpioEnable(uint8_t reg_no, uint32_t* val)
{
	ASSERT(NULL != val);
	ASSERT(AG903_SSC_GPIOEN_REG_NUM > reg_no);

	(*val) = AG903_SSC->PIN_GPIO_ENABLE[reg_no];

	return;
}

/**
 * @brief           SSCPIN_GPIO_PIN_PUñWX^Write
 * @param           reg_no [in] WX^No.i0`3j
 * @param           val    [in] ݒl
 * @return          void
 * @description     SSCPIN_GPIO_PIN_PUñWX^Write
*/
void AG903_SSCPrmSetPinGpioPullUp(uint8_t reg_no, uint32_t val)
{
	ASSERT(AG903_SSC_GPIOPU_REG_NUM > reg_no);
	AG903_SSC->PIN_GPIO_PIN_PU[reg_no] = val;
	return;
}

/**
 * @brief           SSCPIN_GPIO_PIN_PUñWX^Read
 * @param           reg_no [in] WX^No.i0`3j
 * @param           val    [out] ݒl
 * @return          void
 * @description     SSCPIN_GPIO_PIN_PUñWX^Read
*/
void AG903_SSCPrmGetPinGpioPullUp(uint8_t reg_no, uint32_t* val)
{
	ASSERT(NULL != val);
	ASSERT(AG903_SSC_GPIOPU_REG_NUM > reg_no);

	(*val) = AG903_SSC->PIN_GPIO_PIN_PU[reg_no];

	return;
}

/**
 * @brief           SSCPIN_GPIO_PIN_PDñWX^Write
 * @param           reg_no [in] WX^No.i0`3j
 * @param           val    [in] ݒl
 * @return          void
 * @description     SSCPIN_GPIO_PIN_PDñWX^Write
*/
void AG903_SSCPrmSetPinGpioPullDown(uint8_t reg_no, uint32_t val)
{
	ASSERT(AG903_SSC_GPIOPD_REG_NUM > reg_no);
	AG903_SSC->PIN_GPIO_PIN_PD[reg_no] = val;
	return;
}

/**
 * @brief           SSCPIN_GPIO_PIN_PDñWX^Read
 * @param           reg_no [in] WX^No.i0`3j
 * @param           val    [out] ݒl
 * @return          void
 * @description     SSCPIN_GPIO_PIN_PDñWX^Read
*/
void AG903_SSCPrmGetPinGpioPullDown(uint8_t reg_no, uint32_t* val)
{
	ASSERT(NULL != val);
	ASSERT(AG903_SSC_GPIOPD_REG_NUM > reg_no);

	(*val) = AG903_SSC->PIN_GPIO_PIN_PD[reg_no];

	return;
}

/**
 * @brief           SSCCA5_RUN_SETUP̃WX^Write
 * @param           clk  [in] NbN (true=)
 * @param           rst  [in] Zbg (true=Zbg)
 * @param           vbit [in] Vrbgݒl (0 / 1)
 * @return          void
 * @description     SSCCA5_RUN_SETUP̃WX^Write
 * @note            vbit̓e͎CPURAZbgVrbgɔf܂B
*/
void AG903_SSCPrmSetCpuSetup(uint8_t clk, uint8_t rst, uint8_t vbit)
{
	uint32_t	val = 0;

	val |= (vbit << AG903_SSC_CA5_RUN_SETUP_VINITHI_POS) & AG903_SSC_CA5_RUN_SETUP_VINITHI_MSK;
	
	if(true == rst) {
		val |= (1 << AG903_SSC_CA5_RUN_SETUP_RESET_POS);
	}
	
	if(true == clk) {
		val |= (1 << AG903_SSC_CA5_RUN_SETUP_GCLK_POS);
	}
	
	AG903_SSC->CA5_RUN_SETUP = val;
	
	return;
}

/**
 * @brief           SSCCA5_RUN_SETUP̃WX^Read
 * @param           clk  [out] NbN (true=)
 * @param           rst  [out] Zbg (true=Zbg)
 * @param           vbit [out] Vrbgݒl (0 / 1)
 * @return          void
 * @description     SSCCA5_RUN_SETUP̃WX^Read
*/
void AG903_SSCPrmGetCpuSetup(uint8_t* clk, uint8_t* rst, uint8_t* vbit)
{
	uint32_t	val;

	ASSERT(NULL != clk);
	ASSERT(NULL != rst);
	ASSERT(NULL != vbit);

	val = AG903_SSC->CA5_RUN_SETUP;

	if(AG903_SSC_CA5_RUN_SETUP_GCLK_MSK & val) {
		(*clk) = true;
	}
	else {
		(*clk) = false;
	}
	
	if(AG903_SSC_CA5_RUN_SETUP_RESET_MSK & val) {
		(*rst) = true;
	}
	else {
		(*rst) = false;
	}
	
	(*vbit) = (uint8_t)((val & AG903_SSC_CA5_RUN_SETUP_VINITHI_MSK) >> AG903_SSC_CA5_RUN_SETUP_VINITHI_POS);

	return;
}

/**
 * @brief           SSCCA5_RUN_CTRL̃WX^Write
 * @param           none
 * @return          void
 * @description     SSCCA5_RUN_CTRL̃WX^Write
*/
void AG903_SSCPrmClrCpuEventOutStatus(void)
{
	AG903_SSC->CA5_RUN_CTRL = (1 << AG903_SSC_CA5_RUN_CTRL_CLREV_POS);
	return;
}

/**
 * @brief           SSCCA5_RUN_STATUS̃WX^Read
 * @param           status [out] CA5Xe[^X
 * @return          void
 * @description     SSCCA5_RUN_STATUS̃WX^Read
*/
void AG903_SSCPrmGetCpuStatus(AG903_SSCPrmCpuRunStatus* status)
{
	uint32_t	val;
	
	ASSERT(NULL != status);
	
	val = AG903_SSC->CA5_RUN_STATUS;
	
	if(AG903_SSC_CA5_RUN_STATUS_EVENTO_MSK & val) {
		status->sev_op = true;
	}
	else {
		status->sev_op = false;
	}

	if(AG903_SSC_CA5_RUN_STATUS_WFI_MSK & val) {
		status->wfi_stby = true;
	}
	else {
		status->wfi_stby = false;
	}

	if(AG903_SSC_CA5_RUN_STATUS_WFE_MSK & val) {
		status->wfe_stby = true;
	}
	else {
		status->wfe_stby = false;
	}

	if(AG903_SSC_CA5_RUN_STATUS_RESET_MSK & val) {
		status->reset = true;
	}
	else {
		status->reset = false;
	}

	if(AG903_SSC_CA5_RUN_STATUS_GCLK_MSK & val) {
		status->clk = true;
	}
	else {
		status->clk = false;
	}

	return;
}

/**
 * @brief           SSCCA5_ACP_SETUP̃WX^Write
 * @param           acp_addr [in] ݒl (3rbg([31:29])̂ݗL)
 * @return          void
 * @description     SSCCA5_ACP_SETUP̃WX^Write
*/
void AG903_SSCPrmSetCpuAcpSetup(uint32_t acp_addr)
{
	AG903_SSC->CA5_ACP_SETUP = acp_addr & AG903_SSC_CA5_ACP_SETUP_ACPADDR_MSK;
	return;
}

/**
 * @brief           SSCCA5_ACP_SETUP̃WX^Read
 * @param           acp_addr [out] ݒl
 * @return          void
 * @description     SSCCA5_ACP_SETUP̃WX^Read
*/
void AG903_SSCPrmGetCpuAcpSetup(uint32_t* acp_addr)
{
	ASSERT(NULL != acp_addr);
	(*acp_addr) = AG903_SSC->CA5_ACP_SETUP & AG903_SSC_CA5_ACP_SETUP_ACPADDR_MSK;
	return;
}

/**
 * @brief           SSCDMA_SELECT0,SSCDMA_SELECT1,SSCDMA_SELECT2,SSCDMA_SELECT3̃WX^Write
 * @param           reg_no [in] WX^No.i0`3j
 * @param           val    [in] ݒl
 * @return          void
 * @description     SSCDMA_SELECT0,SSCDMA_SELECT1,SSCDMA_SELECT2,SSCDMA_SELECT3̃WX^Write

*/
void AG903_SSCPrmSetDmaInterface(uint8_t reg_no, uint32_t val)
{
	ASSERT(AG903_SSC_DMASEL_REG_NUM > reg_no);

	switch(reg_no) {
		case 0:
			AG903_SSC->DMA_SELECT0 = val;
			break;
		case 1:
			AG903_SSC->DMA_SELECT1 = val;
			break;
		case 2:
			AG903_SSC->DMA_SELECT2 = val;
			break;
		case 3:
			AG903_SSC->DMA_SELECT3 = val;
			break;
		default:
			/* ASSERTŃ`FbNĂ̂łɂ͗ȂB */
			break;
	}
	
	return;
}

/**
 * @brief           SSCDMA_SELECT0,SSCDMA_SELECT1,SSCDMA_SELECT2,SSCDMA_SELECT3̃WX^Read
 * @param           reg_no [in] WX^No.i0`3j
 * @param           val    [out] ݒl
 * @return          void
 * @description     SSCDMA_SELECT0,SSCDMA_SELECT1,SSCDMA_SELECT2,SSCDMA_SELECT3̃WX^Read
*/
void AG903_SSCPrmGetDmaInterface(uint8_t reg_no, uint32_t* val)
{
	ASSERT(NULL != val);
	ASSERT(AG903_SSC_DMASEL_REG_NUM > reg_no);

	switch(reg_no) {
		case 0:
			(*val) = AG903_SSC->DMA_SELECT0;
			break;
		case 1:
			(*val) = AG903_SSC->DMA_SELECT1;
			break;
		case 2:
			(*val) = AG903_SSC->DMA_SELECT2;
			break;
		case 3:
			(*val) = AG903_SSC->DMA_SELECT3;
			break;
		default:
			/* ASSERTŃ`FbNĂ̂łɂ͗ȂB */
			break;
	}
	
	return;
}

/**
 * @brief           SSCDMA_SELECT_PBD̃WX^Write
 * @param           val    [in] ݒl
 * @return          void
 * @description     SSCDMA_SELECT_PBD̃WX^Write
*/
void AG903_SSCPrmSetPbdDmaInterface(uint8_t val)
{
	AG903_SSC->DMA_SELECT_PBD = (val << AG903_SSC_DMA_SELECT_PBD_DMASELECT_POS) & AG903_SSC_DMA_SELECT_PBD_DMASELECT_MSK;
	return;
}

/**
 * @brief           SSCDMA_SELECT_PBD̃WX^Read
 * @param           val    [in] ݒl
 * @return          void
 * @description     SSCDMA_SELECT_PBD̃WX^Read
*/
void AG903_SSCPrmGetPbdDmaInterface(uint8_t* val)
{
	ASSERT(NULL != val);
	(*val) = (uint8_t)((AG903_SSC->DMA_SELECT_PBD & AG903_SSC_DMA_SELECT_PBD_DMASELECT_MSK) >> AG903_SSC_DMA_SELECT_PBD_DMASELECT_POS);
	return;
}

/**
 * @brief           SSCDSP_SETUP̃WX^Write
 * @param           dir [in] o͐ݒ
 * @return          void
 * @description     SSCDSP_SETUP̃WX^Write
*/
void AG903_SSCPrmSetDspPinDir(AG903_SSCPrmDspSetup* dir)
{
	uint32_t	val = 0;
	
	ASSERT(NULL != dir);

	if(AG903_SSC_PINDIR_OUTPUT == dir->vsync0) {
		val |= (1 << AG903_SSC_DSP_SETUP_VSYNC0_DIR_POS);
	}
	if(AG903_SSC_PINDIR_OUTPUT == dir->field0) {
		val |= (1 << AG903_SSC_DSP_SETUP_FIELD0_DIR_POS);
	}
	if(AG903_SSC_PINDIR_OUTPUT == dir->dot0) {
		val |= (1 << AG903_SSC_DSP_SETUP_DOT0_DIR_POS);
	}
	if(AG903_SSC_PINDIR_OUTPUT == dir->vsync1) {
		val |= (1 << AG903_SSC_DSP_SETUP_VSYNC1_DIR_POS);
	}
	if(AG903_SSC_PINDIR_OUTPUT == dir->field1) {
		val |= (1 << AG903_SSC_DSP_SETUP_FIELD1_DIR_POS);
	}
	if(AG903_SSC_PINDIR_OUTPUT == dir->dot1) {
		val |= (1 << AG903_SSC_DSP_SETUP_DOT1_DIR_POS);
	}

	AG903_SSC->DSP_SETUP = val;
	
	return;
}

/**
 * @brief           SSCDSP_SETUP̃WX^Read
 * @param           dir [out] o͐ݒ
 * @return          void
 * @description     SSCDSP_SETUP̃WX^Read
*/
void AG903_SSCPrmGetDspPinDir(AG903_SSCPrmDspSetup* dir)
{
	uint32_t	val;

	ASSERT(NULL != dir);
	
	val = AG903_SSC->DSP_SETUP;
	
	if(AG903_SSC_DSP_SETUP_VSYNC0_DIR_MSK & val) {
		dir->vsync0 = AG903_SSC_PINDIR_OUTPUT;
	}
	else {
		dir->vsync0 = AG903_SSC_PINDIR_INPUT;
	}

	if(AG903_SSC_DSP_SETUP_FIELD0_DIR_MSK & val) {
		dir->field0 = AG903_SSC_PINDIR_OUTPUT;
	}
	else {
		dir->field0 = AG903_SSC_PINDIR_INPUT;
	}

	if(AG903_SSC_DSP_SETUP_DOT0_DIR_MSK & val) {
		dir->dot0 = AG903_SSC_PINDIR_OUTPUT;
	}
	else {
		dir->dot0 = AG903_SSC_PINDIR_INPUT;
	}

	if(AG903_SSC_DSP_SETUP_VSYNC1_DIR_MSK & val) {
		dir->vsync1 = AG903_SSC_PINDIR_OUTPUT;
	}
	else {
		dir->vsync1 = AG903_SSC_PINDIR_INPUT;
	}

	if(AG903_SSC_DSP_SETUP_FIELD1_DIR_MSK & val) {
		dir->field1 = AG903_SSC_PINDIR_OUTPUT;
	}
	else {
		dir->field1 = AG903_SSC_PINDIR_INPUT;
	}

	if(AG903_SSC_DSP_SETUP_DOT1_DIR_MSK & val) {
		dir->dot1 = AG903_SSC_PINDIR_OUTPUT;
	}
	else {
		dir->dot1 = AG903_SSC_PINDIR_INPUT;
	}

	return;
}

/**
 * @brief           SSCTICKn_SETUP̃WX^Write
 * @param           ch     [in] TICKJE^`l
 * @param           clksel [in] NbN\[X
 * @param           div    [in] 
 * @return          void
 * @description     SSCTICKn_SETUP̃WX^Write
*/
void AG903_SSCPrmSetTickSetup(uint8_t ch, uint8_t clksel, uint32_t div)
{
	uint32_t	val = 0;

	ASSERT(AG903_SSC_TICKCNT_CH_NUM > ch);

	switch(ch) {
		case 0:
			val |= (div    << AG903_SSC_TICK0_SETUP_DIVNUM_POS) & AG903_SSC_TICK0_SETUP_DIVNUM_MSK;
			val |= (clksel << AG903_SSC_TICK0_SETUP_CLKSEL_POS) & AG903_SSC_TICK0_SETUP_CLKSEL_MSK;
			AG903_SSC->TICK0_SETUP  = val;
			break;
		case 1:
			val |= (div    << AG903_SSC_TICK1_SETUP_DIVNUM_POS) & AG903_SSC_TICK1_SETUP_DIVNUM_MSK;
			val |= (clksel << AG903_SSC_TICK1_SETUP_CLKSEL_POS) & AG903_SSC_TICK1_SETUP_CLKSEL_MSK;
			AG903_SSC->TICK1_SETUP  = val;
			break;
		default:
			/* ASSERTŃ`FbNĂ̂łɂ͗ȂB */
			break;
	}

	return;
}

/**
 * @brief           SSCTICKn_SETUP̃WX^Read
 * @param           ch     [in] TICKJE^`l
 * @param           clksel [out] NbN\[X
 * @param           div    [out] 
 * @return          void
 * @description     SSCTICKn_SETUP̃WX^Read
*/
void AG903_SSCPrmGetTickSetup(uint8_t ch, uint8_t* clksel, uint32_t* div)
{
	uint32_t	val;

	ASSERT(NULL != clksel);
	ASSERT(NULL != div);
	ASSERT(AG903_SSC_TICKCNT_CH_NUM > ch);

	switch(ch) {
		case 0:
			val = AG903_SSC->TICK0_SETUP;
			(*clksel) = (uint8_t)((val & AG903_SSC_TICK0_SETUP_CLKSEL_MSK) >> AG903_SSC_TICK0_SETUP_CLKSEL_POS);
			(*div)    = (val & AG903_SSC_TICK0_SETUP_DIVNUM_MSK) >> AG903_SSC_TICK0_SETUP_DIVNUM_POS;
			break;
		case 1:
			val = AG903_SSC->TICK1_SETUP;
			(*clksel) = (uint8_t)((val & AG903_SSC_TICK1_SETUP_CLKSEL_MSK) >> AG903_SSC_TICK1_SETUP_CLKSEL_POS);
			(*div)    = (val & AG903_SSC_TICK1_SETUP_DIVNUM_MSK) >> AG903_SSC_TICK1_SETUP_DIVNUM_POS;
			break;
		default:
			/* ASSERTŃ`FbNĂ̂łɂ͗ȂB */
			break;
	}

	return;
}

/**
 * @brief           SSCTIM_SETUP̃WX^Write
 * @param           tim_dir [in] ^C}O[qo͐ݒ (rbg0`3)
 * @return          void
 * @description     SSCTIM_SETUP̃WX^Write
*/
void AG903_SSCPrmSetTimerPinDir(uint8_t tim_dir)
{
	uint32_t	val = 0;
	
	if(AG903_SSC_PINDIR_OUTPUT == ((tim_dir >> 0) & 0x01)) {	/* ext0 */
		val |= (1 << 0);
	}
	if(AG903_SSC_PINDIR_OUTPUT == ((tim_dir >> 1) & 0x01)) {	/* ext1 */
		val |= (1 << 1);
	}
	if(AG903_SSC_PINDIR_OUTPUT == ((tim_dir >> 2) & 0x01)) {	/* ext2 */
		val |= (1 << 2);
	}
	if(AG903_SSC_PINDIR_OUTPUT == ((tim_dir >> 3) & 0x01)) {	/* ext3 */
		val |= (1 << 3);
	}

	AG903_SSC->TIM_SETUP = (val << AG903_SSC_TIM_SETUP_TIM_DIR_POS) & AG903_SSC_TIM_SETUP_TIM_DIR_MSK;

	return;
}

/**
 * @brief           SSCTIM_SETUP̃WX^Read
 * @param           tim_dir [out] ^C}O[qo͐ݒ (rbg0`3)
 * @return          void
 * @description     SSCTIM_SETUP̃WX^Read
*/
void AG903_SSCPrmGetTimerPinDir(uint8_t* tim_dir)
{
	uint8_t	val;

	ASSERT(NULL != tim_dir);

	val = (uint8_t)((AG903_SSC->TIM_SETUP & AG903_SSC_TIM_SETUP_TIM_DIR_MSK) >> AG903_SSC_TIM_SETUP_TIM_DIR_POS);
	
	(*tim_dir) = 0;
	if((1<<0) & val) {
		(*tim_dir) |= (AG903_SSC_PINDIR_OUTPUT << 0);
	}
	if((1<<1) & val) {
		(*tim_dir) |= (AG903_SSC_PINDIR_OUTPUT << 1);
	}
	if((1<<2) & val) {
		(*tim_dir) |= (AG903_SSC_PINDIR_OUTPUT << 2);
	}
	if((1<<3) & val) {
		(*tim_dir) |= (AG903_SSC_PINDIR_OUTPUT << 3);
	}

	return;
}

/**
 * @brief           SSCSSP_SETUP̃WX^Write
 * @param           ch    [in] SSP`l
 * @param           param [in] ݒp[^
 * @return          void
 * @description     SSCSSP_SETUP̃WX^Write
*/
void AG903_SSCPrmSetSspSetup(uint8_t ch, AG903_SSCPrmSspSetup* param)
{
	uint32_t	val;

	ASSERT(NULL != param);
	ASSERT(AG903_SSP_CH_NUM > ch);

	val  = AG903_SSC->SSP_SETUP;
	val &= ~(0xFF << (ch*8));

	switch(ch) {
		case 0:
			val |= (param->mclk_div << AG903_SSC_SSP_SETUP_SSP0_MCLK_DIV_POS) & AG903_SSC_SSP_SETUP_SSP0_MCLK_DIV_MSK;
			if(AG903_SSC_PINDIR_OUTPUT == param->mclk_dir) {
				val |= (1 << AG903_SSC_SSP_SETUP_SSP0_MCLK_DIR_POS);
			}
			if(AG903_SSC_SSPMODE_HLFDPLX == param->mode) {
				val |= (1 << AG903_SSC_SSP_SETUP_SSP0_MODE_POS);
			}
			if(AG903_SSC_PINDIR_OUTPUT == param->dir) {
				val |= (1 << AG903_SSC_SSP_SETUP_SSP0_DIR_POS);
			}
			break;
		case 1:
			val |= (param->mclk_div << AG903_SSC_SSP_SETUP_SSP1_MCLK_DIV_POS) & AG903_SSC_SSP_SETUP_SSP1_MCLK_DIV_MSK;
			if(AG903_SSC_PINDIR_OUTPUT == param->mclk_dir) {
				val |= (1 << AG903_SSC_SSP_SETUP_SSP1_MCLK_DIR_POS);
			}
			if(AG903_SSC_SSPMODE_HLFDPLX == param->mode) {
				val |= (1 << AG903_SSC_SSP_SETUP_SSP1_MODE_POS);
			}
			if(AG903_SSC_PINDIR_OUTPUT == param->dir) {
				val |= (1 << AG903_SSC_SSP_SETUP_SSP1_DIR_POS);
			}
			break;
		case 2:
			val |= (param->mclk_div << AG903_SSC_SSP_SETUP_SSP2_MCLK_DIV_POS) & AG903_SSC_SSP_SETUP_SSP2_MCLK_DIV_MSK;
			if(AG903_SSC_PINDIR_OUTPUT == param->mclk_dir) {
				val |= (1 << AG903_SSC_SSP_SETUP_SSP2_MCLK_DIR_POS);
			}
			if(AG903_SSC_SSPMODE_HLFDPLX == param->mode) {
				val |= (1 << AG903_SSC_SSP_SETUP_SSP2_MODE_POS);
			}
			if(AG903_SSC_PINDIR_OUTPUT == param->dir) {
				val |= (1 << AG903_SSC_SSP_SETUP_SSP2_DIR_POS);
			}
			break;
		case 3:
			val |= (param->mclk_div << AG903_SSC_SSP_SETUP_SSP3_MCLK_DIV_POS) & AG903_SSC_SSP_SETUP_SSP3_MCLK_DIV_MSK;
			if(AG903_SSC_PINDIR_OUTPUT == param->mclk_dir) {
				val |= (1 << AG903_SSC_SSP_SETUP_SSP3_MCLK_DIR_POS);
			}
			if(AG903_SSC_SSPMODE_HLFDPLX == param->mode) {
				val |= (1 << AG903_SSC_SSP_SETUP_SSP3_MODE_POS);
			}
			if(AG903_SSC_PINDIR_OUTPUT == param->dir) {
				val |= (1 << AG903_SSC_SSP_SETUP_SSP3_DIR_POS);
			}
			break;
		default:
			/* ASSERTŃ`FbNĂ̂łɂ͗ȂB */
			break;
	}
	
	AG903_SSC->SSP_SETUP = val;

	return;
}

/**
 * @brief           SSCSSP_SETUP̃WX^Read
 * @param           ch    [in] SSP`l
 * @param           param [out] ݒp[^
 * @return          void
 * @description     SSCSSP_SETUP̃WX^Read
*/
void AG903_SSCPrmGetSspSetup(uint8_t ch, AG903_SSCPrmSspSetup* param)
{
	uint32_t	val;

	ASSERT(NULL != param);
	ASSERT(AG903_SSP_CH_NUM > ch);

	val = AG903_SSC->SSP_SETUP;

	switch(ch) {
		case 0:
			param->mclk_div = (uint8_t)((val & AG903_SSC_SSP_SETUP_SSP0_MCLK_DIV_MSK) >> AG903_SSC_SSP_SETUP_SSP0_MCLK_DIV_POS);
			if(AG903_SSC_SSP_SETUP_SSP0_MCLK_DIR_MSK & val) {
				param->mclk_dir = AG903_SSC_PINDIR_OUTPUT;
			}
			else {
				param->mclk_dir = AG903_SSC_PINDIR_INPUT;
			}
			if(AG903_SSC_SSP_SETUP_SSP0_MODE_MSK & val) {
				param->mode = AG903_SSC_SSPMODE_HLFDPLX;
			}
			else {
				param->mode = AG903_SSC_SSPMODE_FLLDPLX;
			}
			if(AG903_SSC_SSP_SETUP_SSP0_DIR_MSK & val) {
				param->dir = AG903_SSC_PINDIR_OUTPUT;
			}
			else {
				param->dir = AG903_SSC_PINDIR_INPUT;
			}
			break;
		case 1:
			param->mclk_div = (uint8_t)((val & AG903_SSC_SSP_SETUP_SSP1_MCLK_DIV_MSK) >> AG903_SSC_SSP_SETUP_SSP1_MCLK_DIV_POS);
			if(AG903_SSC_SSP_SETUP_SSP1_MCLK_DIR_MSK & val) {
				param->mclk_dir = AG903_SSC_PINDIR_OUTPUT;
			}
			else {
				param->mclk_dir = AG903_SSC_PINDIR_INPUT;
			}
			if(AG903_SSC_SSP_SETUP_SSP1_MODE_MSK & val) {
				param->mode = AG903_SSC_SSPMODE_HLFDPLX;
			}
			else {
				param->mode = AG903_SSC_SSPMODE_FLLDPLX;
			}
			if(AG903_SSC_SSP_SETUP_SSP1_DIR_MSK & val) {
				param->dir = AG903_SSC_PINDIR_OUTPUT;
			}
			else {
				param->dir = AG903_SSC_PINDIR_INPUT;
			}
			break;
		case 2:
			param->mclk_div = (uint8_t)((val & AG903_SSC_SSP_SETUP_SSP2_MCLK_DIV_MSK) >> AG903_SSC_SSP_SETUP_SSP2_MCLK_DIV_POS);
			if(AG903_SSC_SSP_SETUP_SSP2_MCLK_DIR_MSK & val) {
				param->mclk_dir = AG903_SSC_PINDIR_OUTPUT;
			}
			else {
				param->mclk_dir = AG903_SSC_PINDIR_INPUT;
			}
			if(AG903_SSC_SSP_SETUP_SSP2_MODE_MSK & val) {
				param->mode = AG903_SSC_SSPMODE_HLFDPLX;
			}
			else {
				param->mode = AG903_SSC_SSPMODE_FLLDPLX;
			}
			if(AG903_SSC_SSP_SETUP_SSP2_DIR_MSK & val) {
				param->dir = AG903_SSC_PINDIR_OUTPUT;
			}
			else {
				param->dir = AG903_SSC_PINDIR_INPUT;
			}
			break;
		case 3:
			param->mclk_div = (uint8_t)((val & AG903_SSC_SSP_SETUP_SSP3_MCLK_DIV_MSK) >> AG903_SSC_SSP_SETUP_SSP3_MCLK_DIV_POS);
			if(AG903_SSC_SSP_SETUP_SSP3_MCLK_DIR_MSK & val) {
				param->mclk_dir = AG903_SSC_PINDIR_OUTPUT;
			}
			else {
				param->mclk_dir = AG903_SSC_PINDIR_INPUT;
			}
			if(AG903_SSC_SSP_SETUP_SSP3_MODE_MSK & val) {
				param->mode = AG903_SSC_SSPMODE_HLFDPLX;
			}
			else {
				param->mode = AG903_SSC_SSPMODE_FLLDPLX;
			}
			if(AG903_SSC_SSP_SETUP_SSP3_DIR_MSK & val) {
				param->dir = AG903_SSC_PINDIR_OUTPUT;
			}
			else {
				param->dir = AG903_SSC_PINDIR_INPUT;
			}
			break;
		default:
			/* ASSERTŃ`FbNĂ̂łɂ͗ȂB */
			break;
	}

	return;
}

/**
 * @brief           SSCPBD_ADDRCHECK_START,SSCPBD_ADDRCHECK_END̃WX^Write
 * @param           sta_addr [in] ĎJnAhX (2rbg)
 * @param           end_addr [in] ĎIAhX (2rbg)
 * @return          void
 * @description     SSCPBD_ADDRCHECK_START,SSCPBD_ADDRCHECK_END̃WX^Write
*/
void AG903_SSCPrmSetPbdCheckAddr(uint32_t sta_addr, uint32_t end_addr)
{
	AG903_SSC->PBD_ADDRCHECK_START = sta_addr & AG903_SSC_PBD_ADDRCHECK_START_START_ADDR_MSK;	/* rbg[1:0]Write0 */
	AG903_SSC->PBD_ADDRCHECK_END   = end_addr & AG903_SSC_PBD_ADDRCHECK_END_END_ADDR_MSK;		/* rbg[1:0]Write1 */
	return;
}

/**
 * @brief           SSCPBD_ADDRCHECK_START,SSCPBD_ADDRCHECK_END̃WX^Read
 * @param           sta_addr [out] ĎJnAhX
 * @param           end_addr [out] ĎIAhX
 * @return          void
 * @description     SSCPBD_ADDRCHECK_START,SSCPBD_ADDRCHECK_END̃WX^Read
*/
void AG903_SSCPrmGetPbdCheckAddr(uint32_t* sta_addr, uint32_t* end_addr)
{
	ASSERT(NULL != sta_addr);
	ASSERT(NULL != end_addr);
	(*sta_addr) = AG903_SSC->PBD_ADDRCHECK_START;
	(*end_addr) = AG903_SSC->PBD_ADDRCHECK_END;
	return;
}

/**
 * @brief           SSCPBD_ADDRCHECK_ENABLẼWX^Write
 * @param           enable [in] Ďݒ (TRUE=Ď)
 * @return          void
 * @description     SSCPBD_ADDRCHECK_ENABLẼWX^Write
*/
void AG903_SSCPrmSetPbdAddrCheckEnable(uint8_t enable)
{
	if(true == enable) {
		AG903_SSC->PBD_ADDRCHECK_ENABLE = (1 << AG903_SSC_PBD_ADDRCHECK_ENABLE_ENABLE_POS);
	}
	else {
		AG903_SSC->PBD_ADDRCHECK_ENABLE = 0;
	}
	return;
}

/**
 * @brief           SSCPBD_ADDRCHECK_ENABLẼWX^Read
 * @param           enable [out] Ďݒ (TRUE=Ď)
 * @return          void
 * @description     SSCPBD_ADDRCHECK_ENABLẼWX^Read
*/
void AG903_SSCPrmGetPbdAddrCheckEnable(uint8_t* enable)
{
	uint32_t	val;
	
	ASSERT(NULL != enable);
	
	val = AG903_SSC->PBD_ADDRCHECK_ENABLE;
	if(AG903_SSC_PBD_ADDRCHECK_ENABLE_ENABLE_MSK & val) {
		(*enable) = true;
	}
	else {
		(*enable) = false;
	}

	return;
}

/**
 * @brief           SSCPBD_ADDRCHECK_STATUS̃WX^Read
 * @param           status [out] ĎXe[^X (TRUE=ΏۃAhXANZX)
 * @return          void
 * @description     SSCPBD_ADDRCHECK_STATUS̃WX^Read
*/
void AG903_SSCPrmGetPbdAddrCheckStatus(uint8_t* status)
{
	uint32_t	val;
	
	ASSERT(NULL != status);
	
	val = AG903_SSC->PBD_ADDRCHECK_STATUS;
	if(AG903_SSC_PBD_ADDRCHECK_STATUS_STATUS_MSK & val) {
		(*status) = true;
	}
	else {
		(*status) = false;
	}

	return;
}

/**
 * @brief           SSCBMU_STATUS̃WX^Read
 * @param           src_req [out] \[X݃NGXg [10:0]
 * @param           src_rdy [out] \[X݃fB [10:0]
 * @param           snk_req [out] VN݃NGXg [3:0]
 * @param           snk_rdy [out] VN݃fB [3:0]
 * @return          void
 * @description     SSCBMU_STATUS̃WX^Read
*/
void AG903_SSCPrmGetBmuStatus(uint16_t* src_req, uint16_t* src_rdy, uint8_t* snk_req, uint8_t* snk_rdy)
{
	uint32_t	val;
	
	ASSERT(NULL != src_req);
	ASSERT(NULL != src_rdy);
	ASSERT(NULL != snk_req);
	ASSERT(NULL != snk_rdy);

	val = AG903_SSC->BMU_STATUS;

	(*src_req) = (uint16_t)((val & AG903_SSC_BMU_STATUS_SRCREQ_MSK) >> AG903_SSC_BMU_STATUS_SRCREQ_POS);
	(*src_rdy) = (uint16_t)((val & AG903_SSC_BMU_STATUS_SRCRDY_MSK) >> AG903_SSC_BMU_STATUS_SRCRDY_POS);
	(*snk_req) = (uint8_t)((val & AG903_SSC_BMU_STATUS_SNKREQ_MSK) >> AG903_SSC_BMU_STATUS_SNKREQ_POS);
	(*snk_rdy) = (uint8_t)((val & AG903_SSC_BMU_STATUS_SNKRDY_MSK) >> AG903_SSC_BMU_STATUS_SNKRDY_POS);
	
	return;
}

/**
 * @brief           SSCBMU_CONTROL̃WX^Write
 * @param           set_src [in] \[X݃NGXgZbg [10:0]
 * @param           clr_src [in] \[X݃fB [10:0]
 * @param           set_snk [in] VN݃NGXg [3:0]
 * @param           clr_snk [in] VN݃fB [3:0]
 * @return          void
 * @description     SSCBMU_CONTROL̃WX^Write
 * @note            "1"ŃZbg̓NAB"0"͉eȂB
*/
void AG903_SSCPrmSetBmuControl(uint16_t set_src, uint16_t clr_src, uint8_t set_snk, uint8_t clr_snk)
{
	uint32_t	val = 0;
	
	val |= (set_src << AG903_SSC_BMU_CONTROL_SET_SRC_POS) & AG903_SSC_BMU_CONTROL_SET_SRC_MSK;
	val |= (clr_src << AG903_SSC_BMU_CONTROL_CLR_SRC_POS) & AG903_SSC_BMU_CONTROL_CLR_SRC_MSK;
	val |= (set_snk << AG903_SSC_BMU_CONTROL_SET_SNK_POS) & AG903_SSC_BMU_CONTROL_SET_SNK_MSK;
	val |= (clr_snk << AG903_SSC_BMU_CONTROL_CLR_SNK_POS) & AG903_SSC_BMU_CONTROL_CLR_SNK_MSK;
	
	AG903_SSC->BMU_CONTROL = val;
	
	return;
}

/**
 * @brief           SSCBMU_TRIG_STATUS̃WX^Read
 * @param           src_req [out] \[X݃NGXg gKXe[^X [10:0]
 * @param           src_rdy [out] \[X݃fB gKXe[^X [10:0]
 * @param           snk_req [out] VN݃NGXg gKXe[^X [3:0]
 * @param           snk_rdy [out] VN݃fB gKXe[^X [3:0]
 * @return          void
 * @description     SSCBMU_TRIG_STATUS̃WX^Read
*/
void AG903_SSCPrmGetBmuTriggerStatus(uint16_t* src_req, uint16_t* src_rdy, uint8_t* snk_req, uint8_t* snk_rdy)
{
	uint32_t	val;
	
	ASSERT(NULL != src_req);
	ASSERT(NULL != src_rdy);
	ASSERT(NULL != snk_req);
	ASSERT(NULL != snk_rdy);

	val = AG903_SSC->BMU_TRIG_STATUS;

	(*src_req) = (uint16_t)((val & AG903_SSC_BMU_TRIG_STATUS_SRCREQ_MSK) >> AG903_SSC_BMU_TRIG_STATUS_SRCREQ_POS);
	(*src_rdy) = (uint16_t)((val & AG903_SSC_BMU_TRIG_STATUS_SRCRDY_MSK) >> AG903_SSC_BMU_TRIG_STATUS_SRCRDY_POS);
	(*snk_req) = (uint8_t)((val & AG903_SSC_BMU_TRIG_STATUS_SNKREQ_MSK) >> AG903_SSC_BMU_TRIG_STATUS_SNKREQ_POS);
	(*snk_rdy) = (uint8_t)((val & AG903_SSC_BMU_TRIG_STATUS_SNKRDY_MSK) >> AG903_SSC_BMU_TRIG_STATUS_SNKRDY_POS);

	return;
}

/**
 * @brief           SSCBMU_TRIG_CLEAR̃WX^Write
 * @param           src_req [in] \[X݃NGXg gKXe[^X [10:0]
 * @param           src_rdy [in] \[X݃fB gKXe[^X [10:0]
 * @param           snk_req [in] VN݃NGXg gKXe[^X [3:0]
 * @param           snk_rdy [in] VN݃fB gKXe[^X [3:0]
 * @return          void
 * @description     SSCBMU_TRIG_CLEAR̃WX^Write
*/
void AG903_SSCPrmClrBmuTriggerStatus(uint16_t src_req, uint16_t src_rdy, uint8_t snk_req, uint8_t snk_rdy)
{
	uint32_t	val = 0;
	
	val |= (src_req << AG903_SSC_BMU_TRIG_CLEAR_SRCREQ_POS) & AG903_SSC_BMU_TRIG_CLEAR_SRCREQ_MSK;
	val |= (src_rdy << AG903_SSC_BMU_TRIG_CLEAR_SRCRDY_POS) & AG903_SSC_BMU_TRIG_CLEAR_SRCRDY_MSK;
	val |= (snk_req << AG903_SSC_BMU_TRIG_CLEAR_SNKREQ_POS) & AG903_SSC_BMU_TRIG_CLEAR_SNKREQ_MSK;
	val |= (snk_rdy << AG903_SSC_BMU_TRIG_CLEAR_SNKRDY_POS) & AG903_SSC_BMU_TRIG_CLEAR_SNKRDY_MSK;
	
	AG903_SSC->BMU_TRIG_CLEAR = val;

	return;
}

/**
 * @brief           SSCPORT_WAIT̃WX^Write
 * @param           wait [in] EGCgl
 * @return          void
 * @description     SSCPORT_WAIT̃WX^Write
 * @note            EGCgl~(1/200MHz)Ԃ̃EGCg}Bi200MHz쎞j
*/
void AG903_SSCPrmSetPortWait(uint16_t wait)
{
	AG903_SSC->PORT_WAIT = (uint32_t)(wait & AG903_SSC_PORT_WAIT_CYCLE_MSK);
	return;
}

/**
 * @brief           SSCPORT_WAIT̃WX^Read
 * @param           wait [out] EGCgl
 * @return          void
 * @description     SSCPORT_WAIT̃WX^Read
*/
void AG903_SSCPrmGetPortWait(uint16_t* wait)
{
	uint32_t	val;

	ASSERT(NULL != wait);
	
	val = AG903_SSC->PORT_WAIT;
	(*wait) = (uint16_t)((val & AG903_SSC_PORT_WAIT_CYCLE_MSK) >> AG903_SSC_PORT_WAIT_CYCLE_POS);

	return;
}

/**
 * @brief           SSCPBH_MODẼWX^Write
 * @param           wait_sel [in] EGCg[qI
 * @return          void
 * @description     SSCPBH_MODẼWX^Write
*/
void AG903_SSCPrmSetPbhMode(uint8_t wait_sel)
{
	uint32_t	val = 0;

	if(AG903_SSC_PBHWAIT_BLP0BLGT == wait_sel) {
		val |= (1 << AG903_SSC_PBH_MODE_WAIT_POS);
	}
	AG903_SSC->PBH_MODE = val;

	return;
}

/**
 * @brief           SSCPBH_MODẼWX^Read
 * @param           wait_sel [out] EGCg[qI
 * @return          void
 * @description     SSCPBH_MODẼWX^Read
*/
void AG903_SSCPrmGetPbhMode(uint8_t* wait_sel)
{
	uint32_t	val;
	
	ASSERT(NULL != wait_sel);
	
	val = AG903_SSC->PBH_MODE;
	if(AG903_SSC_PBH_MODE_WAIT_MSK & val) {
		(*wait_sel) = AG903_SSC_PBHWAIT_BLP0BLGT;
	}
	else {
		(*wait_sel) = AG903_SSC_PBHWAIT_HWAIT;
	}
	
	return;
}

/**
 * @brief           SSCTDMn_SETUP̃WX^Write
 * @param           ch [in] SSP`l
 * @param           pram [in] ݒp[^
 * @return          void
 * @description     SSCTDMn_SETUP̃WX^Write
 * @note            ݒp[^̊elΉ郌WX^rbgɂ̂܂Write܂B
*/
void AG903_SSCPrmSetTdmSetup(uint8_t ch, AG903_SSCPrmTdmParam* param)
{
	uint32_t	val = 0;

	ASSERT(AG903_SSP_CH_NUM > ch);
	ASSERT(NULL != param);

	switch(ch) {
		case 0:
			val |= (param->enable   << AG903_SSC_TDM0_SETUP_TDM_EN_POS)	& AG903_SSC_TDM0_SETUP_TDM_EN_MSK;
			val |= (param->sclk_pol << AG903_SSC_TDM0_SETUP_SCLK_POL_POS)	& AG903_SSC_TDM0_SETUP_SCLK_POL_MSK;
			val |= (param->fs_pol	<< AG903_SSC_TDM0_SETUP_FS_POL_POS)	& AG903_SSC_TDM0_SETUP_FS_POL_MSK;
			val |= (param->fs_txpw	<< AG903_SSC_TDM0_SETUP_FS_TXPW_POS)	& AG903_SSC_TDM0_SETUP_FS_TXPW_MSK;
			val |= (param->fs_dist	<< AG903_SSC_TDM0_SETUP_FS_DIST_POS)	& AG903_SSC_TDM0_SETUP_FS_DIST_MSK;
			val |= (param->dir_tx	<< AG903_SSC_TDM0_SETUP_DIR_TX_POS)	& AG903_SSC_TDM0_SETUP_DIR_TX_MSK;
			val |= (param->tdm_n	<< AG903_SSC_TDM0_SETUP_TDM_N_POS)	& AG903_SSC_TDM0_SETUP_TDM_N_MSK;
			val |= (param->sclk_n	<< AG903_SSC_TDM0_SETUP_SCLK_N_POS)	& AG903_SSC_TDM0_SETUP_SCLK_N_MSK;
			AG903_SSC->TDM0_SETUP = val;
			break;
		case 1:
			val |= (param->enable   << AG903_SSC_TDM1_SETUP_TDM_EN_POS)	& AG903_SSC_TDM1_SETUP_TDM_EN_MSK;
			val |= (param->sclk_pol << AG903_SSC_TDM1_SETUP_SCLK_POL_POS)	& AG903_SSC_TDM1_SETUP_SCLK_POL_MSK;
			val |= (param->fs_pol	<< AG903_SSC_TDM1_SETUP_FS_POL_POS)	& AG903_SSC_TDM1_SETUP_FS_POL_MSK;
			val |= (param->fs_txpw	<< AG903_SSC_TDM1_SETUP_FS_TXPW_POS)	& AG903_SSC_TDM1_SETUP_FS_TXPW_MSK;
			val |= (param->fs_dist	<< AG903_SSC_TDM1_SETUP_FS_DIST_POS)	& AG903_SSC_TDM1_SETUP_FS_DIST_MSK;
			val |= (param->dir_tx	<< AG903_SSC_TDM1_SETUP_DIR_TX_POS)	& AG903_SSC_TDM1_SETUP_DIR_TX_MSK;
			val |= (param->tdm_n	<< AG903_SSC_TDM1_SETUP_TDM_N_POS)	& AG903_SSC_TDM1_SETUP_TDM_N_MSK;
			val |= (param->sclk_n	<< AG903_SSC_TDM1_SETUP_SCLK_N_POS)	& AG903_SSC_TDM1_SETUP_SCLK_N_MSK;
			AG903_SSC->TDM1_SETUP = val;
			break;
		case 2:
			val |= (param->enable   << AG903_SSC_TDM2_SETUP_TDM_EN_POS)	& AG903_SSC_TDM2_SETUP_TDM_EN_MSK;
			val |= (param->sclk_pol << AG903_SSC_TDM2_SETUP_SCLK_POL_POS)	& AG903_SSC_TDM2_SETUP_SCLK_POL_MSK;
			val |= (param->fs_pol	<< AG903_SSC_TDM2_SETUP_FS_POL_POS)	& AG903_SSC_TDM2_SETUP_FS_POL_MSK;
			val |= (param->fs_txpw	<< AG903_SSC_TDM2_SETUP_FS_TXPW_POS)	& AG903_SSC_TDM2_SETUP_FS_TXPW_MSK;
			val |= (param->fs_dist	<< AG903_SSC_TDM2_SETUP_FS_DIST_POS)	& AG903_SSC_TDM2_SETUP_FS_DIST_MSK;
			val |= (param->dir_tx	<< AG903_SSC_TDM2_SETUP_DIR_TX_POS)	& AG903_SSC_TDM2_SETUP_DIR_TX_MSK;
			val |= (param->tdm_n	<< AG903_SSC_TDM2_SETUP_TDM_N_POS)	& AG903_SSC_TDM2_SETUP_TDM_N_MSK;
			val |= (param->sclk_n	<< AG903_SSC_TDM2_SETUP_SCLK_N_POS)	& AG903_SSC_TDM2_SETUP_SCLK_N_MSK;
			AG903_SSC->TDM2_SETUP = val;
			break;
		case 3:
			val |= (param->enable   << AG903_SSC_TDM3_SETUP_TDM_EN_POS)	& AG903_SSC_TDM3_SETUP_TDM_EN_MSK;
			val |= (param->sclk_pol << AG903_SSC_TDM3_SETUP_SCLK_POL_POS)	& AG903_SSC_TDM3_SETUP_SCLK_POL_MSK;
			val |= (param->fs_pol	<< AG903_SSC_TDM3_SETUP_FS_POL_POS)	& AG903_SSC_TDM3_SETUP_FS_POL_MSK;
			val |= (param->fs_txpw	<< AG903_SSC_TDM3_SETUP_FS_TXPW_POS)	& AG903_SSC_TDM3_SETUP_FS_TXPW_MSK;
			val |= (param->fs_dist	<< AG903_SSC_TDM3_SETUP_FS_DIST_POS)	& AG903_SSC_TDM3_SETUP_FS_DIST_MSK;
			val |= (param->dir_tx	<< AG903_SSC_TDM3_SETUP_DIR_TX_POS)	& AG903_SSC_TDM3_SETUP_DIR_TX_MSK;
			val |= (param->tdm_n	<< AG903_SSC_TDM3_SETUP_TDM_N_POS)	& AG903_SSC_TDM3_SETUP_TDM_N_MSK;
			val |= (param->sclk_n	<< AG903_SSC_TDM3_SETUP_SCLK_N_POS)	& AG903_SSC_TDM3_SETUP_SCLK_N_MSK;
			AG903_SSC->TDM3_SETUP = val;
			break;
		default:
			/* ASSERTŃ`FbNĂ̂łɂ͗ȂB */
			break;
	}

	return;
}

/**
 * @brief           SSCTDMn_SETUP̃WX^Read
 * @param           ch [in] SSP`l
 * @param           pram [out] ݒp[^
 * @param           stat [out] 
 * @return          void
 * @description     SSCTDMn_SETUP̃WX^Write
 * @note            ݒp[^̊el͑Ή郌WX^rbgReadl̂̂łB
*/
void AG903_SSCPrmGetTdmSetup(uint8_t ch, AG903_SSCPrmTdmParam* param, AG903_SSCPrmTdmStatus* stat)
{
	uint32_t	val;

	ASSERT(AG903_SSP_CH_NUM > ch);
	ASSERT(NULL != param);
	ASSERT(NULL != stat);

	switch(ch) {
		case 0:
			val = AG903_SSC->TDM0_SETUP;
			param->enable 	= (uint8_t)((val & AG903_SSC_TDM0_SETUP_TDM_EN_MSK)   >> AG903_SSC_TDM0_SETUP_TDM_EN_POS);
			param->sclk_pol	= (uint8_t)((val & AG903_SSC_TDM0_SETUP_SCLK_POL_MSK) >> AG903_SSC_TDM0_SETUP_SCLK_POL_POS);
			param->fs_pol	= (uint8_t)((val & AG903_SSC_TDM0_SETUP_FS_POL_MSK)   >> AG903_SSC_TDM0_SETUP_FS_POL_POS);
			param->fs_txpw	= (uint8_t)((val & AG903_SSC_TDM0_SETUP_FS_TXPW_MSK)  >> AG903_SSC_TDM0_SETUP_FS_TXPW_POS);
			param->fs_dist	= (uint8_t)((val & AG903_SSC_TDM0_SETUP_FS_DIST_MSK)  >> AG903_SSC_TDM0_SETUP_FS_DIST_POS);
			param->dir_tx	= (uint8_t)((val & AG903_SSC_TDM0_SETUP_DIR_TX_MSK)   >> AG903_SSC_TDM0_SETUP_DIR_TX_POS);
			param->tdm_n	= (uint8_t)((val & AG903_SSC_TDM0_SETUP_TDM_N_MSK)    >> AG903_SSC_TDM0_SETUP_TDM_N_POS);
			param->sclk_n	= (uint8_t)((val & AG903_SSC_TDM0_SETUP_SCLK_N_MSK)   >> AG903_SSC_TDM0_SETUP_SCLK_N_POS);
			stat->rx_err	= (uint8_t)((val & AG903_SSC_TDM0_SETUP_FS_RXERR_MSK) >> AG903_SSC_TDM0_SETUP_FS_RXERR_POS);
			stat->tx_stat	= (uint8_t)((val & AG903_SSC_TDM0_SETUP_ST_TX_MSK)    >> AG903_SSC_TDM0_SETUP_ST_TX_POS);
			stat->rx_stat	= (uint8_t)((val & AG903_SSC_TDM0_SETUP_ST_RX_MSK)    >> AG903_SSC_TDM0_SETUP_ST_RX_POS);
			break;
		case 1:
			val = AG903_SSC->TDM1_SETUP;
			param->enable 	= (uint8_t)((val & AG903_SSC_TDM1_SETUP_TDM_EN_MSK)   >> AG903_SSC_TDM1_SETUP_TDM_EN_POS);
			param->sclk_pol	= (uint8_t)((val & AG903_SSC_TDM1_SETUP_SCLK_POL_MSK) >> AG903_SSC_TDM1_SETUP_SCLK_POL_POS);
			param->fs_pol	= (uint8_t)((val & AG903_SSC_TDM1_SETUP_FS_POL_MSK)   >> AG903_SSC_TDM1_SETUP_FS_POL_POS);
			param->fs_txpw	= (uint8_t)((val & AG903_SSC_TDM1_SETUP_FS_TXPW_MSK)  >> AG903_SSC_TDM1_SETUP_FS_TXPW_POS);
			param->fs_dist	= (uint8_t)((val & AG903_SSC_TDM1_SETUP_FS_DIST_MSK)  >> AG903_SSC_TDM1_SETUP_FS_DIST_POS);
			param->dir_tx	= (uint8_t)((val & AG903_SSC_TDM1_SETUP_DIR_TX_MSK)   >> AG903_SSC_TDM1_SETUP_DIR_TX_POS);
			param->tdm_n	= (uint8_t)((val & AG903_SSC_TDM1_SETUP_TDM_N_MSK)    >> AG903_SSC_TDM1_SETUP_TDM_N_POS);
			param->sclk_n	= (uint8_t)((val & AG903_SSC_TDM1_SETUP_SCLK_N_MSK)   >> AG903_SSC_TDM1_SETUP_SCLK_N_POS);
			stat->rx_err	= (uint8_t)((val & AG903_SSC_TDM1_SETUP_FS_RXERR_MSK) >> AG903_SSC_TDM1_SETUP_FS_RXERR_POS);
			stat->tx_stat	= (uint8_t)((val & AG903_SSC_TDM1_SETUP_ST_TX_MSK)    >> AG903_SSC_TDM1_SETUP_ST_TX_POS);
			stat->rx_stat	= (uint8_t)((val & AG903_SSC_TDM1_SETUP_ST_RX_MSK)    >> AG903_SSC_TDM1_SETUP_ST_RX_POS);
			break;
		case 2:
			val = AG903_SSC->TDM2_SETUP;
			param->enable 	= (uint8_t)((val & AG903_SSC_TDM2_SETUP_TDM_EN_MSK)   >> AG903_SSC_TDM2_SETUP_TDM_EN_POS);
			param->sclk_pol	= (uint8_t)((val & AG903_SSC_TDM2_SETUP_SCLK_POL_MSK) >> AG903_SSC_TDM2_SETUP_SCLK_POL_POS);
			param->fs_pol	= (uint8_t)((val & AG903_SSC_TDM2_SETUP_FS_POL_MSK)   >> AG903_SSC_TDM2_SETUP_FS_POL_POS);
			param->fs_txpw	= (uint8_t)((val & AG903_SSC_TDM2_SETUP_FS_TXPW_MSK)  >> AG903_SSC_TDM2_SETUP_FS_TXPW_POS);
			param->fs_dist	= (uint8_t)((val & AG903_SSC_TDM2_SETUP_FS_DIST_MSK)  >> AG903_SSC_TDM2_SETUP_FS_DIST_POS);
			param->dir_tx	= (uint8_t)((val & AG903_SSC_TDM2_SETUP_DIR_TX_MSK)   >> AG903_SSC_TDM2_SETUP_DIR_TX_POS);
			param->tdm_n	= (uint8_t)((val & AG903_SSC_TDM2_SETUP_TDM_N_MSK)    >> AG903_SSC_TDM2_SETUP_TDM_N_POS);
			param->sclk_n	= (uint8_t)((val & AG903_SSC_TDM2_SETUP_SCLK_N_MSK)   >> AG903_SSC_TDM2_SETUP_SCLK_N_POS);
			stat->rx_err	= (uint8_t)((val & AG903_SSC_TDM2_SETUP_FS_RXERR_MSK) >> AG903_SSC_TDM2_SETUP_FS_RXERR_POS);
			stat->tx_stat	= (uint8_t)((val & AG903_SSC_TDM2_SETUP_ST_TX_MSK)    >> AG903_SSC_TDM2_SETUP_ST_TX_POS);
			stat->rx_stat	= (uint8_t)((val & AG903_SSC_TDM2_SETUP_ST_RX_MSK)    >> AG903_SSC_TDM2_SETUP_ST_RX_POS);
			break;
		case 3:
			val = AG903_SSC->TDM3_SETUP;
			param->enable 	= (uint8_t)((val & AG903_SSC_TDM3_SETUP_TDM_EN_MSK)   >> AG903_SSC_TDM3_SETUP_TDM_EN_POS);
			param->sclk_pol	= (uint8_t)((val & AG903_SSC_TDM3_SETUP_SCLK_POL_MSK) >> AG903_SSC_TDM3_SETUP_SCLK_POL_POS);
			param->fs_pol	= (uint8_t)((val & AG903_SSC_TDM3_SETUP_FS_POL_MSK)   >> AG903_SSC_TDM3_SETUP_FS_POL_POS);
			param->fs_txpw	= (uint8_t)((val & AG903_SSC_TDM3_SETUP_FS_TXPW_MSK)  >> AG903_SSC_TDM3_SETUP_FS_TXPW_POS);
			param->fs_dist	= (uint8_t)((val & AG903_SSC_TDM3_SETUP_FS_DIST_MSK)  >> AG903_SSC_TDM3_SETUP_FS_DIST_POS);
			param->dir_tx	= (uint8_t)((val & AG903_SSC_TDM3_SETUP_DIR_TX_MSK)   >> AG903_SSC_TDM3_SETUP_DIR_TX_POS);
			param->tdm_n	= (uint8_t)((val & AG903_SSC_TDM3_SETUP_TDM_N_MSK)    >> AG903_SSC_TDM3_SETUP_TDM_N_POS);
			param->sclk_n	= (uint8_t)((val & AG903_SSC_TDM3_SETUP_SCLK_N_MSK)   >> AG903_SSC_TDM3_SETUP_SCLK_N_POS);
			stat->rx_err	= (uint8_t)((val & AG903_SSC_TDM3_SETUP_FS_RXERR_MSK) >> AG903_SSC_TDM3_SETUP_FS_RXERR_POS);
			stat->tx_stat	= (uint8_t)((val & AG903_SSC_TDM3_SETUP_ST_TX_MSK)    >> AG903_SSC_TDM3_SETUP_ST_TX_POS);
			stat->rx_stat	= (uint8_t)((val & AG903_SSC_TDM3_SETUP_ST_RX_MSK)    >> AG903_SSC_TDM3_SETUP_ST_RX_POS);
			break;
		default:
			/* ASSERTŃ`FbNĂ̂łɂ͗ȂB */
			break;
	}
	
	return;
}

/**
 * @brief           SSCUSB_POWER_SETUP̃WX^Write
 * @param           mode      [in] 샂[h
 * @param           int_ena   [in] dی@\ (true=Cl[u)
 * @param           prtct_ena [in] dی@\ (true=Cl[u)
 * @return          void
 * @description     SSCUSB_POWER_SETUP̃WX^Write
*/
void AG903_SSCPrmSetUsbPowerSetup(uint8_t mode, uint8_t int_ena, uint8_t prtct_ena)
{
	uint32_t	val = 0;
	
	if(AG903_SSC_USBMODE_DEVICE == mode) {
		val |= (1 << AG903_SSC_USB_POWER_SETUP_PWR_MODE_POS);
	}
	if(true == int_ena) {
		val |= (1 << AG903_SSC_USB_POWER_SETUP_PWR_INT_EN_POS);
	}
	if(true == prtct_ena) {
		val |= (1 << AG903_SSC_USB_POWER_SETUP_PWR_PROT_EN_POS);
	}
	AG903_SSC->USB_POWER_SETUP = val;
	
	return;
	
}

/**
 * @brief           SSCUSB_POWER_SETUP̃WX^Read
 * @param           mode      [out] 샂[h
 * @param           int_ena   [out] dی@\ (true=Cl[u)
 * @param           prtct_ena [out] dی@\ (true=Cl[u)
 * @return          void
 * @description     SSCUSB_POWER_SETUP̃WX^Read
*/
void AG903_SSCPrmGetUsbPowerSetup(uint8_t* mode, uint8_t* int_ena, uint8_t* prtct_ena)
{
	uint32_t	val;

	ASSERT(NULL != mode);
	ASSERT(NULL != int_ena);
	ASSERT(NULL != prtct_ena);

	val = AG903_SSC->USB_POWER_SETUP;
	if(AG903_SSC_USB_POWER_SETUP_PWR_MODE_MSK & val) {
		(*mode) = AG903_SSC_USBMODE_DEVICE;
	}
	else {
		(*mode) = AG903_SSC_USBMODE_HOST;
	}
	if(AG903_SSC_USB_POWER_SETUP_PWR_INT_EN_MSK & val) {
		(*int_ena) = true;
	}
	else {
		(*int_ena) = false;
	}
	if(AG903_SSC_USB_POWER_SETUP_PWR_PROT_EN_MSK & val) {
		(*prtct_ena) = true;
	}
	else {
		(*prtct_ena) = false;
	}

	return;
}

/**
 * @brief           SSCUSB_POWER_STATUS̃WX^Read
 * @param           vbus_lv    [out] VBUS[qx
 * @param           prtct_lv   [out] dیĎ[qx
 * @param           prtct_stat [out] dی
 * @return          void
 * @description     SSCUSB_POWER_STATUS̃WX^Read
*/
void AG903_SSCPrmGetUsbPowerStatus(uint8_t* vbus_lv, uint8_t* prtct_lv, uint8_t* prtct_stat)
{
	uint32_t	val;
	
	ASSERT(NULL != vbus_lv);
	ASSERT(NULL != prtct_lv);
	ASSERT(NULL != prtct_stat);
	
	val = AG903_SSC->USB_POWER_STATUS;
	(*vbus_lv)    = (uint8_t)((val & AG903_SSC_USB_POWER_STATUS_VBUS_PIN_MSK)       >> AG903_SSC_USB_POWER_STATUS_VBUS_PIN_POS);
	(*prtct_lv)   = (uint8_t)((val & AG903_SSC_USB_POWER_STATUS_PWR_PROT_PIN_MSK)   >> AG903_SSC_USB_POWER_STATUS_PWR_PROT_PIN_POS);
	(*prtct_stat) = (uint8_t)((val & AG903_SSC_USB_POWER_STATUS_PWR_PROT_STATE_MSK) >> AG903_SSC_USB_POWER_STATUS_PWR_PROT_STATE_POS);
	
	return;
}

/**
 * @brief           SSCUSB_POWER_CLEAR̃WX^Write
 * @param           none
 * @return          void
 * @description     SSCUSB_POWER_CLEAR̃WX^Write
*/
void AG903_SSCPrmClrUsbPowerStatus(void)
{
	AG903_SSC->USB_POWER_CLEAR = (1 << AG903_SSC_USB_POWER_CLEAR_PWR_PROT_CLR_POS);
	return;
}

/**
 * @brief           SSCSD_POWER_SETUP̃WX^Write
 * @param           int_ena   [in] dی@\ (true=Cl[u)
 * @param           prtct_ena [in] dی@\ (true=Cl[u)
 * @return          void
 * @description     SSCSD_POWER_SETUP̃WX^Write
*/
void AG903_SSCPrmSetSdPowerSetup(uint8_t int_ena, uint8_t prtct_ena)
{
	uint32_t	val = 0;
	
	if(true == int_ena) {
		val |= (1 << AG903_SSC_SD_POWER_SETUP_PWR_INT_EN_POS);
	}
	if(true == prtct_ena) {
		val |= (1 << AG903_SSC_SD_POWER_SETUP_PWR_PROT_EN_POS);
	}
	AG903_SSC->SD_POWER_SETUP = val;

	return;
}

/**
 * @brief           SSCSD_POWER_SETUP̃WX^Read
 * @param           int_ena   [out] dی@\ (true=Cl[u)
 * @param           prtct_ena [out] dی@\ (true=Cl[u)
 * @return          void
 * @description     SSCSD_POWER_SETUP̃WX^Read
*/
void AG903_SSCPrmGetSdPowerSetup(uint8_t* int_ena, uint8_t* prtct_ena)
{
	uint32_t	val;
	
	ASSERT(NULL != int_ena);
	ASSERT(NULL != prtct_ena);
	
	val = AG903_SSC->SD_POWER_SETUP;
	if(AG903_SSC_SD_POWER_SETUP_PWR_INT_EN_MSK & val) {
		(*int_ena) = true;
	}
	else {
		(*int_ena) = false;
	}
	if(AG903_SSC_SD_POWER_SETUP_PWR_PROT_EN_MSK & val) {
		(*prtct_ena) = true;
	}
	else {
		(*prtct_ena) = false;
	}
	
	return;
}

/**
 * @brief           SSCSD_POWER_STATUS̃WX^Read
 * @param           prtct_lv   [out] dیĎ[qx
 * @param           prtct_stat [out] dی
 * @return          void
 * @description     SSCSD_POWER_STATUS̃WX^Read
*/
void AG903_SSCPrmGetSdPowerStatus(uint8_t* prtct_lv, uint8_t* prtct_stat)
{
	uint32_t	val = 0;
	
	ASSERT(NULL != prtct_lv);
	ASSERT(NULL != prtct_stat);
	
	val = AG903_SSC->SD_POWER_STATUS;
	(*prtct_lv)   = (uint8_t)((val & AG903_SSC_SD_POWER_STATUS_PWR_PROT_PIN_MSK)   >> AG903_SSC_SD_POWER_STATUS_PWR_PROT_PIN_POS);
	(*prtct_stat) = (uint8_t)((val & AG903_SSC_SD_POWER_STATUS_PWR_PROT_STATE_MSK) >> AG903_SSC_SD_POWER_STATUS_PWR_PROT_STATE_POS);
	
	return;
}

/**
 * @brief           SSCSD_POWER_CLEAR̃WX^Write
 * @param           none
 * @return          void
 * @description     SSCSD_POWER_CLEAR̃WX^Write
*/
void AG903_SSCPrmClrSdPowerStatus(void)
{
	AG903_SSC->SD_POWER_CLEAR = (1 << AG903_SSC_SD_POWER_CLEAR_PWR_PROT_CLR_POS);
	return;
}

/**
 * @brief           SSCVIDEOADC_SETUP̃WX^Write
 * @param           pram [in] ݒp[^
 * @return          void
 * @description     SSCVIDEOADC_SETUP̃WX^Write
 * @note            ݒp[^̊elΉ郌WX^rbgɂ̂܂Write܂B
*/
void AG903_SSCPrmSetVideoAdcSetup(AG903_SSCPrmVideoAdcParam* param)
{
	uint32_t	val = 0;

	ASSERT(NULL != param);

	val |= (param->acq_clr  << AG903_SSC_VIDEOADC_SETUP_ACQCLR_POS) & AG903_SSC_VIDEOADC_SETUP_ACQCLR_MSK;
	val |= (param->acq_en   << AG903_SSC_VIDEOADC_SETUP_ACQEN_POS)  & AG903_SSC_VIDEOADC_SETUP_ACQEN_MSK;
	val |= (param->dat_en   << AG903_SSC_VIDEOADC_SETUP_DATEN_POS)  & AG903_SSC_VIDEOADC_SETUP_DATEN_MSK;
	val |= (param->clmp_tmg << AG903_SSC_VIDEOADC_SETUP_CTHO_POS)   & AG903_SSC_VIDEOADC_SETUP_CTHO_MSK;
	val |= (param->clmp_lv  << AG903_SSC_VIDEOADC_SETUP_CLHO_POS)   & AG903_SSC_VIDEOADC_SETUP_CLHO_MSK;
	val |= (param->pga_ctl  << AG903_SSC_VIDEOADC_SETUP_GHO_POS)    & AG903_SSC_VIDEOADC_SETUP_GHO_MSK;
	AG903_SSC->VIDEOADC_SETUP = val;

	return;
}

/**
 * @brief           SSCVIDEOADC_SETUP̃WX^Read
 * @param           pram [out] ݒp[^
 * @return          void
 * @description     SSCVIDEOADC_SETUP̃WX^Read
 * @note            ݒp[^̊el͑Ή郌WX^rbgReadl̂̂łB
*/
void AG903_SSCPrmGetVideoAdcSetup(AG903_SSCPrmVideoAdcParam* param)
{
	uint32_t	val;

	ASSERT(NULL != param);

	val = AG903_SSC->VIDEOADC_SETUP;
	param->acq_clr  = (uint8_t)((val & AG903_SSC_VIDEOADC_SETUP_ACQCLR_MSK) >> AG903_SSC_VIDEOADC_SETUP_ACQCLR_POS);
	param->acq_en   = (uint8_t)((val & AG903_SSC_VIDEOADC_SETUP_ACQEN_MSK)  >> AG903_SSC_VIDEOADC_SETUP_ACQEN_POS);
	param->dat_en   = (uint8_t)((val & AG903_SSC_VIDEOADC_SETUP_DATEN_MSK)  >> AG903_SSC_VIDEOADC_SETUP_DATEN_POS);
	param->clmp_tmg = (uint8_t)((val & AG903_SSC_VIDEOADC_SETUP_CTHO_MSK)   >> AG903_SSC_VIDEOADC_SETUP_CTHO_POS);
	param->clmp_lv  = (uint8_t)((val & AG903_SSC_VIDEOADC_SETUP_CLHO_MSK)   >> AG903_SSC_VIDEOADC_SETUP_CLHO_POS);
	param->pga_ctl  = (uint8_t)((val & AG903_SSC_VIDEOADC_SETUP_GHO_MSK)    >> AG903_SSC_VIDEOADC_SETUP_GHO_POS);

	return;
}

/**
 * @brief           SSCCVBSDEC_ADDR̃WX^Write,SSCCVBSDEC_DATÃWX^Write
 * @param           ch   [in] R|WbgrfIfR[_CH (0`3)
 * @param           reg  [in] WX^AhX (0x000`0xFFF)
 * @param           data [in] ݃f[^
 * @return          void
 * @description     SSCCVBSDEC_ADDR̃WX^Write,SSCCVBSDEC_DATÃWX^Write
*/
void AG903_SSCPrmWriteCmpstVideoDec(uint8_t ch, uint16_t reg, uint16_t data)
{
	uint32_t	val = 0;

	ASSERT(AG903_SSC_CVBSDEC_CH_NUM > ch);

	val |= (ch  << AG903_SSC_CVBSDEC_ADDR_CH_POS)   & AG903_SSC_CVBSDEC_ADDR_CH_MSK;
	val |= (reg << AG903_SSC_CVBSDEC_ADDR_ADDR_POS) & AG903_SSC_CVBSDEC_ADDR_ADDR_MSK;
	AG903_SSC->CVBSDEC_ADDR = val;	/* WX^I */

	AG903_SSC->CVBSDEC_DATA = (uint32_t)(data & AG903_SSC_CVBSDEC_DATA_DATA_MSK);

	return;
}

/**
 * @brief           SSCCVBSDEC_ADDR̃WX^Write,SSCCVBSDEC_DATÃWX^Read
 * @param           ch   [in] R|WbgrfIfR[_CH (0`3)
 * @param           reg  [in] WX^AhX (0x000`0xFFF)
 * @param           data [out] Ǎ݃f[^
 * @return          void
 * @description     SSCCVBSDEC_ADDR̃WX^Write,SSCCVBSDEC_DATÃWX^Read
*/
void AG903_SSCPrmReadCmpstVideoDec(uint8_t ch, uint16_t reg, uint16_t* data)
{
	uint32_t	val = 0;

	ASSERT(AG903_SSC_CVBSDEC_CH_NUM > ch);
	ASSERT(NULL != data);

	val |= (ch  << AG903_SSC_CVBSDEC_ADDR_CH_POS)   & AG903_SSC_CVBSDEC_ADDR_CH_MSK;
	val |= (reg << AG903_SSC_CVBSDEC_ADDR_ADDR_POS) & AG903_SSC_CVBSDEC_ADDR_ADDR_MSK;
	AG903_SSC->CVBSDEC_ADDR = val;	/* WX^I */

	val = AG903_SSC->CVBSDEC_DATA;
	(*data) = (uint16_t)((val & AG903_SSC_CVBSDEC_DATA_DATA_MSK) >> AG903_SSC_CVBSDEC_DATA_DATA_POS);

	return;
}

/**
 * @brief           SSCVIDEOADC_MAXMINñWX^Read
 * @param           ch  [in] R|WbgrfIfR[_CH (0`3)
 * @param           max [out] ől
 * @param           min [out] ŏl
 * @return          void
 * @description     SSCVIDEOADC_MAXMINñWX^Read
*/
void AG903_SSCPrmGetVideoAdcMaxMin(uint8_t ch, uint16_t* max, uint16_t* min)
{
	uint32_t	val;
	
	ASSERT(AG903_SSC_CVBSDEC_CH_NUM > ch);
	ASSERT(NULL != max);
	ASSERT(NULL != min);
	
	val = AG903_SSC->VIDEOADC_MAXMIN[ch];
	(*max) = (uint16_t)((val & AG903_SSC_VIDEOADC_MAXMIN_MAX_MSK) >> AG903_SSC_VIDEOADC_MAXMIN_MAX_POS);
	(*min) = (uint16_t)((val & AG903_SSC_VIDEOADC_MAXMIN_MIN_MSK) >> AG903_SSC_VIDEOADC_MAXMIN_MIN_POS);
	
	return;
}

/**
 * @brief           SSCSGI_STATUS̃WX^Read
 * @param           stat [out] \tg݃Xe[^X
 * @return          void
 * @description     SSCSGI_STATUS̃WX^Read
*/
void AG903_SSCPrmGetSgiStatus(uint32_t* stat)
{
	ASSERT(NULL != stat);
	(*stat) = AG903_SSC->SGI_STATUS;
	return;
}

/**
 * @brief           SSCSGI_SET̃WX^Write
 * @param           set [in] \tgݗv [31:0]
 * @return          void
 * @description     SSCSGI_SET̃WX^Write
 * @note            "1"ŃZbgB"0"͉eȂB
*/
void AG903_SSCPrmSetSgi(uint32_t set)
{
	AG903_SSC->SGI_SET = set;
	return;
}

/**
 * @brief           SSCSGI_CLEAR̃WX^Write
 * @param           clr [in] \tgݗv [31:0]
 * @return          void
 * @description     SSCSGI_CLEAR̃WX^Write
 * @note            "1"ŃNAB"0"͉eȂB
*/
void AG903_SSCPrmClrSgi(uint32_t clr)
{
	AG903_SSC->SGI_CLEAR = clr;
	return;
}

/**
 * @brief           SSCCA5_JUMPADDR̃WX^Write
 * @param           addr [in] WvAhX
 * @return          void
 * @description     SSCCA5_JUMPADDR̃WX^Write
*/
void AG903_SSCPrmSetCpuJumpAddr(uint32_t addr)
{
	AG903_SSC->CA5_JUMPADDR = addr;
	return;
}

/**
 * @brief           SSCCA5_JUMPADDR̃WX^Read
 * @param           addr [out] WvAhX
 * @return          void
 * @description     SSCCA5_JUMPADDR̃WX^Read
*/
void AG903_SSCPrmGetCpuJumpAddr(uint32_t* addr)
{
	ASSERT(NULL != addr);
	(*addr) = AG903_SSC->CA5_JUMPADDR;
	return;
}

/**
 * @brief           SSCCOUNT64_LOWER,SSCCOUNT64_UPPER̃WX^Write
 * @param           upper [in] JE^l [63:32]
 * @param           lower [in] JE^l [0:31]
 * @return          void
 * @description     SSCCOUNT64_LOWER,SSCCOUNT64_UPPER̃WX^Write
*/
void AG903_SSCPrmSetFreeRunCnt(uint32_t upper, uint32_t lower)
{
	AG903_SSC->COUNT64_LOWER = lower;
	AG903_SSC->COUNT64_UPPER = upper;
	return;
}

/**
 * @brief           SSCCOUNT64_LOWER,SSCCOUNT64_UPPER̃WX^Read
 * @param           upper [out] JE^l [63:32]
 * @param           lower [out] JE^l [0:31]
 * @return          void
 * @description     SSCCOUNT64_LOWER,SSCCOUNT64_UPPER̃WX^Read
*/
void AG903_SSCPrmGetFreeRunCnt(uint32_t* upper, uint32_t* lower)
{
	(*lower) = AG903_SSC->COUNT64_LOWER;
	(*upper) = AG903_SSC->COUNT64_UPPER;
	return;
}
