/*
 * RC-S620/S sample library for uC3
 *
 * 本ドライバは、下記ライブラリを参考にuC3向けにポーティングしたサンプルドライバです。
 * 各関数の説明は、下記ライブラリに同梱される「RCS620S-sjis.txt」を参照ください。
 *
 * 「Arduino向けRC-S620/S制御ライブラリ」
 * http://blog.felicalauncher.com/sdk_for_air/?page_id=2699
 */

#include "kernel.h"
#include "DDR_COM.h"

#include <string.h>
#include <inttypes.h>

#include "RCS620S_UC3.h"

/* --------------------------------
 * Constant
 * -------------------------------- */

#define RCS620S_DEFAULT_TIMEOUT  1000

/* --------------------------------
 * Variable
 * -------------------------------- */
static unsigned long timeout_;
static ID            comid_;
static uint8_t       response_[RCS620S_MAX_RW_RESPONSE_LEN];

uint8_t idm_[8];
uint8_t pmm_[8];

/* --------------------------------
 * Prototype Declaration
 * -------------------------------- */
static int     RCS620S_rwCommand(const uint8_t *command, uint16_t commandLen, uint8_t response[RCS620S_MAX_RW_RESPONSE_LEN], uint16_t *responseLen);
static void    RCS620S_cancel(void);
static uint8_t RCS620S_calcDCS(const uint8_t *data, uint16_t len);
static void    RCS620S_writeSerial(const uint8_t *data, uint16_t len);
static int     RCS620S_readSerial(uint8_t *data, uint16_t len);
static void    RCS620S_flushSerial(void);

static int RCS620S_initSerial(void);
static int RCS620S_command(const char *command, uint16_t commandLen, const char *response, uint16_t responseLen);

/* --------------------------------
 * Macro
 * -------------------------------- */

/* --------------------------------
 * Function
 * -------------------------------- */

/* ------------------------
 * public
 * ------------------------ */

int RCS620S_initDevice(ID comid)
{
    int      ret;

    // RCS620S constructor
    timeout_ = RCS620S_DEFAULT_TIMEOUT;

    // Serial.begin(115200) for uC3 COM Driver
    comid_ = comid;
    RCS620S_initSerial();

    // 8.2.6 Reset コマンド
    ret = RCS620S_command("\xd4\x18\x01", 3, "\xd5\x19", 2);
    if (!ret) {
        return 0;
    }

    /* RFConfiguration (various timings) */
    ret = RCS620S_command("\xd4\x32\x02\x00\x00\x00", 6, "\xd5\x33", 2);
    if (!ret) {
        return 0;
    }

    /* RFConfiguration (max retries) */
    ret = RCS620S_command("\xd4\x32\x05\x00\x00\x00", 6, "\xd5\x33", 2);
    if (!ret) {
        return 0;
    }

    /* RFConfiguration (additional wait time = 24ms) */
    ret = RCS620S_command("\xd4\x32\x81\xb7", 4, "\xd5\x33", 2);
    if (!ret) {
        return 0;
    }

    return 1;
}

int RCS620S_polling(uint16_t systemCode)
{
    int ret;
    uint8_t buf[9];
    uint8_t response[RCS620S_MAX_RW_RESPONSE_LEN];
    uint16_t responseLen;

    /* InListPassiveTarget */
    memcpy(buf, "\xd4\x4a\x01\x01\x00\xff\xff\x00\x00", 9);
    buf[5] = (uint8_t)((systemCode >> 8) & 0xff);
    buf[6] = (uint8_t)((systemCode >> 0) & 0xff);

    ret = RCS620S_rwCommand(buf, 9, response, &responseLen);
    if (!ret || (responseLen != 22) ||
        (memcmp(response, "\xd5\x4b\x01\x01\x12\x01", 6) != 0)) {
        return 0;
    }

    memcpy(idm_, response + 6, 8);
    memcpy(pmm_, response + 14, 8);

    return 1;
}

#if 0
// 本サンプルドライバでは、下記関数の動作検証は行っていません。ご注意ください。
//   -  RCS620S_cardCommand
//   -  RCS619S_rfOff
//   -  RCS620S_push
int RCS620S_cardCommand(const uint8_t *command, uint8_t commandLen, uint8_t response[RCS620S_MAX_CARD_RESPONSE_LEN], uint8_t *responseLen)
{
    int      ret;
    uint16_t commandTimeout;
    uint8_t  buf[RCS620S_MAX_RW_RESPONSE_LEN];
    uint16_t len;

    if (timeout_ >= (0x10000 / 2)) {
        commandTimeout = 0xffff;
    } else {
        commandTimeout = (uint16_t)(timeout_ * 2);
    }

    /* CommunicateThruEX */
    buf[0] = 0xd4;
    buf[1] = 0xa0;
    buf[2] = (uint8_t)((commandTimeout >> 0) & 0xff);
    buf[3] = (uint8_t)((commandTimeout >> 8) & 0xff);
    buf[4] = (uint8_t)(commandLen + 1);
    memcpy(buf + 5, command, commandLen);

    ret = RCS620S_rwCommand(buf, 5 + commandLen, buf, &len);
    if (!ret || (len < 4) ||
        (buf[0] != 0xd5) || (buf[1] != 0xa1) || (buf[2] != 0x00) ||
        (len != (3 + buf[3]))) {
        return 0;
    }

    *responseLen = (uint8_t)(buf[3] - 1);
    memcpy(response, buf + 4, *responseLen);

    return 1;
}

int RCS620S_rfOff(void)
{
    int      ret;
    uint8_t  response[RCS620S_MAX_RW_RESPONSE_LEN];
    uint16_t responseLen;

    /* RFConfiguration (RF field) */
    ret = RCS620S_rwCommand((const uint8_t *)"\xd4\x32\x01\x00", 4,
        response, &responseLen);
    if (!ret || (responseLen != 2) ||
        (memcmp(response, "\xd5\x33", 2) != 0)) {
        return 0;
    }

    return 1;
}

int RCS620S_push(const uint8_t *data, uint8_t dataLen)
{
    int     ret;
    uint8_t buf[RCS620S_MAX_CARD_RESPONSE_LEN];
    uint8_t responseLen;

    if (dataLen > 224) {
        return 0;
    }

    /* Push */
    buf[0] = 0xb0;
    memcpy(buf + 1, idm_, 8);
    buf[9] = dataLen;
    memcpy(buf + 10, data, dataLen);

    ret = RCS620S_cardCommand(buf, 10 + dataLen, buf, &responseLen);
    if (!ret || (responseLen != 10) || (buf[0] != 0xb1) ||
        (memcmp(buf + 1, idm_, 8) != 0) || (buf[9] != dataLen)) {
        return 0;
    }

    buf[0] = 0xa4;
    memcpy(buf + 1, idm_, 8);
    buf[9] = 0x00;

    ret = RCS620S_cardCommand(buf, 10, buf, &responseLen);
    if (!ret || (responseLen != 10) || (buf[0] != 0xa5) ||
        (memcmp(buf + 1, idm_, 8) != 0) || (buf[9] != 0x00)) {
        return 0;
    }

    dly_tsk(1000);

    return 1;
}
#endif

/* ------------------------
 * private
 * ------------------------ */

static int RCS620S_rwCommand(const uint8_t *command, uint16_t commandLen, uint8_t response[RCS620S_MAX_RW_RESPONSE_LEN], uint16_t *responseLen)
{
    int ret;
    uint8_t buf[9];

    RCS620S_flushSerial();

    uint8_t dcs = RCS620S_calcDCS(command, commandLen);

    /* transmit the command */
    buf[0] = 0x00;
    buf[1] = 0x00;
    buf[2] = 0xff;
    if (commandLen <= 255) {
        /* normal frame */
        buf[3] = commandLen;
        buf[4] = (uint8_t)-buf[3];
        RCS620S_writeSerial(buf, 5);
    } else {
        /* extended frame */
        buf[3] = 0xff;
        buf[4] = 0xff;
        buf[5] = (uint8_t)((commandLen >> 8) & 0xff);
        buf[6] = (uint8_t)((commandLen >> 0) & 0xff);
        buf[7] = (uint8_t)-(buf[5] + buf[6]);
        RCS620S_writeSerial(buf, 8);
    }
    RCS620S_writeSerial(command, commandLen);
    buf[0] = dcs;
    buf[1] = 0x00;
    RCS620S_writeSerial(buf, 2);

    RCS620S_flushSerial();

    /* receive an ACK */
    ret = RCS620S_readSerial(buf, 6);
    if (!ret || (memcmp(buf, "\x00\x00\xff\x00\xff\x00", 6) != 0)) {
        RCS620S_cancel();
        return 0;
    }

    /* receive a response */
    ret = RCS620S_readSerial(buf, 5);
    if (!ret) {
        RCS620S_cancel();
        return 0;
    } else if  (memcmp(buf, "\x00\x00\xff", 3) != 0) {
        return 0;
    }
    if ((buf[3] == 0xff) && (buf[4] == 0xff)) {
        ret = RCS620S_readSerial(buf + 5, 3);
        if (!ret || (((buf[5] + buf[6] + buf[7]) & 0xff) != 0)) {
            return 0;
        }
        *responseLen = (((uint16_t)buf[5] << 8) |
                        ((uint16_t)buf[6] << 0));
    } else {
        if (((buf[3] + buf[4]) & 0xff) != 0) {
            return 0;
        }
        *responseLen = buf[3];
    }
    if (*responseLen > RCS620S_MAX_RW_RESPONSE_LEN) {
        return 0;
    }

    ret = RCS620S_readSerial(response, *responseLen);
    if (!ret) {
        RCS620S_cancel();
        return 0;
    }

    dcs = RCS620S_calcDCS(response, *responseLen);

    ret = RCS620S_readSerial(buf, 2);
    if (!ret || (buf[0] != dcs) || (buf[1] != 0x00)) {
        RCS620S_cancel();
        return 0;
    }

    return 1;
}

static void RCS620S_cancel(void)
{
    /* transmit an ACK */
    RCS620S_writeSerial((const uint8_t*)"\x00\x00\xff\x00\xff\x00", 6);
    dly_tsk(1);
    RCS620S_flushSerial();
}

static uint8_t RCS620S_calcDCS(const uint8_t *data, uint16_t len)
{
    uint8_t sum = 0;

    for (uint16_t i = 0; i < len; i++) {
        sum += data[i];
    }

    return (uint8_t) - (sum & 0xff);
}

static void RCS620S_writeSerial(const uint8_t *data, uint16_t len)
{
    UINT scnt = len;
    (void)puts_com(comid_, (VB const *)data, &scnt, TMO_FEVR);
}

static int RCS620S_readSerial(uint8_t *data, uint16_t len)
{
    UINT rcnt = len;
    ER ercd;
    static UB sbuf[256];

    ercd = gets_com(comid_, (VB *)data, sbuf, -1, &rcnt, timeout_);

    return (ercd == E_OK);
}

static void RCS620S_flushSerial(void)
{
    (void)ctr_com(comid_, CLN_TXBUF, TMO_FEVR);
}

// Initialize uC3 standard communication driver.
static int RCS620S_initSerial(void)
{
    const T_COM_SMOD com_ini = { 115200U, BLEN8, PAR_NONE, SBIT1, FLW_NONE };
    ER               ercd;

    // uC3のUARTドライバを初期化
    ercd = ini_com(comid_, &com_ini);
    if (ercd != E_OK) {
        return 0;
    }

    // uC3のUARTドライバを有効化
    ercd = ctr_com(comid_, STA_COM, 0);
    if (ercd != E_OK) {
        return 0;
    }

    return 1;
}

// RCS620S_rwCommand wrapper
static int RCS620S_command(const char *command, uint16_t commandLen, const char *response, uint16_t responseLen)
{
    int ret;
    uint16_t responseLenRecv;

    // ホストコマンドパケットを送信し、ホストレスポンスパケットを受信
    ret = RCS620S_rwCommand((const uint8_t*)command, commandLen, response_, &responseLenRecv);

    // 受信したホストレスポンスパケットが正しいかどうかチェック
    if (!ret || (responseLenRecv != responseLen) ||
        (memcmp(response_, response, responseLen) != 0)) {
        return 0;
    }

    return 1;
}
