/**
 * @file    STM32L4xx_UC3.h
 * @brief   STMicroelectronics STM32L4xx (Cortex-M4 core) registers definitions
 * @date    2020.11.30
 * @author  Copyright (c) 2015-2020, eForce Co.,Ltd.  All rights reserved.
 *
 *********************************************************************************
 * @par     History
 *          - rev 1.0 (2015.02.12) i-cho
 *            Reference manual, RM0351 STM32L4x6 Rev.0.2
 *            Initial version. 
 *          - rev 1.1 (2017.01.20) i-cho
 *            Supported STM32L43x, STM32L462, STM32L452
 *          - rev 1.2 (2017.06.07) i-cho
 *            Supported STM32L496
 *          - rev 1.3 (2020.09.28) sakaguchi
 *            Fixed the structure t_dma,t_dma_ch to work with WLAN library.
 *          - rev 1.4 (2020.11.30) s-makino
 *            Corrected IRQ number on HASH processor.
 *********************************************************************************
 */

#ifndef STM32L4XX_UC3_H_
#define STM32L4XX_UC3_H_

#include "Cortex-M4.h"


/** 
 * Peripheral_memory_map
 */
#define PERIPH_BASE     (0x40000000UL)
#define APB1_BASE       (PERIPH_BASE)
#define APB2_BASE       (PERIPH_BASE + 0x00010000UL)
#define AHB1_BASE       (PERIPH_BASE + 0x00020000UL)
#define AHB2_BASE       (PERIPH_BASE + 0x08000000UL)


/* CRC registers (CRC) */

struct t_crc {
    UW      DR;         /* Data register                                */
    UW      IDR;        /* Independent data register                    */
    UW      CR;         /* Control register                             */
    UW      rsv2;
    UW      INIT;       /* Initial CRC value register,                  */
    UW      POL;        /* CRC polynomial register,                     */
};

#define REG_CRC     (*(volatile struct t_crc *)(AHB1_BASE + 0x3000UL))


/* Power control registers (PWR) */

struct t_pwr {
    UW      CR1;   /* power control register 1                          */
    UW      CR2;   /* power control register 2                          */
    UW      CR3;   /* power control register 3                          */
    UW      CR4;   /* power control register 4                          */
    UW      SR1;   /* power status register 1                           */
    UW      SR2;   /* power status register 2                           */
    UW      SCR;   /* power status reset register                       */
    UW      rsv0;
    UW      PUCRA; /* Pull_up control register of portA                 */
    UW      PDCRA; /* Pull_Down control register of portA               */
    UW      PUCRB; /* Pull_up control register of portB                 */
    UW      PDCRB; /* Pull_Down control register of portB               */
    UW      PUCRC; /* Pull_up control register of portC                 */
    UW      PDCRC; /* Pull_Down control register of portC               */
    UW      PUCRD; /* Pull_up control register of portD                 */
    UW      PDCRD; /* Pull_Down control register of portD               */
    UW      PUCRE; /* Pull_up control register of portE                 */
    UW      PDCRE; /* Pull_Down control register of portE               */
    UW      PUCRF; /* Pull_up control register of portF                 */
    UW      PDCRF; /* Pull_Down control register of portF               */
    UW      PUCRG; /* Pull_up control register of portG                 */
    UW      PDCRG; /* Pull_Down control register of portG               */
    UW      PUCRH; /* Pull_up control register of portH                 */
    UW      PDCRH; /* Pull_Down control register of portH               */
};

#define REG_PWR     (*(volatile struct t_pwr *)(APB1_BASE + 0x7000UL))


/* QuadSPI interface (QUADSPI)  */

struct t_quadspi {
    UW      CR;     /* Control register,                                */
    UW      DCR;    /* Device Configuration register,                   */
    UW      SR;     /* Status register,                                 */
    UW      FCR;    /* Flag Clear register,                             */
    UW      DLR;    /* Data Length register,                            */
    UW      CCR;    /* Communication Configuration register,            */
    UW      AR;     /* Address register,                                */
    UW      ABR;    /* Alternate Bytes register,                        */
    UW      DR;     /* Data register,                                   */
    UW      PSMKR;  /* Polling Status Mask register,                    */
    UW      PSMAR;  /* Polling Status Match register,                   */
    UW      PIR;    /* Polling Interval register,                       */
    UW      LPTR;   /* Low Power Timeout register,                      */
};

#define REG_QUADSPI     (*(volatile struct t_quadspi *)0xA0001000UL)


/* Reset and clock control registers (RCC) */

struct t_rcc {
    UW      CR;         /* Reset control register                       */
    UW      ICSCR;      /* Internal Clock Sources Calibration Register  */
    UW      CFGR;       /* Clock configuration register                 */
    UW      PLLCFGR;    /* PLL configuration register                   */
    UW      PLLSAI1CFGR;    /* PLL SAI1 Configuration Register          */
    UW      PLLSAI2CFGR;    /* PLL SAI2 Configuration Register          */
    UW      CIER;       /* Clock Interrupt Enable Register              */
    UW      CIFR;       /* Clock Interrupt Flag Register                */
    UW      CICR;       /* Clock Interrupt Clear Register               */
    UW      rsv0;
    UW      AHB1RSTR;   /* AHB1 peripheral reset register               */
    UW      AHB2RSTR;   /* AHB2 peripheral reset register               */
    UW      AHB3RSTR;   /* AHB3 peripheral reset register               */
    UW      rsv1;
    UW      APB1RSTR1;  /* LowSpeed APB1 macrocells resets              */
    UW      APB2RSTR2;  /* LowSpeed APB1 macrocells resets              */
    UW      APB2RSTR;   /* High Speed APB macrocells resets             */
    UW      rsv2;
    UW      AHB1ENR;    /* AHB1 peripheral clock enable register        */
    UW      AHB2ENR;    /* AHB2 peripheral clock enable register        */
    UW      AHB3ENR;    /* AHB3 peripheral clock enable register        */
    UW      rsv3;
    UW      APB1ENR1;   /* LowSpeed APB1 macrocells clock enables       */
    UW      APB2ENR2;   /* LowSpeed APB1 macrocells clock enables       */
    UW      APB2ENR;    /* High Speed APB macrocells clock enabled      */
    UW      rsv4;
    UW      AHB1SMENR;  /* AHB1 macrocells clocks enables in sleep mode */
    UW      AHB2SMENR;  /* AHB2 macrocells clock enables in sleep mode  */
    UW      AHB3SMENR;  /* AHB3 macrocells clock enables in sleep mode  */
    UW      rsv5;
    UW      APB1SMENR1; /* LowSpeed APB1 macrocells clock enables in sleep mode     */
    UW      APB1SMENR2; /* LowSpeed APB1 macrocells clock enables in sleep mode     */
    UW      APB2SMENR;  /* High Speed APB macrocells clock enabled in sleep mode    */
    UW      rsv6;
    UW      CCIPR;      /* IPs Clocks Configuration Register            */
    UW      rsv7;
    UW      BDCR;       /* Vswitch Backup Domain Control Register       */
    UW      CSR;        /* clock control & status register              */
};

#define REG_RCC     (*(volatile struct t_rcc *)(AHB1_BASE + 0x1000UL))


/* Flash interface registers                                            */

struct t_flash {
    UW      ACR;        /* access control register                      */
    UW      PDKEYR;     /* power down key register                      */
    UW      KEYR;       /* key register                                 */
    UW      OPTKEYR;    /* option key register                          */
    UW      SR;         /* status register                              */
    UW      CR;         /* control register                             */
    UW      ECCR;       /* ECC register                                 */
    UW      rsv0;
    UW      OPTR;       /* option register                              */
    UW      PCROP1SR;   /* bank1 PCROP start address register           */
    UW      PCROP1ER;   /* bank1 PCROP end address register             */
    UW      WRP1AR;     /* bank1 WRP area A address register            */
    UW      WRP1BR;     /* bank1 WRP area B address register            */
    UW      rsv1[4];
    UW      PCROP2SR;   /* bank2 PCROP start address register           */
    UW      PCROP2ER;   /* bank2 PCROP end address register             */
    UW      WRP2AR;     /* bank2 WRP area A address register            */
    UW      WRP2BR;     /* bank2 WRP area B address register            */
};

#define REG_FLASH   (*(volatile struct t_flash *)(AHB1_BASE + 0x2000UL))


/* General purpose and alternate function I/O registers (GPIO)          */

struct t_gpio {
    UW      MODER;      /* Port mode register                           */
    UW      OTYPER;     /* Port output type register                    */
    UW      OSPEEDER;   /* Port output speed register                   */
    UW      PUPDR;      /* Port pull-up/pull-down register              */
    UW      IDR;        /* Port input data register                     */
    UW      ODR;        /* Port output data register                    */
    UW      BSRR;       /* Port bit set/reset register                  */
    UW      LCKR;       /* Port configuration lock register             */
    UW      AFRL;       /* Alternate function low register              */
    UW      AFRH;       /* Alternate function high register             */
    UW      BRR;        /* Bit Reset register                           */
    UW      ASCR;       /* analog switch control register               */
};

#define REG_GPIOA   (*(volatile struct t_gpio *)(AHB2_BASE + 0x0000UL))
#define REG_GPIOB   (*(volatile struct t_gpio *)(AHB2_BASE + 0x0400UL))
#define REG_GPIOC   (*(volatile struct t_gpio *)(AHB2_BASE + 0x0800UL))
#define REG_GPIOD   (*(volatile struct t_gpio *)(AHB2_BASE + 0x0C00UL))
#define REG_GPIOE   (*(volatile struct t_gpio *)(AHB2_BASE + 0x1000UL))
#define REG_GPIOF   (*(volatile struct t_gpio *)(AHB2_BASE + 0x1400UL))
#define REG_GPIOG   (*(volatile struct t_gpio *)(AHB2_BASE + 0x1800UL))
#define REG_GPIOH   (*(volatile struct t_gpio *)(AHB2_BASE + 0x1C00UL))
#define REG_GPIOI   (*(volatile struct t_gpio *)(AHB2_BASE + 0x2000UL))


/* System configuration controller registers (SYSCFG) */

struct t_syscfg {
    UW      MEMRMP;     /* Memory remap register                        */
    UW      CFGR1;      /* configuration register 1                     */
    UW      EXTICR1;    /* External interrupt configuration register 1  */
    UW      EXTICR2;    /* External interrupt configuration register 2  */
    UW      EXTICR3;    /* External interrupt configuration register 3  */
    UW      EXTICR4;    /* External interrupt configuration register 4  */
    UW      SCSR;       /* SRAM2 control and status register            */
    UW      CFGR2;      /* configuration register 2                     */
    UW      SWPR;       /* SRAM2 write protection register              */
    UW      SKR;        /* SRAM2 key register                           */
};

#define REG_SYSCFG  (*(volatile struct t_syscfg *)(APB2_BASE + 0x0000UL))


/* External interrupt/event controller registers (EXTI) */

struct t_exti {
    UW      IMR1;       /* Interrupt mask register 1,                   */
    UW      EMR1;       /* Event mask register 1,                       */
    UW      RTSR1;      /* Rising trigger selection register 1,         */
    UW      FTSR1;      /* Falling trigger selection register 1,        */
    UW      SWIER1;     /* Software interrupt event register 1,         */
    UW      PR1;        /* Pending register 1,                          */
    UW      rsv1;
    UW      rsv2;
    UW      IMR2;       /* Interrupt mask register 2,                   */
    UW      EMR2;       /* Event mask register 2,                       */
    UW      RTSR2;      /* Rising trigger selection register 2,         */
    UW      FTSR2;      /* Falling trigger selection register 2,        */
    UW      SWIER2;     /* Software interrupt event register 2,         */
    UW      PR2;        /* Pending register 2,                          */
};

#define REG_EXTI    (*(volatile struct t_exti *)(APB2_BASE + 0x0400UL))


/* Firewall (FW) */

struct t_fw {
    UW      CSSA;       /* Code Segment Start Address register          */
    UW      CSL;        /* Code Segment Length register,                */
    UW      NVDSSA;     /* NON volatile data Segment Start Address register */
    UW      NVDSL;      /* NON volatile data Segment Length register    */
    UW      VDSSA;      /* Volatile data Segment Start Address register */
    UW      VDSL;       /* Volatile data Segment Length register,       */
    UW      rsv1;
    UW      rsv2;
    UW      CR;         /* Configuration  register,                     */
};

#define REG_FW      (*(volatile struct t_fw *)(APB2_BASE + 0x1C00UL))


/* DMA control registers (DMA) */

typedef struct t_dma_ch {
    UW      CCR;        /* channel x configuration register             */
    UW      CNDTR;      /* channel x number of data register            */
    UW      CPAR;       /* channel x peripheral address register        */
    UW      CMAR;       /* channel x memory address register            */
    UW      rsv0;
} T_DMA_CH;

struct t_dma {
    UW      ISR;        /* interrupt status register,                   */
    UW      IFCR;       /* interrupt flag clear register,               */
    T_DMA_CH Channel[7];    /* DMA Stream 1~7                           */
};

#define REG_DMA1    (*(volatile struct t_dma *)(AHB1_BASE + 0x0000UL))
#define REG_DMA2    (*(volatile struct t_dma *)(AHB1_BASE + 0x0400UL))


/* Analog/digital converter (ADC) */

struct t_adc {
    UW      ISR;        /* Interrupt and Status Register            */
    UW      IER;        /* Interrupt Enable Register,               */
    UW      CR;         /* control register,                        */
    UW      CFGR;       /* Configuration register,                  */
    UW      CFGR2;      /* Configuration register 2,                */
    UW      SMPR1;      /* sample time register 1,                  */
    UW      SMPR2;      /* sample time register 2,                  */
    UW      rsv1;
    UW      TR1;        /* watchdog threshold register 1,           */
    UW      TR2;        /* watchdog threshold register 2,           */
    UW      TR3;        /* watchdog threshold register 3,           */
    UW      rsv2;
    UW      SQR1;       /* regular sequence register 1,             */
    UW      SQR2;       /* regular sequence register 2,             */
    UW      SQR3;       /* regular sequence register 3,             */
    UW      SQR4;       /* regular sequence register 4,             */
    UW      DR;         /* regular data register,                   */
    UW      rsv3;
    UW      rsv4;
    UW      JSQR;       /* injected sequence register,              */
    UW      rsv5[4];
    UW      OFR1;       /* offset register 1,                       */
    UW      OFR2;       /* offset register 2,                       */
    UW      OFR3;       /* offset register 3,                       */
    UW      OFR4;       /* offset register 4,                       */
    UW      rsv6[4];
    UW      JDR1;       /* injected data register 1,                */
    UW      JDR2;       /* injected data register 2,                */
    UW      JDR3;       /* injected data register 3,                */
    UW      JDR4;       /* injected data register 4,                */
    UW      rsv7[4];
    UW      AWD2CR;     /* Analog Watchdog 2 Configuration Register,*/
    UW      AWD3CR;     /* Analog Watchdog 3 Configuration Register,*/
    UW      rsv8;
    UW      rsv9;
    UW      DIFSEL;     /* Differential Mode Selection Register,    */
    UW      CALFACT;    /* Calibration Factors,                     */
};

struct t_adc_common {
    UW      CSR;        /* Common status register,                  */
    UW      rsv1;
    UW      CCR;        /* common control register,                 */
    UW      CDR;        /* common regular data register for dual    */
};

#define REG_ADC1        (*(volatile struct t_adc *)(AHB2_BASE + 0x08040000UL))
#define REG_ADC2        (*(volatile struct t_adc *)(AHB2_BASE + 0x08040100UL))
#define REG_ADC3        (*(volatile struct t_adc *)(AHB2_BASE + 0x08040200UL))
#define REG_ADC_COMMON  (*(volatile struct t_adc_common *)(AHB2_BASE + 0x08040300UL))


/* Digital to analog converter (DAC) */

struct t_dac {
    UW      CR;         /* Control register                             */
    UW      SWTRIGR;    /* Software Trigger Register                    */
    UW      DHR12R1;    /* Channel 1 12-bit right-aligned data register */
    UW      DHR12L1;    /* Channel 1 12-bit left-aligned data register  */
    UW      DHR8R1;     /* Channel 1 8-bit right-aligned data register  */
    UW      DHR12R2;    /* Channel 2 12-bit right-aligned data register */
    UW      DHR12L2;    /* Channel 2 12-bit left-aligned data register  */
    UW      DHR8R2;     /* Channel 2 8-bit right-aligned data register  */
    UW      DHR12RD;    /* Dual 12-bit right-aligned data register      */
    UW      DHR12LD;    /* Dual 12-bit left-aligned data register       */
    UW      DHR8RD;     /* Dual 8-bit right-aligned data register       */
    UW      DOR1;       /* Channel 1 data output register               */
    UW      DOR2;       /* Channel 2 data output register               */
    UW      SR;         /* Status register                              */
    UW      CCR;        /* calibration control register,                */
    UW      MCR;        /* mode control register,                       */
    UW      SHSR1;      /* Sample and Hold sample time register 1,      */
    UW      SHSR2;      /* Sample and Hold sample time register 2,      */
    UW      SHHR;       /* Sample and Hold hold time register,          */
    UW      SHRR;       /* Sample and Hold refresh time register,       */
};

#define REG_DAC1        (*(volatile struct t_dac *)(APB1_BASE + 0x7400UL))


/* Digital filter for sigma delta modulators (DFSDM) */

struct t_dfsdm_ch {
    UW      R1;         /* channel configuration register1              */
    UW      R2;         /* channel configuration register2,             */
    UW      AWSCDR;     /* channel analog watchdog and short circuit detector register  */
    UW      WDATAR;     /* channel watchdog filter data register        */
    UW      DATINR;     /* channel data input register,                 */
};

#define REG_DFSDM1_CH0   (*(volatile struct t_dfsdm_ch *)(APB2_BASE + 0x6000UL))
#define REG_DFSDM1_CH1   (*(volatile struct t_dfsdm_ch *)(APB2_BASE + 0x6020UL))
#define REG_DFSDM1_CH2   (*(volatile struct t_dfsdm_ch *)(APB2_BASE + 0x6040UL))
#define REG_DFSDM1_CH3   (*(volatile struct t_dfsdm_ch *)(APB2_BASE + 0x6060UL))
#define REG_DFSDM1_CH4   (*(volatile struct t_dfsdm_ch *)(APB2_BASE + 0x6080UL))
#define REG_DFSDM1_CH5   (*(volatile struct t_dfsdm_ch *)(APB2_BASE + 0x60A0UL))
#define REG_DFSDM1_CH6   (*(volatile struct t_dfsdm_ch *)(APB2_BASE + 0x60C0UL))
#define REG_DFSDM1_CH7   (*(volatile struct t_dfsdm_ch *)(APB2_BASE + 0x60E0UL))

struct t_dfsdm_flt {
    UW      CR1;        /* control register1,                           */
    UW      CR2;        /* control register2,                           */
    UW      ISR;        /* interrupt and status register,               */
    UW      ICR;        /* interrupt flag clear register,               */
    UW      JCHGR;      /* injected channel group selection register,   */
    UW      FCR;        /* filter control register,                     */
    UW      JDATAR;     /* data register for injected group,            */
    UW      RDATAR;     /* data register for regular group,             */
    UW      AWHTR;      /* analog watchdog high threshold register,     */
    UW      AWLTR;      /* analog watchdog low threshold register,      */
    UW      AWSR;       /* analog watchdog status register              */
    UW      AWCFR;      /* analog watchdog clear flag register          */
    UW      EXMAX;      /* extreme detector maximum register,           */
    UW      EXMIN;      /* extreme detector minimum register            */
    UW      CNVTIMR;    /* conversion timer,                            */
};

#define REG_DFSDM1_FLT0  (*(volatile struct t_dfsdm_flt *)(APB2_BASE + 0x6100UL))
#define REG_DFSDM1_FLT1  (*(volatile struct t_dfsdm_flt *)(APB2_BASE + 0x6180UL))
#define REG_DFSDM1_FLT2  (*(volatile struct t_dfsdm_flt *)(APB2_BASE + 0x6200UL))
#define REG_DFSDM1_FLT3  (*(volatile struct t_dfsdm_flt *)(APB2_BASE + 0x6280UL))


/* Timer control (TIM) */

struct t_tim {
    UW      CR1;        /* Control register 1                           */
    UW      CR2;        /* Control register 2                           */
    UW      SMCR;       /* Slave mode control register                  */
    UW      DIER;       /* DMA/Interrupt enable register                */
    UW      SR;         /* Status register                              */
    UW      EGR;        /* Event generation register                    */
    UW      CCMR1;      /* Capture/compare mode register 1              */
    UW      CCMR2;      /* Capture/compare mode register 2              */
    UW      CCER;       /* Capture/compare enable register              */
    UW      CNT;        /* Counter                                      */
    UW      PSC;        /* Prescaler                                    */
    UW      ARR;        /* Auto-reload register                         */
    UW      RCR;        /* Repetition counter register                  */
    UW      CCR1;       /* Capture/compare register 1                   */
    UW      CCR2;       /* Capture/compare register 2                   */
    UW      CCR3;       /* Capture/compare register 3                   */
    UW      CCR4;       /* Capture/compare register 4                   */
    UW      BDTR;       /* Break and dead-time register                 */
    UW      DCR;        /* DMA control register                         */
    UW      DMAR;       /* DMA address for full transfer                */
    UW      OR1;        /* option register 1,                           */
    UW      CCMR3;      /* capture/compare mode register 3,             */
    UW      CCR5;       /* capture/compare register5,                   */
    UW      CCR6;       /* capture/compare register6,                   */
    UW      OR2;        /* option register 2,                           */
    UW      OR3;        /* option register 3,                           */
};

#define REG_TIM1    (*(volatile struct t_tim *)(APB2_BASE + 0x2C00UL))
#define REG_TIM2    (*(volatile struct t_tim *)(APB1_BASE + 0x0000UL))
#define REG_TIM3    (*(volatile struct t_tim *)(APB1_BASE + 0x0400UL))
#define REG_TIM4    (*(volatile struct t_tim *)(APB1_BASE + 0x0800UL))
#define REG_TIM5    (*(volatile struct t_tim *)(APB1_BASE + 0x0C00UL))
#define REG_TIM6    (*(volatile struct t_tim *)(APB1_BASE + 0x1000UL))
#define REG_TIM7    (*(volatile struct t_tim *)(APB1_BASE + 0x1400UL))
#define REG_TIM8    (*(volatile struct t_tim *)(APB2_BASE + 0x3400UL))
#define REG_TIM15   (*(volatile struct t_tim *)(APB2_BASE + 0x4000UL))
#define REG_TIM16   (*(volatile struct t_tim *)(APB2_BASE + 0x4400UL))
#define REG_TIM17   (*(volatile struct t_tim *)(APB2_BASE + 0x4800UL))


/* Touch Sensing Controller (TSC) */

struct t_tsc {
    UW      CR;         /* control register                             */
    UW      IER;        /* interrupt enable register,                   */
    UW      ICR;        /* interrupt clear register,                    */
    UW      ISR;        /* interrupt status register,                   */
    UW      IOHCR;      /* I/O hysteresis control register,             */
    UW      rsv1;
    UW      IOASCR;     /* I/O analog switch control register,          */
    UW      rsv2;
    UW      IOSCR;      /* I/O sampling control register,               */
    UW      rsv3;
    UW      IOCCR;      /* I/O channel control register,                */
    UW      rsv4;
    UW      IOGCSR;     /* I/O group control status register,           */
    UW      IOGXCR[8];  /* I/O group x counter register,                */
};

#define REG_TSC     (*(volatile struct t_tsc *)(AHB1_BASE + 0x4000UL))


/* Chrom-Art Accelerator controller (DMA2D) */

struct t_dma2d {
    UW      CR;             /* Control Register,                        */
    UW      ISR;            /* Interrupt Status Register,               */
    UW      IFCR;           /* Interrupt Flag Clear Register,           */
    UW      FGMAR;          /* Foreground Memory Address Register,      */
    UW      FGOR;           /* Foreground Offset Register,              */
    UW      BGMAR;          /* Background Memory Address Register,      */
    UW      BGOR;           /* Background Offset Register,              */
    UW      FGPFCCR;        /* Foreground PFC Control Register,         */
    UW      FGCOLR;         /* Foreground Color Register,               */
    UW      BGPFCCR;        /* Background PFC Control Register,         */
    UW      BGCOLR;         /* Background Color Register,               */
    UW      FGCMAR;         /* Foreground CLUT Memory Address Register, */
    UW      BGCMAR;         /* Background CLUT Memory Address Register, */
    UW      OPFCCR;         /* Output PFC Control Register,             */
    UW      OCOLR;          /* Output Color Register,                   */
    UW      OMAR;           /* Output Memory Address Register,          */
    UW      OOR;            /* Output Offset Register,                  */
    UW      NLR;            /* Number of Line Register,                 */
    UW      LWR;            /* Line Watermark Register,                 */
    UW      AMTCR;          /* AHB Master Timer Configuration Register, */
    UW      rsv1[236];
    UW      FGCLUT[256];    /* Foreground CLUT,                         */
    UW      BGCLUT[256];    /* Background CLUT,                         */
};

#define REG_DMA2D   (*(volatile struct t_dma2d *)(AHB1_BASE + 0xB000UL))


/* Real-time clock (RTC)    */

struct t_rtc {
    UW      TR;         /* time register                                */
    UW      DR;         /* date register,                               */
    UW      CR;         /* control register,                            */
    UW      ISR;        /* initialization and status register,          */
    UW      PRER;       /* prescaler register,                          */
    UW      WUTR;       /* wakeup timer register,                       */
    UW      rsv1;
    UW      ALRMAR;     /* alarm A register,                            */
    UW      ALRMBR;     /* alarm B register,                            */
    UW      WPR;        /* write protection register,                   */
    UW      SSR;        /* sub second register,                         */
    UW      SHIFTR;     /* shift control register,                      */
    UW      TSTR;       /* time stamp time register,                    */
    UW      TSDR;       /* time stamp date register,                    */
    UW      TSSSR;      /* time-stamp sub second register,              */
    UW      CALR;       /* calibration register,                        */
    UW      TAMPCR;     /* tamper configuration register,               */
    UW      ALRMASSR;   /* alarm A sub second register,                 */
    UW      ALRMBSSR;   /* alarm B sub second register,                 */
    UW      OR;         /* option register,                             */
    UW      BKP0R;      /* backup register 0,                           */
    UW      BKP1R;      /* backup register 1,                           */
    UW      BKP2R;      /* backup register 2,                           */
    UW      BKP3R;      /* backup register 3,                           */
    UW      BKP4R;      /* backup register 4,                           */
    UW      BKP5R;      /* backup register 5,                           */
    UW      BKP6R;      /* backup register 6,                           */
    UW      BKP7R;      /* backup register 7,                           */
    UW      BKP8R;      /* backup register 8,                           */
    UW      BKP9R;      /* backup register 9,                           */
    UW      BKP10R;     /* backup register 10,                          */
    UW      BKP11R;     /* backup register 11,                          */
    UW      BKP12R;     /* backup register 12,                          */
    UW      BKP13R;     /* backup register 13,                          */
    UW      BKP14R;     /* backup register 14,                          */
    UW      BKP15R;     /* backup register 15,                          */
    UW      BKP16R;     /* backup register 16,                          */
    UW      BKP17R;     /* backup register 17,                          */
    UW      BKP18R;     /* backup register 18,                          */
    UW      BKP19R;     /* backup register 19,                          */
    UW      BKP20R;     /* backup register 20,                          */
    UW      BKP21R;     /* backup register 21,                          */
    UW      BKP22R;     /* backup register 22,                          */
    UW      BKP23R;     /* backup register 23,                          */
    UW      BKP24R;     /* backup register 24,                          */
    UW      BKP25R;     /* backup register 25,                          */
    UW      BKP26R;     /* backup register 26,                          */
    UW      BKP27R;     /* backup register 27,                          */
    UW      BKP28R;     /* backup register 28,                          */
    UW      BKP29R;     /* backup register 29,                          */
    UW      BKP30R;     /* backup register 30,                          */
    UW      BKP31R;     /* backup register 31,                          */
};

#define REG_RTC     (*(volatile struct t_rtc *)(APB1_BASE + 0x2800UL))


/* Serial audio interface (SAI) */

struct t_sai {
    UW      GCR;        /* SAI global configuration register            */
};

struct t_sai_block {
    UW      CR1;        /* SAI block x configuration register 1         */
    UW      CR2;        /* SAI block x configuration register 2         */
    UW      FRCR;       /* SAI block x frame configuration register     */
    UW      SLOTR;      /* SAI block x slot register                    */
    UW      IMR;        /* SAI block x interrupt mask register          */
    UW      SR;         /* SAI block x status register                  */
    UW      CLRFR;      /* SAI block x clear flag register              */
    UW      DR;         /* SAI block x data register                    */
};

#define REG_SAI1        (*(volatile struct t_sai *)(APB2_BASE + 0x5400UL))
#define REG_SAI1_BLOCKA (*(volatile struct t_sai_block *)(APB2_BASE + 0x5404UL))
#define REG_SAI1_BLOCKB (*(volatile struct t_sai_block *)(APB2_BASE + 0x5424UL))
#define REG_SAI2        (*(volatile struct t_sai *)(APB2_BASE + 0x5800UL))
#define REG_SAI2_BLOCKA (*(volatile struct t_sai_block *)(APB2_BASE + 0x5804UL))
#define REG_SAI2_BLOCKB (*(volatile struct t_sai_block *)(APB2_BASE + 0x5824UL))


/* Independent watchdog (IWDG) */

struct t_iwdg {
    UW      KR;         /* Key register                                 */
    UW      PR;         /* Prescaler register                           */
    UW      RLR;        /* Reload register                              */
    UW      SR;         /* Status register                              */
    UW      WINR;       /* Window register                              */
};

#define REG_IWDG    (*(volatile struct t_iwdg *)(APB1_BASE + 0x3000UL))


/* Window watchdog (WWDG) */

struct t_wwdg {
    UW      CR;         /* Control register                             */
    UW      CFR;        /* Configuration register                       */
    UW      SR;         /* Status register                              */
};

#define REG_WWDG    (*(volatile struct t_wwdg *)(APB1_BASE + 0x2C00UL))


/* Liquid crystal display controller (LCD)   */
  
struct t_lcd {
    UW      CR;         /* control register,                            */
    UW      FCR;        /* frame control register,                      */
    UW      SR;         /* status register,                             */
    UW      CLR;        /* clear register,                              */
    UW      rsv1;
    UW      RAM[16];    /*!< LCD display memory,                        */
};  

#define REG_LCD     (*(volatile struct t_lcd *)(APB1_BASE + 0x2400UL))


/* Low power timer (LPTIM)  */

struct t_lptim {
    UW      ISR;        /* Interrupt and Status register                */
    UW      ICR;        /* Interrupt Clear register,                    */
    UW      IER;        /* Interrupt Enable register,                   */
    UW      CFGR;       /* Configuration register,                      */
    UW      CR;         /* Control register,                            */
    UW      CMP;        /* Compare register,                            */
    UW      ARR;        /* Autoreload register,                         */
    UW      CNT;        /* Counter register,                            */
    UW      OR;         /* Option register,                             */
};

#define REG_LPTIM1  (*(volatile struct t_lptim *)(APB1_BASE + 0x7C00UL))
#define REG_LPTIM2  (*(volatile struct t_lptim *)(APB1_BASE + 0x9400UL))


/* Operational amplifiers (OPAMP) */

struct t_opamp {
    UW      CSR;        /* control/status register                      */
    UW      OTR;        /* offset trimming register for normal mode,    */
    UW      LPOTR;      /* offset trimming register for low power mode, */
};

#define REG_OPAMP1  (*(volatile struct t_opamp *)(APB1_BASE + 0x7800UL))
#define REG_OPAMP2  (*(volatile struct t_opamp *)(APB1_BASE + 0x7810UL))


/* Random number generator (RNG) */

struct t_rng {
    UW      CR;         /* Control register                             */
    UW      SR;         /* Status register                              */
    UW      DR;         /* Data register                                */
};

#define REG_RNG     (*(volatile struct t_rng *)(AHB2_BASE + 0x08060800UL))


/* HASH processor (HASH) */

struct t_hash {
    UW      CR;         /* Control register                             */
    UW      DIN;        /* Data input register                          */
    UW      STR;        /* Start register                               */
    UW      HR[5];      /* Digest register 0~4                          */
    UW      IMR;        /* Interrupt mask register                      */
    UW      SR;         /* Status register                              */
    UW      rsv0[52];
    UW      CSR[54];    /* Context swap register 0~53                   */
    UW      rsv1[4];
    UW      HASH_HR[8]; /* Digest register 0~7                          */
};

#define REG_HASH    (*(volatile struct t_hash *)(AHB2_BASE + 0x08060400UL))


/* Advanced encryption standard hardware accelerator (AES) */

struct t_aes {
    UW      CR;         /* control register                             */
    UW      SR;         /* status register                              */
    UW      DIN;        /* data input register                          */
    UW      DOUT;       /* data output register                         */
    UW      KEYR0;      /* key register 0                               */
    UW      KEYR1;      /* key register 1                               */
    UW      KEYR2;      /* key register 2                               */
    UW      KEYR3;      /* key register 3                               */
    UW      IVR0;       /* initialization vector register 0             */
    UW      IVR1;       /* initialization vector register 1             */
    UW      IVR2;       /* initialization vector register 2             */
    UW      IVR3;       /* initialization vector register 3             */
    UW      KEYR4;      /* key register 4                               */
    UW      KEYR5;      /* key register 5                               */
    UW      KEYR6;      /* key register 6                               */
    UW      KEYR7;      /* key register 7                               */
    UW      SUSP0R;     /* Suspend register 0 ~ 7                       */
    UW      SUSP1R;
    UW      SUSP2R;
    UW      SUSP3R;
    UW      SUSP4R;
    UW      SUSP5R;
    UW      SUSP6R;
    UW      SUSP7R;
};

#define REG_AES     (*(volatile struct t_aes *)(AHB2_BASE + 0x08060000UL))


/* Digital camera interface (DCMI) */

struct t_dcmi {
    UW      CR;         /* control register                         */
    UW      SR;         /* status register,                         */
    UW      RIS;        /* raw interrupt status register,           */
    UW      IER;        /* interrupt enable register,               */
    UW      MIS;        /* masked interrupt status register,        */
    UW      ICR;        /* interrupt clear register,                */
    UW      ESCR;       /* embedded synchronization code register,  */
    UW      ESUR;       /* embedded synchronization unmask register */
    UW      CWSTRT;     /* crop window start,                       */
    UW      CWSIZE;     /* crop window size,                        */
    UW      DR;         /* data register,                           */
};

#define REG_DCMI    (*(volatile struct t_dcmi *)(AHB2_BASE + 0x08050000UL))


/* Inter-integrated circuit interface registers (I2C) */

struct t_i2c {
    UW      CR1;        /* Control register 1                           */
    UW      CR2;        /* Control register 2,                          */
    UW      OAR1;       /* Own address 1 register,                      */
    UW      OAR2;       /* Own address 2 register,                      */
    UW      TIMINGR;    /* Timing register,                             */
    UW      TIMEOUTR;   /* Timeout register,                            */
    UW      ISR;        /* Interrupt and status register,               */
    UW      ICR;        /* Interrupt clear register,                    */
    UW      PECR;       /* PEC register,                                */
    UW      RXDR;       /* Receive data register,                       */
    UW      TXDR;       /* Transmit data register,                      */
};

#define REG_I2C1    (*(volatile struct t_i2c *)(APB1_BASE + 0x5400UL))
#define REG_I2C2    (*(volatile struct t_i2c *)(APB1_BASE + 0x5800UL))
#define REG_I2C3    (*(volatile struct t_i2c *)(APB1_BASE + 0x5C00UL))
#define REG_I2C4    (*(volatile struct t_i2c *)(APB1_BASE + 0x8400UL))


/* Universal synchronous asynchronous receiver transmitter (USART) */

struct t_usart {
    UW      CR1;        /* Control register 1                           */
    UW      CR2;        /* Control register 2                           */
    UW      CR3;        /* Control register 3                           */
    UW      BRR;        /* Baud rate register                           */
    UW      GTPR;       /* Guard time and prescaler register            */
    UW      RTOR;       /* Receiver Time Out register                   */  
    UW      RQR;        /* Request register                             */
    UW      ISR;        /* Interrupt and status register                */
    UW      ICR;        /* Interrupt flag Clear register                */
    UW      RDR;        /* Receive Data register                        */
    UW      TDR;        /* Transmit Data register                       */
};

#define REG_USART1  (*(volatile struct t_usart *)(APB2_BASE+0x00003800UL))
#define REG_USART2  (*(volatile struct t_usart *)(APB1_BASE+0x00004400UL))
#define REG_USART3  (*(volatile struct t_usart *)(APB1_BASE+0x00004800UL))
#define REG_UART4   (*(volatile struct t_usart *)(APB1_BASE+0x00004C00UL))
#define REG_UART5   (*(volatile struct t_usart *)(APB1_BASE+0x00005000UL))
#define REG_LPUART1 (*(volatile struct t_usart *)(APB1_BASE+0x00008000UL))


/* Serial peripheral interface (SPI) */

struct t_spi {
    UW      CR1;        /* SPI control register 1                       */
    UW      CR2;        /* SPI control register 2                       */
    UW      SR;         /* SPI status register                          */
    UW      DR;         /* SPI data register                            */
    UW      CRCPR;      /* SPI CRC polynomial register                  */
    UW      RXCRCR;     /* SPI Rx CRC register                          */
    UW      TXCRCR;     /* SPI Tx CRC register                          */
    UW      I2SCFGR;    /* I2S Configuration register                   */
    UW      I2SPR;      /* I2S Prescaler register                       */
};

#define REG_SPI1    (*(volatile struct t_spi *)(APB2_BASE + 0x3000UL))
#define REG_SPI2    (*(volatile struct t_spi *)(APB1_BASE + 0x3800UL))
#define REG_SPI3    (*(volatile struct t_spi *)(APB1_BASE + 0x3C00UL))


/* Single Wire Protocol Master Interface (SWPMI) */

struct t_spwmi {
    UW      CR;         /* Configuration/Control register               */
    UW      BRR;        /* bitrate register,                            */
    UW      rsv1;
    UW      ISR;        /* Interrupt and Status register,               */
    UW      ICR;        /* Interrupt Flag Clear register,               */
    UW      IER;        /* Interrupt Enable register,                   */
    UW      RFL;        /* Receive Frame Length register,               */
    UW      TDR;        /* Transmit data register,                      */
    UW      RDR;        /* Receive data register,                       */
    UW      OR;         /* Option register,                             */
};

#define REG_SWPMI1  (*(volatile struct t_spwmi *)(APB1_BASE + 0x8800UL))


/* SD/SDIO/MMC card host interface (SDMMC) */

struct t_sdmmc {
    UW      POWER;      /* Power control register                       */
    UW      CLKCR;      /* Clock control register                       */
    UW      ARG;        /* Argument register                            */
    UW      CMD;        /* Command register                             */
    UW      RESPCMD;    /* Command response register                    */
    UW      RESP[4];    /* Response register 1..4                       */
    UW      DTIMER;     /* Data timer register                          */
    UW      DLEN;       /* Data length register                         */
    UW      DCTRL;      /* Data control register                        */
    UW      DCOUNT;     /* Data counter register                        */
    UW      STA;        /* Status register                              */
    UW      ICR;        /* Interrupt clear register                     */
    UW      MASK;       /* Mask register                                */
    UW      rsv0[2];
    UW      FIFOCNT;    /* FIFO counter register                        */
    UW      rsv1[13];
    UW      FIFO;       /* Data FIFO register                           */
};

#define REG_SDMMC   (*(volatile struct t_sdmmc *)(APB2_BASE + 0x2800UL))


/* Controller area network registers (bxCAN) */

typedef struct t_txmbx {
    UW      TIxR;       /* Tx mailbox identifier register               */
    UW      TDTxR;      /* Mailbox data length control and time stamp   */
    UW      TDLxR;      /* Mailbox data low register                    */
    UW      TDHxR;      /* Mailbox data high register                   */
} T_TxMBX;

typedef struct t_rxmbx {
    UW      RIxR;       /* Rx mailbox identifier register               */
    UW      RDTxR;      /* Receive FIFO mailbox data length and time    */
    UW      RDLxR;      /* Receive FIFO mailbox data low register       */
    UW      RDHxR;      /* Receive FIFO mailbox data high register      */
} T_RxMBX;

typedef struct t_filt {
    UW      FR1;        /* Filter registers                             */
    UW      FR2;
} T_FILT;

struct t_bxcan {
    UW      MCR;        /* CAN master control register                  */
    UW      MSR;        /* CAN master status register                   */
    UW      TSR;        /* CAN transmit status register                 */
    UW      RF0R;       /* CAN receive FIFO 0 register                  */
    UW      RF1R;       /* CAN receive FIFO 1 register                  */
    UW      IER;        /* CAN interrupt enable register                */
    UW      ESR;        /* CAN error status register                    */
    UW      BTR;        /* CAN bit timing register                      */
    UW      rsv0[88];
    T_TxMBX txmbx[3];   /* Tx Mailboxes                                 */
    T_RxMBX rxmbx[2];   /* Rx Mailboxes                                 */
    UW      rsv1[12];
    UW      FMR;        /* CAN filter master register                   */
    UW      FM1R;       /* CAN filter mode register                     */
    UW      rsv2;
    UW      FS1R;       /* CAN filter scale register                    */
    UW      rsv3;
    UW      FFA1R;      /* CAN filter FIFO assignment register          */
    UW      rsv4;
    UW      FA1R;       /* CAN filter activation register               */
    UW      rsv5[8];
    T_FILT  filter[28]; /* Filter bank registers                        */
                        /* 0-27 for connectivity line devices           */
                        /* 0-14 for other devices                       */
};

#define REG_bxCAN1  (*(volatile struct t_bxcan *)(APB1_BASE + 0x6400UL))
#define REG_bxCAN2  (*(volatile struct t_bxcan *)(APB1_BASE + 0x6800UL))


/* Clock recovery system (CRS) */

struct t_crs {
    UW      CR;         /* ccontrol register                            */
    UW      CFGR;       /* configuration register                       */
    UW      ISR;        /* interrupt and status register                */
    UW      ICR;        /* interrupt flag clear register                */
};

#define REG_CRS     (*(volatile struct t_crs *)(APB1_BASE + 0x6000UL))


/* Comparator */

struct t_comp {
    UW      CSR;        /* COMP comparator control and status register  */
};

#define REG_COMP1   (*(volatile struct t_comp *)(APB2_BASE + 0x0200UL))
#define REG_COMP2   (*(volatile struct t_comp *)(APB2_BASE + 0x0204UL))


/* USB on-the-go full-speed (OTG_FS) */

/* USB_OTG_Host_Channel_Specific_Registers */
typedef struct t_otg_fs_hc {
    UW      CHAR;       /* Host channel characteristics register        */
    UW      rsv0;
    UW      HCINT;      /* Host channel interrupt register              */
    UW      HCINTMSK;   /* Host channel interrupt mask register         */
    UW      TSIZ;       /* Host channel transfer size register          */
    UW      rsv1[3];
} T_OTG_FS_HC;

/* USB_OTG_IN_Endpoint-Specific_Register */
typedef struct t_otg_fs_diep {
    UW      CTL;        /* Device IN endpoint control register          */
    UW      rsv0;
    UW      IEPINT;     /* Device IN endpoint interrupt register        */
    UW      rsv1;
    UW      TSIZ;       /* Device IN endpoint transfer size register    */
    UW      rsv2;
    UW      TXFSTS;     /* Device IN endpoint transmit FIFO status register */
    UW      rsv3;
} T_OTG_FS_DIEP;

/* USB_OTG_OUT_Endpoint-Specific_Registers */
typedef struct t_otg_fs_doep {
    UW      CTL;        /* Device OUT endpoint control register         */
    UW      rsv0;
    UW      OEPINT;     /* Device OUT endpoint interrupt register       */
    UW      rsv1;
    UW      TSIZ;       /* Device OUT endpoint transfer size register   */
    UW      rsv2[3];
} T_OTG_FS_DOEP;

struct t_otg_fs {
    /* Global register */
    UW      GOTGCTL;    /* Control and status register                  */
    UW      GOTGINT;    /* Interrupt register                           */
    UW      GAHBCFG;    /* AHB configuration register                   */
    UW      GUSBCFG;    /* USB configuration register                   */
    UW      GRSTCTL;    /* Reset register                               */
    UW      GINTSTS;    /* Core interrupt register                      */
    UW      GINTMSK;    /* Interrupt mask register                      */
    UW      GRXSTSR;    /* Receive status read register                 */
    UW      GRXSTSPR;   /* Receive status pop register                  */
    UW      GRXFSIZ;    /* Receive FIFO size register                   */
    UW      HNPTXFSIZ_DIEPTXF0;
    UW      HNPTXSTS;   /* Non-periodic transmit FIFO/queue status register     */
    UW      rsv0[2];
    UW      GCCFG;      /* General core configuration register          */
    UW      CID;        /* Core ID register                             */
    UW      rsv1[5];
    UW      GLPMCFG;    /* LPM Register                                 */
    UW      GPWRDN;     /* Power Down Register                          */
    UW      GDFIFOCFG;  /* DFIFO Software Config Register               */
    UW      GADPCTL;    /* ADP Timer, Control and Status Register       */
    UW      rsv2[39];
    UW      HPTXFSIZ;   /* Host periodic transmit FIFO size register    */
    UW      DIEPTXF1;   /* Device IN endpoint transmit FIFO size register 1..3  */
    UW      DIEPTXF2;
    UW      DIEPTXF3;
    UW      DIEPTXF4;
    UW      DIEPTXF5;
    UW      rsv3[186];
    /* Host-mode register 0x400 */
    UW      HCFG;       /* Host configuration register                  */
    UW      HFIR;       /* Host frame interval register                 */
    UW      HFNUM;      /* Host frame number/frame time remaining register      */
    UW      rsv4[1];
    UW      HPTXSTS;    /* Host periodic transmit FIFO/queue status register    */
    UW      HAINT;      /* Host all channels interrupt register         */
    UW      HAINTMSK;   /* Host all channels interrupt mask register    */
    UW      rsv5[9];
    UW      HPRT;       /* Host port control and status register        */
    UW      rsv6[47];
    /* Host channel register 0x500 */
    T_OTG_FS_HC HostCh[12];
    UW      rsv7[96];
    /* Device-mode register 0x800 */
    UW      DCFG;       /* Device configuration register                */
    UW      DCTL;       /* Device control register                      */
    UW      DSTS;       /* Device status register                       */
    UW      rsv8;
    UW      DIEPMSK;    /* Device IN endpoint common interrupt mask register    */
    UW      DOEPMSK;    /* Device OUT endpoint common interrupt mask register   */
    UW      DAINT;      /* Device all endpoints interrupt register      */
    UW      DAINTMSK;   /* All endpoint interrupt mask register         */
    UW      rsv9;
    UW      rsv10;
    UW      DVBUSDIS;   /* Vbus discharge time register                 */
    UW      DVBUSPULSE; /* Vbus pulsing time register                   */
    UW      DTHRCTL;
    UW      DIEPEMPMSK;
    UW      DEACHINT;
    UW      DEACHMSK;
    UW      rsv11[50];
    T_OTG_FS_DIEP iep[4];   /* Device IN endpoint 0..3                  */
    UW      rsv12[96];
    T_OTG_FS_DOEP oep[4];   /* Device OUT endpoint 0..3                 */
    UW      rsv13[160];
    UW      PCGCCTL;    /* Power and clock gating control register      */
};

#define REG_OTG_FS  (*(volatile struct t_otg_fs *)0x50000000UL)


/* Voltage reference buffer (VREF)  */

struct t_vref {
    UW      CSR;        /* control and status register                  */
    UW      CCR;        /* calibration and control register,            */
};

#define REG_VREF    (*(volatile struct t_vref *)(APB2_BASE + 0x0030UL))


/**
  * Flexible memory controller (FMC) 
  */
struct t_fmc {
    /* Bank1    */
    UW      BCR1;       /* SRAM/NOR-Flash chip-select control register 1*/
    UW      BTR1;       /* SRAM/NOR-Flash chip-select timing register 1 */
    UW      BCR2;       /* SRAM/NOR-Flash chip-select control register 2*/
    UW      BTR2;       /* SRAM/NOR-Flash chip-select timing register 2 */
    UW      BCR3;       /* SRAM/NOR-Flash chip-select control register 3*/
    UW      BTR3;       /* SRAM/NOR-Flash chip-select timing register 3 */
    UW      BCR4;       /* SRAM/NOR-Flash chip-select control register 4*/
    UW      BTR4;       /* SRAM/NOR-Flash chip-select timing register 4 */
    UW      rsv0[16];
    /* Bank2    */
    UW      PCR2;       /* PC Card/NAND Flash control register 2..4     */
    UW      SR2;        /* FIFO status and interrupt register 2..4      */
    UW      PMEM2;      /* Common memory space timing register 2..4     */
    UW      PATT2;      /* Attribute memory space timing register 2..4  */
    UW      rsv1;
    UW      ECRR2;      /* ECC result register 2..3                     */
    UW      rsv2[2];
    /* Bank3    */
    UW      PCR3;
    UW      SR3;
    UW      PMEM3;
    UW      PATT3;
    UW      rsv3;
    UW      ECRR3;
    UW      rsv4[2];
    /* Bank 4   */
    UW      PCR4;
    UW      SR4;
    UW      PMEM4;
    UW      PATT4;
    UW      PIO4;       /* I/O space timing register 4                  */
    UW      rsv5[20];
    /* Timing   */
    UW      BWTR1;      /* SRAM/NOR-Flash chip-select timing register 1 */
    UW      rsv6;
    UW      BWTR2;      /* SRAM/NOR-Flash chip-select timing register 2 */
    UW      rsv7;
    UW      BWTR3;      /* SRAM/NOR-Flash chip-select timing register 3 */
    UW      rsv8;
    UW      BWTR4;      /* SRAM/NOR-Flash chip-select timing register 4 */
    UW      rsv9[8];
    /* SDRAM    */
    UW      SDCR1;      /* SDRAM Control registers 1                    */
    UW      SDCR2;      /* SDRAM Control registers 2                    */
    UW      SDTR1;      /* SDRAM Timing registers 1                     */
    UW      SDTR2;      /* SDRAM Timing registers 2                     */
    UW      SDCMR;      /* SDRAM Command Mode register                  */
    UW      SDRTR;      /* SDRAM Refresh Timer register                 */
    UW      SDSR;       /* SDRAM Status register                        */
};

#define REG_FMC     (*(volatile struct t_fmc *)0xA0000000UL)


/* Debug support (DBG) */

struct t_dbgmcu {
    UW      IDCODE;     /* ID codes and locking mechanism               */
    UW      CR;         /* Configuration register                       */
    UW      APB1FZR1;   /* APB1 freeze register1                        */
    UW      APB1FZR2;   /* APB1 freeze register2                        */
    UW      APB2FZ;     /* APB2 freeze register                         */
};

#define REG_DBGMCU  (*(volatile struct t_dbgmcu *)0xE0042000UL)


/* IRQ Interrupt vector */

#define IRQ_WWDG            (IRQ0)      /* Window Watchdog                  */
#define IRQ_PVD_PVM         (IRQ1)      /* PVD/PVM1/PVM2/PVM3/PVM4 through EXTI Line detection      */
#define IRQ_TAMP_STAMP      (IRQ2)      /* Tamper and TimeStamp interrupts through the EXTI line    */
#define IRQ_RTC_WKUP        (IRQ3)      /* RTC Wakeup interrupt through the EXTI line   */
#define IRQ_FLASH           (IRQ4)      /* Flash global                     */
#define IRQ_RCC             (IRQ5)      /* RCC global                       */
#define IRQ_EXTI0           (IRQ6)      /* EXTI Line0                       */
#define IRQ_EXTI1           (IRQ7)      /* EXTI Line1                       */
#define IRQ_EXTI2           (IRQ8)      /* EXTI Line2                       */
#define IRQ_EXTI3           (IRQ9)      /* EXTI Line3                       */
#define IRQ_EXTI4           (IRQ10)     /* EXTI Line4                       */
#define IRQ_DMA1_Channel1   (IRQ11)     /* DMA1 Channel 1 global            */
#define IRQ_DMA1_Channel2   (IRQ12)     /* DMA1 Channel 2 global            */
#define IRQ_DMA1_Channel3   (IRQ13)     /* DMA1 Channel 3 global            */
#define IRQ_DMA1_Channel4   (IRQ14)     /* DMA1 Channel 4 global            */
#define IRQ_DMA1_Channel5   (IRQ15)     /* DMA1 Channel 5 global            */
#define IRQ_DMA1_Channel6   (IRQ16)     /* DMA1 Channel 6 global            */
#define IRQ_DMA1_Channel7   (IRQ17)     /* DMA1 Channel 7 global            */
#define IRQ_ADC1_2          (IRQ18)     /* ADC1, ADC2 SAR global            */
#define IRQ_CAN1_TX         (IRQ19)     /* CAN TX                           */
#define IRQ_CAN1_RX0        (IRQ20)     /* CAN RX0                          */
#define IRQ_CAN1_RX1        (IRQ21)     /* CAN RX1                          */
#define IRQ_CAN1_SCE        (IRQ22)     /* CAN SCE                          */
#define IRQ_EXTI9_5         (IRQ23)     /* EXTI Line[9:5]                   */
#define IRQ_TIM1_BRK_TIM15  (IRQ24)     /* TIM1 Break interrupt and TIM15 global    */
#define IRQ_TIM1_UP_TIM16   (IRQ25)     /* TIM1 Update Interrupt and TIM16 global   */
#define IRQ_TIM1_TRG_COM_TIM17  (IRQ26) /* TIM1 Trigger and Commutation Interrupt and TIM17 global  */
#define IRQ_TIM1_CC         (IRQ27)     /* TIM1 Capture Compare             */
#define IRQ_TIM2            (IRQ28)     /* TIM2 global                      */
#define IRQ_TIM3            (IRQ29)     /* TIM3 global                      */
#define IRQ_TIM4            (IRQ30)     /* TIM4 global                      */
#define IRQ_I2C1_EV         (IRQ31)     /* I2C1 event                       */
#define IRQ_I2C1_ER         (IRQ32)     /* I2C1 error                       */
#define IRQ_I2C2_EV         (IRQ33)     /* I2C2 event                       */
#define IRQ_I2C2_ER         (IRQ34)     /* I2C2 error                       */
#define IRQ_SPI1            (IRQ35)     /* SPI1 global                      */
#define IRQ_SPI2            (IRQ36)     /* SPI2 global                      */
#define IRQ_USART1          (IRQ37)     /* USART1 global                    */
#define IRQ_USART2          (IRQ38)     /* USART2 global                    */
#define IRQ_USART3          (IRQ39)     /* USART3 global                    */
#define IRQ_EXTI15_10       (IRQ40)     /* EXTI Line[15:10]                 */
#define IRQ_RTCAlarm        (IRQ41)     /* RTC alarm through EXTI line      */
#define IRQ_DFSDM3          (IRQ42)     /* SD Filter 3 global               */
#define IRQ_TIM8_BRK        (IRQ43)     /* TIM8 Break Interrupt             */
#define IRQ_TIM8            (IRQ44)     /* TIM8 Update Interrupt            */
#define IRQ_TIM8_TRG_COM    (IRQ45)     /* TIM8 Trigger and Commutation     */
#define IRQ_TIM8_CC         (IRQ46)     /* TIM8 Capture Compare             */
#define IRQ_ADC3            (IRQ47)     /* ADC3 global                      */
#define IRQ_FMC             (IRQ48)     /* FMC global                       */
#define IRQ_SDMMC           (IRQ49)     /* SDMMC global                     */
#define IRQ_TIM5            (IRQ50)     /* TIM5 global                      */
#define IRQ_SPI3            (IRQ51)     /* SPI3 global                      */
#define IRQ_UART4           (IRQ52)     /* UART4 global                     */
#define IRQ_UART5           (IRQ53)     /* UART5 global                     */
#define IRQ_TIM6_DAC        (IRQ54)     /* TIM6 global and DAC1,2 error     */
#define IRQ_TIM7            (IRQ55)     /* TIM7 global                      */
#define IRQ_DMA2_Channel1   (IRQ56)     /* DMA2 Channel 1 global            */
#define IRQ_DMA2_Channel2   (IRQ57)     /* DMA2 Channel 2 global            */
#define IRQ_DMA2_Channel3   (IRQ58)     /* DMA2 Channel 3 global            */
#define IRQ_DMA2_Channel4   (IRQ59)     /* DMA2 Channel 4 global            */
#define IRQ_DMA2_Channel5   (IRQ60)     /* DMA2 Channel 5 global            */
#define IRQ_DFSDM0          (IRQ61)     /* SD Filter 0 global               */
#define IRQ_DFSDM1          (IRQ62)     /* SD Filter 1 global               */
#define IRQ_DFSDM2          (IRQ63)     /* SD Filter 2 global               */
#define IRQ_COMP            (IRQ64)     /* COMP1 and COMP2 Interrupts       */
#define IRQ_LPTIM1          (IRQ65)     /* LP TIM1                          */
#define IRQ_LPTIM2          (IRQ66)     /* LP TIM2                          */
#define IRQ_OTG_FS          (IRQ67)     /* USB On-The-Go FS global          */
#define IRQ_DMA2_Channel6   (IRQ68)     /* DMA2 Channel 6 global            */
#define IRQ_DMA2_Channel7   (IRQ69)     /* DMA2 Channel 7 global            */
#define IRQ_LPUART1         (IRQ70)     /* LP UART1 interrupt               */
#define IRQ_QUADSPI         (IRQ71)     /* Quad SPI global                  */
#define IRQ_I2C3_EV         (IRQ72)     /* I2C3 event                       */
#define IRQ_I2C3_ER         (IRQ73)     /* I2C3 error                       */
#define IRQ_SAI1            (IRQ74)     /* Serial Audio Interface 1 global  */
#define IRQ_SAI2            (IRQ75)     /* Serial Audio Interface 2 global  */
#define IRQ_SWPMI1          (IRQ76)     /* Serial Wire Interface 1 global   */
#define IRQ_TSC             (IRQ77)     /* Touch Sense Controller global    */
#define IRQ_LCDC            (IRQ78)     /* LCD global interrupt             */
#define IRQ_AES             (IRQ79)     /* AES global                       */
#define IRQ_RNG             (IRQ80)     /* RNG global                       */
#define IRQ_HASH            (IRQ80)     /* HASH global (same IRQ as RNG)    */
#define IRQ_FPU             (IRQ81)     /* FPU global                       */
#define IRQ_CRS             (IRQ82)     /* CRS interrupt                    */
#define IRQ_I2C4_EV         (IRQ83)     /* I2C4 event interrupt, wakeup through EXTI line 40 */
#define IRQ_I2C4_ER         (IRQ84)     /* I2C4 error interrupt             */
#define IRQ_DCMI            (IRQ85)     /* DCMI global interrupt            */
#define IRQ_CAN2_TX         (IRQ86)     /* CAN2 TX interrupt                */
#define IRQ_CAN2_RX0        (IRQ87)     /* CAN2 RX0 interrupt               */
#define IRQ_CAN2_RX1        (IRQ88)     /* CAN2 RX1 interrupt               */
#define IRQ_CAN2_SCE        (IRQ89)     /* CAN SCE interrupt                */
#define IRQ_DMA2D           (IRQ90)     /* DMA2D global interrupt           */

#endif  /* STM32L4XX_UC3_H_ */
