/**
 * @file    DDR_COM.c
 * @brief   MICRO C CUBE / COMPACT, DEVICE DRIVER
 *          Standard Communication Interface
 * @date    2021.03.04
 * @author  Copyright (c) 2008-2021, eForce Co., Ltd. All rights reserved.
 *
 ****************************************************************************
 * @par     History
 *          - rev 1.0 (2008.03.31)
 *            Initial version.
 *          - rev 1.3 (2016.12.04) y-kim
 *            Added the error check for parameters.
 *            Fixed the IPA warnings.
 *          - rev 1.4 (2017.02.21) s-tamura
 *            Deleted the error check for parameters in ini_com function.
 *          - rev 1.5 (2021.03.04) sugawara
 *            Added the error check for *scnt and *rcnt.
 ****************************************************************************
 */
#include "kernel.h"
#include "DDR_COM.h"


extern ER ctr_dev(ID DevID, ID FuncID, VP_INT ControlData);


ER ini_com(ID DevID, T_COM_SMOD const *pk_SerialMode)
{

    return ctr_dev(DevID, TA_COM_INI, (VP_INT)pk_SerialMode);
}

ER ref_com(ID DevID, T_COM_REF *pk_SerialRef)
{
    if (pk_SerialRef == 0) {
        return E_PAR;
    }
    return ctr_dev(DevID, TA_COM_REF, (VP_INT)pk_SerialRef);
}

ER ctr_com(ID DevID, UH command, TMO tmout)
{
    T_COM_CTR SerialData;

    SerialData.command = command;
    SerialData.time = tmout;
    return ctr_dev(DevID, TA_COM_CTR, (VP_INT)&SerialData);
}

ER putc_com(ID DevID, VB chr, TMO tmout)
{
    T_COM_SND TransmiteData;

    TransmiteData.tbuf = &chr;
    TransmiteData.tcnt = 1U;
    TransmiteData.time = tmout;
    return ctr_dev(DevID, TA_COM_PUTS, (VP_INT)&TransmiteData);
}

ER puts_com(ID DevID, VB const *schr, UINT *scnt, TMO tmout)
{
    T_COM_SND TransmiteData;
    ER ercd;

    if (schr == 0) {
        return E_PAR;
    }
    if (scnt == 0) {
        return E_PAR;
    }
    if (*scnt == 0U) {
        return E_PAR;
    }

    TransmiteData.tbuf = schr;
    TransmiteData.tcnt = (UH)*scnt;
    TransmiteData.time = tmout;
    ercd = ctr_dev(DevID, TA_COM_PUTS, (VP_INT)&TransmiteData);
    *scnt -= (UINT)TransmiteData.tcnt;
    return ercd;
}

ER getc_com(ID DevID, VB *rbuf, UB *sbuf, TMO tmout)
{
    T_COM_RCV ReceiveData;

    if (rbuf == 0) {
        return E_PAR;
    }

    ReceiveData.rbuf = rbuf;
    ReceiveData.sbuf = sbuf;
    ReceiveData.rcnt = 1U;
    ReceiveData.eos = 0;
    ReceiveData.time = tmout;
    return ctr_dev(DevID, TA_COM_GETS, (VP_INT)&ReceiveData);
}

ER gets_com(ID DevID, VB *rbuf, UB *sbuf, INT eos, UINT *rcnt, TMO tmout)
{
    T_COM_RCV ReceiveData;
    T_COM_EOS EndOfStr;
    ER ercd;

    if (rbuf == 0) {
        return E_PAR;
    }
    if (rcnt == 0) {
        return E_PAR;
    }
    if (*rcnt == 0U) {
        return E_PAR;
    }

    if (eos >= 0) {
        EndOfStr.chr[0] = (VB)eos;
        EndOfStr.flg[0] = 1U;
        EndOfStr.flg[1] = 0U;
        EndOfStr.flg[2] = 0U;
        EndOfStr.flg[3] = 0U;
    }

    ReceiveData.rbuf = rbuf;
    ReceiveData.sbuf = sbuf;
    ReceiveData.rcnt = (UH)*rcnt;
    ReceiveData.eos = (eos < 0) ? 0 : &EndOfStr;
    ReceiveData.time = tmout;
    ercd = ctr_dev(DevID, TA_COM_GETS, (VP_INT)&ReceiveData);
    *rcnt -= (UINT)ReceiveData.rcnt;
    return ercd;
}
