/**
 * @file    sample.c
 * @brief   NFC Sample program for STM32L496G-DISCO
 * @date    2021.12.02
 * @author  Copyright (c) 2021, eForce Co.,Ltd.  All rights reserved.
 ****************************************************************************
 */

/* {{UC3_INCLUDE */
#include "kernel.h"
#include "kernel_id.h"
#include "hw_dep.h"
/* }}UC3_INCLUDE */

#include <string.h> // strlen

#include "RCS620S_UC3.h"
#include "log.h"

/* Private typedef -----------------------------------------------------------*/

/* Private define ------------------------------------------------------------*/
#define LD2_GPIO    (REG_GPIOB.ODR)
#define LD3_GPIO    (REG_GPIOA.ODR)
#define LD2_PIN     ((UH)0x2000U)  /* PB13 selected  */
#define LD3_PIN     ((UH)0x0020U)  /* PA5  selected */

/* Private macro -------------------------------------------------------------*/

/* Private variables ---------------------------------------------------------*/

/* Private function prototypes -----------------------------------------------*/


extern void cortex_m_init_peripheral(void);
extern void init_peripheral(void);

/* {{UC3_CODE */

#define ASSERT(exp) do{if(!(exp)) int_abort();}while(0)

/******************************************************************
    Default interrupt handler
     Reached when the interrupt is accepted even though it is not the generation of an
     interrupt service routine also define the interrupt handler.
 ******************************************************************/

void int_abort(void)
{
    for(;;);
}

/*******************************
        TaskNfc
 *******************************/
/* {{UC3_TASK(TaskNfc) */
void TaskNfc(VP_INT exinf)
{
    int ret = 0;

    // UARTドライバ初期化とRC-S620/S初期化シーケンスの実行
    ret = RCS620S_initDevice(ID_USART_NFC);
    ASSERT(ret);
    log_str("TaskNfc - NFC reader initialized.\r\n");

    // Passive communication mode の通信でTarget IDの取得（Targetの捕捉）を行うためのコマンド
    for (;;) {
        ret = RCS620S_polling(0xFFFF);
        if (ret) {
            // Targetを捕捉できた場合は、読み取ったIDMとPMMを出力
            log_str("TaskNfc - IC card detected.\r\n");
            log_str("  IDM: ");
            log_idm_pmm(idm_);
            log_str("  PMM: ");
            log_idm_pmm(pmm_);
            dly_tsk(5000); // 5秒後にTargetの捕捉を再開
        } else {
            // Targetを捕捉できなかった場合は、1秒後にリトライ
            log_str("TaskNfc - waiting IC card...\r\n");
            dly_tsk(1000);
        }
    }
}
/* }}UC3_TASK */

/*******************************
        TaskLog
 *******************************/
/* {{UC3_TASK(TaskLog) */
void TaskLog(VP_INT exinf)
{
    T_MSGBLK *blk;

    const VB         demo_str[] = "\r\n\teForce uC3/Compact - NFC Sample Program\r\n";
    const T_COM_SMOD com_ini = { 115200U, BLEN8, PAR_NONE, SBIT1, FLW_NONE };

    UINT txcnt;

    (void)ini_com(ID_USART_LOG, &com_ini);
    (void)ctr_com(ID_USART_LOG, STA_COM, 0);

    // タスク起動時にタイトルメッセージを表示
    txcnt = strlen(demo_str);
    (void)puts_com(ID_USART_LOG, (VB *)demo_str, &txcnt, TMO_FEVR);

    for (;;) {
        // メッセージボックスの文字列をシリアルに出力
        (void)rcv_mbx(ID_MBX_LOG, (T_MSG **)&blk);
        (void)puts_com(ID_USART_LOG, blk->buf, &blk->cnt, TMO_FEVR);
        (void)rel_mpf(ID_MPF_LOG, (VP)blk);
    }
}
/* }}UC3_TASK */

/*******************************
        TaskLed
 *******************************/
/* {{UC3_TASK(TaskLed) */
void TaskLed(VP_INT exinf)
{
    for (;;) {
        // 1秒間隔でLEDを反転
        LD2_GPIO ^= LD2_PIN;
        LD3_GPIO ^= LD3_PIN;
        dly_tsk(1000);
    }
}
/* }}UC3_TASK */


/* }}UC3_CODE */


/*******************************
        Main entry
 *******************************/

int main(void)
{
    ER ret = E_OK;

    /* Initialize hardware */
    cortex_m_init_peripheral();
    init_peripheral();

    /* Generate system and start */
    ret = start_uC3();

    return ret;
}

/* end */
