/**
 * @file    DDR_STM32L4_USART0.c
 * @brief   Serial Interface STMicroelectronics STM32L4/L4+ USART
 * @date    2020.08.31
 * @author  Copyright (c) 2015-2020, eForce Co.,Ltd.  All rights reserved.
 *
 ****************************************************************************
 * @par     History
 *          - rev 1.0 (2015.02.05) i-cho
 *            Initial version.
 *          - rev 1.1 (2015.02.26) t-yokota
 *            add low power mode.
 *          - rev 1.2 (2017.01.25) i-cho
 *            Fixed the IPA warnings.
 *          - rev 1.3 (2020.08.31) sugawara
 *            Add FIFO mode for STM32L4+ series.
 *            Add UART4, UART5.
 ****************************************************************************
 */

#include "kernel.h"
#include "kernel_id.h"

#include "DDR_COM.h"
#include "DDR_STM32L4_USART0_cfg.h"
#include "DDR_STM32L4_USART0.h"

#include <string.h>

/**
 * local functions
 */
static BOOL _ddr_stm32l4_usart0_check_chr(const T_COM_RCV *ReceiveData, VB chr, UB sts);
static UB _ddr_stm32l4_usart0_getssr(UW ssr, VB chr, volatile struct t_usart *port);
static void _ddr_stm32l4_usart0_tci(T_STM32L4_USART0_MNG *pk_USART0mng);
static BOOL _ddr_stm32l4_usart0_copy(T_STM32L4_USART0_DMNG *data, T_STM32L4_USART0_CMNG const *cdata, T_COM_SND *TransmiteData);
static void _ddr_stm32l4_usart0_chk_rxoff(T_STM32L4_USART0_DMNG *data, T_STM32L4_USART0_CMNG const *cdata, volatile struct t_usart *port);
static void _ddr_stm32l4_usart0_chk_rxon(T_STM32L4_USART0_DMNG *data, T_STM32L4_USART0_CMNG const *cdata, volatile struct t_usart *port);
static void _ddr_stm32l4_usart0_send_local_buf(T_STM32L4_USART0_MNG const *pk_USART0mng, ID semid);
static void _ddr_stm32l4_usart0_send_drv_buf(T_STM32L4_USART0_MNG const *pk_USART0mng);
static void _ddr_stm32l4_usart0_eri(T_STM32L4_USART0_MNG *pk_USART0mng);
static void _ddr_stm32l4_usart0_rxi(T_STM32L4_USART0_MNG *pk_USART0mng);
static void _ddr_stm32l4_usart0_txi(T_STM32L4_USART0_MNG *pk_USART0mng);
static BOOL _ddr_stm32l4_usart0_recv_strings(T_STM32L4_USART0_DMNG *data, T_STM32L4_USART0_CMNG const *cdata, T_COM_RCV *ReceiveData, volatile struct t_usart *port);
static ER _ddr_stm32l4_usart0_snd(T_COM_SND *TransmiteData, T_STM32L4_USART0_MNG const *pk_USART0mng);
static ER _ddr_stm32l4_usart0_rcv(T_COM_RCV *ReceiveData,  T_STM32L4_USART0_MNG const *pk_USART0mng);
static ER _ddr_stm32l4_usart0_cln_tx_buf(T_STM32L4_USART0_MNG const *pk_USART0mng, TMO tim);
static ER _ddr_stm32l4_usart0_rst_tx_buf(T_STM32L4_USART0_MNG const *pk_USART0mng);
static ER _ddr_stm32l4_usart0_rst_rx_buf(T_STM32L4_USART0_MNG const *pk_USART0mng);
static ER _ddr_stm32l4_usart0_dis_send(T_STM32L4_USART0_MNG const *pk_USART0mng);
static ER _ddr_stm32l4_usart0_dis_rcv(T_STM32L4_USART0_MNG const *pk_USART0mng);
static ER _ddr_stm32l4_usart0_ena_rcv(T_STM32L4_USART0_MNG const *pk_USART0mng);
static ER _ddr_stm32l4_usart0_snd_brk(T_STM32L4_USART0_MNG const *pk_USART0mng, TMO tim);
static ER _ddr_stm32l4_usart0_lock_trans(T_STM32L4_USART0_MNG const *pk_USART0mng);
static ER _ddr_stm32l4_usart0_lock_recv(T_STM32L4_USART0_MNG const *pk_USART0mng);
static ER _ddr_stm32l4_usart0_unlock_trans(T_STM32L4_USART0_MNG const *pk_USART0mng);
static ER _ddr_stm32l4_usart0_unlock_recv(T_STM32L4_USART0_MNG const *pk_USART0mng);
static ER _ddr_stm32l4_usart0_ctr(T_COM_CTR const *pk_SerialData,  T_STM32L4_USART0_MNG const *pk_USART0mng);
static ER _ddr_stm32l4_usart0_ref(T_COM_REF *pk_SerialRef,  T_STM32L4_USART0_MNG const *pk_USART0mng);
static ER _ddr_stm32l4_usart0_ini(T_COM_SMOD const *pk_SerialMode,  T_STM32L4_USART0_MNG const *pk_USART0mng);


#define T_COM_NOISE (T_COM_EROVB)

/**
 * USART1 Configurations
 */
#ifdef USART_1
#ifndef USART1SEL
#define USART1SEL   0
#endif
#if (TXBUF_SZ1==0)
#define _ddr_stm32l4_usart0_tbuf1    0
#else
VB  _ddr_stm32l4_usart0_tbuf1[TXBUF_SZ1];
#endif
#if (RXBUF_SZ1==0)
#define _ddr_stm32l4_usart0_rbuf1    0
#define _ddr_stm32l4_usart0_sbuf1    0
#else
VB  _ddr_stm32l4_usart0_rbuf1[RXBUF_SZ1];
UB  _ddr_stm32l4_usart0_sbuf1[RXBUF_SZ1];
#endif
#ifndef CTSPORT1
#define CTSPORT1    (*(volatile UW *)0)
#endif
#ifndef CTSBIT1
#define CTSBIT1     0
#endif
#ifndef FIFO_EN1
#define FIFO_EN1    0
#endif
#ifndef FIFO_TXFT1
#define FIFO_TXFT1  0
#endif
#ifndef FIFO_RXFT1
#define FIFO_RXFT1  0
#endif
T_STM32L4_USART0_DMNG _ddr_stm32l4_usart0_data1;
T_STM32L4_USART0_CMNG const _ddr_stm32l4_usart0_const1 = {_ddr_stm32l4_usart0_tbuf1, _ddr_stm32l4_usart0_rbuf1,
            _ddr_stm32l4_usart0_sbuf1, TXBUF_SZ1, RXBUF_SZ1, XOFF_SZ1, XON_SZ1, TXSEM1, RXSEM1,
#ifdef LPMODE
            USART1SEL,
#else
            USART1_CLK,
#endif
            &REG_RCC.APB2ENR, &REG_RCC.APB2RSTR, 0x00004000, &CTSPORT1, CTSBIT1, FIFO_EN1, FIFO_TXFT1, FIFO_RXFT1};
T_STM32L4_USART0_MNG const _ddr_stm32l4_usart0_mng1 = {(volatile struct t_usart *)&REG_USART1,
            &_ddr_stm32l4_usart0_const1, &_ddr_stm32l4_usart0_data1};
#endif

/**
 * USART2 Configurations
 */
#ifdef USART_2
#ifndef USART2SEL
#define USART2SEL   0
#endif
#if (TXBUF_SZ2==0)
#define _ddr_stm32l4_usart0_tbuf2    0
#else
VB  _ddr_stm32l4_usart0_tbuf2[TXBUF_SZ2];
#endif
#if (RXBUF_SZ2==0)
#define _ddr_stm32l4_usart0_rbuf2    0
#define _ddr_stm32l4_usart0_sbuf2    0
#else
VB  _ddr_stm32l4_usart0_rbuf2[RXBUF_SZ2];
UB  _ddr_stm32l4_usart0_sbuf2[RXBUF_SZ2];
#endif
#ifndef CTSPORT2
#define CTSPORT2    (*(volatile UW *)0)
#endif
#ifndef CTSBIT2
#define CTSBIT2     0
#endif
#ifndef FIFO_EN2
#define FIFO_EN2    0
#endif
#ifndef FIFO_TXFT2
#define FIFO_TXFT2  0
#endif
#ifndef FIFO_RXFT2
#define FIFO_RXFT2  0
#endif
T_STM32L4_USART0_DMNG _ddr_stm32l4_usart0_data2;
T_STM32L4_USART0_CMNG const _ddr_stm32l4_usart0_const2 = {_ddr_stm32l4_usart0_tbuf2, _ddr_stm32l4_usart0_rbuf2,
            _ddr_stm32l4_usart0_sbuf2, TXBUF_SZ2, RXBUF_SZ2, XOFF_SZ2, XON_SZ2, TXSEM2, RXSEM2,
#ifdef LPMODE
            USART2SEL,
#else
            USART2_CLK,
#endif
            &REG_RCC.APB1ENR1, &REG_RCC.APB1RSTR1, 0x00020000, &CTSPORT2, CTSBIT2, FIFO_EN2, FIFO_TXFT2, FIFO_RXFT2};
T_STM32L4_USART0_MNG const _ddr_stm32l4_usart0_mng2 = {(volatile struct t_usart *)&REG_USART2,
            &_ddr_stm32l4_usart0_const2, &_ddr_stm32l4_usart0_data2};
#endif

/**
 * USART3 Configurations
 */
#ifdef USART_3
#ifndef USART3SEL
#define USART3SEL   0
#endif
#if (TXBUF_SZ3==0)
#define _ddr_stm32l4_usart0_tbuf3    0
#else
VB  _ddr_stm32l4_usart0_tbuf3[TXBUF_SZ3];
#endif
#if (RXBUF_SZ3==0)
#define _ddr_stm32l4_usart0_rbuf3    0
#define _ddr_stm32l4_usart0_sbuf3    0
#else
VB  _ddr_stm32l4_usart0_rbuf3[RXBUF_SZ3];
UB  _ddr_stm32l4_usart0_sbuf3[RXBUF_SZ3];
#endif
#ifndef CTSPORT3
#define CTSPORT3    (*(volatile UW *)0)
#endif
#ifndef CTSBIT3
#define CTSBIT3     0
#endif
#ifndef FIFO_EN3
#define FIFO_EN3    0
#endif
#ifndef FIFO_TXFT3
#define FIFO_TXFT3  0
#endif
#ifndef FIFO_RXFT3
#define FIFO_RXFT3  0
#endif
T_STM32L4_USART0_DMNG _ddr_stm32l4_usart0_data3;
T_STM32L4_USART0_CMNG const _ddr_stm32l4_usart0_const3 = {_ddr_stm32l4_usart0_tbuf3, _ddr_stm32l4_usart0_rbuf3,
            _ddr_stm32l4_usart0_sbuf3, TXBUF_SZ3, RXBUF_SZ3, XOFF_SZ3, XON_SZ3, TXSEM3, RXSEM3,
#ifdef LPMODE
            USART3SEL,
#else
            USART3_CLK,
#endif
            &REG_RCC.APB1ENR1, &REG_RCC.APB1RSTR1, 0x00040000, &CTSPORT3, CTSBIT3, FIFO_EN3, FIFO_TXFT3, FIFO_RXFT3};
T_STM32L4_USART0_MNG const _ddr_stm32l4_usart0_mng3 = {(volatile struct t_usart *)&REG_USART3,
            &_ddr_stm32l4_usart0_const3, &_ddr_stm32l4_usart0_data3};
#endif

/**
 * USART4 Configurations
 */
#ifdef USART_4
#ifndef USART4SEL
#define USART4SEL   0
#endif
#if (TXBUF_SZ4==0)
#define _ddr_stm32l4_usart0_tbuf4    0
#else
VB  _ddr_stm32l4_usart0_tbuf4[TXBUF_SZ4];
#endif
#if (RXBUF_SZ4==0)
#define _ddr_stm32l4_usart0_rbuf4    0
#define _ddr_stm32l4_usart0_sbuf4    0
#else
VB  _ddr_stm32l4_usart0_rbuf4[RXBUF_SZ4];
UB  _ddr_stm32l4_usart0_sbuf4[RXBUF_SZ4];
#endif
#ifndef CTSPORT4
#define CTSPORT4    (*(volatile UW *)0)
#endif
#ifndef CTSBIT4
#define CTSBIT4     0
#endif
#ifndef FIFO_EN4
#define FIFO_EN4    0
#endif
#ifndef FIFO_TXFT4
#define FIFO_TXFT4  0
#endif
#ifndef FIFO_RXFT4
#define FIFO_RXFT4  0
#endif
T_STM32L4_USART0_DMNG _ddr_stm32l4_usart0_data4;
T_STM32L4_USART0_CMNG const _ddr_stm32l4_usart0_const4 = {_ddr_stm32l4_usart0_tbuf4, _ddr_stm32l4_usart0_rbuf4,
            _ddr_stm32l4_usart0_sbuf4, TXBUF_SZ4, RXBUF_SZ4, XOFF_SZ4, XON_SZ4, TXSEM4, RXSEM4,
#ifdef LPMODE
            USART4SEL,
#else
            USART4_CLK,
#endif
            &REG_RCC.APB1ENR1, &REG_RCC.APB1RSTR1, 0x00080000, &CTSPORT4, CTSBIT4, FIFO_EN4, FIFO_TXFT4, FIFO_RXFT4};
T_STM32L4_USART0_MNG const _ddr_stm32l4_usart0_mng4 = {(volatile struct t_usart *)&REG_UART4,
            &_ddr_stm32l4_usart0_const4, &_ddr_stm32l4_usart0_data4};
#endif

/**
 * USART5 Configurations
 */
#ifdef USART_5
#ifndef USART5SEL
#define USART5SEL   0
#endif
#if (TXBUF_SZ5==0)
#define _ddr_stm32l4_usart0_tbuf5    0
#else
VB  _ddr_stm32l4_usart0_tbuf5[TXBUF_SZ5];
#endif
#if (RXBUF_SZ5==0)
#define _ddr_stm32l4_usart0_rbuf5    0
#define _ddr_stm32l4_usart0_sbuf5    0
#else
VB  _ddr_stm32l4_usart0_rbuf5[RXBUF_SZ5];
UB  _ddr_stm32l4_usart0_sbuf5[RXBUF_SZ5];
#endif
#ifndef CTSPORT5
#define CTSPORT5    (*(volatile UW *)0)
#endif
#ifndef CTSBIT5
#define CTSBIT5     0
#endif
#ifndef FIFO_EN5
#define FIFO_EN5    0
#endif
#ifndef FIFO_TXFT5
#define FIFO_TXFT5  0
#endif
#ifndef FIFO_RXFT5
#define FIFO_RXFT5  0
#endif
T_STM32L4_USART0_DMNG _ddr_stm32l4_usart0_data5;
T_STM32L4_USART0_CMNG const _ddr_stm32l4_usart0_const5 = {_ddr_stm32l4_usart0_tbuf5, _ddr_stm32l4_usart0_rbuf5,
            _ddr_stm32l4_usart0_sbuf5, TXBUF_SZ5, RXBUF_SZ5, XOFF_SZ5, XON_SZ5, TXSEM5, RXSEM5,
#ifdef LPMODE
            USART5SEL,
#else
            USART5_CLK,
#endif
            &REG_RCC.APB1ENR1, &REG_RCC.APB1RSTR1, 0x00100000, &CTSPORT5, CTSBIT5, FIFO_EN5, FIFO_TXFT5, FIFO_RXFT5};
T_STM32L4_USART0_MNG const _ddr_stm32l4_usart0_mng5 = {(volatile struct t_usart *)&REG_UART5,
            &_ddr_stm32l4_usart0_const5, &_ddr_stm32l4_usart0_data5};
#endif

/***************************************
        Initialize Deivce Driver
 ***************************************/

void _ddr_stm32l4_usart0_init(void)
{
#ifdef USART_1
    memset(&_ddr_stm32l4_usart0_data1, 0x00, sizeof(T_STM32L4_USART0_DMNG));
    (void)loc_cpu();
    REG_RCC.CCIPR    &= ~0x3U;
    REG_RCC.CCIPR    |= USART1SEL;
    REG_RCC.APB2ENR  |= 0x00004000U;
    REG_RCC.APB2RSTR  = 0x00004000U;
    REG_RCC.APB2RSTR  = 0x00000000U;
    REG_USART1.CR1 = 0U;
    (void)vset_ipl(IRQ_USART1, PROPRITY1);
    (void)ena_int(IRQ_USART1);
    (void)unl_cpu();
#endif
#ifdef USART_2
    memset(&_ddr_stm32l4_usart0_data2, 0x00, sizeof(T_STM32L4_USART0_DMNG));
    (void)loc_cpu();
    REG_RCC.CCIPR     &= ~((UW)0x3U << 2);
    REG_RCC.CCIPR     |= ((UW)USART2SEL << 2);
    REG_RCC.APB1ENR1  |= 0x00020000U;
    REG_RCC.APB1RSTR1  = 0x00020000U;
    REG_RCC.APB1RSTR1  = 0x00000000U;
    REG_USART2.CR1 = 0U;
    (void)vset_ipl(IRQ_USART2, PROPRITY2);
    (void)ena_int(IRQ_USART2);
    (void)unl_cpu();
#endif
#ifdef USART_3
    memset(&_ddr_stm32l4_usart0_data3, 0x00, sizeof(T_STM32L4_USART0_DMNG));
    (void)loc_cpu();
    REG_RCC.CCIPR     &= ~((UW)0x3U << 4);
    REG_RCC.CCIPR     |= ((UW)USART3SEL << 4);
    REG_RCC.APB1ENR1  |= 0x00040000U;
    REG_RCC.APB1RSTR1  = 0x00040000U;
    REG_RCC.APB1RSTR1  = 0x00000000U;
    REG_USART3.CR1 = 0U;
    (void)vset_ipl(IRQ_USART3, PROPRITY3);
    (void)ena_int(IRQ_USART3);
    (void)unl_cpu();
#endif
#ifdef USART_4
    memset(&_ddr_stm32l4_usart0_data4, 0x00, sizeof(T_STM32L4_USART0_DMNG));
    (void)loc_cpu();
    REG_RCC.CCIPR     &= ~((UW)0x3U << 6);
    REG_RCC.CCIPR     |= ((UW)USART4SEL << 6);
    REG_RCC.APB1ENR1  |= 0x00080000U;
    REG_RCC.APB1RSTR1  = 0x00080000U;
    REG_RCC.APB1RSTR1  = 0x00000000U;
    REG_UART4.CR1 = 0U;
    (void)vset_ipl(IRQ_UART4, PROPRITY4);
    (void)ena_int(IRQ_UART4);
    (void)unl_cpu();
#endif
#ifdef USART_5
    memset(&_ddr_stm32l4_usart0_data5, 0x00, sizeof(T_STM32L4_USART0_DMNG));
    (void)loc_cpu();
    REG_RCC.CCIPR     &= ~((UW)0x3U << 8);
    REG_RCC.CCIPR     |= ((UW)USART5SEL << 8);
    REG_RCC.APB1ENR1  |= 0x00100000U;
    REG_RCC.APB1RSTR1  = 0x00100000U;
    REG_RCC.APB1RSTR1  = 0x00000000U;
    REG_UART5.CR1 = 0U;
    (void)vset_ipl(IRQ_UART5, PROPRITY5);
    (void)ena_int(IRQ_UART5);
    (void)unl_cpu();
#endif
}

/***************************************
        USARTMImF
 ***************************************/

static BOOL _ddr_stm32l4_usart0_check_chr(const T_COM_RCV *ReceiveData, VB chr, UB sts)
{
    BOOL flag = FALSE;

    if ((sts & (T_COM_EROR|T_COM_ERP|T_COM_ERF|T_COM_BRK)) != 0U) {
        flag = TRUE;
    } else if (ReceiveData->rcnt == 0U) {
        flag = TRUE;
    } else {
        T_COM_EOS * eos = ReceiveData->eos;
        if (eos != 0U) {
            if ((eos->flg[0] != 0U) && (eos->chr[0] == chr)) {
                flag = TRUE;
            } else if ((eos->flg[1] != 0U) && (eos->chr[1] == chr)) {
                flag = TRUE;
            } else if ((eos->flg[2] != 0U) && (eos->chr[2] == chr)) {
                flag = TRUE;
            } else if ((eos->flg[3] != 0U) && (eos->chr[3] == chr)) {
                flag = TRUE;
            } else {
                /* do nothing */
            }
        }
    }
    return flag;
}

/***************************************
        USARTMG[Xe[^X
 ***************************************/

static UB _ddr_stm32l4_usart0_getssr(UW ssr, VB chr, volatile struct t_usart *port)
{
    UB sts = 0U;
    
    if ((ssr & 0x00000001U) != 0U) {
        port->ICR |= 0x00000001U;
        sts |= T_COM_ERP;
    }
    if ((ssr & 0x00000002U) != 0U) {
        port->ICR |= 0x00000002U;
        if (chr == '\0') {
            sts |= T_COM_BRK;
        } else {
            sts |= T_COM_ERF;
        }
    }
    if ((ssr & 0x00000004U) != 0U) {
        port->ICR |= 0x00000004U;
        sts |= T_COM_NOISE;
    }
    if ((ssr & 0x00000008U) != 0U) {
        port->ICR |= 0x00000008U;
        sts |= T_COM_EROR;
    }
    return sts;
}

/***********************************
        USARTMIݏ
 ***********************************/

static void _ddr_stm32l4_usart0_tci(T_STM32L4_USART0_MNG *pk_USART0mng)
{
    volatile struct t_usart *port;
    T_STM32L4_USART0_CMNG const *cdata;
    T_STM32L4_USART0_DMNG *data;

    port = pk_USART0mng->port;
    data = pk_USART0mng->data;
    if (((port->ISR & 0x00000040U) != 0U) &&
        ( data->tcnt             == 0U)) {
        port->CR1 &= ~0x00000040U;
        if (data->status.bit.cln_wait == 1U) {
            data->status.bit.cln_wait = 0U;
            cdata = pk_USART0mng->cdata;
            (void)isig_sem((ID)cdata->tsemid);
        }
    }
}

/*******************************
        obt@Rs[
 *******************************/

static BOOL _ddr_stm32l4_usart0_copy(T_STM32L4_USART0_DMNG *data, T_STM32L4_USART0_CMNG const *cdata, T_COM_SND *TransmiteData)
{
    while (TransmiteData->tcnt != 0U) {
        if (data->tcnt < cdata->tsize) {
            cdata->tbuf[data->sndp++] = *TransmiteData->tbuf++;
            TransmiteData->tcnt--;
            if (data->sndp >= cdata->tsize) {
                data->sndp = 0U;
            }
            data->tcnt++;
        } else {
            break;
        }
    }
    return (TransmiteData->tcnt == 0U) ? TRUE : FALSE;
}

/***********************************
        MXOFF̃`FbN
 ***********************************/

static void _ddr_stm32l4_usart0_chk_rxoff(T_STM32L4_USART0_DMNG *data, T_STM32L4_USART0_CMNG const *cdata, volatile struct t_usart *port)
{
    if ((data->status.bit.rx_xoff == 0U) &&
        (data->rcnt >= cdata->xoff_size)) {
        data->status.bit.rx_xoff = 1U;
        if ((port->ISR & 0x00000080U) != 0U) {
            port->TDR = XOFF;
        } else {
            data->status.bit.req_xon_xoff = 1U;
        }
    }
}

/***********************************
        MXOÑ`FbN
 ***********************************/

static void _ddr_stm32l4_usart0_chk_rxon(T_STM32L4_USART0_DMNG *data, T_STM32L4_USART0_CMNG const *cdata, volatile struct t_usart *port)
{
    if ((data->status.bit.rx_xoff == 1U) &&
        (data->rcnt <= cdata->xon_size)) {
        data->status.bit.rx_xoff = 0U;
        if ((port->ISR & 0x00000080U) != 0U) {
            port->TDR = XON;
        } else {
            data->status.bit.req_xon_xoff = 1U;
        }
    }
}

/***********************************************
        J[Mobt@̑M
 ***********************************************/

static void _ddr_stm32l4_usart0_send_local_buf(T_STM32L4_USART0_MNG const *pk_USART0mng, ID semid)
{
    volatile struct t_usart *port;
    T_STM32L4_USART0_CMNG const *cdata;
    T_STM32L4_USART0_DMNG *data;

    data = pk_USART0mng->data;
    cdata = pk_USART0mng->cdata;
    port = pk_USART0mng->port;

    do {
        port->TDR = (UW)*data->SndData->tbuf++;
        if (--data->SndData->tcnt == 0U) {
            data->SndData = 0;
            (void)isig_sem(semid);
            break;
        }
    } while ((cdata->fifo_en) && (port->ISR & 0x00000080U));
}

/***********************************************
        Mobt@̑M
 ***********************************************/

static void _ddr_stm32l4_usart0_send_drv_buf(T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    volatile struct t_usart *port;
    T_STM32L4_USART0_CMNG const *cdata;
    T_STM32L4_USART0_DMNG *data;
    UH sndp;

    data = pk_USART0mng->data;
    cdata = pk_USART0mng->cdata;
    port = pk_USART0mng->port;

    sndp = (UH)(data->sndp - data->tcnt);
    if (data->tcnt > data->sndp) {
        sndp = (UH)(sndp + cdata->tsize);
    }

    do {
        port->TDR = (UW)cdata->tbuf[sndp];
        if (++sndp >= cdata->tsize)
            sndp = 0U;
        if (--data->tcnt == 0U)
            break;
    } while ((cdata->fifo_en) && (port->ISR & 0x00000080U));

    if (data->SndData != 0) {
        if (_ddr_stm32l4_usart0_copy(data, cdata, data->SndData) != FALSE) {
            data->SndData = 0;
            (void)isig_sem((ID)cdata->tsemid);
        }
    }
}

/***************************************
        USARTMG[ݏ
 ***************************************/

static void _ddr_stm32l4_usart0_eri(T_STM32L4_USART0_MNG *pk_USART0mng)
{
    volatile struct t_usart *port;
    T_STM32L4_USART0_CMNG const *cdata;
    T_STM32L4_USART0_DMNG *data;
    UW ssr;
    UH rcvp;
    UB sts;
    VB chr;
    BOOL iflag = FALSE;

    data = pk_USART0mng->data;
    cdata = pk_USART0mng->cdata;
    port = pk_USART0mng->port;

    ssr = port->ISR;

    if (cdata->fifo_en) {
        if ((ssr & 0x0000000FU) && (port->CR3 & 0x10000000U)) {
            iflag = TRUE;
        }
    } else {
        if ((ssr & 0x0000000FU) && (port->CR1 & 0x00000020U)) {
            iflag = TRUE;
        }
    }

    if (iflag) {
        if ((data->RcvData != 0) ||
            (data->rcnt < cdata->rsize)) {
            chr = (VB)(port->RDR & data->mask);
            sts = _ddr_stm32l4_usart0_getssr(ssr, chr, port);
            if (data->status.bit.er_buf_ovr == 1U) {
                sts |= T_COM_EROVB;
            }
            if ((data->status.bit.sns_brk == 0U) || ((sts & T_COM_BRK) == 0U)) {
                data->status.bit.sns_brk = ((sts & T_COM_BRK) != 0U) ? 1U : 0U;
                if (data->RcvData != 0) {
                    data->status.bit.er_buf_ovr = 0U;
                    if (data->RcvData->sbuf != 0) {
                        data->RcvData->rcnt--;
                        *data->RcvData->sbuf++ = sts;
                        *data->RcvData->rbuf++ = chr;
                        data->RcvData = 0;
                        (void)isig_sem((ID)cdata->rsemid);
                    }
                } else if (data->rcnt < cdata->rsize) {
                    data->status.bit.er_buf_ovr = 0U;
                    if (cdata->sbuf != 0) {
                        rcvp = data->rcvp + data->rcnt;
                        if (rcvp >= cdata->rsize) {
                            rcvp -= cdata->rsize;
                        }
                        cdata->sbuf[rcvp] = sts;
                        cdata->rbuf[rcvp] = chr;
                        data->rcnt++;
                        if (data->status.bit.sft_flw == 1U) {
                            _ddr_stm32l4_usart0_chk_rxoff(data, cdata, port);
                        }
                    }
                } else {
                    data->status.bit.er_buf_ovr = 1U;
                }
            }
        } else {
            if (cdata->fifo_en) {
                port->CR1 &= ~0x00000100U;
                port->CR3 &= ~0x10000001U;
            } else {
                port->CR1 &= ~0x00000120U;
                port->CR3 &= ~0x00000001U;
            }
        }
    }
}

/*******************************************
        USARTMf[^tݏ
 *******************************************/

static void _ddr_stm32l4_usart0_rxi(T_STM32L4_USART0_MNG *pk_USART0mng)
{
    volatile struct t_usart *port;
    T_STM32L4_USART0_CMNG const *cdata;
    T_STM32L4_USART0_DMNG *data;
    UH rcvp;
    VB chr;
    UB sts;
    BOOL iflag = FALSE;

    data = pk_USART0mng->data;
    cdata = pk_USART0mng->cdata;
    port = pk_USART0mng->port;

    if (cdata->fifo_en) {
        if (((port->ISR & 0x04000000U)) &&
            ((port->CR3 & 0x10000000U))) {
            iflag = TRUE;
        }
    } else {
        if (((port->ISR & 0x00000020U)) &&
            ((port->CR1 & 0x00000020U))) {
            iflag = TRUE;
        }
    }

    if (iflag) {
        if ((data->RcvData != 0) ||
            (data->rcnt < cdata->rsize)) {
            do {
                chr = (VB)(port->RDR & data->mask);
                if ((data->status.bit.sft_flw == 1U) && (chr == XON)) {
                    data->status.bit.tx_xoff = 0U;
                    if ((port->ISR & 0x00000080U) != 0U) {
                        if (data->status.bit.req_xon_xoff == 1U) {
                            port->TDR = (data->status.bit.rx_xoff == 0U) ? (UW)XON : (UW)XOFF;
                            data->status.bit.req_xon_xoff = 0U;
                        } else if (data->tcnt != 0U) {
                            _ddr_stm32l4_usart0_send_drv_buf(pk_USART0mng);
                        } else if (data->SndData != 0) {
                            _ddr_stm32l4_usart0_send_local_buf(pk_USART0mng, (ID)cdata->tsemid);
                        } else {
                            /* do nothing */
                        }
                    }
                    if ((data->tcnt == 0U) && (data->SndData == 0)) {
                        if (cdata->fifo_en) {
                            port->CR3 &= ~0x00800000U;
                        } else {
                            port->CR1 &= ~0x00000080U;
                        }
                    } else {
                        if (cdata->fifo_en) {
                            port->CR3 |= 0x00800000U;
                        } else {
                            port->CR1 |= 0x00000080U;
                        }
                    }
                    if (data->status.bit.cln_wait == 1U) {
                        port->CR1 |= 0x00000040U;
                    }
                    break;
                } else if ((data->status.bit.sft_flw == 1U) && (chr == XOFF)) {
                    data->status.bit.tx_xoff = 1U;
                    if (cdata->fifo_en) {
                        port->CR3 &= ~0x00800000U;
                    } else {
                        port->CR1 &= ~0x00000080U;
                    }
                    break;
                } else {
                    if (data->RcvData != 0) {
                        data->RcvData->rcnt--;
                        *data->RcvData->rbuf++ = chr;
                        sts = (data->status.bit.er_buf_ovr == 0U) ? (UB)0U : (UB)T_COM_EROVB;
                        if (data->RcvData->sbuf != 0) {
                            *data->RcvData->sbuf++ = sts;
                        }
                        data->status.bit.er_buf_ovr = 0U;
                        if (_ddr_stm32l4_usart0_check_chr(data->RcvData, chr, sts) != FALSE) {
                            data->RcvData = 0;
                            (void)isig_sem((ID)cdata->rsemid);
                            break;
                        }
                    } else if (data->rcnt < cdata->rsize) {
                        rcvp = data->rcvp + data->rcnt;
                        if (rcvp >= cdata->rsize) {
                            rcvp -= cdata->rsize;
                        }
                        cdata->rbuf[rcvp] = chr;
                        if (cdata->sbuf != 0) {
                            cdata->sbuf[rcvp] = (data->status.bit.er_buf_ovr == 0U) ? (UB)0U : (UB)T_COM_EROVB;
                        }
                        data->status.bit.er_buf_ovr = 0U;
                        data->rcnt++;
                        if (data->status.bit.sft_flw == 1U) {
                            _ddr_stm32l4_usart0_chk_rxoff(data, cdata, port);
                        }
                        if (data->rcnt == cdata->rsize) {
                            break;
                        }
                    } else {
                        data->status.bit.er_buf_ovr = 1U;
                    }
                }
            } while ((cdata->fifo_en) && (port->ISR & 0x00000020U));
        } else {
            if (cdata->fifo_en) {
                port->CR1 &= ~0x00000100U;
                port->CR3 &= ~0x10000001U;
            } else {
                port->CR1 &= ~0x00000120U;
                port->CR3 &= ~0x00000001U;
            }
        }
    }
}

/***********************************************
        USARTMf[^GveBݏ
 ***********************************************/

static void _ddr_stm32l4_usart0_txi(T_STM32L4_USART0_MNG *pk_USART0mng)
{
    volatile struct t_usart *port;
    T_STM32L4_USART0_CMNG const *cdata;
    T_STM32L4_USART0_DMNG *data;
    BOOL iflag = FALSE;

    data = pk_USART0mng->data;
    cdata = pk_USART0mng->cdata;
    port = pk_USART0mng->port;

    if (cdata->fifo_en) {
        if ((port->ISR & 0x08000000U) &&
            (port->CR3 & 0x00800000U)) {
            iflag = TRUE;
        }
    } else {
        if ((port->ISR & 0x00000080U) &&
            (port->CR1 & 0x00000080U)) {
            iflag = TRUE;
        }
    }

    if (iflag) {
        if (data->status.bit.req_xon_xoff == 1U) {
            port->TDR = (data->status.bit.rx_xoff == 0U) ? (UW)XON : (UW)XOFF;
            data->status.bit.req_xon_xoff = 0U;
            if ((data->tcnt == 0U) && (data->SndData == 0)) {
                if (cdata->fifo_en) {
                    port->CR3 &= ~0x00800000U;
                } else {
                    port->CR1 &= ~0x00000080U;
                }
            }
        } else if (data->status.bit.tx_xoff == 0U) {
            if (cdata->tsize == 0U) {
                if (data->SndData != 0) {
                    _ddr_stm32l4_usart0_send_local_buf(pk_USART0mng, (ID)cdata->tsemid);
                }
            } else {
                if (data->tcnt != 0U) {
                    _ddr_stm32l4_usart0_send_drv_buf(pk_USART0mng);
                }
            }
            if ((data->tcnt == 0U) && (data->SndData == 0)) {
                if (cdata->fifo_en) {
                    port->CR3 &= ~0x00800000U;
                } else {
                    port->CR1 &= ~0x00000080U;
                }
            }
        } else {
            if (cdata->fifo_en) {
                port->CR1 &= ~0x00000040U;
                port->CR3 &= ~0x00800000U;
            } else {
                port->CR1 &= ~0x000000C0U;
            }
        }
    }
}

/***************************************
        USARTݏ
 ***************************************/

void _ddr_stm32l4_usart0_intr(T_STM32L4_USART0_MNG *pk_USART0mng)
{
    _ddr_stm32l4_usart0_eri(pk_USART0mng);
    _ddr_stm32l4_usart0_rxi(pk_USART0mng);
    _ddr_stm32l4_usart0_txi(pk_USART0mng);
    _ddr_stm32l4_usart0_tci(pk_USART0mng);
}

/*******************************
        ̎o
 *******************************/

static BOOL _ddr_stm32l4_usart0_recv_strings(T_STM32L4_USART0_DMNG *data, T_STM32L4_USART0_CMNG const *cdata, T_COM_RCV *ReceiveData, volatile struct t_usart *port)
{
    BOOL flag;
    VB chr;
    UB sts;

    for (flag = FALSE; flag == FALSE; ) {
        if (ReceiveData->rcnt == 0U) {
            flag = TRUE;
        } else if (data->rcnt == 0U) {
            break;
        } else {
            *ReceiveData->rbuf++ = chr = cdata->rbuf[data->rcvp];
            sts = cdata->sbuf[data->rcvp];
            if (ReceiveData->sbuf != 0)
                *ReceiveData->sbuf++ = sts;
            if (++data->rcvp >= cdata->rsize) {
                data->rcvp = 0U;
            }
            data->rcnt--;
            ReceiveData->rcnt--;
            if (data->status.bit.sft_flw == 1U) {
                _ddr_stm32l4_usart0_chk_rxon(data, cdata, port);
            }
            flag = _ddr_stm32l4_usart0_check_chr(ReceiveData, chr, sts);
        }
    }
    return flag;
}

/*******************************
        USARTM
 *******************************/

static ER _ddr_stm32l4_usart0_snd(T_COM_SND *TransmiteData, T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    volatile struct t_usart *port;
    T_STM32L4_USART0_CMNG const *cdata;
    T_STM32L4_USART0_DMNG *data;
    ID tskid;
    ER ercd = E_OK;

    data = pk_USART0mng->data;
    if ((data->status.bit.init_flg == 0U) || (sns_dpn() != FALSE)) {
        ercd = E_OBJ;
    } else {
        (void)get_tid(&tskid);
        (void)dis_dsp();
        if (((data->tlockid == 0U) || (data->tlockid == (UH)tskid)) &&
            (data->status.bit.ena_tx  == 1U) &&
            (data->SndData == 0)) {
            port = pk_USART0mng->port;
            cdata = pk_USART0mng->cdata;
            (void)loc_cpu();
            if (cdata->fifo_en) {
                port->CR3 &= ~0x00800000U;
            } else {
                port->CR1 &= ~0x00000080U;
            }
            (void)unl_cpu();
            if (_ddr_stm32l4_usart0_copy(data, cdata, TransmiteData) == FALSE) {
                data->SndData = TransmiteData;
                (void)loc_cpu();
                if (cdata->fifo_en) {
                    port->CR3 |= 0x00800000U;
                } else {
                    port->CR1 |= 0x00000080U;
                }
                (void)unl_cpu();
                (void)ena_dsp();
                ercd = twai_sem((ID)cdata->tsemid, TransmiteData->time);
                if (ercd != E_OK) {
                    (void)loc_cpu();
                    if (cdata->fifo_en) {
                        port->CR3 &= ~0x00800000U;
                    } else {
                        port->CR1 &= ~0x00000080U;
                    }
                    data->SndData = 0;
                    (void)unl_cpu();
                    ercd = pol_sem((ID)cdata->tsemid);
                }
            } else {
                if (data->tcnt != 0U) {
                    (void)loc_cpu();
                    if (cdata->fifo_en) {
                        port->CR3 |= 0x00800000U;
                    } else {
                        port->CR1 |= 0x00000080U;
                    }
                    (void)unl_cpu();
                }
                (void)ena_dsp();
            }
        } else {
            (void)ena_dsp();
            ercd = E_OBJ;
        }
    }
    return ercd;
}

/*******************************
        USARTM
 *******************************/

static ER _ddr_stm32l4_usart0_rcv(T_COM_RCV *ReceiveData,  T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    volatile struct t_usart *port;
    T_STM32L4_USART0_CMNG const *cdata;
    T_STM32L4_USART0_DMNG *data;
    ID tskid;
    ER ercd = E_OK;

    cdata = pk_USART0mng->cdata;
    data = pk_USART0mng->data;
    if ((data->status.bit.init_flg == 0U) || (sns_dpn() != FALSE) ||
        (data->status.bit.ena_rx   == 0U)) {
        ercd = E_OBJ;
    } else {
        (void)get_tid(&tskid);
        (void)dis_dsp();
        port = pk_USART0mng->port;
        (void)loc_cpu();
        if (cdata->fifo_en) {
            port->CR1 &= ~0x00000100U;
            port->CR3 &= ~0x10000001U;
        } else {
            port->CR1 &= ~0x00000120U;
            port->CR3 &= ~0x00000001U;
        }
        (void)unl_cpu();
        if (((data->rlockid == 0U) || (data->rlockid == (UH)tskid)) &&
            (data->RcvData  == 0)) {
            if (_ddr_stm32l4_usart0_recv_strings(data, cdata, ReceiveData, port) == FALSE) {
                (void)loc_cpu();
                data->RcvData = ReceiveData;
                if (cdata->fifo_en) {
                    port->CR1 |= 0x00000100U;
                    port->CR3 |= 0x10000001U;
                } else {
                    port->CR1 |= 0x00000120U;
                    port->CR3 |= 0x00000001U;
                }
                (void)unl_cpu();
                (void)ena_dsp();
                ercd = twai_sem((ID)cdata->rsemid, ReceiveData->time);
                if (ercd != E_OK) {
                    data->RcvData = 0;
                    ercd = pol_sem((ID)cdata->rsemid);
                }
            } else {
                (void)loc_cpu();
                if (cdata->fifo_en) {
                    port->CR1 |= 0x00000100U;
                    port->CR3 |= 0x10000001U;
                } else {
                    port->CR1 |= 0x00000120U;
                    port->CR3 |= 0x00000001U;
                }
                (void)unl_cpu();
                (void)ena_dsp();
            }
        } else {
            (void)loc_cpu();
            if (cdata->fifo_en) {
                port->CR1 |= 0x00000100U;
                port->CR3 |= 0x10000001U;
            } else {
                port->CR1 |= 0x00000120U;
                port->CR3 |= 0x00000001U;
            }
            (void)unl_cpu();
            (void)ena_dsp();
            ercd = E_OBJ;
        }
    }
    return ercd;
}

/***********************************
        Mobt@̑o
 ***********************************/

static ER _ddr_stm32l4_usart0_cln_tx_buf(T_STM32L4_USART0_MNG const *pk_USART0mng, TMO tim)
{
    volatile struct t_usart *port;
    T_STM32L4_USART0_CMNG const *cdata;
    T_STM32L4_USART0_DMNG *data;
    ER ercd;
    ID tskid;

    data = pk_USART0mng->data;
    (void)get_tid(&tskid);
    if (((data->tlockid == 0U) ||
         (data->tlockid == (UH)tskid)) &&
        ( data->SndData == 0)) {
        port = pk_USART0mng->port;
        (void)loc_cpu();
        if (((port->ISR & 0x00000040U) == 0U) ||
            (data->tcnt != 0U)) {
            port->CR1 |= 0x00000040U;
            data->status.bit.cln_wait = 1U;
            (void)unl_cpu();
            cdata = pk_USART0mng->cdata;
            ercd = twai_sem((ID)cdata->tsemid, tim);
            if (ercd != E_OK) {
                (void)loc_cpu();
                data->status.bit.cln_wait = 0U;
                (void)unl_cpu();
                ercd = pol_sem((ID)cdata->tsemid);
            }
        } else {
            (void)unl_cpu();
            ercd = E_OK;
        }
    } else {
        ercd = E_OBJ;
    }

    return ercd;
}

/***************************************
        Mobt@̃Zbg
 ***************************************/

static ER _ddr_stm32l4_usart0_rst_tx_buf(T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    volatile struct t_usart *port;
    T_STM32L4_USART0_DMNG *data;
    ER ercd;
    ID tskid;

    data = pk_USART0mng->data;
    (void)get_tid(&tskid);
    if (((data->tlockid == 0U) ||
         (data->tlockid == (UH)tskid)) &&
        ( data->SndData == 0)) {
        if (pk_USART0mng->cdata->fifo_en) {
            port = pk_USART0mng->port;
            (void)loc_cpu();
            port->RQR |= 0x00000010U;
            (void)unl_cpu();
        }
        data->tcnt = 0U;
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }

    return ercd;
}

/***************************************
        Mobt@̃Zbg
 ***************************************/

static ER _ddr_stm32l4_usart0_rst_rx_buf(T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    volatile struct t_usart *port;
    T_STM32L4_USART0_DMNG *data;
    ER ercd;
    ID tskid;

    data = pk_USART0mng->data;
    (void)get_tid(&tskid);
    if (((data->rlockid == 0U) ||
         (data->rlockid == (UH)tskid)) &&
        ( data->RcvData == 0)) {
        if (pk_USART0mng->cdata->fifo_en) {
            port = pk_USART0mng->port;
            (void)loc_cpu();
            port->RQR |= 0x00000008U;
            (void)unl_cpu();
        }
        data->rcnt = 0U;
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/*******************************
        M̋֎~
 *******************************/

static ER _ddr_stm32l4_usart0_dis_send(T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    T_STM32L4_USART0_CMNG const *cdata;
    volatile struct t_usart *port;
    T_STM32L4_USART0_DMNG *data;
    ER ercd;
    ID tskid;

    cdata = pk_USART0mng->cdata;
    data = pk_USART0mng->data;
    (void)get_tid(&tskid);
    if (((data->tlockid == 0U) ||
         (data->tlockid == (UH)tskid)) &&
        ( data->SndData == 0) &&
        ( data->tcnt    == 0U)) {
        port = pk_USART0mng->port;
        if ((port->ISR & 0x00000040U) != 0U) {
            (void)loc_cpu();
            data->status.bit.ena_tx = 0U;
            if (cdata->fifo_en) {
                port->CR1 &= ~0x00000048U;
                port->CR3 &= ~0x00800000U;
            } else {
                port->CR1 &= ~0x000000C8U;
            }
            (void)unl_cpu();
            ercd = E_OK;
        } else {
            ercd = E_OBJ;
        }
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/*******************************
        M̋֎~
 *******************************/

static ER _ddr_stm32l4_usart0_dis_rcv(T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    T_STM32L4_USART0_CMNG const *cdata;
    T_STM32L4_USART0_DMNG *data;
    ER ercd;
    ID tskid;

    cdata = pk_USART0mng->cdata;
    data = pk_USART0mng->data;
    (void)get_tid(&tskid);
    if (((data->rlockid == 0U) ||
         (data->rlockid == (UH)tskid)) &&
        ( data->RcvData == 0) &&
        ( data->rcnt    == 0U)) {
        (void)loc_cpu();
        data->status.bit.ena_rx = 0U;
        if (cdata->fifo_en) {
            pk_USART0mng->port->CR1 &= ~0x00000100U;
            pk_USART0mng->port->CR3 &= ~0x10000001U;
        } else {
            pk_USART0mng->port->CR1 &= ~0x00000120U;
            pk_USART0mng->port->CR3 &= ~0x00000001U;
        }
        (void)unl_cpu();
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/*******************************
        M̋
 *******************************/

static ER _ddr_stm32l4_usart0_ena_send(T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    T_STM32L4_USART0_DMNG *data;
    ER ercd;
    ID tskid;

    data = pk_USART0mng->data;
    (void)get_tid(&tskid);
    if ((data->tlockid == 0U) ||
        (data->tlockid == (UH)tskid)) {
        (void)loc_cpu();
        pk_USART0mng->data->status.bit.ena_tx = 1U;
        pk_USART0mng->port->CR1 |= 0x00000008U;
        (void)unl_cpu();
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/*******************************
        M̋
 *******************************/

static ER _ddr_stm32l4_usart0_ena_rcv(T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    T_STM32L4_USART0_CMNG const *cdata;
    T_STM32L4_USART0_DMNG *data;
    ER ercd;
    ID tskid;

    cdata = pk_USART0mng->cdata;
    data = pk_USART0mng->data;
    (void)get_tid(&tskid);
    if ((data->rlockid == 0U) ||
        (data->rlockid == (UH)tskid)) {
        (void)loc_cpu();
        pk_USART0mng->data->status.bit.ena_rx = 1U;
        if (cdata->fifo_en) {
            pk_USART0mng->port->CR1 |= 0x00000004U;
            pk_USART0mng->port->CR3 |= 0x10000001U;
        } else {
            pk_USART0mng->port->CR1 |= 0x00000024U;
            pk_USART0mng->port->CR3 |= 0x00000001U;
        }
        (void)unl_cpu();
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/*******************************
        u[N̑o
 *******************************/

static ER _ddr_stm32l4_usart0_snd_brk(T_STM32L4_USART0_MNG const *pk_USART0mng, TMO tim)
{
    volatile struct t_usart *port;
    T_STM32L4_USART0_DMNG *data;
    ER ercd;
    ID tskid;

    data = pk_USART0mng->data;
    (void)get_tid(&tskid);
    if (((data->tlockid == 0U) ||
         (data->tlockid == (UH)tskid)) &&
        ( data->SndData == 0) &&
        ( data->tcnt    == 0U)) {
        port = pk_USART0mng->port;
        if ((port->ISR & 0x00000040U) != 0U) {
            (void)loc_cpu();
            port->RQR |= 0x00000002U;
            (void)unl_cpu();
            (void)dly_tsk((RELTIM)tim);
            (void)loc_cpu();
            port->RQR &= ~0x00000002U;
            (void)unl_cpu();
            ercd = E_OK;
        } else {
            ercd = E_OBJ;
        }
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/***********************************
        M`l̃bN
 ***********************************/

static ER _ddr_stm32l4_usart0_lock_trans(T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    T_STM32L4_USART0_DMNG *data;
    ER ercd;
    ID tskid;

    data = pk_USART0mng->data;
    (void)get_tid(&tskid);
    (void)loc_cpu();
    if ((data->tlockid == 0U) ||
        (data->SndData == 0)) {
        data->tlockid = (UH)tskid;
        ercd = E_OK;
    } else if (data->tlockid == (UH)tskid) {
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    (void)unl_cpu();
    return ercd;
}

/***********************************
        M`l̃bN
 ***********************************/

static ER _ddr_stm32l4_usart0_lock_recv(T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    T_STM32L4_USART0_DMNG *data;
    ER ercd;
    ID tskid;

    data = pk_USART0mng->data;
    (void)get_tid(&tskid);
    (void)loc_cpu();
    if ((data->rlockid == 0U) ||
        (data->RcvData == 0)) {
        data->rlockid = (UH)tskid;
        ercd = E_OK;
    } else if (data->rlockid == (UH)tskid) {
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    (void)unl_cpu();
    return ercd;
}

/***************************************
        M`l̃bN
 ***************************************/

static ER _ddr_stm32l4_usart0_unlock_trans(T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    T_STM32L4_USART0_DMNG *data;
    ER ercd;
    ID tskid;

    data = pk_USART0mng->data;
    (void)get_tid(&tskid);
    if (data->tlockid == (UH)tskid) {
        data->tlockid = 0U;
        ercd = E_OK;
    } else if (data->tlockid == 0U) {
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/***************************************
        M`l̃bN
 ***************************************/

static ER _ddr_stm32l4_usart0_unlock_recv(T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    T_STM32L4_USART0_DMNG *data;
    ER ercd;
    ID tskid;

    data = pk_USART0mng->data;
    (void)get_tid(&tskid);
    if (data->rlockid == (UH)tskid) {
        data->rlockid = 0U;
        ercd = E_OK;
    } else if (data->rlockid == 0U) {
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/*******************************
        USARTM
 *******************************/

static ER _ddr_stm32l4_usart0_ctr(T_COM_CTR const *pk_SerialData,  T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    ER ercd = E_OK;

    if (pk_USART0mng->data->status.bit.init_flg == 0U) {
        ercd = E_OBJ;
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & CLN_TXBUF) == CLN_TXBUF) {
            ercd = _ddr_stm32l4_usart0_cln_tx_buf(pk_USART0mng, pk_SerialData->time);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & RST_TXBUF) == RST_TXBUF) {
            ercd = _ddr_stm32l4_usart0_rst_tx_buf(pk_USART0mng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & RST_RXBUF) == RST_RXBUF) {
            ercd = _ddr_stm32l4_usart0_rst_rx_buf(pk_USART0mng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & STP_TX) == STP_TX) {
            ercd = _ddr_stm32l4_usart0_dis_send(pk_USART0mng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & STP_RX) == STP_RX) {
            ercd = _ddr_stm32l4_usart0_dis_rcv(pk_USART0mng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & STA_TX) == STA_TX) {
            ercd = _ddr_stm32l4_usart0_ena_send(pk_USART0mng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & STA_RX) == STA_RX) {
            ercd = _ddr_stm32l4_usart0_ena_rcv(pk_USART0mng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & SND_BRK) == SND_BRK) {
            ercd = _ddr_stm32l4_usart0_snd_brk(pk_USART0mng, pk_SerialData->time);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & LOC_TX) == LOC_TX) {
            ercd = _ddr_stm32l4_usart0_lock_trans(pk_USART0mng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & LOC_RX) == LOC_RX) {
            ercd = _ddr_stm32l4_usart0_lock_recv(pk_USART0mng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & UNL_TX) == UNL_TX) {
            ercd = _ddr_stm32l4_usart0_unlock_trans(pk_USART0mng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & UNL_RX) == UNL_RX) {
            ercd = _ddr_stm32l4_usart0_unlock_recv(pk_USART0mng);
        }
    }

    return ercd;
}

/*******************************
        USARTԎQƏ
 *******************************/

static ER _ddr_stm32l4_usart0_ref(T_COM_REF *pk_SerialRef,  T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    T_STM32L4_USART0_CMNG const *cdata;
    T_STM32L4_USART0_DMNG *data;
    UH status = 0U;

    data = pk_USART0mng->data;
    if (data->status.bit.init_flg == 1U) {
        cdata = pk_USART0mng->cdata;
        pk_SerialRef->rxcnt = data->rcnt;
        pk_SerialRef->txcnt = data->tcnt;

        status |= T_COM_DSR|T_COM_INIT;
        if ((cdata->ctsport != 0) && (cdata->ctsbit != 0U)) {
            if ((*cdata->ctsport & cdata->ctsbit) == 0U) {
                status |= T_COM_CTS;
            }
        } else {
            status |= T_COM_CTS;
        }
        if (data->status.bit.ena_tx == 1U) {
            status |= T_COM_ENATX;
        }
        if (data->status.bit.ena_rx == 1U) {
            status |= T_COM_ENARX;
        }
        if (data->status.bit.rx_xoff == 1U) {
            status |= T_COM_RXOFF;
        }
        if (data->status.bit.tx_xoff == 1U) {
            status |= T_COM_TXOFF;
        }
    }
    pk_SerialRef->status = status;
    return E_OK;
}

/*******************************
        USART
 *******************************/

static ER _ddr_stm32l4_usart0_ini(T_COM_SMOD const *pk_SerialMode,  T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    volatile struct t_usart *port;
    T_STM32L4_USART0_MSTS status;
    T_RSEM pk_rsem;
    ER ercd = E_OK;
    UW cr1 = 0U;
    UW cr2 = 0U;
    UW cr3 = 0U;
    UW brr;
    UW tmp;
#ifdef LPMODE
    T_REFCLK rclk;
#endif

    port = pk_USART0mng->port;
    if (pk_SerialMode != 0) {

        if (pk_USART0mng->data->status.bit.init_flg == 0U) {
            port->CR1 = 0U;
            status.word = 0U;
#ifdef LPMODE
            if (pk_USART0mng->cdata->pclk == 0U) {
                /* PCLK */
                if (port == &REG_USART1) {
                    _ddr_stm32l4_ref_clk(CLK_PCLK2_CLOCK, &rclk);
                } else {
                    _ddr_stm32l4_ref_clk(CLK_PCLK1_CLOCK, &rclk);
                }
                tmp =  (rclk.frq * 10U) / ((pk_SerialMode->baud / 100U) * 16U);
            } else if (pk_USART0mng->cdata->pclk == 1U) {
                /* SYSCLK */
                tmp =  (SYSCLK * 10U) / ((pk_SerialMode->baud / 100U) * 16U);
            } else if (pk_USART0mng->cdata->pclk == 2) {
                /* HSI16 */
                tmp =  (HSI16 * 10U) / ((pk_SerialMode->baud / 100U) * 16U);
            }
#else
            tmp = (pk_USART0mng->cdata->pclk * 10U) / ((pk_SerialMode->baud / 100U) * 16U);
#endif
            brr = (tmp / 1000U) << 4;
            tmp %= 1000U;
            tmp += 32U;
            tmp *= 16U;
            brr += (tmp / 1000U);

            switch (pk_SerialMode->blen) {
                case BLEN7:
                    tmp = 0x7FU;
                    switch (pk_SerialMode->par) {
                        case PAR_ODD:
                            cr1 |= 0x00000600U;
                            break;
                        case PAR_EVEN:
                            cr1 |= 0x00000400U;
                            break;
                        default:
                            ercd = E_PAR;
                            break;
                    }
                    break;
                case BLEN8:
                    tmp = 0xFFU;
                    switch (pk_SerialMode->par) {
                        case PAR_ODD:
                            cr1 |= 0x00001600U;
                            break;
                        case PAR_EVEN:
                            cr1 |= 0x00001400U;
                            break;
                        default:
                            break;
                    }
                    break;
                default:
                    ercd = E_PAR;
                    break;
            }

            switch (pk_SerialMode->sbit) {
                case SBIT2:
                    cr2 = 0x00002000U;
                    break;
                case SBIT15:
                    cr2= 0x00003000U;
                    break;
                case SBIT1:
                    break;
                default:
                    ercd = E_PAR;
                    break;
            }

            switch (pk_SerialMode->flow) {
                case FLW_XON:
                    status.bit.sft_flw = 1U;
                    break;
                case FLW_HARD:
                    if ((pk_USART0mng->cdata->ctsport != 0) && (pk_USART0mng->cdata->ctsbit != 0U)) {
                        cr3 |= 0x00000300U;
                        status.bit.hrd_flw = 1U;
                    } else {
                        ercd = E_PAR;
                    }
                    break;
                case FLW_NONE:
                    break;
                default:
                    ercd = E_PAR;
                    break;
            }

            if (pk_USART0mng->cdata->fifo_en) {
                cr1 |= 0x20000000U;

                cr3 |= ((UW)pk_USART0mng->cdata->fifo_rxft << 25U);
                cr3 |= ((UW)pk_USART0mng->cdata->fifo_txft << 29U);
            }

            if (ercd == E_OK) {
                (void)loc_cpu();
                status.bit.init_flg = 1U;
                *pk_USART0mng->cdata->enr |= pk_USART0mng->cdata->apbbit;
                *pk_USART0mng->cdata->rstr |= pk_USART0mng->cdata->apbbit;
                *pk_USART0mng->cdata->rstr &= ~pk_USART0mng->cdata->apbbit;
                pk_USART0mng->data->status.word = status.word;
                pk_USART0mng->data->mask = tmp;
                port->CR1 &= ~0x00000001U;
                port->CR2 = cr2;
                port->CR1 = cr1;
                port->CR3 = cr3;
                port->BRR = brr;
                port->CR1 = cr1 | 0x00000001U;
                (void)unl_cpu();
            }
        } else {
            ercd = E_OBJ;
        }
    } else {
        if (pk_USART0mng->data->status.bit.init_flg == 1U) {
            (void)loc_cpu();
            *pk_USART0mng->cdata->rstr |= pk_USART0mng->cdata->apbbit;
            *pk_USART0mng->cdata->rstr &= ~pk_USART0mng->cdata->apbbit;
            if ((pk_USART0mng->data->SndData != 0) && (pk_USART0mng->data->status.bit.cln_wait != 0U)) {
                if (ref_sem((ID)pk_USART0mng->cdata->tsemid, &pk_rsem) == E_OK) {
                    if (pk_rsem.wtskid != 0) {
                        (void)rel_wai(pk_rsem.wtskid);
                    }
                    for (; pk_rsem.semcnt > 0U; pk_rsem.semcnt--) {
                        (void)pol_sem((ID)pk_USART0mng->cdata->tsemid);
                    }
                }
            }
            if (pk_USART0mng->data->RcvData == 0) {
                if (ref_sem((ID)pk_USART0mng->cdata->rsemid, &pk_rsem) == E_OK) {
                    if (pk_rsem.wtskid != 0) {
                        (void)rel_wai(pk_rsem.wtskid);
                    }
                    for (; pk_rsem.semcnt > 0U; pk_rsem.semcnt--) {
                        (void)pol_sem((ID)pk_USART0mng->cdata->rsemid);
                    }
                }
            }
            memset(pk_USART0mng->data, 0x00, sizeof(T_STM32L4_USART0_DMNG));
            port->CR1 &= ~0x00000001U;
            (void)unl_cpu();
        }
    }
    return ercd;
}

/*******************************
    USARThCoC
 *******************************/

ER _ddr_stm32l4_usart0drv(ID FuncID, VP_INT pk_ControlData, T_STM32L4_USART0_MNG const *pk_USART0mng)
{
    ER ercd;

    switch(FuncID) {
        case TA_COM_INI:
            ercd = _ddr_stm32l4_usart0_ini((T_COM_SMOD const *)pk_ControlData, pk_USART0mng);
            break;
        case TA_COM_REF:
            ercd = _ddr_stm32l4_usart0_ref((T_COM_REF *)pk_ControlData, pk_USART0mng);
            break;
        case TA_COM_CTR:
            ercd = _ddr_stm32l4_usart0_ctr((T_COM_CTR const *)pk_ControlData, pk_USART0mng);
            break;
        case TA_COM_PUTS:
            ercd = _ddr_stm32l4_usart0_snd((T_COM_SND *)pk_ControlData, pk_USART0mng);
            break;
        case TA_COM_GETS:
            ercd = _ddr_stm32l4_usart0_rcv((T_COM_RCV *)pk_ControlData, pk_USART0mng);
            break;
        default:
            ercd = E_NOSPT;
            break;
    }
    return ercd;
}
