/**
 * @file    DDR_RZT1_SCIF.h
 * @brief   RZ/T1 Serial communication interface
 * @date    2015.04.15
 * @author  Copyright (c) 2014-2015, eForce Co.,Ltd.  All rights reserved.
 *
 * @par     History
 *          - rev 1.0 (2014.10.03) i-cho
 *            Initial version.
 *          - rev 1.1 (2015.04.15) i-cho
 *            Corresponding to Renesas Electronics Corporation iodefine.h V0.8.
 */
#include "kernel.h"

#include "RZT1_UC3.h"
#include "DDR_COM.h"
#include "DDR_RZT1_SCIF.h"
#include "DDR_RZT1_CPG.h"
/* User config header */
#include "DDR_RZT1_SCIF_cfg.h"

#include <string.h>

/* External function prototypes -----------------------------------------------*/
extern ER set_type_int(INTNO intno, int edge_mode);

/* Private function prototypes -----------------------------------------------*/
void _ddr_rzt1_scif_intr(T_RZT1_SCIF_MNG *);
void _ddr_rzt1_scif_rxi(T_RZT1_SCIF_MNG *);
void _ddr_rzt1_scif_bri(T_RZT1_SCIF_MNG *);
void _ddr_rzt1_scif_txi(T_RZT1_SCIF_MNG *);
ER _ddr_rzt1_scifdrv(ID, VP, T_RZT1_SCIF_MNG *);

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/

#ifdef SCIF_0

#if (RSTRG_0 == 14)
#define TRSTRG_0    (7<<8)
#elif (RSTRG_0 == 12)
#define TRSTRG_0    (6<<8)
#elif (RSTRG_0 == 10)
#define TRSTRG_0    (5<<8)
#elif (RSTRG_0 == 8)
#define TRSTRG_0    (4<<8)
#elif (RSTRG_0 == 6)
#define TRSTRG_0    (3<<8)
#elif (RSTRG_0 == 4)
#define TRSTRG_0    (2<<8)
#elif (RSTRG_0 == 1)
#define TRSTRG_0    (1<<8)
#else
#define TRSTRG_0    (0<<8)
#endif

#if (RTRG_0 == 14)
#define TRTRG_0     (3<<6)
#elif (RTRG_0 == 8)
#define TRTRG_0     (2<<6)
#elif (RTRG_0 == 4)
#define TRTRG_0     (1<<6)
#else
#define TRTRG_0     (0<<6)
#endif

#if (TTRG_0 == 0)
#define TTTRG_0     (3<<4)
#elif (TTRG_0 == 2)
#define TTTRG_0     (2<<4)
#elif (TTRG_0 == 4)
#define TTTRG_0     (1<<4)
#else
#define TTTRG_0     (0<<4)
#endif

#if (TXBUF_SZ0==0)
#define _ddr_rzt1_scif_tbuf0  0
#else
VB  _ddr_rzt1_scif_tbuf0[TXBUF_SZ0];
#endif
VB  _ddr_rzt1_scif_rbuf0[RXBUF_SZ0];
UB  _ddr_rzt1_scif_sbuf0[RXBUF_SZ0];
T_RZT1_SCIF_MNG _ddr_rzt1_scif_data0;

const T_CDEV _ddr_rzt1_scif_cdev0 = {&_ddr_rzt1_scif_data0, (FP)_ddr_rzt1_scifdrv};
const T_CFLG _ddr_rzt1_scif_cflg0 = {TA_TFIFO|TA_WMUL, 0x00000000};
const T_CISR _ddr_rzt1_scif_cisr0[4] = {
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data0, INT_SCIFA_BRI_ERI0,    (FP)_ddr_rzt1_scif_intr, IPL_SCIF0},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data0, INT_SCIFA_RXI0,        (FP)_ddr_rzt1_scif_intr, IPL_SCIF0},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data0, INT_SCIFA_TXI0,        (FP)_ddr_rzt1_scif_intr, IPL_SCIF0},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data0, INT_SCIFA_TEI_DRI0,    (FP)_ddr_rzt1_scif_intr, IPL_SCIF0}};
#endif

#ifdef SCIF_1

#if (RSTRG_1 == 14)
#define TRSTRG_1    (7<<8)
#elif (RSTRG_1 == 12)
#define TRSTRG_1    (6<<8)
#elif (RSTRG_1 == 10)
#define TRSTRG_1    (5<<8)
#elif (RSTRG_1 == 8)
#define TRSTRG_1    (4<<8)
#elif (RSTRG_1 == 6)
#define TRSTRG_1    (3<<8)
#elif (RSTRG_1 == 4)
#define TRSTRG_1    (2<<8)
#elif (RSTRG_1 == 1)
#define TRSTRG_1    (1<<8)
#else
#define TRSTRG_1    (0<<8)
#endif

#if (RTRG_1 == 14)
#define TRTRG_1     (3<<6)
#elif (RTRG_1 == 8)
#define TRTRG_1     (2<<6)
#elif (RTRG_1 == 4)
#define TRTRG_1     (1<<6)
#else
#define TRTRG_1     (0<<6)
#endif

#if (TTRG_1 == 0)
#define TTTRG_1     (3<<4)
#elif (TTRG_1 == 2)
#define TTTRG_1     (2<<4)
#elif (TTRG_1 == 4)
#define TTTRG_1     (1<<4)
#else
#define TTTRG_1     (0<<4)
#endif

#if (TXBUF_SZ1==0)
#define _ddr_rzt1_scif_tbuf1  0
#else
VB  _ddr_rzt1_scif_tbuf1[TXBUF_SZ1];
#endif
VB  _ddr_rzt1_scif_rbuf1[RXBUF_SZ1];
UB  _ddr_rzt1_scif_sbuf1[RXBUF_SZ1];
T_RZT1_SCIF_MNG _ddr_rzt1_scif_data1;

const T_CDEV _ddr_rzt1_scif_cdev1 = {&_ddr_rzt1_scif_data1, (FP)_ddr_rzt1_scifdrv};
const T_CFLG _ddr_rzt1_scif_cflg1 = {TA_TFIFO|TA_WMUL, 0x00000000};
const T_CISR _ddr_rzt1_scif_cisr1[4] = {
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data1, INT_SCIFA_BRI_ERI1,    (FP)_ddr_rzt1_scif_intr, IPL_SCIF1},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data1, INT_SCIFA_RXI1,        (FP)_ddr_rzt1_scif_intr, IPL_SCIF1},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data1, INT_SCIFA_TXI1,        (FP)_ddr_rzt1_scif_intr, IPL_SCIF1},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data1, INT_SCIFA_TEI_DRI1,    (FP)_ddr_rzt1_scif_intr, IPL_SCIF1}};
#endif

#ifdef SCIF_2

#if (RSTRG_2 == 14)
#define TRSTRG_2    (7<<8)
#elif (RSTRG_2 == 12)
#define TRSTRG_2    (6<<8)
#elif (RSTRG_2 == 10)
#define TRSTRG_2    (5<<8)
#elif (RSTRG_2 == 8)
#define TRSTRG_2    (4<<8)
#elif (RSTRG_2 == 6)
#define TRSTRG_2    (3<<8)
#elif (RSTRG_2 == 4)
#define TRSTRG_2    (2<<8)
#elif (RSTRG_2 == 1)
#define TRSTRG_2    (1<<8)
#else
#define TRSTRG_2    (0<<8)
#endif

#if (RTRG_2 == 14)
#define TRTRG_2     (3<<6)
#elif (RTRG_2 == 8)
#define TRTRG_2     (2<<6)
#elif (RTRG_2 == 4)
#define TRTRG_2     (1<<6)
#else
#define TRTRG_2     (0<<6)
#endif

#if (TTRG_2 == 0)
#define TTTRG_2     (3<<4)
#elif (TTRG_2 == 2)
#define TTTRG_2     (2<<4)
#elif (TTRG_2 == 4)
#define TTTRG_2     (1<<4)
#else
#define TTTRG_2     (0<<4)
#endif

#if (TXBUF_SZ2==0)
#define _ddr_rzt1_scif_tbuf2  0
#else
VB  _ddr_rzt1_scif_tbuf2[TXBUF_SZ2];
#endif
VB  _ddr_rzt1_scif_rbuf2[RXBUF_SZ2];
UB  _ddr_rzt1_scif_sbuf2[RXBUF_SZ2];
T_RZT1_SCIF_MNG _ddr_rzt1_scif_data2;

const T_CDEV _ddr_rzt1_scif_cdev2 = {&_ddr_rzt1_scif_data2, (FP)_ddr_rzt1_scifdrv};
const T_CFLG _ddr_rzt1_scif_cflg2 = {TA_TFIFO|TA_WMUL, 0x00000000};
const T_CISR _ddr_rzt1_scif_cisr2[4] = {
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data2, INT_SCIFA_BRI_ERI2,    (FP)_ddr_rzt1_scif_intr, IPL_SCIF2},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data2, INT_SCIFA_RXI2,        (FP)_ddr_rzt1_scif_intr, IPL_SCIF2},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data2, INT_SCIFA_TXI2,        (FP)_ddr_rzt1_scif_intr, IPL_SCIF2},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data2, INT_SCIFA_TEI_DRI2,    (FP)_ddr_rzt1_scif_intr, IPL_SCIF2}};
#endif

#ifdef SCIF_3

#if (RSTRG_3 == 14)
#define TRSTRG_3    (7<<8)
#elif (RSTRG_3 == 12)
#define TRSTRG_3    (6<<8)
#elif (RSTRG_3 == 10)
#define TRSTRG_3    (5<<8)
#elif (RSTRG_3 == 8)
#define TRSTRG_3    (4<<8)
#elif (RSTRG_3 == 6)
#define TRSTRG_3    (3<<8)
#elif (RSTRG_3 == 4)
#define TRSTRG_3    (2<<8)
#elif (RSTRG_3 == 1)
#define TRSTRG_3    (1<<8)
#else
#define TRSTRG_3    (0<<8)
#endif

#if (RTRG_3 == 14)
#define TRTRG_3     (3<<6)
#elif (RTRG_3 == 8)
#define TRTRG_3     (2<<6)
#elif (RTRG_3 == 4)
#define TRTRG_3     (1<<6)
#else
#define TRTRG_3     (0<<6)
#endif

#if (TTRG_3 == 0)
#define TTTRG_3     (3<<4)
#elif (TTRG_3 == 2)
#define TTTRG_3     (2<<4)
#elif (TTRG_3 == 4)
#define TTTRG_3     (1<<4)
#else
#define TTTRG_3     (0<<4)
#endif

#if (TXBUF_SZ3==0)
#define _ddr_rzt1_scif_tbuf3  0
#else
VB  _ddr_rzt1_scif_tbuf3[TXBUF_SZ3];
#endif
VB  _ddr_rzt1_scif_rbuf3[RXBUF_SZ3];
UB  _ddr_rzt1_scif_sbuf3[RXBUF_SZ3];
T_RZT1_SCIF_MNG _ddr_rzt1_scif_data3;

const T_CDEV _ddr_rzt1_scif_cdev3 = {&_ddr_rzt1_scif_data3, (FP)_ddr_rzt1_scifdrv};
const T_CFLG _ddr_rzt1_scif_cflg3 = {TA_TFIFO|TA_WMUL, 0x00000000};
const T_CISR _ddr_rzt1_scif_cisr3[4] = {
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data3, INT_SCIFA_BRI_ERI3,    (FP)_ddr_rzt1_scif_intr, IPL_SCIF3},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data3, INT_SCIFA_RXI3,        (FP)_ddr_rzt1_scif_intr, IPL_SCIF3},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data3, INT_SCIFA_TXI3,        (FP)_ddr_rzt1_scif_intr, IPL_SCIF3},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data3, INT_SCIFA_TEI_DRI3,    (FP)_ddr_rzt1_scif_intr, IPL_SCIF3}};
#endif

#ifdef SCIF_4

#if (RSTRG_4 == 14)
#define TRSTRG_4    (7<<8)
#elif (RSTRG_4 == 12)
#define TRSTRG_4    (6<<8)
#elif (RSTRG_4 == 10)
#define TRSTRG_4    (5<<8)
#elif (RSTRG_4 == 8)
#define TRSTRG_4    (4<<8)
#elif (RSTRG_4 == 6)
#define TRSTRG_4    (3<<8)
#elif (RSTRG_4 == 4)
#define TRSTRG_4    (2<<8)
#elif (RSTRG_4 == 1)
#define TRSTRG_4    (1<<8)
#else
#define TRSTRG_4    (0<<8)
#endif

#if (RTRG_4 == 14)
#define TRTRG_4     (3<<6)
#elif (RTRG_4 == 8)
#define TRTRG_4     (2<<6)
#elif (RTRG_4 == 4)
#define TRTRG_4     (1<<6)
#else
#define TRTRG_4     (0<<6)
#endif

#if (TTRG_4 == 0)
#define TTTRG_4     (3<<4)
#elif (TTRG_4 == 2)
#define TTTRG_4     (2<<4)
#elif (TTRG_4 == 4)
#define TTTRG_4     (1<<4)
#else
#define TTTRG_4     (0<<4)
#endif

#if (TXBUF_SZ4==0)
#define _ddr_rzt1_scif_tbuf4  0
#else
VB  _ddr_rzt1_scif_tbuf4[TXBUF_SZ4];
#endif
VB  _ddr_rzt1_scif_rbuf4[RXBUF_SZ4];
UB  _ddr_rzt1_scif_sbuf4[RXBUF_SZ4];
T_RZT1_SCIF_MNG _ddr_rzt1_scif_data4;

const T_CDEV _ddr_rzt1_scif_cdev4 = {&_ddr_rzt1_scif_data4, (FP)_ddr_rzt1_scifdrv};
const T_CFLG _ddr_rzt1_scif_cflg4 = {TA_TFIFO|TA_WMUL, 0x00000000};
const T_CISR _ddr_rzt1_scif_cisr4[4] = {
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data4, INT_SCIFA_BRI_ERI4,    (FP)_ddr_rzt1_scif_intr, IPL_SCIF4},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data4, INT_SCIFA_RXI4,        (FP)_ddr_rzt1_scif_intr, IPL_SCIF4},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data4, INT_SCIFA_TXI4,        (FP)_ddr_rzt1_scif_intr, IPL_SCIF4},
        {TA_HLNG, (VP_INT)&_ddr_rzt1_scif_data4, INT_SCIFA_TEI_DRI4,    (FP)_ddr_rzt1_scif_intr, IPL_SCIF4}};
#endif

/* Private functions ---------------------------------------------------------*/


/**
  * @brief Clock supply for SCIF module.
  * @param[in] channel
  */
static ER _ddr_rzt1_scif_clk_supply(UW channel)
{
    ER ercd;
    volatile UW dummy32;
    UW bit;

    if (channel >= 5){
        ercd = E_PAR;
    } else {
        /* Unlock Register Write Protection */
        SYSTEM.PRCR.LONG = PRCR_MSTP_UNLOCK;
        dummy32 = SYSTEM.PRCR.LONG;

        bit = (0x00000200 >> channel);

        /* Clock Supply */
        SYSTEM.MSTPCRB.LONG &= ~(bit);
        dummy32 = SYSTEM.MSTPCRB.LONG;

        /* Lock Register Write Protection */
        SYSTEM.PRCR.LONG = PRCR_MSTP_LOCK;
        dummy32 = SYSTEM.PRCR.LONG;
    }

    return ercd;
}

/**
  * Initializes device driver.
  */
ER _ddr_rzt1_scif_init(ID devid, volatile struct st_scifa *scif_port)
{
    ER_ID ercd;

  #ifdef SCIF_0
    if (scif_port == &SCIFA0) {
        memset(&_ddr_rzt1_scif_data0, 0x00, sizeof(_ddr_rzt1_scif_data0));
        _ddr_rzt1_scif_data0.port = scif_port;
        _ddr_rzt1_scif_data0.rxi_picbit = (1<<1);   /* VIC.PIC3.BIT.PIC97 */
        _ddr_rzt1_scif_data0.txi_picbit = (1<<2);   /* VIC.PIC3.BIT.PIC98 */
        _ddr_rzt1_scif_data0.tbuf = _ddr_rzt1_scif_tbuf0;
        _ddr_rzt1_scif_data0.rbuf = _ddr_rzt1_scif_rbuf0;
        _ddr_rzt1_scif_data0.sbuf = _ddr_rzt1_scif_sbuf0;
        _ddr_rzt1_scif_data0.tsize = TXBUF_SZ0;
        _ddr_rzt1_scif_data0.rsize = RXBUF_SZ0;
        _ddr_rzt1_scif_data0.xoff_size = XOFF_SZ0;
        _ddr_rzt1_scif_data0.xon_size = XON_SZ0;
        _ddr_rzt1_scif_data0.aux[0] = TRSTRG_0 | TTTRG_0 | TRTRG_0;
        _ddr_rzt1_scif_data0.devhdr = (FP)_ddr_rzt1_scifdrv;
        if ((ercd = acre_flg((T_CFLG *)&_ddr_rzt1_scif_cflg0)) > E_OK) {
            _ddr_rzt1_scif_data0.flgid = (UH)ercd;
            ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr0[0]);
            if (ercd > E_OK) {
                _ddr_rzt1_scif_data0.isrid[0] = (UH)ercd;
                ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr0[1]);
                if (ercd > E_OK) {
                    _ddr_rzt1_scif_data0.isrid[1] = (UH)ercd;
                    ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr0[2]);
                    if (ercd > E_OK) {
                        _ddr_rzt1_scif_data0.isrid[2] = (UH)ercd;
                        ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr0[3]);
                        if (ercd > E_OK) {
                            _ddr_rzt1_scif_data0.isrid[3] = (UH)ercd;
                            ercd = vdef_dev(devid, (VP)&_ddr_rzt1_scif_cdev0);
                            if (ercd >= E_OK) {
                                loc_cpu();
                                _ddr_rzt1_scif_clk_supply(0);
                                SCIFA0.SCR.WORD = 0;
                                SCIFA0.SPTR.WORD = 0x00C3;
                                unl_cpu();
                                ena_int(INT_SCIFA_BRI_ERI0);
                                set_type_int(INT_SCIFA_RXI0, IRQ_TYPE_EDGE);
                                ena_int(INT_SCIFA_RXI0);
                                set_type_int(INT_SCIFA_TXI0, IRQ_TYPE_EDGE);
                                ena_int(INT_SCIFA_TXI0);
                                ena_int(INT_SCIFA_TEI_DRI0);
                            } else {
                                del_flg((ID)_ddr_rzt1_scif_data0.flgid);
                                del_isr((ID)_ddr_rzt1_scif_data0.isrid[0]);
                                del_isr((ID)_ddr_rzt1_scif_data0.isrid[1]);
                                del_isr((ID)_ddr_rzt1_scif_data0.isrid[2]);
                                del_isr((ID)_ddr_rzt1_scif_data0.isrid[3]);
                            }
                        } else {
                            del_flg((ID)_ddr_rzt1_scif_data0.flgid);
                            del_isr((ID)_ddr_rzt1_scif_data0.isrid[0]);
                            del_isr((ID)_ddr_rzt1_scif_data0.isrid[1]);
                            del_isr((ID)_ddr_rzt1_scif_data0.isrid[2]);
                        }
                    } else {
                        del_flg((ID)_ddr_rzt1_scif_data0.flgid);
                        del_isr((ID)_ddr_rzt1_scif_data0.isrid[0]);
                        del_isr((ID)_ddr_rzt1_scif_data0.isrid[1]);
                    }
                } else {
                    del_flg((ID)_ddr_rzt1_scif_data0.flgid);
                    del_isr((ID)_ddr_rzt1_scif_data0.isrid[0]);
                }
            } else {
                del_flg((ID)_ddr_rzt1_scif_data0.flgid);
            }
        }
    } else
  #endif


  #ifdef SCIF_1
    if (scif_port == &SCIFA1) {
        memset(&_ddr_rzt1_scif_data1, 0x00, sizeof(_ddr_rzt1_scif_data1));
        _ddr_rzt1_scif_data1.port = scif_port;
        _ddr_rzt1_scif_data1.rxi_picbit = (1<<5);   /* VIC.PIC3.BIT.PIC101 */
        _ddr_rzt1_scif_data1.txi_picbit = (1<<6);   /* VIC.PIC3.BIT.PIC102 */
        _ddr_rzt1_scif_data1.tbuf = _ddr_rzt1_scif_tbuf1;
        _ddr_rzt1_scif_data1.rbuf = _ddr_rzt1_scif_rbuf1;
        _ddr_rzt1_scif_data1.sbuf = _ddr_rzt1_scif_sbuf1;
        _ddr_rzt1_scif_data1.tsize = TXBUF_SZ1;
        _ddr_rzt1_scif_data1.rsize = RXBUF_SZ1;
        _ddr_rzt1_scif_data1.xoff_size = XOFF_SZ1;
        _ddr_rzt1_scif_data1.xon_size = XON_SZ1;
        _ddr_rzt1_scif_data1.aux[0] = TRSTRG_1 | TTTRG_1 | TRTRG_1;
        _ddr_rzt1_scif_data1.devhdr = (FP)_ddr_rzt1_scifdrv;
        if ((ercd = acre_flg((T_CFLG *)&_ddr_rzt1_scif_cflg1)) > E_OK) {
            _ddr_rzt1_scif_data1.flgid = (UH)ercd;
            ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr1[0]);
            if (ercd > E_OK) {
                _ddr_rzt1_scif_data1.isrid[0] = (UH)ercd;
                ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr1[1]);
                if (ercd > E_OK) {
                    _ddr_rzt1_scif_data1.isrid[1] = (UH)ercd;
                    ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr1[2]);
                    if (ercd > E_OK) {
                        _ddr_rzt1_scif_data1.isrid[2] = (UH)ercd;
                        ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr1[3]);
                        if (ercd > E_OK) {
                            _ddr_rzt1_scif_data1.isrid[3] = (UH)ercd;
                            ercd = vdef_dev(devid, (VP)&_ddr_rzt1_scif_cdev1);
                            if (ercd >= E_OK) {
                                loc_cpu();
                                _ddr_rzt1_scif_clk_supply(1);
                                SCIFA1.SCR.WORD = 0;
                                SCIFA1.SPTR.WORD = 0x00C3;
                                unl_cpu();
                                ena_int(INT_SCIFA_BRI_ERI1);
                                set_type_int(INT_SCIFA_RXI1, IRQ_TYPE_EDGE);
                                ena_int(INT_SCIFA_RXI1);
                                set_type_int(INT_SCIFA_TXI1, IRQ_TYPE_EDGE);
                                ena_int(INT_SCIFA_TXI1);
                                ena_int(INT_SCIFA_TEI_DRI1);
                            } else {
                                del_flg((ID)_ddr_rzt1_scif_data1.flgid);
                                del_isr((ID)_ddr_rzt1_scif_data1.isrid[0]);
                                del_isr((ID)_ddr_rzt1_scif_data1.isrid[1]);
                                del_isr((ID)_ddr_rzt1_scif_data1.isrid[2]);
                                del_isr((ID)_ddr_rzt1_scif_data1.isrid[3]);
                            }
                        } else {
                            del_flg((ID)_ddr_rzt1_scif_data1.flgid);
                            del_isr((ID)_ddr_rzt1_scif_data1.isrid[0]);
                            del_isr((ID)_ddr_rzt1_scif_data1.isrid[1]);
                            del_isr((ID)_ddr_rzt1_scif_data1.isrid[2]);
                        }
                    } else {
                        del_flg((ID)_ddr_rzt1_scif_data1.flgid);
                        del_isr((ID)_ddr_rzt1_scif_data1.isrid[0]);
                        del_isr((ID)_ddr_rzt1_scif_data1.isrid[1]);
                    }
                } else {
                    del_flg((ID)_ddr_rzt1_scif_data1.flgid);
                    del_isr((ID)_ddr_rzt1_scif_data1.isrid[0]);
                }
            } else {
                del_flg((ID)_ddr_rzt1_scif_data1.flgid);
            }
        }
    } else
  #endif


  #ifdef SCIF_2
    if (scif_port == &SCIFA2) {
        memset(&_ddr_rzt1_scif_data2, 0x00, sizeof(_ddr_rzt1_scif_data2));
        _ddr_rzt1_scif_data2.port = scif_port;
        _ddr_rzt1_scif_data2.rxi_picbit = (1<<14);   /* VIC.PIC3.BIT.PIC110 */
        _ddr_rzt1_scif_data2.txi_picbit = (1<<15);   /* VIC.PIC3.BIT.PIC111 */
        _ddr_rzt1_scif_data2.tbuf = _ddr_rzt1_scif_tbuf2;
        _ddr_rzt1_scif_data2.rbuf = _ddr_rzt1_scif_rbuf2;
        _ddr_rzt1_scif_data2.sbuf = _ddr_rzt1_scif_sbuf2;
        _ddr_rzt1_scif_data2.tsize = TXBUF_SZ2;
        _ddr_rzt1_scif_data2.rsize = RXBUF_SZ2;
        _ddr_rzt1_scif_data2.xoff_size = XOFF_SZ2;
        _ddr_rzt1_scif_data2.xon_size = XON_SZ2;
        _ddr_rzt1_scif_data2.aux[0] = TRSTRG_2 | TTTRG_2 | TRTRG_2;
        _ddr_rzt1_scif_data2.devhdr = (FP)_ddr_rzt1_scifdrv;
        if ((ercd = acre_flg((T_CFLG *)&_ddr_rzt1_scif_cflg2)) > E_OK) {
            _ddr_rzt1_scif_data2.flgid = (UH)ercd;
            ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr2[0]);
            if (ercd > E_OK) {
                _ddr_rzt1_scif_data2.isrid[0] = (UH)ercd;
                ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr2[1]);
                if (ercd > E_OK) {
                    _ddr_rzt1_scif_data2.isrid[1] = (UH)ercd;
                    ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr2[2]);
                    if (ercd > E_OK) {
                        _ddr_rzt1_scif_data2.isrid[2] = (UH)ercd;
                        ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr2[3]);
                        if (ercd > E_OK) {
                            _ddr_rzt1_scif_data2.isrid[3] = (UH)ercd;
                            ercd = vdef_dev(devid, (VP)&_ddr_rzt1_scif_cdev2);
                            if (ercd >= E_OK) {
                                loc_cpu();
                                _ddr_rzt1_scif_clk_supply(2);
                                SCIFA2.SCR.WORD  = 0;
                                SCIFA2.SPTR.WORD = 0x00C3;
                                unl_cpu();
                                ena_int(INT_SCIFA_BRI_ERI2);
                                set_type_int(INT_SCIFA_RXI2, IRQ_TYPE_EDGE);
                                ena_int(INT_SCIFA_RXI2);
                                set_type_int(INT_SCIFA_TXI2, IRQ_TYPE_EDGE);
                                ena_int(INT_SCIFA_TXI2);
                                ena_int(INT_SCIFA_TEI_DRI2);
                            } else {
                                del_flg((ID)_ddr_rzt1_scif_data2.flgid);
                                del_isr((ID)_ddr_rzt1_scif_data2.isrid[0]);
                                del_isr((ID)_ddr_rzt1_scif_data2.isrid[1]);
                                del_isr((ID)_ddr_rzt1_scif_data2.isrid[2]);
                                del_isr((ID)_ddr_rzt1_scif_data2.isrid[3]);
                            }
                        } else {
                            del_flg((ID)_ddr_rzt1_scif_data2.flgid);
                            del_isr((ID)_ddr_rzt1_scif_data2.isrid[0]);
                            del_isr((ID)_ddr_rzt1_scif_data2.isrid[1]);
                            del_isr((ID)_ddr_rzt1_scif_data2.isrid[2]);
                        }
                    } else {
                        del_flg((ID)_ddr_rzt1_scif_data2.flgid);
                        del_isr((ID)_ddr_rzt1_scif_data2.isrid[0]);
                        del_isr((ID)_ddr_rzt1_scif_data2.isrid[1]);
                    }
                } else {
                    del_flg((ID)_ddr_rzt1_scif_data2.flgid);
                    del_isr((ID)_ddr_rzt1_scif_data2.isrid[0]);
                }
            } else {
                del_flg((ID)_ddr_rzt1_scif_data2.flgid);
            }
        }
    } else
  #endif


  #ifdef SCIF_3
    if (scif_port == &SCIFA3) {
        memset(&_ddr_rzt1_scif_data3, 0x00, sizeof(_ddr_rzt1_scif_data3));
        _ddr_rzt1_scif_data3.port = scif_port;
        _ddr_rzt1_scif_data3.rxi_picbit = (1<<18);   /* VIC.PIC3.BIT.PIC114 */
        _ddr_rzt1_scif_data3.txi_picbit = (1<<19);   /* VIC.PIC3.BIT.PIC115 */
        _ddr_rzt1_scif_data3.tbuf = _ddr_rzt1_scif_tbuf3;
        _ddr_rzt1_scif_data3.rbuf = _ddr_rzt1_scif_rbuf3;
        _ddr_rzt1_scif_data3.sbuf = _ddr_rzt1_scif_sbuf3;
        _ddr_rzt1_scif_data3.tsize = TXBUF_SZ3;
        _ddr_rzt1_scif_data3.rsize = RXBUF_SZ3;
        _ddr_rzt1_scif_data3.xoff_size = XOFF_SZ3;
        _ddr_rzt1_scif_data3.xon_size = XON_SZ3;
        _ddr_rzt1_scif_data3.aux[0] = TRSTRG_3 | TTTRG_3 | TRTRG_3;
        _ddr_rzt1_scif_data3.devhdr = (FP)_ddr_rzt1_scifdrv;
        if ((ercd = acre_flg((T_CFLG *)&_ddr_rzt1_scif_cflg3)) > E_OK) {
            _ddr_rzt1_scif_data3.flgid = (UH)ercd;
            ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr3[0]);
            if (ercd > E_OK) {
                _ddr_rzt1_scif_data3.isrid[0] = (UH)ercd;
                ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr3[1]);
                if (ercd > E_OK) {
                    _ddr_rzt1_scif_data3.isrid[1] = (UH)ercd;
                    ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr3[2]);
                    if (ercd > E_OK) {
                        _ddr_rzt1_scif_data3.isrid[2] = (UH)ercd;
                        ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr3[3]);
                        if (ercd > E_OK) {
                            _ddr_rzt1_scif_data3.isrid[3] = (UH)ercd;
                            ercd = vdef_dev(devid, (VP)&_ddr_rzt1_scif_cdev3);
                            if (ercd >= E_OK) {
                                loc_cpu();
                                _ddr_rzt1_scif_clk_supply(3);
                                SCIFA3.SCR.WORD = 0;
                                SCIFA3.SPTR.WORD = 0x00C3;
                                unl_cpu();
                                ena_int(INT_SCIFA_BRI_ERI3);
                                set_type_int(INT_SCIFA_RXI3, IRQ_TYPE_EDGE);
                                ena_int(INT_SCIFA_RXI3);
                                set_type_int(INT_SCIFA_TXI3, IRQ_TYPE_EDGE);
                                ena_int(INT_SCIFA_TXI3);
                                ena_int(INT_SCIFA_TEI_DRI3);
                            } else {
                                del_flg((ID)_ddr_rzt1_scif_data3.flgid);
                                del_isr((ID)_ddr_rzt1_scif_data3.isrid[0]);
                                del_isr((ID)_ddr_rzt1_scif_data3.isrid[1]);
                                del_isr((ID)_ddr_rzt1_scif_data3.isrid[2]);
                                del_isr((ID)_ddr_rzt1_scif_data3.isrid[3]);
                            }
                        } else {
                            del_flg((ID)_ddr_rzt1_scif_data3.flgid);
                            del_isr((ID)_ddr_rzt1_scif_data3.isrid[0]);
                            del_isr((ID)_ddr_rzt1_scif_data3.isrid[1]);
                            del_isr((ID)_ddr_rzt1_scif_data3.isrid[2]);
                        }
                    } else {
                        del_flg((ID)_ddr_rzt1_scif_data3.flgid);
                        del_isr((ID)_ddr_rzt1_scif_data3.isrid[0]);
                        del_isr((ID)_ddr_rzt1_scif_data3.isrid[1]);
                    }
                } else {
                    del_flg((ID)_ddr_rzt1_scif_data3.flgid);
                    del_isr((ID)_ddr_rzt1_scif_data3.isrid[0]);
                }
            } else {
                del_flg((ID)_ddr_rzt1_scif_data3.flgid);
            }
        }
    } else
  #endif


  #ifdef SCIF_4
    if (scif_port == &SCIFA4) {
        memset(&_ddr_rzt1_scif_data4, 0x00, sizeof(_ddr_rzt1_scif_data4));
        _ddr_rzt1_scif_data4.port = scif_port;
        _ddr_rzt1_scif_data4.rxi_picbit = (1<<21);   /* VIC.PIC3.BIT.PIC117 */
        _ddr_rzt1_scif_data4.txi_picbit = (1<<22);   /* VIC.PIC3.BIT.PIC118 */
        _ddr_rzt1_scif_data4.tbuf = _ddr_rzt1_scif_tbuf4;
        _ddr_rzt1_scif_data4.rbuf = _ddr_rzt1_scif_rbuf4;
        _ddr_rzt1_scif_data4.sbuf = _ddr_rzt1_scif_sbuf4;
        _ddr_rzt1_scif_data4.tsize = TXBUF_SZ4;
        _ddr_rzt1_scif_data4.rsize = RXBUF_SZ4;
        _ddr_rzt1_scif_data4.xoff_size = XOFF_SZ4;
        _ddr_rzt1_scif_data4.xon_size = XON_SZ4;
        _ddr_rzt1_scif_data4.aux[0] = TRSTRG_4 | TTTRG_4 | TRTRG_4;
        _ddr_rzt1_scif_data4.devhdr = (FP)_ddr_rzt1_scifdrv;
        if ((ercd = acre_flg((T_CFLG *)&_ddr_rzt1_scif_cflg4)) > E_OK) {
            _ddr_rzt1_scif_data4.flgid = (UH)ercd;
            ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr4[0]);
            if (ercd > E_OK) {
                _ddr_rzt1_scif_data4.isrid[0] = (UH)ercd;
                ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr4[1]);
                if (ercd > E_OK) {
                    _ddr_rzt1_scif_data4.isrid[1] = (UH)ercd;
                    ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr4[2]);
                    if (ercd > E_OK) {
                        _ddr_rzt1_scif_data4.isrid[2] = (UH)ercd;
                        ercd = acre_isr((T_CISR *)&_ddr_rzt1_scif_cisr4[3]);
                        if (ercd > E_OK) {
                            _ddr_rzt1_scif_data4.isrid[3] = (UH)ercd;
                            ercd = vdef_dev(devid, (VP)&_ddr_rzt1_scif_cdev4);
                            if (ercd >= E_OK) {
                                loc_cpu();
                                _ddr_rzt1_scif_clk_supply(4);
                                SCIFA4.SCR.WORD = 0;
                                SCIFA4.SPTR.WORD = 0x00C3;
                                unl_cpu();
                                ena_int(INT_SCIFA_BRI_ERI4);
                                set_type_int(INT_SCIFA_RXI4, IRQ_TYPE_EDGE);
                                ena_int(INT_SCIFA_RXI4);
                                set_type_int(INT_SCIFA_TXI4, IRQ_TYPE_EDGE);
                                ena_int(INT_SCIFA_TXI4);
                                ena_int(INT_SCIFA_TEI_DRI4);
                            } else {
                                del_flg((ID)_ddr_rzt1_scif_data4.flgid);
                                del_isr((ID)_ddr_rzt1_scif_data4.isrid[0]);
                                del_isr((ID)_ddr_rzt1_scif_data4.isrid[1]);
                                del_isr((ID)_ddr_rzt1_scif_data4.isrid[2]);
                                del_isr((ID)_ddr_rzt1_scif_data4.isrid[3]);
                            }
                        } else {
                            del_flg((ID)_ddr_rzt1_scif_data4.flgid);
                            del_isr((ID)_ddr_rzt1_scif_data4.isrid[0]);
                            del_isr((ID)_ddr_rzt1_scif_data4.isrid[1]);
                            del_isr((ID)_ddr_rzt1_scif_data4.isrid[2]);
                        }
                    } else {
                        del_flg((ID)_ddr_rzt1_scif_data4.flgid);
                        del_isr((ID)_ddr_rzt1_scif_data4.isrid[0]);
                        del_isr((ID)_ddr_rzt1_scif_data4.isrid[1]);
                    }
                } else {
                    del_flg((ID)_ddr_rzt1_scif_data4.flgid);
                    del_isr((ID)_ddr_rzt1_scif_data4.isrid[0]);
                }
            } else {
                del_flg((ID)_ddr_rzt1_scif_data4.flgid);
            }
        }
    } else
  #endif
        ercd = E_NOEXS;

    return ercd;
}

#if (defined(SCIF_0)||defined(SCIF_1)||defined(SCIF_2)||defined(SCIF_3)||defined(SCIF_4))

/**
  * @brief Check receipt end.
  * @param[in] port
  */
BOOL _ddr_rzt1_scif_check_chr(T_COM_RCV *ReceiveData, VB chr, UB sts)
{
    BOOL flag;

    if ((sts & (T_COM_EROR|T_COM_ERP|T_COM_ERF|T_COM_BRK)) != 0) {
        flag = TRUE;
    } else if (ReceiveData->rcnt == 0) {
        flag = TRUE;
    } else if (ReceiveData->eos != 0) {
        if ((ReceiveData->eos->flg[0] != 0) && (ReceiveData->eos->chr[0] == chr)) {
            flag = TRUE;
        } else if ((ReceiveData->eos->flg[1] != 0) && (ReceiveData->eos->chr[1] == chr)) {
            flag = TRUE;
        } else if ((ReceiveData->eos->flg[2] != 0) && (ReceiveData->eos->chr[2] == chr)) {
            flag = TRUE;
        } else if ((ReceiveData->eos->flg[3] != 0) && (ReceiveData->eos->chr[3] == chr)) {
            flag = TRUE;
        } else {
            flag = FALSE;
        }
    } else {
        flag = FALSE;
    }
    return flag;
}

/**
  * @brief Analyze SCIF status.
  * @param[in] port
  */
UB _ddr_rzt1_scif_getssr(volatile struct st_scifa *port)
{
    UH ssr;
    UB sts = 0;

    ssr = port->FSR.WORD;
    if ((ssr & 0x0004) != 0) {
        sts |= T_COM_ERP;
    }
    if ((ssr & 0x0008) != 0) {
        sts |= T_COM_ERF;
    }
    if ((ssr & 0x0010) != 0) {
        sts |= T_COM_BRK;
        port->FSR.WORD = ~0x0010;
    }
    return sts;
}

/**
  * @brief Copy buffer (from local to internal).
  * @param[in/out] pk_SCIFmng
  * @param[in] TransmiteData
  */
BOOL _ddr_rzt1_scif_copy(T_RZT1_SCIF_MNG *pk_SCIFmng, T_COM_SND *TransmiteData)
{
    for(; TransmiteData->tcnt != 0; ) {
        if (pk_SCIFmng->tcnt < pk_SCIFmng->tsize) {
            pk_SCIFmng->tbuf[pk_SCIFmng->sndp++] = *TransmiteData->tbuf++;
            TransmiteData->tcnt--;
            if (pk_SCIFmng->sndp >= pk_SCIFmng->tsize) {
                pk_SCIFmng->sndp = 0;
            }
            pk_SCIFmng->tcnt++;
        } else {
            break;
        }
    }
    return (TransmiteData->tcnt == 0) ? TRUE : FALSE;
}

/**
  * @brief Check XOFF
  * @param[in] pk_SCIFmng
  */
void _ddr_rzt1_scif_chk_rxoff(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    if ((pk_SCIFmng->status.bit.rx_xoff == 0) &&
        (pk_SCIFmng->rcnt >= pk_SCIFmng->xoff_size)) {
        pk_SCIFmng->status.bit.rx_xoff = 1;
        if (((pk_SCIFmng->port->FDR.WORD >> 8) & 0x001F) < 16) {
            pk_SCIFmng->port->FTDR = XOFF;
        } else {
            loc_cpu();
            pk_SCIFmng->status.bit.req_xon_xoff = 1;
            pk_SCIFmng->port->SCR.WORD |= 0x0080;
            unl_cpu();
        }
    }
}

/**
  * @brief Check XON
  * @param[in] pk_SCIFmng
  */
void _ddr_rzt1_scif_chk_rxon(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    if ((pk_SCIFmng->status.bit.rx_xoff == 1) &&
        (pk_SCIFmng->rcnt <= pk_SCIFmng->xon_size)) {
        pk_SCIFmng->status.bit.rx_xoff = 0;
        if (((pk_SCIFmng->port->FDR.WORD >> 8) & 0x001F) < 16) {
            pk_SCIFmng->port->FTDR = XON;
        } else {
            loc_cpu();
            pk_SCIFmng->status.bit.req_xon_xoff = 1;
            pk_SCIFmng->port->SCR.WORD |= 0x0080;
            unl_cpu();
        }
    }
}

/**
  * @brief Transmit data from local buffer(app buffer)
  * @param[in] pk_SCIFmng
  */
void _ddr_rzt1_scif_send_local_buf(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    for(; ((pk_SCIFmng->port->FDR.WORD >> 8) & 0x001F) < 16; ) {
        pk_SCIFmng->port->FTDR = *pk_SCIFmng->SndData->tbuf++;
        if ((pk_SCIFmng->port->FSR.WORD & 0x0020) != 0) {
            pk_SCIFmng->port->FSR.WORD = ~0x0020;   /* Clear TDFE */
        }
        if ((pk_SCIFmng->port->FSR.WORD & 0x0040) != 0) {
            pk_SCIFmng->port->FSR.WORD = ~0x0040;   /* Clear TEND */
        }
        if (--pk_SCIFmng->SndData->tcnt == 0) {
            pk_SCIFmng->SndData = 0;
            pk_SCIFmng->port->SCR.WORD &= ~0x0080;
            iset_flg((ID)pk_SCIFmng->flgid, TXI_FLG);
            break;
        }
    }
}

/**
  * @brief Transmit data from internal buffer
  * @param[in] pk_SCIFmng
  */
void _ddr_rzt1_scif_send_drv_buf(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    UH sndp;

    sndp = pk_SCIFmng->sndp - pk_SCIFmng->tcnt;
    if (pk_SCIFmng->tcnt > pk_SCIFmng->sndp) {
        sndp += pk_SCIFmng->tsize;
    }
    for(; ((pk_SCIFmng->port->FDR.WORD >> 8) & 0x001F) < 16; ) {
        pk_SCIFmng->port->FTDR = pk_SCIFmng->tbuf[sndp];
        if ((pk_SCIFmng->port->FSR.WORD & 0x0020) != 0) {
            pk_SCIFmng->port->FSR.WORD = ~0x0020;   /* Clear TDFE */
        }
        if ((pk_SCIFmng->port->FSR.WORD & 0x0040) != 0) {
            pk_SCIFmng->port->FSR.WORD = ~0x0040;   /* Clear TEND */
        }
        if (++sndp >= pk_SCIFmng->tsize) {
            sndp = 0;
        }
        if (--pk_SCIFmng->tcnt == 0) {
            break;
        }
    }
    if (pk_SCIFmng->SndData != 0) {
        if (_ddr_rzt1_scif_copy(pk_SCIFmng, pk_SCIFmng->SndData) == TRUE) {
            pk_SCIFmng->SndData = 0;
            iset_flg((ID)pk_SCIFmng->flgid, TXI_FLG);
        }
    }
    if ((pk_SCIFmng->tcnt == 0) && (pk_SCIFmng->SndData == 0)) {
        pk_SCIFmng->port->SCR.WORD &= ~0x0080;
    }
}

/**
  * @brief BRI interrupt
  * @param[in] pk_SCIFmng
  */
void _ddr_rzt1_scif_bri(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    UH rcvp;

    /* Overrun */
    if ((pk_SCIFmng->port->LSR.WORD & 0x0001) != 0) {
        pk_SCIFmng->port->LSR.WORD = 0;
        pk_SCIFmng->status.bit.er_buf_ovr = 1;
    }

    /* Break */
    if ((pk_SCIFmng->port->FSR.WORD & 0x0010) != 0) {
        pk_SCIFmng->port->FSR.WORD = ~0x0010;
        if (pk_SCIFmng->RcvData != 0) {
            if (pk_SCIFmng->RcvData->sbuf != 0) {
                pk_SCIFmng->RcvData->rcnt--;
                *pk_SCIFmng->RcvData->rbuf++ = 0;
                *pk_SCIFmng->RcvData->sbuf++ = (pk_SCIFmng->status.bit.er_buf_ovr == 0) ? T_COM_BRK : (T_COM_BRK | T_COM_EROVB);
                pk_SCIFmng->status.bit.er_buf_ovr = 0;
            }
        } else if (pk_SCIFmng->rcnt < pk_SCIFmng->rsize) {
            rcvp = pk_SCIFmng->rcvp + pk_SCIFmng->rcnt;
            if (rcvp >= pk_SCIFmng->rsize) {
                rcvp -= pk_SCIFmng->rsize;
            }
            if (pk_SCIFmng->sbuf != 0) {
                pk_SCIFmng->rbuf[rcvp] = 0;
                pk_SCIFmng->sbuf[rcvp] = (pk_SCIFmng->status.bit.er_buf_ovr == 0) ? T_COM_BRK : (T_COM_BRK | T_COM_EROVB);
                pk_SCIFmng->status.bit.er_buf_ovr = 0;
                if (pk_SCIFmng->status.bit.sft_flw == 1) {
                    _ddr_rzt1_scif_chk_rxoff(pk_SCIFmng);
                }
                if (++pk_SCIFmng->rcnt == pk_SCIFmng->rsize) {
                    pk_SCIFmng->port->SCR.WORD &= ~0x0040;
                }
            }
        }
    }
}

/**
  * @brief TXI interrupt
  * @param[in] pk_SCIFmng
  */
void _ddr_rzt1_scif_txi(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    if (pk_SCIFmng->status.bit.req_xon_xoff == 1) {
        pk_SCIFmng->port->FTDR = (pk_SCIFmng->status.bit.rx_xoff == 0) ? XON : XOFF;
        if ((pk_SCIFmng->port->FSR.WORD & 0x0020) != 0) {
            pk_SCIFmng->port->FSR.WORD = ~0x0020;   /* Clear TDFE */
        }
        if ((pk_SCIFmng->port->FSR.WORD & 0x0040) != 0) {
            pk_SCIFmng->port->FSR.WORD = ~0x0040;   /* Clear TEND */
        }
        pk_SCIFmng->status.bit.req_xon_xoff = 0;
    }
    if (pk_SCIFmng->status.bit.tx_xoff == 0) {
        if (pk_SCIFmng->tsize == 0) {
            if (pk_SCIFmng->SndData != 0) {
                _ddr_rzt1_scif_send_local_buf(pk_SCIFmng);
            } else {
                pk_SCIFmng->port->SCR.WORD &= ~0x0080;
            }
        } else {
            if (pk_SCIFmng->tcnt != 0) {
                _ddr_rzt1_scif_send_drv_buf(pk_SCIFmng);
            } else {
                pk_SCIFmng->port->SCR.WORD &= ~0x0080;
            }
       }
    } else {
        pk_SCIFmng->port->SCR.WORD &= ~0x0080;
    }

    if ((pk_SCIFmng->tcnt == 0) && (pk_SCIFmng->aux[1] != 0)) {
        iset_flg((ID)pk_SCIFmng->flgid, TEI_FLG);
        pk_SCIFmng->aux[1] = 0;
    }
}

/**
  * @brief RXI, DRI, ERI interrupt
  * @param[in] pk_SCIFmng
  */
void _ddr_rzt1_scif_rxi(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    UH rcvp;
    VB chr;
    UB sts;

    if ((pk_SCIFmng->rcnt < pk_SCIFmng->rsize) ||
        (pk_SCIFmng->RcvData != 0)) {
        for(; (pk_SCIFmng->port->FDR.WORD & 0x001F) != 0; ) {
            sts = _ddr_rzt1_scif_getssr(pk_SCIFmng->port);
            chr = pk_SCIFmng->port->FRDR;
            if ((pk_SCIFmng->status.bit.sft_flw == 1) && (chr == XON)) {
                pk_SCIFmng->status.bit.tx_xoff = 0;
                pk_SCIFmng->port->SCR.WORD |= 0x0080;
                if (((pk_SCIFmng->port->FDR.WORD >> 8) & 0x001F) < 16) {
                    _ddr_rzt1_scif_txi(pk_SCIFmng);
                }
            } else if ((pk_SCIFmng->status.bit.sft_flw == 1) && (chr == XOFF)) {
                pk_SCIFmng->status.bit.tx_xoff = 1;
                pk_SCIFmng->port->SCR.WORD &= ~0x0080;
            } else {
                if (pk_SCIFmng->RcvData != 0) {
                    pk_SCIFmng->RcvData->rcnt--;
                    *pk_SCIFmng->RcvData->rbuf++ = chr;
                    if (pk_SCIFmng->RcvData->sbuf != 0) {
                        *pk_SCIFmng->RcvData->sbuf++ = (pk_SCIFmng->status.bit.er_buf_ovr == 0) ? sts : (sts | T_COM_EROVB);
                    }
                    pk_SCIFmng->status.bit.er_buf_ovr = 0;
                    if (_ddr_rzt1_scif_check_chr(pk_SCIFmng->RcvData, chr, sts) == TRUE) {
                        pk_SCIFmng->RcvData = 0;
                        iset_flg((ID)pk_SCIFmng->flgid, RXI_FLG);
                    }
                } else if (pk_SCIFmng->rcnt < pk_SCIFmng->rsize) {
                    rcvp = pk_SCIFmng->rcvp + pk_SCIFmng->rcnt;
                    if (rcvp >= pk_SCIFmng->rsize) {
                        rcvp -= pk_SCIFmng->rsize;
                    }
                    pk_SCIFmng->rbuf[rcvp] = chr;
                    if (pk_SCIFmng->sbuf != 0) {
                        pk_SCIFmng->sbuf[rcvp] = (pk_SCIFmng->status.bit.er_buf_ovr == 0) ? sts : (sts | T_COM_EROVB);
                    }
                    pk_SCIFmng->status.bit.er_buf_ovr = 0;
                    if (pk_SCIFmng->status.bit.sft_flw == 1) {
                        _ddr_rzt1_scif_chk_rxoff(pk_SCIFmng);
                    }
                    if (++pk_SCIFmng->rcnt == pk_SCIFmng->rsize) {
                        pk_SCIFmng->port->SCR.WORD &= ~0x0040;
                        break;
                    }
                }
            }
        }
        if ((pk_SCIFmng->port->SCR.WORD & 0x0040) != 0) {
            if ((pk_SCIFmng->port->FSR.WORD & 0x0002) != 0) {
                pk_SCIFmng->port->FSR.WORD = ~0x0002;   /* Clear RDF */
            }
            if ((pk_SCIFmng->port->FSR.WORD & 0x0001) != 0) {
                pk_SCIFmng->port->FSR.WORD = ~0x0001;   /* Clear DR */
            }
        }
    } else {
        pk_SCIFmng->port->SCR.WORD &= ~0x0040;  /* Disable RIE */
    }
    if ((pk_SCIFmng->port->FSR.WORD & 0x0080) != 0) {
        pk_SCIFmng->port->FSR.WORD = ~0x0080;   /* Clear ER */
    }
}

/**
  * SCIF Interrupt service routine.
  */
void _ddr_rzt1_scif_intr(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
#if 0
    if (pk_SCIFmng->port->FSR.BIT.BRK  == 1 ||
        pk_SCIFmng->port->LSR.BIT.ORER == 1) {
        _ddr_rzt1_scif_bri(pk_SCIFmng);
    }

    if (pk_SCIFmng->port->FSR.BIT.ER  == 1 ||
        pk_SCIFmng->port->FSR.BIT.RDF == 1 ||
        pk_SCIFmng->port->FSR.BIT.DR  == 1) {
        if ((VIC.RAIS3.LONG & pk_SCIFmng->rxi_picbit) != 0) {
            VIC.PIC3.LONG = pk_SCIFmng->rxi_picbit;  /* Clear interrupt edge detection */
        }
        _ddr_rzt1_scif_rxi(pk_SCIFmng);
    }

    if (pk_SCIFmng->port->FSR.BIT.TDFE == 1) {
        VIC.PIC3.LONG = pk_SCIFmng->txi_picbit;  /* Clear interrupt edge detection */
        _ddr_rzt1_scif_txi(pk_SCIFmng);
    }
#else
    if (pk_SCIFmng->port->FSR.BIT.BRK  == 1 ||
        pk_SCIFmng->port->LSR.BIT.ORER == 1) {
        _ddr_rzt1_scif_bri(pk_SCIFmng);
    }

    if (pk_SCIFmng->port->FSR.BIT.ER == 1) {
        _ddr_rzt1_scif_rxi(pk_SCIFmng);
    }

    if (pk_SCIFmng->port->FSR.BIT.RDF == 1) {
        VIC.PIC3.LONG = pk_SCIFmng->rxi_picbit;  /* Clear interrupt edge detection */
        _ddr_rzt1_scif_rxi(pk_SCIFmng);
    }

    if (pk_SCIFmng->port->FSR.BIT.TDFE == 1) {
        VIC.PIC3.LONG = pk_SCIFmng->txi_picbit;  /* Clear interrupt edge detection */
        _ddr_rzt1_scif_txi(pk_SCIFmng);
    }

    if (pk_SCIFmng->port->FSR.BIT.DR == 1) {
        _ddr_rzt1_scif_rxi(pk_SCIFmng);
    }
#endif

}

/**
  * @brief Fills transmit FIFO.
  * @param[in] pk_SCIFmng
  * @param[in] TransmiteData
  */
void _ddr_rzt1_scif_send_char(T_RZT1_SCIF_MNG *pk_SCIFmng, T_COM_SND *TransmiteData)
{
    if (pk_SCIFmng->status.bit.req_xon_xoff == 1) {
        pk_SCIFmng->port->FTDR = (pk_SCIFmng->status.bit.rx_xoff == 0) ? XON : XOFF;
        if ((pk_SCIFmng->port->FSR.WORD & 0x0020) != 0) {
            pk_SCIFmng->port->FSR.WORD = ~0x0020;   /* Clear TDFE */
        }
        pk_SCIFmng->status.bit.req_xon_xoff = 0;
    }
    if (pk_SCIFmng->status.bit.tx_xoff == 0) {
        for(; ((pk_SCIFmng->port->FDR.WORD >> 8) & 0x001F) < 16; ) {
            pk_SCIFmng->port->FTDR = *TransmiteData->tbuf++;
            if ((pk_SCIFmng->port->FSR.WORD & 0x0020) != 0) {
                pk_SCIFmng->port->FSR.WORD = ~0x0020;   /* Clear TDFE */
            }
            if ((pk_SCIFmng->port->FSR.WORD & 0x0040) != 0) {
                pk_SCIFmng->port->FSR.WORD = ~0x0040;   /* Clear TEND */
            }
            if (--TransmiteData->tcnt == 0) {
                break;
            }
        }
    }
}

/**
  * @brief Get strings
  * @param[in] pk_SCIFmng
  * @param[in/out] ReceiveData
  */
BOOL _ddr_rzt1_scif_recv_strings(T_RZT1_SCIF_MNG *pk_SCIFmng, T_COM_RCV *ReceiveData)
{
    BOOL flag;
    VB chr;
    UB sts;

    for (flag = FALSE; flag == FALSE; ) {
        if (ReceiveData->rcnt == 0) {
            flag = TRUE;
        } else if (pk_SCIFmng->rcnt == 0) {
            break;
        } else {
            *ReceiveData->rbuf++ = chr = pk_SCIFmng->rbuf[pk_SCIFmng->rcvp];
            sts = pk_SCIFmng->sbuf[pk_SCIFmng->rcvp];
            if (ReceiveData->sbuf != 0) {
                *ReceiveData->sbuf++ = sts;
            }
            if (++pk_SCIFmng->rcvp >= pk_SCIFmng->rsize) {
                pk_SCIFmng->rcvp = 0;
            }
            pk_SCIFmng->rcnt--;
            ReceiveData->rcnt--;
            if (pk_SCIFmng->status.bit.sft_flw == 1) {
                _ddr_rzt1_scif_chk_rxon(pk_SCIFmng);
            }
            flag = _ddr_rzt1_scif_check_chr(ReceiveData, chr, sts);
        }
    }
    return flag;
}

/**
  * @brief Transmit data
  * @param[in] TransmiteData
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_snd(T_COM_SND *TransmiteData, T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    FLGPTN flgptn;
    ID tskid;
    ER ercd = E_OK;

    if ((pk_SCIFmng->status.bit.init_flg == 0) || (sns_dpn() == TRUE))
        ercd = E_OBJ;
    else {
        get_tid(&tskid);
        dis_dsp();
        if (((pk_SCIFmng->tlockid == 0) || (pk_SCIFmng->tlockid == (UH)tskid)) &&
            (pk_SCIFmng->status.bit.ena_tx == 1) &&
            (pk_SCIFmng->SndData == 0)) {
            loc_cpu();
            pk_SCIFmng->port->SCR.WORD &= ~0x00C0;
            unl_cpu();
            if ((pk_SCIFmng->tcnt == 0) && (((pk_SCIFmng->port->FDR.WORD >> 8) & 0x001F) < 16)) {
                _ddr_rzt1_scif_send_char(pk_SCIFmng, TransmiteData);
            }
            loc_cpu();
            pk_SCIFmng->port->SCR.WORD |= 0x0040;
            unl_cpu();
            if (_ddr_rzt1_scif_copy(pk_SCIFmng, TransmiteData) == FALSE) {
                pk_SCIFmng->SndData = TransmiteData;
                loc_cpu();
                pk_SCIFmng->port->SCR.WORD |= 0x0080;
                unl_cpu();
                ena_dsp();
                if ((ercd = twai_flg((ID)pk_SCIFmng->flgid, TXI_FLG, TWF_ORW, &flgptn, TransmiteData->time)) != E_OK) {
                    loc_cpu();
                    pk_SCIFmng->port->SCR.WORD &= ~0x0080;
                    unl_cpu();
                    pk_SCIFmng->SndData = 0;
                    ercd = pol_flg((ID)pk_SCIFmng->flgid, TXI_FLG, TWF_ORW, &flgptn);
                }
                clr_flg((ID)pk_SCIFmng->flgid, ~TXI_FLG);
            } else {
                if (pk_SCIFmng->tcnt != 0) {
                    loc_cpu();
                    pk_SCIFmng->port->SCR.WORD |= 0x0080;
                    unl_cpu();
                }
                ena_dsp();
            }
        } else {
            ena_dsp();
            ercd = E_OBJ;
        }
    }
    return ercd;
}

/**
  * @brief Receive data
  * @param[in/out] ReceiveData
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_rcv(T_COM_RCV *ReceiveData, T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    FLGPTN flgptn;
    ID tskid;
    ER ercd = E_OK;

    if ((pk_SCIFmng->status.bit.init_flg == 0) || (sns_dpn() == TRUE) ||
        (pk_SCIFmng->status.bit.ena_rx   == 0)) {
        ercd = E_OBJ;
    } else {
        get_tid(&tskid);
        dis_dsp();
        if (((pk_SCIFmng->rlockid == 0) || (pk_SCIFmng->rlockid == (UH)tskid)) &&
            (pk_SCIFmng->RcvData        == 0)) {
            loc_cpu();
            pk_SCIFmng->port->SCR.WORD &= ~0x0040;
            unl_cpu();
            if (_ddr_rzt1_scif_recv_strings(pk_SCIFmng, ReceiveData) == FALSE) {
                pk_SCIFmng->RcvData = ReceiveData;
                loc_cpu();
                pk_SCIFmng->port->SCR.WORD |= 0x0040;
                unl_cpu();
                ena_dsp();
                if ((ercd = twai_flg((ID)pk_SCIFmng->flgid, RXI_FLG, TWF_ORW, &flgptn, ReceiveData->time)) != E_OK) {
                    pk_SCIFmng->RcvData = 0;
                    ercd = pol_flg((ID)pk_SCIFmng->flgid, RXI_FLG, TWF_ORW, &flgptn);
                }
                clr_flg((ID)pk_SCIFmng->flgid, ~RXI_FLG);
            } else {
                loc_cpu();
                pk_SCIFmng->port->SCR.WORD |= 0x0040;
                unl_cpu();
                ena_dsp();
            }
        } else {
            ena_dsp();
            ercd = E_OBJ;
        }
    }
    return ercd;
}

/**
  * @brief Transmit data from transmit FIFO.
  * @param[in] pk_SCIFmng
  * @param[in] time
  */
ER _ddr_rzt1_scif_cln_tx_buf(T_RZT1_SCIF_MNG *pk_SCIFmng, TMO time)
{
    FLGPTN flgptn;
    ID tskid;
    ER ercd;

    get_tid(&tskid);
    if (((pk_SCIFmng->tlockid == 0) ||
         (pk_SCIFmng->tlockid == (UH)tskid)) &&
        ( pk_SCIFmng->SndData == 0)) {
        loc_cpu();
        if (pk_SCIFmng->tcnt != 0) {
            pk_SCIFmng->port->SCR.WORD |= 0x0080;
            pk_SCIFmng->aux[1] = tskid;
            unl_cpu();
            if ((ercd = twai_flg((ID)pk_SCIFmng->flgid, TEI_FLG, TWF_ORW, &flgptn, time)) != E_OK) {
                pk_SCIFmng->aux[1] = 0;
                ercd = pol_flg((ID)pk_SCIFmng->flgid, TEI_FLG, TWF_ORW, &flgptn);
            }
            clr_flg((ID)pk_SCIFmng->flgid, ~TEI_FLG);
        } else {
            unl_cpu();
            ercd = E_OK;
        }
        /* Wait for until FIFO empty */
        for(; ((pk_SCIFmng->port->FSR.WORD & 0x0040) == 0) && (ercd == E_OK); ) {
            ercd = dly_tsk(0);
        }
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/**
  * @brief Reset transmit FIFO data.
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_rst_tx_buf(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    ER ercd;
    ID tskid;

    get_tid(&tskid);
    if (((pk_SCIFmng->tlockid == 0) ||
         (pk_SCIFmng->tlockid == (UH)tskid)) &&
        ( pk_SCIFmng->SndData == 0)) {
        loc_cpu();
        pk_SCIFmng->port->FCR.WORD |= 0x0004;
        pk_SCIFmng->port->FCR.WORD &= ~0x0004;
        pk_SCIFmng->tcnt = 0;
        unl_cpu();
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/**
  * @brief Reset receive FIFO data.
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_rst_rx_buf(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    ER ercd;
    ID tskid;

    get_tid(&tskid);
    if (((pk_SCIFmng->rlockid == 0) ||
         (pk_SCIFmng->rlockid == (UH)tskid)) &&
        ( pk_SCIFmng->RcvData == 0)) {
        loc_cpu();
        pk_SCIFmng->port->FCR.WORD |= 0x0002;
        pk_SCIFmng->port->FCR.WORD &= ~0x0002;
        pk_SCIFmng->rcnt = 0;
        unl_cpu();
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/**
  * @brief Disable transmit operation.
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_dis_send(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    ER ercd;
    ID tskid;

    get_tid(&tskid);
    if (((pk_SCIFmng->tlockid == 0) ||
         (pk_SCIFmng->tlockid == (UH)tskid)) &&
        ( pk_SCIFmng->SndData == 0) &&
        ( pk_SCIFmng->tcnt    == 0)) {
        loc_cpu();
        pk_SCIFmng->status.bit.ena_tx = 0;
        unl_cpu();
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/**
  * @brief Disable recevie operation.
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_dis_rcv(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    ER ercd;
    ID tskid;

    get_tid(&tskid);
    if (((pk_SCIFmng->rlockid == 0) ||
         (pk_SCIFmng->rlockid == (UH)tskid)) &&
        ( pk_SCIFmng->RcvData == 0) &&
        ( pk_SCIFmng->rcnt    == 0) &&
        ((pk_SCIFmng->port->FDR.WORD & 0x001F) != 0)) {
        loc_cpu();
        pk_SCIFmng->status.bit.ena_rx = 0;
        pk_SCIFmng->port->SCR.WORD &= ~0x0050;
        unl_cpu();
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/**
  * @brief Enable transmit operation.
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_ena_send(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    ER ercd;
    ID tskid;

    get_tid(&tskid);
    if ((pk_SCIFmng->rlockid == 0) ||
        (pk_SCIFmng->rlockid == (UH)tskid)) {
        loc_cpu();
        pk_SCIFmng->status.bit.ena_tx = 1;
        pk_SCIFmng->port->SCR.WORD |= 0x0020;
        unl_cpu();
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/**
  * @brief Enable receive operation.
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_ena_rcv(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    ER ercd;
    ID tskid;

    get_tid(&tskid);
    if ((pk_SCIFmng->rlockid == 0) ||
        (pk_SCIFmng->rlockid == (UH)tskid)) {
        loc_cpu();
        pk_SCIFmng->status.bit.ena_rx = 1;
        pk_SCIFmng->port->SCR.WORD |= 0x0050;
        unl_cpu();
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/**
  * @brief Send break signal.
  * @param[in] pk_SCIFmng
  * @param[in] time
  */
ER _ddr_rzt1_scif_snd_brk(T_RZT1_SCIF_MNG *pk_SCIFmng, TMO time)
{
    ER ercd;
    ID tskid;

    get_tid(&tskid);
    if (((pk_SCIFmng->tlockid == 0) ||
         (pk_SCIFmng->tlockid == (UH)tskid)) &&
        ( pk_SCIFmng->SndData == 0) &&
        ( pk_SCIFmng->tcnt    == 0)) {
        if ((pk_SCIFmng->port->FSR.WORD & 0x0040) != 0) {
            loc_cpu();
            pk_SCIFmng->port->SPTR.WORD &= ~0x0001;
            pk_SCIFmng->port->SCR.WORD &= ~0x0020;
            unl_cpu();
            dly_tsk(time);
            loc_cpu();
            pk_SCIFmng->port->SPTR.WORD |= 0x0001;
            pk_SCIFmng->port->SCR.WORD |= 0x0020;
            unl_cpu();
            ercd = E_OK;
        } else {
            ercd = E_OBJ;
        }
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/**
  * @brief Lock transmit channel.
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_lock_trans(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    ER ercd;
    ID tskid;

    get_tid(&tskid);
    loc_cpu();
    if ((pk_SCIFmng->tlockid == 0) ||
        (pk_SCIFmng->SndData == 0)) {
        pk_SCIFmng->tlockid = (UH)tskid;
        ercd = E_OK;
    } else if (pk_SCIFmng->tlockid == (UH)tskid) {
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    unl_cpu();
    return ercd;
}

/**
  * @brief Lock receive channel.
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_lock_recv(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    ER ercd;
    ID tskid;

    get_tid(&tskid);
    loc_cpu();
    if ((pk_SCIFmng->rlockid == 0) ||
        (pk_SCIFmng->RcvData == 0)) {
        pk_SCIFmng->rlockid = (UH)tskid;
        ercd = E_OK;
    } else if (pk_SCIFmng->rlockid == (UH)tskid) {
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    unl_cpu();
    return ercd;
}

/**
  * @brief Unlock transmit channel.
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_unlock_trans(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    ER ercd;
    ID tskid;

    get_tid(&tskid);
    if (pk_SCIFmng->tlockid == (UH)tskid) {
        pk_SCIFmng->tlockid = 0;
        ercd = E_OK;
    } else if (pk_SCIFmng->tlockid == 0) {
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/**
  * @brief Unlock receive channel.
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_unlock_recv(T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    ER ercd;
    ID tskid;

    get_tid(&tskid);
    if (pk_SCIFmng->rlockid == (UH)tskid) {
        pk_SCIFmng->rlockid = 0;
        ercd = E_OK;
    } else if (pk_SCIFmng->rlockid == 0) {
        ercd = E_OK;
    } else {
        ercd = E_OBJ;
    }
    return ercd;
}

/**
  * @brief Control SCIF signal.
  * @param[in] pk_SerialData
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_ctr(const T_COM_CTR *pk_SerialData, T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    ER ercd = E_OK;

    if (pk_SCIFmng->status.bit.init_flg == 0) {
        ercd = E_OBJ;
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & CLN_TXBUF) == CLN_TXBUF) {
            ercd = _ddr_rzt1_scif_cln_tx_buf(pk_SCIFmng, pk_SerialData->time);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & RST_TXBUF) == RST_TXBUF) {
            ercd = _ddr_rzt1_scif_rst_tx_buf(pk_SCIFmng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & RST_RXBUF) == RST_RXBUF) {
            ercd = _ddr_rzt1_scif_rst_rx_buf(pk_SCIFmng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & STP_TX) == STP_TX) {
            ercd = _ddr_rzt1_scif_dis_send(pk_SCIFmng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & STP_RX) == STP_RX) {
            ercd = _ddr_rzt1_scif_dis_rcv(pk_SCIFmng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & STA_TX) == STA_TX) {
            ercd = _ddr_rzt1_scif_ena_send(pk_SCIFmng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & STA_RX) == STA_RX) {
            ercd = _ddr_rzt1_scif_ena_rcv(pk_SCIFmng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & SND_BRK) == SND_BRK) {
            ercd = _ddr_rzt1_scif_snd_brk(pk_SCIFmng, pk_SerialData->time);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & LOC_TX) == LOC_TX) {
            ercd = _ddr_rzt1_scif_lock_trans(pk_SCIFmng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & LOC_RX) == LOC_RX) {
            ercd = _ddr_rzt1_scif_lock_recv(pk_SCIFmng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & UNL_TX) == UNL_TX) {
            ercd = _ddr_rzt1_scif_unlock_trans(pk_SCIFmng);
        }
    }

    if (ercd == E_OK) {
        if ((pk_SerialData->command & UNL_RX) == UNL_RX) {
            ercd = _ddr_rzt1_scif_unlock_recv(pk_SCIFmng);
        }
    }

    return ercd;
}

/**
  * @brief Reference to serial port status.
  * @param[in/out] pk_SerialRef
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scif_ref(T_COM_REF *pk_SerialRef, T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    UH status = 0;

    if (pk_SCIFmng->status.bit.init_flg == 1) {
        loc_cpu();
        pk_SerialRef->rxcnt = pk_SCIFmng->rcnt;
        pk_SerialRef->rxcnt += (pk_SCIFmng->port->FDR.WORD & 0x001F);
        pk_SerialRef->txcnt = pk_SCIFmng->tcnt;
        pk_SerialRef->txcnt += ((pk_SCIFmng->port->FDR.WORD >> 8) & 0x001F);
        unl_cpu();

        status |= T_COM_DSR|T_COM_INIT;
        if ((pk_SCIFmng->port->SPTR.WORD & 0x0010) == 0) {
            status |= T_COM_CTS;
        }
        if (pk_SCIFmng->status.bit.ena_tx == 1) {
            status |= T_COM_ENATX;
        }
        if (pk_SCIFmng->status.bit.ena_rx == 1) {
            status |= T_COM_ENARX;
        }
        if (pk_SCIFmng->status.bit.rx_xoff == 1) {
            status |= T_COM_RXOFF;
        }
        if (pk_SCIFmng->status.bit.tx_xoff == 1) {
            status |= T_COM_TXOFF;
        }
    }
    pk_SerialRef->status = status;
    return E_OK;
}


/**
  * Initializes the UART function of the SCIFA.
  */
ER _ddr_rzt1_scif_ini(const T_COM_SMOD *pk_SerialMode, T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    UW sericlk;
    UW brr;
    ER ercd = E_OK;
    T_RZT1_SCIF_MSTS status;
    UH smr;
    UH fcr;
    volatile int i;

    pk_SCIFmng->port->SCR.WORD = 0;
    pk_SCIFmng->port->FCR.WORD = 0x0006; /* Reset FTDR/FRDR */
    status.word = 0;
    fcr = pk_SCIFmng->aux[0];
    sericlk = _ddr_rzt1_cpg_get_sericlk();
    for (smr = 0; smr < 4; smr++) {
        brr = sericlk / (16 * pk_SerialMode->baud) + 1;
        if (brr <= 512) {
            break;
        }
        sericlk /= 4;
    }
    if (smr == 4)
        ercd = E_PAR;

    switch(pk_SerialMode->blen) {
        case BLEN7:
            smr |= 0x40;
            break;
        case BLEN8:
            break;
        default:
            ercd = E_PAR;
            break;
    }

    switch(pk_SerialMode->par) {
        case PAR_ODD:
            smr |= 0x30;
            break;
        case PAR_EVEN:
            smr |= 0x20;
            break;
        default:
            break;
    }

    switch(pk_SerialMode->sbit) {
        case SBIT2:
            smr |= 0x08;
            break;
        case SBIT1:
            break;
        default:
            ercd = E_PAR;
            break;
    }

    switch(pk_SerialMode->flow) {
        case FLW_HARD:
            fcr |= 0x0008;
            status.bit.hrd_flw = 1;
            break;
        case FLW_XON:
            status.bit.sft_flw = 1;
            break;
        case FLW_NONE:
            break;
        default:
            ercd = E_PAR;
            break;
    }

    if (ercd == E_OK) {
        status.bit.init_flg = 1;
        pk_SCIFmng->status.word = status.word;
        pk_SCIFmng->port->SMR.WORD = smr;
        pk_SCIFmng->port->BRR_MDDR.BRR = (UB)brr/2-1;
        if (sns_dpn() == TRUE) {
            for(i=0; i<10000;i++);
        } else {
            dly_tsk(1);
        }
        pk_SCIFmng->port->FCR.WORD = fcr;
    }
    return ercd;
}

/**
  * @brief Driver main entry
  * @param[in] FuncID
  * @param[in] pk_ControlData
  * @param[in] pk_SCIFmng
  */
ER _ddr_rzt1_scifdrv(ID FuncID, VP pk_ControlData, T_RZT1_SCIF_MNG *pk_SCIFmng)
{
    ER ercd;

    switch(FuncID) {
        case TA_COM_INI:
            ercd = _ddr_rzt1_scif_ini((const T_COM_SMOD *)pk_ControlData, pk_SCIFmng);
            break;
        case TA_COM_REF:
            ercd = _ddr_rzt1_scif_ref((T_COM_REF *)pk_ControlData, pk_SCIFmng);
            break;
        case TA_COM_CTR:
            ercd = _ddr_rzt1_scif_ctr((const T_COM_CTR *)pk_ControlData, pk_SCIFmng);
            break;
        case TA_COM_PUTS:
            ercd = _ddr_rzt1_scif_snd((T_COM_SND *)pk_ControlData, pk_SCIFmng);
            break;
        case TA_COM_GETS:
            ercd = _ddr_rzt1_scif_rcv((T_COM_RCV *)pk_ControlData, pk_SCIFmng);
            break;
        default:
            ercd = E_NOSPT;
            break;
    }
    return ercd;
}
#endif
