/**
  * @file    DDR_CORTEXR_MPU.S
  * @brief   MPU and Cache management driver (Cortex-R)
  * @date    2015.02.03
  * @author  Copyright (c)  2016, eForce Co.,Ltd.  All rights reserved.
  *
  * @par     History
  *          - rev 1.0 (2016.02.03) yokota
  *            Initial version. migrate from ds-5.
  *            Not supported Sub-region disabling.
  */

        .text
        .code 32
        .align 2
/**
  * @brief   Initialize MPU
  *
  * @param[in] r0: Configuration table address
  */

        .globl  _ddr_cortexr_mpu_init
        .type   _ddr_cortexr_mpu_init, %function

_ddr_cortexr_mpu_init:
        mov     r10, r0
        mov     r0,  #0                 @ region number starts from 0.
        mrc     p15, 0, r1, c0, c0, 4   @ get DRegion spec.
        lsr     r1,  r1, #8

@ r10 ... configuration table
mpu_init_1:
        ldmia   r10!,{r7-r9}
        
@ r7 ...... size
@ r8 ...... physical address
@ r9 ...... access permission
@ [r10] ... region attribute

        teq     r7,  #0 @ Check terminate
        beq     mpu_init_e

        cmp     r0, r1  @ 
        bhs     mpu_init_e

mpu_init_2:
        mcr     p15, 0, r0, c6, c2, 0       @ RGNR(MPU Memory Region Number Register)
        isb                                 @ Ensure subsequent insts execute wrt region n settings
        
        mcr     p15, 0, r8, c6, c1, 0       @ DRBAR(Data Region Base Address Register)
        isb

        ldr     r2,  [r10]
        orr     r2,  r2, r9
        mcr     p15, 0, r2, c6, c1, 4       @ DRACR(Data Region Access Control Register)
        isb

        lsl     r7,  r7, #1
        orr     r7,  r7, #1
        mcr     p15, 0, r7, c6, c1, 2       @ DRSR(Data Region Size and Enable Register)
        isb

        add     r0,  r0,  #1                @ update region number
        add     r10, r10, #4*1
        b       mpu_init_1
mpu_init_e:
        dsb
        mov     r0,  #0x0
        mcr     p15,  0,  r0,  c7,  c5,  6  @ Invalidate entire branch predictor array (NOP)
        isb
        
        @ Enable MPU and caches
        mrc     p15,  0,  r0,  c1,  c0,  0  
        orr     r0,  r0, #0x00000001        @ Enable MPU (M bit  0)
        orr     r0,  r0, #0x00000004        @ Enables L1 data cache (C bit  2)
        orr     r0,  r0, #0x00000800        @ Branch prediction enable bit. (SBO)
        orr     r0,  r0, #0x00001000        @ Enables L1 instruction cache (I bit 12)
        @orr     r0,  r0, #0x00020000        @ Enables MPU background region.
        dsb                                 @ Ensure all previous loads/stores have completed
        mcr     p15,  0,  r0,  c1,  c0,  0
        isb                                 @ Ensure subsequent insts execute wrt new MPU settings
        bx      lr


/**
  * @brief   Enable MPU
  */
        .globl  _ddr_cortexr_mpu_ena
        .type   _ddr_cortexr_mpu_ena, %function

_ddr_cortexr_mpu_ena:
        mrc     p15, #0, r0, c1, c0, #0
        orr     r0,  r0, #1
        dsb
        mcr     p15, #0, r0, c1, c0, #0
        isb
        bx      lr


/**
  * @brief   Disable MPU
  */
        .globl  _ddr_cortexr_mpu_dis
        .type   _ddr_cortexr_mpu_dis, %function

_ddr_cortexr_mpu_dis:
        mrc     p15, #0, r0, c1, c0, #0
        bic     r0,  r0, #1
        dsb
        mcr     p15, #0, r0, c1, c0, #0
        isb
        bx      lr


/**
  * @brief   Clean data cache
  *        void _kernel_clean_data_cache(void *, SIZE);
  */
        .globl _kernel_clean_data_cache
        .type  _kernel_clean_data_cache, %function

_kernel_clean_data_cache:
        mrc     p15,  1,  r3,  c0,  c0,  0  @ Read Current Cache Size Identification Register
        and     r3,  r3,  #0x07
        mov     r2,  #16
        cmp     r3,  #2
        moveq   r2,  #64
        cmp     r3,  #1
        moveq   r2,  #32
        teq     r1,  #0
        add     r3,  r2,  #-1
        moveq   r1,  r2

        add     r1,  r1,  r0
        bic     r0,  r0,  r3
        sub     r1,  r1,  #1
        bic     r1,  r1,  r3
clean_data_cache_1:
        mcr     p15,  0,  r0,  c7, c10,  1  @ Clean data cache line to Point-of-Coherency by MVA
        add     r0,  r0,  r2
        cmp     r0,  r1
        bls     clean_data_cache_1
        dsb
        bx      lr


/**
  * @brief   Invalidate data cache
  *        void _kernel_invalid_data_cache(void *, SIZE);
  */

        .globl _kernel_invalid_data_cache
        .type  _kernel_invalid_data_cache, %function

_kernel_invalid_data_cache:
        mrc     p15,  1,  r3,  c0,  c0,  0  @ Read Current Cache Size Identification Register
        and     r3,  r3,  #0x07
        mov     r2,  #16
        cmp     r3,  #2
        moveq   r2,  #64
        cmp     r3,  #1
        moveq   r2,  #32
        teq     r1,  #0
        add     r3,  r2,  #-1
        moveq   r1,  r2

        add     r1,  r1,  r0
        bic     r0,  r0,  r3
        sub     r1,  r1,  #1
        bic     r1,  r1,  r3
invalid_data_cache:
        mcr     p15,  0,  r0,  c7,  c6,  1  @ Invalidate data cache line to Point-of-Coherency by MVA
        add     r0,  r0,  r2
        cmp     r0,  r1
        bls     invalid_data_cache
        dsb
        bx      lr


/**
  * @brief   Invalidate instruction cache
  *
  *     void _kernel_invalid_inst_cache(void *, SIZE);
  *
  */

        .globl _kernel_invalid_inst_cache
        .type  _kernel_invalid_inst_cache, %function

_kernel_invalid_inst_cache:
        teq     r1,  #0
        mcreq   p15,  0,  r1,  c7,  c5,  0  @ Invalidate All Instruction Caches
        bxeq    lr

        mrc     p15,  1,  r3,  c0,  c0,  0
        and     r3,  r3,  #0x07
        mov     r2,  #16
        cmp     r3,  #2
        moveq   r2,  #64
        cmp     r3,  #1
        moveq   r2,  #32
        add     r3,  r2,  #-1

        add     r1,  r1,  r0
        bic     r0,  r0,  r3
        sub     r1,  r1,  #1
        bic     r1,  r1,  r3
invalid_inst_cache:
        mcr     p15,  0,  r0,  c7,  c5,  1  @ Invalidate Instruction Cache Line to Point-of-Unification by MVA
        add     r0,  r0,  r2
        cmp     r0,  r1
        bls     invalid_inst_cache
        dsb
        isb
        bx      lr


/**
  * @brief   Data Synchronization Barrier
  *
  *        void _kernel_synch_cache(void);
  *
  */

        .globl _kernel_synch_cache
        .type  _kernel_synch_cache, %function

_kernel_synch_cache:
        dsb
        bx      lr


/**
  * @brief   Data Memory Barrier
  *
  * void _memory_barrier(void);
  *
  */

        .globl _memory_barrier
        .type  _memory_barrier, %function

_memory_barrier:
        dmb
        bx      lr


/**
 * @brief   Wait for interrupt
 *
 * void _kernel_wait_for_interrupt(void);
 *
 */

        .globl  _kernel_wait_for_interrupt
        .type   _kernel_wait_for_interrupt, %function

_kernel_wait_for_interrupt:
        mov     r0,  #0
        mcr     p15,  0,  r0,  c7, c0,   4  @ Wait For Interrupt (NOP)
        bx      lr


/**
  * @brief   Cache Invalidation code for Cortex-R
  *
  *    void _kernel_invalid_cache(void);
  *
  */

        .globl  _kernel_invalid_cache
        .type   _kernel_invalid_cache, %function

_kernel_invalid_cache:

        @ Invalidate L1 Instruction Cache
        mrc     p15,  1,  r0,  c0,  c0,  1  @ Read Current Cache Level ID Register
        tst     r0,  #0x3
        mov     r0,  #0
        mcrne   p15,  0,  r0,  c7,  c5,  0  @ If cache is implemented, invalidate all instruction caches

        @ Invalidate Data/Unified Caches
        mrc     p15,  1,  r0,  c0,  c0,  1  @ Read Current Cache Level ID Register
        ands    r3,  r0,  #0x7000000
        mov     r3,  r3,  LSR #23
        beq     finished                    @ If neither instruction nor data cache is implemented.

        mov     r10, #0
loop_Lx:
        add     r2,  r10, r10, LSR #1
        mov     r1,  r0,  LSR r2
        and     r1,  r1,  #7
        cmp     r1,  #2
        blt     skip

        mcr     p15,  2, r10,  c0,  c0,  0  @ Write Cache Size Selection Register
        isb
        mrc     p15,  1, r1,  c0,  c0,  0   @ Read Current Cache Size Identification Register
        and     r2,  r1,  #0x7
        add     r2,  r2,  #4
        movw    r4,  #0x3FF
        ands    r4,  r4,  r1,  LSR #3
        clz     r5,  r4
        movw    r7,  #0x00007FFF
        ands    r7,  r7,  r1,  LSR #13

loop_way:
        mov     r9, r4

loop_set:
        orr     r11, r10, r9,  LSL r5
        orr     r11, r11, r7,  LSL r2
        mcr     p15,  0, r11,  c7,  c6,  2  @ Invalidate data cache line by physical address
        subs    r9,  r9,  #1
        bge     loop_set
        subs    r7,  r7,  #1
        bge     loop_way
skip:
        add     r10, r10, #2
        cmp     r3, r10
        bgt     loop_Lx

finished:
        dsb
        bx      lr


