/**
 * @file    DDR_RZT1_CPG.c
 * @brief   RZ/T1 Clock Pulse Generator unit (Cortex-R core)
 * @date    2015.04.30
 * @author  Copyright (c) 2014-2016, eForce Co.,Ltd.  All rights reserved.
 *
 * @par     History
 *          - rev 1.0 (2014.10.03) i-cho
 *            Initial version. 
 *          - rev 1.1 (2015.04.30) i-cho
 *            modified _ddr_rzt1_cpg_chg_cpuclk function argument.
 *            added ETCLKD and ETCLKE changing function.
 *          - rev 1.2 (2016.02.22) yokota
 *            modified for GNUC
 */

#include "uC3sys.h"
#include "RZT1_UC3.h"
#include "DDR_RZT1_CPG.h"

/* User config header */
#include "DDR_RZT1_CPG_cfg.h"

#if 0
/***********  DDR_RZT1_CPG_cfg.h ̒`e  **************************/

/* NbN\[X̎g */
#define MAIN_CLK    25000000u
#define AUDIO_CLK   11289600u

/***********  DDR_RZT1_CPG_cfg.h ̒`e͂܂  ****************/
#endif


/**
  * Set CPU clock and LOCO clock
  */
ER _ddr_rzt1_cpg_chg_cpuclk(UINT cpucksel, UINT ckio)
{
    ER ercd = E_OK;
    volatile UW dummy;
    volatile UW loop;
      
    if (cpucksel <= CPG_CPUCKSEL_600_MHZ && ckio <= CPG_CKIO_18_75_MHZ) {
    
        SYSTEM.PRCR.LONG = PRCR_CPG_UNLOCK;
        dummy = SYSTEM.PRCR.LONG;

        /* Enables LOCO clock operation */
        SYSTEM.LOCOCR.BIT.LCSTP = 0x00000000;

        /* Set CPUCLK to xxxx, and dummy read at three times */
        SYSTEM.PLL1CR.LONG = cpucksel;
        dummy = SYSTEM.PLL1CR.LONG;
        dummy = SYSTEM.PLL1CR.LONG;
        dummy = SYSTEM.PLL1CR.LONG;

        /* Enables PLL1, and wait about 100us for PLL1 (and LOCO)*/
        SYSTEM.PLL1CR2.LONG = 1;     
        for (loop = 0; loop < 20000 ; loop++) {
#ifdef __ICCARM__
            asm("nop");
#endif
#ifdef __CC_ARM
            __asm("nop");
#endif
#ifdef __GNUC__
            asm volatile("nop");
#endif
        }
        /* This loop will be modified to The sequence by using CMT timer */

        /* Selects the PLL1 as clock source */
        SYSTEM.SCKCR2.LONG = 1;

        /* Set BSC CKIO clock */
        SYSTEM.SCKCR.BIT.CKIO = ckio;

        SYSTEM.PRCR.LONG = PRCR_CPG_LOCK;
        dummy = SYSTEM.PRCR.LONG;

        ercd = E_OK;
    } else {
        ercd = E_PAR;
    }

    return ercd;
}


/**
 * Set ETCLKD(Ether MAC) and ETCLKE(Ether-PHY)
 * @param[in] ETCLKD
 * @param[in] ETCLKE
 */
ER _ddr_rzt1_cpg_chg_etclk(UINT d, UINT e)
{
    ER ercd = E_OK;
    volatile UW dummy;

    if (d <=3 && e <= 1) {
        SYSTEM.PRCR.LONG = PRCR_CPG_UNLOCK;
        dummy = SYSTEM.PRCR.LONG;
        SYSTEM.SCKCR.BIT.ETCKD = d;
        SYSTEM.SCKCR.BIT.ETCKE = e;
        SYSTEM.PRCR.LONG = PRCR_CPG_LOCK;
        dummy = SYSTEM.PRCR.LONG;
    } else {
        ercd = E_PAR;
    }

    return ercd;
}


/**
  * Get high speed serial clock.
  * @return SERICLK
  */
UW _ddr_rzt1_cpg_get_sericlk(void)
{
    UW clk;
    
    if (SYSTEM.SCKCR.BIT.SERICK != CPG_SERICK_150_MHZ) {
        /* 120MHz */
        clk = 120000000;
    } else {
        /* 150MHz */
        clk = 150000000;
    }
    
    return clk;
}


/**
  * Get external bus clock
  * @param[in] CPUCKSEL
  */
UW _ddr_rzt1_cpg_get_ckio(void)
{
    UW clk;
    UB ckio;
    
    ckio = SYSTEM.SCKCR.BIT.CKIO;
    switch (ckio) {
    case CPG_CKIO_75_MHZ:
        clk = 75000000;
        break;
    case CPG_CKIO_50_MHZ:
        clk = 50000000;
        break;
    case CPG_CKIO_37_5_MHZ:
        clk = 37500000;
        break;
    case CPG_CKIO_30_MHZ:
        clk = 30000000;
        break;
    case CPG_CKIO_25_MHZ:
        clk = 25000000;
        break;
    case CPG_CKIO_21_43_MHZ:
        clk = 21430000;
        break;
    case CPG_CKIO_18_75_MHZ:
        clk = 18750000;
        break;
    default:
        clk = 0;
        break;
    }
    
    return clk;
}

