/**
 * @file    DDR_RZT1_CMT0.c
 * @brief   RZ/T1 Interval Timer
 * @date    2015.04.27
 * @author  Copyright (c) 2014-2015, eForce Co.,Ltd.  All rights reserved.
 *
 * @par     History
 *          - rev 1.0 (2014.10.03) i-cho
 *            Initial version. 
 *          - rev 1.1 (2015.04.27) i-cho
 *            Supported CMT Uint1 ch1.
 */
 
#include "uC3sys.h"
#include "RZT1_UC3.h"
#include "DDR_RZT1_CPG.h"
#include "DDR_RZT1_VIC.h"

/* User config header */
#include "DDR_RZT1_CMT0_cfg.h"

#if 0
/***********  DDR_RZT1_CMT0_cfg.h ̒`e  ***********************

// Select CMT clock source
#define CLK_SRC    1

// Interrupt priority of timer (0`15)
#define IPL_TIMER 15

// CMT channel (0`3)
#define CH_TIM  0

 ***********  DDR_RZT1_CMT0_cfg.h ̒`e͂܂  *************/
#endif

#ifndef CH_TIM
#define CH_TIM  0
#endif

#ifndef IPL_TIMER
#define IPL_TIMER   15
#endif

/* Peripheral module clock(Hz) It's fixed value */
#define PCLKD   75000000u  


/**
 * Select CMT clock source
 */
#if (CLK_SRC==0)
    #define CKSbit        0 /* PCLKD/8 */
#elif (CLK_SRC==1)
    #define CKSbit        1 /* PCLKD/32 */
#elif (CLK_SRC==2)
    #define CKSbit        2 /* PCLKD/128 */
#elif (CLK_SRC==3)
    #define CKSbit        3 /* PCLKD/512 */
#else
    /* default! */
    #define CKSbit        0 /* PCLKD/8 */
#endif

/* CMT Uint0 ch0 */
#if CH_TIM==0
    #define INT_CMIn        INT_CMT_CMI0
    #define REG_VIC_PICn    VIC.PIC0.BIT.PIC21
    #define REG_VIC_RAISn   VIC.RAIS0.BIT.RAI21
    #define STRn            CMT.CMSTR0.BIT.STR0
    #define CKSn            CMT0.CMCR.BIT.CKS
    #define CMIEn           CMT0.CMCR.BIT.CMIE
    #define CMCNTn          CMT0.CMCNT
    #define CMCORn          CMT0.CMCOR
    #define MSTPCRn         SYSTEM.MSTPCRA.LONG 
    #define MSTPCRn_bit     (0x00000010)
/* CMT Uint0 ch1 */
#elif CH_TIM==1
    #define INT_CMIn        INT_CMT_CMI1
    #define REG_VIC_PICn    VIC.PIC0.BIT.PIC22
    #define REG_VIC_RAISn   VIC.RAIS0.BIT.RAI22
    #define STRn            CMT.CMSTR0.BIT.STR1
    #define CKSn            CMT1.CMCR.BIT.CKS
    #define CMIEn           CMT1.CMCR.BIT.CMIE
    #define CMCNTn          CMT1.CMCNT
    #define CMCORn          CMT1.CMCOR
    #define MSTPCRn         SYSTEM.MSTPCRA.LONG 
    #define MSTPCRn_bit     (0x00000010)
/* CMT Uint1 ch0 */
#elif CH_TIM==2
    #define INT_CMIn        INT_CMT_CMI2
    #define REG_VIC_PICn    VIC.PIC0.BIT.PIC23
    #define REG_VIC_RAISn   VIC.RAIS0.BIT.RAI23
    #define STRn            CMT.CMSTR1.BIT.STR2
    #define CKSn            CMT2.CMCR.BIT.CKS
    #define CMIEn           CMT2.CMCR.BIT.CMIE
    #define CMCNTn          CMT2.CMCNT
    #define CMCORn          CMT2.CMCOR
    #define MSTPCRn         SYSTEM.MSTPCRA.LONG 
    #define MSTPCRn_bit     (0x00000008)
/* CMT Uint1 ch1 */
#elif CH_TIM==3
    #define INT_CMIn        INT_CMT_CMI3
    #define REG_VIC_PICn    VIC.PIC0.BIT.PIC24
    #define REG_VIC_RAISn   VIC.RAIS0.BIT.RAI24
    #define STRn            CMT.CMSTR1.BIT.STR3
    #define CKSn            CMT3.CMCR.BIT.CKS
    #define CMIEn           CMT3.CMCR.BIT.CMIE
    #define CMCNTn          CMT3.CMCNT
    #define CMCORn          CMT3.CMCOR
    #define MSTPCRn         SYSTEM.MSTPCRA.LONG 
    #define MSTPCRn_bit     (0x00000008)
#if 0
/* CMT Uint2 ch0 */
#elif CH_TIM==4
    #define INT_CMIn        INT_CMT_CMI4
    #define REG_VIC_PICn    VIC.PIC9.BIT.PIC299;
    #define REG_VIC_RAISn   VIC.RAIS9.BIT.RAI299;
    #define STRn            CMT.CMSTR2.BIT.STR4
    #define CKSn            CMT4.CMCR.BIT.CKS
    #define CMIEn           CMT4.CMCR.BIT.CMIE
    #define CMCNTn          CMT4.CMCNT
    #define CMCORn          CMT4.CMCOR
    #define MSTPCRn         SYSTEM.MSTPCRA.LONG 
    #define MSTPCRn_bit     (0x00000004)
/* CMT Uint2 ch1 */
#elif CH_TIM==5
    #define INT_CMIn        INT_CMT_CMI5
    #define REG_VIC_PICn    VIC.PIC9.BIT.PIC300;
    #define REG_VIC_RAISn   VIC.RAIS9.BIT.RAI300;
    #define STRn            CMT.CMSTR2.BIT.STR5
    #define CKSn            CMT5.CMCR.BIT.CKS
    #define CMIEn           CMT5.CMCR.BIT.CMIE
    #define CMCNTn          CMT5.CMCNT
    #define CMCORn          CMT5.CMCOR
    #define MSTPCRn         SYSTEM.MSTPCRA.LONG 
    #define MSTPCRn_bit     (0x00000004)
#endif
#else
  #error invalid chanel number!
#endif


/**
 * Private variable
 */
UW _ddr_rzt1_cmt0_systim;
UW _ddr_rzt1_cmt0_rate;
UW _ddr_rzt1_cmt0_tick = 0;
UH _ddr_rzt1_cmt0_save;


/*--------------- private function ------------------------------*/

/**
  * @brief Interval timer ISR
  */
void _ddr_rzt1_cmt0_isr(void)
{
    loc_cpu();

    /* Clear interrupt edge detection */
    REG_VIC_PICn = 1;

    _ddr_rzt1_cmt0_systim += _ddr_rzt1_cmt0_tick;

    unl_cpu();
    isig_tim();
}


/**
  * @brief Initialization routine
  */

T_CISR const _ddr_rzt1_cmt0_cisr = {TA_NULL, (VP_INT)0, INT_CMIn, _ddr_rzt1_cmt0_isr, IPL_TIMER};

void _ddr_rzt1_cmt0_init(UINT tick)
{
    UW tcnt;
    BOOL iFlag;
    volatile UW dummy32;

    if (acre_isr((T_CISR *)&_ddr_rzt1_cmt0_cisr) >= E_OK) {

        iFlag = sns_loc();
        if (iFlag != TRUE)
            loc_cpu();

        /* 1msec interval operation */
        if (CKSbit == 0) {
            tcnt = ((PCLKD / 8000) * tick);   /* PCLKD/8 */
        } else if (CKSbit == 1) {
            tcnt = ((PCLKD / 32000) * tick);  /* PCLKD/32 */
        } else if (CKSbit == 2) {
            tcnt = ((PCLKD / 128000) * tick); /* PCLKD/128 */
        } else if (CKSbit == 3) {
            tcnt = ((PCLKD / 512000) * tick); /* PCLKD/512 */
        } else {
            ;
        }
        _ddr_rzt1_cmt0_tick = (tick * 1000);
        _ddr_rzt1_cmt0_rate = tcnt-1;
        _ddr_rzt1_cmt0_systim = 0;
        
        /* Clock Supply */
        SYSTEM.PRCR.LONG = PRCR_MSTP_UNLOCK;    /* Unlock Register Write Protection */
        dummy32 = SYSTEM.PRCR.LONG;

        MSTPCRn &= ~(MSTPCRn_bit);
        dummy32 = MSTPCRn;

        SYSTEM.PRCR.LONG = PRCR_MSTP_LOCK;  /* Lock Register Write Protection */
        dummy32 = SYSTEM.PRCR.LONG;
        
        /* Set CMT register */
        CKSn = CKSbit;      /* Count clock */
        CMCNTn = 0x0000;    /* Clear CMCNT counter */
        CMCORn = tcnt - 1;
        CMIEn = 1;          /* Enable compare match interrupt */

        set_type_int(INT_CMIn, IRQ_TYPE_EDGE);
        ena_int(INT_CMIn);
        
        STRn = 1;   /* Start count */

        if (iFlag != TRUE)
            unl_cpu();
    }
}


/**
  * @brief Get system micro time
  */
UW _kernel_micro_systim(void)
{
    UW clk = 0;
    UW tcn;

    if (_ddr_rzt1_cmt0_tick != 0) {
        if (REG_VIC_RAISn != 0) {
            clk = _ddr_rzt1_cmt0_systim + _ddr_rzt1_cmt0_tick;
            tcn = CMCNTn;
        } else {
            clk = _ddr_rzt1_cmt0_systim;
            tcn = CMCNTn;
            if (REG_VIC_RAISn != 0) {
                clk += _ddr_rzt1_cmt0_tick;
                tcn = CMCNTn;
            }
        }
        clk += ((tcn * _ddr_rzt1_cmt0_tick) / (_ddr_rzt1_cmt0_rate+1));
    }
    return clk;
}


/**
  * @brief Suspend system timer
  */
void _kernel_micro_systim_stop(void)
{
    uint16_t wstop;

    wstop = CMCNTn;
    STRn = 0;
    CMCNTn = wstop;
}


/**
  * @brief Resume system timer
  */
void _kernel_micro_systim_start(void)
{
    STRn = 1;
}
