/**
  * @file    DDR_RZT1_VIC.c
  * @brief   RZ/T1 Vectored Interrupt Controller (Cortex-R core)
  * @date    2014.10.03
  * @author  Copyright (c) 2014-2016, eForce Co.,Ltd.  All rights reserved.
  *
  * @par     History
  *          - rev 1.0 (2014.10.03) i-cho
  *            Initial version.
  *          - rev 1.1 (2015.07.02) yokota
  *            add clr_int. 
  *          - rev 1.2 (2016.02.26) i-cho
  *            Fixed. Handler address of interrupt number 256 or more is not registered.
  */
#include "uC3sys.h"
#include "RZT1_UC3.h"

extern void _kernel_set_cp15ctrl(UW cp15);
extern UW _kernel_get_cp15ctrl(void);

#define VIC_SIZE    295
#define VEC_NUM     (VIC_SIZE-1)    /* vector number 1 ~ 294 */

/**
  * Relationship of priority level with the vector number
  * Vector number        Priority level
  *---------------------------------------
  * 1   ~ 255              PRLn
  * 256 ~ 294, 300         PRLn + 16
  *---------------------------------------
  * PRLn 0(high) ~ 15(low)
  *
  */
#define PRL_MAX     15      /* priority level */


/**
  * Vector tables
  */
#ifdef __ICCARM__
#pragma section = "VINFTBL"
int _kernel_dummy_2 @ "VINFTBL";
#pragma section = "VECTTBL"
T_CORTEXR_DEFINT vecttbl[VIC_SIZE] @ "VECTTBL";
#endif

#ifdef __CC_ARM
int _kernel_dummy_2 __attribute__ ((section ("VINFTBL"), zero_init));
T_CORTEXR_DEFINT vecttbl[VIC_SIZE] __attribute__ ((section ("VECTTBL"), zero_init));
#endif

#ifdef __GNUC__
int _kernel_dummy_2 __attribute__ ((section(".vinftbl")));
T_CORTEXR_DEFINT vecttbl[VIC_SIZE] __attribute__ ((section (".vecttbl")));
#endif


void _ddr_rzt1_vic_init_vad(INTNO intno)
{
    volatile uint32_t *prl_base;
    volatile uint32_t *vad_base;
    uint32_t reg_offset = 0;

    reg_offset = intno;
    if (intno <= 255) { /* Vector number : 1 to 255 */
        prl_base = (uint32_t *)&(VIC.PRL1.LONG);
        vad_base = (uint32_t *)&(VIC.VAD1.LONG);
        
    } else {    /* Vector number : 256 to 294 */
        prl_base = (uint32_t *)&(VIC.PRL256.LONG);
        vad_base = (uint32_t *)&(VIC.VAD256.LONG);

        reg_offset -= 255;   /* Offset (PRLn and VADn base is changed (eg. VAD1 to VAD256) */
    }
    
    /* Set interrupt priority level (0 to 15) or (16 to 31) */
    prl_base += (reg_offset - 1);
    *prl_base = PRL_MAX;

    /* Set interrupt service routine address */
    vad_base += (reg_offset - 1);
    *vad_base = (UW)&vecttbl[intno];  /* handler address */
}

/**
  * @brief Interrupt controller initialization routine
  */
void _ddr_rzt1_vic_init(void)
{
    INTNO intno;

    _kernel_dummy_2 = 0;

    /* Mask all interrupt */
    VIC.IEC0.LONG = 0xFFFFFFFF;
    VIC.IEC1.LONG = 0xFFFFFFFF;
    VIC.IEC2.LONG = 0xFFFFFFFF;
    VIC.IEC3.LONG = 0xFFFFFFFF;
    VIC.IEC4.LONG = 0xFFFFFFFF;
    VIC.IEC5.LONG = 0xFFFFFFFF;
    VIC.IEC6.LONG = 0xFFFFFFFF;
    VIC.IEC7.LONG = 0xFFFFFFFF;
    VIC.IEC8.LONG = 0xFFFFFFFF;
    VIC.IEC9.LONG = 0xFFFFFFFF;
    /* Clear egde detection */
    VIC.PIC0.LONG = 0xFFFFFFFF;
    VIC.PIC1.LONG = 0xFFFFFFFF;
    VIC.PIC2.LONG = 0xFFFFFFFF;
    VIC.PIC3.LONG = 0xFFFFFFFF;
    VIC.PIC4.LONG = 0xFFFFFFFF;
    VIC.PIC5.LONG = 0xFFFFFFFF;
    VIC.PIC6.LONG = 0xFFFFFFFF;
    VIC.PIC7.LONG = 0xFFFFFFFF;
    VIC.PIC8.LONG = 0xFFFFFFFF;
    VIC.PIC9.LONG = 0xFFFFFFFF;
    /* Clear priority level mask (edge type only) */
    VIC.PRLC0.LONG = 0x0000FFFF;
    VIC.PRLC1.LONG = 0x0000FFFF;
    
    /* set all the vector and vector priority register to default */
    /* NOTE: vector number start from 1 !!! */
    for (intno = 1; intno < VIC_SIZE; intno++ ) {
        _ddr_rzt1_vic_init_vad(intno);
    }
    _kernel_synch_cache();

    /* Set VE-bit of CP15 control register */
    _kernel_set_cp15ctrl(_kernel_get_cp15ctrl() | 0x01000000);
}

/**
  * @brief  Set external interrupt pin.
  * @param[in] pin ........ External interrupt pin number (0 ~  15)
  * @param[in] detect ..... Interrupt pin detection sense
  * @param[in] clksel ..... sampling clock
  */
ER set_irqpin_int(int pin, int detect, int clksel)
{
    volatile UW *irqcr_base;

    if (pin > 15) {
        return E_PAR;
    }
    
    _kernel_lock();
    
    irqcr_base = (void *)(&(ICU.IRQCR0.LONG));    
    
    /* Disable digital noise filter (Clear IRQFLTEn bit (n = pin_num))*/
    ICU.IRQFLTE.LONG &= (0x0000FFFF & ~(1 << pin));

    /* Set IRQ detection sense */
    irqcr_base += pin;      /* Specify IRQCRn register address */
    *irqcr_base = detect;   /* Set IRQCRn.IRQMD[1:0]           */

    /* Set digital noise filter and enable */
    if (IRQ_FLTC_CLK_NONE != clksel) {
        /* Set digital noise filter */
        ICU.IRQFLTC.LONG &= ~(3 << (pin * 2));      /* Clear FCLKSELn[1:0] */
        ICU.IRQFLTC.LONG |= (clksel << (pin * 2));  /* Set FCLKSELn[1:0] to dnf_set value */

        /* Enable digital noise filter */
        ICU.IRQFLTE.LONG |= (1 << pin);
    }

    _kernel_synch_cache();
    _kernel_unlock();

    return E_OK;
}

/**
  * @brief  Set interrupt detection type.
  * @param  intno ........ Interrupt line numbe
  * @param  edge_mode .... Trigger mode (0: Level, 1: Edge)
  * @return E_OK ......... Trigger mode set
  *         E_PAR ........ Invalid interrupt number
  */
ER set_type_int(INTNO intno, int edge_mode)
{
    volatile uint32_t *pls_base;
    uint32_t reg_num;
    uint32_t bit_num;
    
    if (intno >= VIC_SIZE) {
        return E_PAR;
    }

    _kernel_lock();
    
    if (intno <= 255) { /* Vector number : 1 to 255 */
        pls_base = (uint32_t *)&(VIC.PLS0.LONG);
        reg_num = (intno / 32);    /* PLSn, PICn (n = reg_num) */
        bit_num = (intno % 32);    /* PLSn.PLSm, PICn.PICm (m = bit_num) */
    
    } else {    /* Vector number : 256 to 294 */
        pls_base = (uint32_t *)&(VIC.PLS8.LONG);
        reg_num  = ((intno / 32) - 8);    /* PLSn, PICn (n = 8 + reg_num) */
        bit_num  = (intno % 32);          /* PLSn.PLSm, PICn.PICm (m = bit_num) */
    }
    
    pls_base += reg_num;
    
    if (edge_mode != IRQ_TYPE_LEVEL) {
        *pls_base |= (1 << bit_num);
    } else {
        *pls_base &= ~(1 << bit_num);
    }
    
    _kernel_synch_cache();
    _kernel_unlock();
    
    return E_OK;
}

/**
  * @brief  Enable specified interrupt event generation
  * @param  INTNO intno ........ Interrupt line number
  * @return E_OK ............... Interrupt line activated
  *         E_PAR .............. Invalid interrupt number / priority
  */
ER ena_int(INTNO intno)
{
    ER ercd;
    volatile uint32_t *prl_base;
    volatile uint32_t *ien_base;
    uint32_t reg_num;
    uint32_t bit_num;
    uint32_t reg_offset = 0;

    reg_offset = intno;
    if (intno <= VEC_NUM) {
        _kernel_lock();

        if (intno <= 255) { /* Vector number : 1 to 255 */
            prl_base = (uint32_t *)&(VIC.PRL1.LONG);
            ien_base = (uint32_t *)&(VIC.IEN0.LONG);

            reg_num = intno / 32;    /* IENn (n = reg_num) */
            bit_num = intno % 32;    /* IENn.IENm (m = bit_num) */

        } else {    /* Vector number : 256 to 294 */

            prl_base = (uint32_t *)&(VIC.PRL256.LONG);
            ien_base = (uint32_t *)&(VIC.IEN8.LONG);

            reg_num = ((intno / 32) - 8);   /* IENn (n = 8 + reg_num)  */
            bit_num = (intno % 32);         /* IENn.IENm (m = bit_num) */
            reg_offset -= 255;              /* Offset (PRLn and VADn base is changed (eg. VAD1 to VAD256) */
        }
        /* Set interrupt priority level (0 to 15) or (16 to 31) */
        prl_base += (reg_offset - 1);
        *prl_base = (vecttbl[intno].vinftbl.intinfo & 0x0F);

        /* Set interrupt enable register (enable interrupt) */
        ien_base += reg_num;
        *ien_base |= (0x00000001 << bit_num);
        
        _kernel_synch_cache();
        _kernel_unlock();
        ercd = E_OK;
    } else {
        ercd = E_PAR;
    }
    return ercd;
}


/**
  * @brief  Disable specified interrupt event generation
  * @param  INTNO intno ........ Interrupt line number
  * @return E_OK ............... Interrupt line disabled
  *         E_PAR .............. Invalid interrupt number
  */
ER dis_int(INTNO intno)
{
    ER ercd;
    volatile uint32_t *iec_base;
    uint32_t reg_num;
    uint32_t bit_num;
    
    if (intno <= VEC_NUM) {
        _kernel_lock();
        
        if (intno <= 255) { /* Vector number : 1 to 255 */
            iec_base = (uint32_t *)&(VIC.IEC0.LONG);
            reg_num = (intno / 32);    /* IECn (n = reg_num) */
            bit_num = (intno % 32);    /* IECn.IECm (m = bit_num) */
            
        } else {    /* Vector number : 256 to 294 */
            
            iec_base = (uint32_t *)&(VIC.IEC8.LONG);
            reg_num = ((intno / 32) - 8);    /* IECn (n = 8 + reg_num)  */
            bit_num = (intno % 32);          /* IECn.IECm (m = bit_num) */
        }

        iec_base += reg_num;
        *iec_base |= (0x00000001 << bit_num);

        _kernel_synch_cache();
        _kernel_unlock();
        ercd = E_OK;
        
    } else {
        ercd = E_PAR;
    }
    return ercd;
}

/**
  * @brief Pre-Handler routine for interrupt event
  * @return always TURE. If TRUE, allow nested interrupts.
  */
BOOL _kernel_pre_inthdr(T_INTPARA *intpara)
{
    return TRUE;
}

/**
  * @brief Post-Handler routine for interrupt event
  */
void _kernel_post_inthdr(UW savedt)
{
    /* End interrupt sequence */
    VIC.HVA0.LONG = 0;
}

/**
  * @brief  clear edge detection interrupt 
  * @param  INTNO intno ........ Interrupt line number
  * @return E_OK ............... clear interrupt
  *         E_PAR .............. Invalid interrupt number
  */
ER clr_int(INTNO intno)
{
    volatile uint32_t *pic_base;
    uint32_t reg_num;
    uint32_t bit_num;

    if (intno >= VIC_SIZE) {
        return E_PAR;
    }

    _kernel_lock();
    
    if (intno <= 255) { /* Vector number : 1 to 255 */
        pic_base = (uint32_t *)&(VIC.PIC0.LONG);
        reg_num = (intno / 32);    /* PLSn, PICn (n = reg_num) */
        bit_num = (intno % 32);    /* PLSn.PLSm, PICn.PICm (m = bit_num) */
    
    } else {    /* Vector number : 256 to 294 */
        pic_base = (uint32_t *)&(VIC.PIC8.LONG);
        reg_num  = ((intno / 32) - 8);    /* PLSn, PICn (n = 8 + reg_num) */
        bit_num  = (intno % 32);          /* PLSn.PLSm, PICn.PICm (m = bit_num) */
    }

    pic_base += reg_num;
    *pic_base |= (1 << bit_num);

    _kernel_synch_cache();
    _kernel_unlock();

    return E_OK;
}
